//   

/* output::tableForm --- printing a set or list in table form

tableForm(CONT <, seperator> <, OPTIONS >)

CONT      : set or list
seperator : (optional) string
OPTIONS   : (optional) one or a set of options

tableForm(CONT) prints the objects of CONT in table form.
The width of the table depends on the size of TEXTWIDTH.

If seperator is given then it is appended to each object.
Appending spaces to the separator results additionally space
between columns.

OPTIONS is one or a set of

Unquoted            - strings will be printed without quotes
Unique              - all columns are of the same width
Left, Center, Right - the entries will be aligned left, center or right
Width = w           - the table has the max width 'w'
Columns = c          - the table has 'c' columns
Append = char       - append 'char' to each entry
Sort = procedure    - the entries will be sorted with the given procedure
                      'procedure' is no procedure => no sorting
Output = file       - output into a file
                      if 'file' is a string => file named 'file' will be
                      opened and overwritten and closed after writing
                      if 'file' is a file descriptor the table will be
                      appended to 'file' without closing 'file'
Format = 

Without "Sort" the objects will be converted to strings an then sorted
alphabetically.


*/


output::tableForm:=
  proc(CONT = [], SEP = FAIL, OPT = FAIL)
    local COLW, COLWM, COLN, k, PE, FILE, FC, i, APP, SEPlength, firstOpt,
          allOptions, optionTypes;
  begin
    allOptions := table(Unquoted = FALSE,
                        Unique   = FALSE,
                        Left     = TRUE,
                        Center   = FALSE,
                        Right    = FALSE,
                        Width    = -1,
                        Columns  = -1,
                        Output   = FAIL,
                        Append   = "",
                        Sort     = ((X, Y) -> expr2text(X) < expr2text(Y)),
                        String   = FALSE):
    optionTypes := table(Width   = Type::PosInt,
                         Columns = Type::PosInt,
                         Output  = Type::Union(DOM_STRING, Type::NonNegInt),
                         Append  = DOM_STRING,
                         Sort    = Type::Union(DOM_PROC, DOM_FUNC_ENV, Type::Boolean)
                        ):
    
    if args(0) > 2 and testtype(SEP, DOM_TABLE) then
      firstOpt := 2;
    elif args(0) > 1 and testtype(SEP, DOM_STRING) or testtype(SEP, DOM_LIST) then
      firstOpt := 3;
    else
      firstOpt := 2;
    end_if;

    if domtype(OPT) <> DOM_TABLE then
      OPT := prog::getOptions(firstOpt, [args()], allOptions, TRUE, optionTypes)[1];
    end_if;

    if contains([DOM_LIST, DOM_SET, DOM_TABLE, DOM_DOMAIN], type(CONT)) = 0 then
      error("wrong type of first argument")
    end_if;
    if args(0) > 1 then
      if type(SEP) = DOM_STRING or type(SEP) = DOM_LIST then
        if nops(OPT) < args(0) - 2 then // too few arguments
          error("wrong type of additional arguments");
        end_if;
      else // no separator given, but additional arguments
        if nops(OPT) < args(0) - 1 then // too few arguments
          error("wrong type of additional arguments")
        end_if          
      end_if;
    end_if;

    if type(SEP) <> DOM_STRING and type(SEP) <> DOM_LIST then // OPTIONS without SEPARATOR
      SEP:= " ";
      SEPlength:= 1
    else
      OPT[Separator]:= SEP; // separator set explicitly
      if type(SEP) = DOM_LIST then
        SEPlength:= max(op(map(SEP, length)))
      else
        SEPlength:= length(SEP)
      end_if
    end_if;
       
    APP:= OPT[Append];
    
    // file
    FC:= FALSE;
    FILE:=
      (if domtype(OPT[Output]) = DOM_STRING then
         if (k:= fopen(Text, OPT[Output], Write)) <> FAIL then
           FC:= TRUE;
           k
         else
           error("cannot open file '".OPT[Output]."'")
         end_if
       elif domtype(OPT[Output]) = DOM_INT then
         if traperror(fprint(Unquoted, OPT[Output], "")) = 0 then
           OPT[Output]
         else
           error("cannot write into given file")
         end_if
       else
         0
       end_if);

    // different input objetcs
    if type(CONT) <> DOM_LIST then
      if type(CONT) = DOM_SET then
        CONT:= [op(CONT)]
      elif type(CONT) = DOM_TABLE or type(CONT) = DOM_DOMAIN then
        if OPT[Columns] = -1 then
          OPT[Columns]:= 2
        end_if;
        if not contains(OPT, Separator) then
          SEP:= " = "
        end_if;
        CONT:= map(sort([op(CONT)], OPT[Sort]), op);
        OPT[Sort]:= FALSE; // no further sorting!
        return(output::tableForm(CONT, SEP, OPT))
//       elif type(CONT) = piecewise then
//         if not contains(OPT, hold(Columns)) then
//           OPT[hold(Columns)]:= 2
//         end_if;
//         if not contains(OPT, Separator) then
//           SEP:= " if "
//         end_if;
//         output::tableForm(map([op(CONT)], X -> stdlib::branch::expression(X),
//                                                stdlib::branch::condition(X)),
//                           SEP, OPT)
      end_if
    else
      // DOM_LIST
    end_if;

    // sort list
    // assert(domtype(CONT, DOM_LIST));
    if OPT[Sort] <> FALSE then
      CONT:= sort(CONT, OPT[Sort])
    end_if;
    
    // container empty
    if nops(CONT) = 0 then
      return()
    end_if;

    CONT:= map(CONT, X -> if OPT[Unquoted] = TRUE and type(X) = DOM_STRING then
                            X
                          elif domtype(X) = DOM_PROC then
                            DOM_PROC::print(X)
                          else
                            expr2text(X)
                          end_if);

    // col width
    // determine number of columns
    COLWM:= 0;
    for k in CONT do
      if length(k) > COLWM then
        COLWM:= length(k)
      end_if
    end_for;
    COLWM:= COLWM + length(APP) + SEPlength;

    if COLWM < 1 then
      warning("the table is too wide");
      COLWM:= 1
    end_if;
    
    //
    if OPT[Columns] <> -1 then
      COLN:= OPT[Columns]
    elif OPT[Width] <> -1 then
      COLN:= OPT[Width] div COLWM
    else
      COLN:= (TEXTWIDTH + length(APP) + SEPlength) div COLWM
    end_if;

    if COLN < 1 then
      warning("the table is too wide");
      COLN:= 1
    end_if;

    // create separator list
    if domtype(SEP) <> DOM_LIST then
      SEP:= [""].[SEP $ COLN]
    else
      SEP:= [""].SEP.(SEP $ (COLN - nops(SEP) + 1) div nops(SEP)); // fill up too short list
    end_if;
    
    // 
    COLW:= [COLWM $ COLN];
    if not OPT[Unique] then
      (COLW[i]:= max(op(map(zip(CONT, [k $k = 1..nops(CONT)],
                                (X, Y) -> if ((Y - 1) mod COLN) = (i - 1) then X else null() end),
                            length)), 1)
       + length(APP) + length(SEP[i])) $ i=1..COLN
    end_if;
    
    // fill up with empty entries
    if (k:= nops(CONT) - (nops(CONT) div COLN)*COLN) > 0 then
      (CONT:= append(CONT, "")) $ i = 1..COLN - k
    end_if;
    
    PE:=
      proc(X, Y, i)
        local app, sep;
      begin
        sep:= "";
        (if i > 1 and X <> "" then
           sep:= SEP[i]
         else
           ""
         end_if).
        (if X <> "" then
           if Y = "" or Y = FAIL then // end of table
             // FIX according to bug # 1745: The append symbol was
             // not appended to the last element of the output list. 
             // Before the fix the line of code was 
             //   >> app:= "" //(_concat(" " $ length(APP))) <<
             app:= APP."" //(_concat(" " $ length(APP)))
           else
             app:= APP
           end_if;
           if i = COLN and OPT[Left] then
             X.app
           else
             if OPT[Right] then
               (" " $ COLW[i] - length(app) - length(sep) - length(X)).X.app
             elif OPT[Center] then
               Y:= COLW[i] - length(app) - length(sep) - length(X);
                 // FIX according to bug # 1745: The 'elif i = 1'-branch has been 
                 // inserted. 
                 // 
               // >> start of bug fix <<
               if i = 1 then 
                 (" " $ Y div 2).X.app.(if i = COLN then "" else " " $ Y - (Y div 2) - 1 end_if)
               else 
               // >> end of bug fix <<
                 (" " $ Y div 2).X.app.(if i = COLN then "" else " " $ Y - (Y div 2) end_if)
               end_if;
             else // left
               if i = COLN then // end of row
                 X.app
                 // FIX according to bug # 1745: The 'elif i = 1'-branch has been 
                 // inserted. 
                 // 
               // >> start of bug fix <<
               elif i = 1 then 
                 X.app.(" " $ COLW[i] - length(app) - length(sep) - length(X) - 1)
               // >> end of bug fix <<
               else
                 X.app.(" " $ COLW[i] - length(app) - length(sep) - length(X))
               end_if
             end_if
           end_if
         else
           " " $ COLW[i]
         end_if)
      end_proc;

    // printing
    if OPT[String] then
      return(_concat((PE(op(CONT, (k - 1)*COLN + i), op(CONT, (k - 1)*COLN + i + 1), i)
                      $ i = 1..COLN, "\n") $ k = 1..nops(CONT) div COLN))
    else
      for k from 1 to nops(CONT) div COLN do
        fprint(Unquoted, FILE,
               PE(op(CONT, (k - 1)*COLN + i), op(CONT, (k - 1)*COLN + i + 1), i) $ i = 1..COLN);
      end_for
    end_if;
    
    if FC then fclose(FILE) end_if;
    
    null()
  end_proc:
