// 

/* -----------------------------------------------------------
 
 Reflect3d -- reflection about a plane or a point
 
 Syntax (cf. Plane):
 Reflect3d(point, normal, <options>, <objects>)
 Reflect3d(point, <options>, <objects>)
 
 point1, normal   : lists of 3 expressions
 
 plot(plot::Reflect3d([1, 1, 1], [2, a, 3], a=-2..2, plot::Function3d(sin@_plus)));
 plot(plot::Reflect3d([1, 1, 1], plot::Function3d(sin@_plus)));

----------------------------------------------------------- */

plot::createPlotDomain("Reflect3d",
                       "Reflection about a plane",
                       3,
                       [PositionX, PositionY, PositionZ, Position,
                        NormalX, NormalY, NormalZ, Normal,
                        Matrix3d, ShiftX, ShiftY, ShiftZ, Shift],
                       "Transform"):
  
plot::objectHierarchy["Reflect3d"] := plot::objectHierarchy["Rotate3d"]:
     
plot::Reflect3d::hints := {Scaling = Constrained}:
//-----------------------------------------------------------------
plot::Reflect3d::hiddenSlots({Matrix3d, Shift, ShiftX, ShiftY, ShiftZ}):
// rebuild InspectorEntry without hidden slots
plot::buildInspectorEntry("Reflect3d", 3,
			  [PositionX, PositionY, PositionZ, Position,
			   NormalX, NormalY, NormalZ, Normal], "Transform"):
//-----------------------------------------------------------------
// Konsistenz zu plot::Plane:
plot::Reflect3d::styleSheet:= table(
      PositionX = 0,
      PositionY = 0,
      PositionZ = 0,
      NormalX = 0,
      NormalY = 0,
      NormalZ = 1
):
//-----------------------------------------------------------------
			       
plot::Reflect3d::new :=
proc()
  local object, other, i;
begin
  // check all known options from the argument list
  object := dom::checkArgs([], args());
  
  // get arguments which are not yet processed
  other := object::other;
    
  // convert matrix cols to list of coordinates
  if nops(other) > 0 and (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
    if nops(other[1]) = 3 then
       // other[1] provides 'Position' but not 'Normal'
       other[1]:= [op(other[1])]
    else
       // other[1] provides 'Position' and 'Normal'
       other[1] := [op(linalg::col(other[1],i))] $ i=1..linalg::ncols(other[1])
    end_if;
  end_if;
  
  if nops(other) > 0 then
    if nops(other) = 1 then // reflection at a point
       object::Position:= other[1];
       object::Normal  := [0, 0, 0]:
    end_if;
    if nops(other) = 2 then // reflection about a plane
       object::Position:= other[1];
       object::Normal  := other[2];
    end_if;
  end_if;

  map(object::children,
      c -> if c::dom::dimension <> 3 then
	     context(hold(error)("only 3-dimensional plot objects allowed"))
	   end_if);
  
  // calculate internal values
  dom::changeNotifier(object, "NormalX"=object::NormalX);
  
  // semantically check for validity
  dom::checkObject(object);
end_proc:

plot::Reflect3d::print :=
obj -> hold(plot::Reflect3d)(obj::Position, obj::Normal, op(obj::children),
                             dom::printAttributes(obj, {Position, Normal})):

plot::Reflect3d::changeNotifier :=
proc(obj, eq)
    local slotName, newval, n, x0;
begin
  [slotName, newval] := [op(eq)];
  case slotName
    of "Position"  do
    of "PositionX" do
    of "PositionY" do
    of "PositionZ" do
    of "Normal"    do
    of "NormalX"   do
    of "NormalY"   do
    of "NormalZ"   do
      dom::extslot(obj, slotName, newval);
      // normal:
      n := obj::Normal;
      if iszero(n[1]) and iszero(n[2]) and iszero(n[3]) then
	dom::extslot(obj, "Matrix3d", [-1,  0,  0,
				        0, -1,  0,
				        0,  0, -1]);
	dom::extslot(obj, "ShiftX", 2*obj::PositionX);
	dom::extslot(obj, "ShiftY", 2*obj::PositionY);
	dom::extslot(obj, "ShiftZ", 2*obj::PositionZ);
      else
         n := map(n, `/`, sqrt(n[1]^2+n[2]^2+n[3]^2));
         dom::extslot(obj, "Matrix3d",
   		   eval([1-2*n[1]^2, -2*n[1]*n[2], -2*n[1]*n[3],
   			 -2*n[1]*n[2], 1-2*n[2]^2, -2*n[2]*n[3],
   			 -2*n[1]*n[3], -2*n[2]*n[3], 1-2*n[3]^2]));
         x0 := 2*n[1]*obj::PositionX + 
               2*n[2]*obj::PositionY + 
               2*n[3]*obj::PositionZ;
         dom::extslot(obj, "ShiftX", x0*n[1]);
         dom::extslot(obj, "ShiftY", x0*n[2]);
         dom::extslot(obj, "ShiftZ", x0*n[3]);
      end_if;
      return(FALSE);
  end_case;
  
  return(TRUE);
end_proc:
