//      

/* -----------------------------------------------------------
    Rotate3d -- a 3D rotation

    Syntax:
    Rotate3d(angle, center = [0, 0, 0], axis = [0, 0, 1]
             child1, child2, ...  <, op1, op2, ...>)

    angle        : angle of rotation (in radians)
    center       : first point defining the rotation axis
    axis         : second point defining the rotation axis
                   default axis is the z-axis defined 
                   by [0,0,0] and [0,0,1]               
    child1,      : plot objects (the objects to be rotated
    child2, ...
    op1, op2, ...: plot options of the form 'option = value'
---------------------------------------------------------------- */ 

plot::createPlotDomain("Rotate3d", "3D rotation"):

  
plot::Rotate3d::styleSheet := table(AxisX = 0,   AxisY = 0,   AxisZ = 1,
                                    CenterX = 0, CenterY = 0, CenterZ = 0):

plot::Rotate3d::hints := {Scaling = Constrained}:

// methods yet to be implemented:  convert, convert_to, expr

plot::Rotate3d::hiddenSlots({Matrix3d, Shift, ShiftX, ShiftY, ShiftZ}):

// here we have to do something
plot::Rotate3d::new:=
  proc()
    local object, other, children, dimensions;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;
    
    if nops(other) > 0 then
      object::Angle := other[1];
      if nops(other) > 1 then 
	object::Center  := other[2];
	if nops(other) > 2 then
	  object::Axis    := other[3];
	  if nops(other) > 3 then
	    error("unexpected argument ".other[4]);
	  end_if;
	end_if;
      end_if;
    end_if;
    
    if object::Angle = FAIL then
      error("No rotation angle given");
    end_if;
    dom::changeNotifier(object, "Angle"=object::Angle);
    
    children := object::children;
    if nops(children) = 0 then
      // empty Rotate3d
      return(dom::checkObject(object))
    end_if;
    
    dimensions := map({op(children)}, x -> x::dom::dimension);
    if nops(dimensions) <> 1 or dimensions <> {3} then
      error("only 3-dimensional plot objects allowed")
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

plot::Rotate3d::changeNotifier :=
  proc(obj, eq)
    local slotName, newval, c, s, t, x, y, z, cx, cy, cz, n, angle;
  begin
    [slotName, newval] := [op(eq)];
    case slotName
      of "Angle"   do
      of "Axis"    do
      of "AxisX"   do
      of "AxisY"   do
      of "AxisZ"   do
      of "Center"  do
      of "CenterX" do
      of "CenterY" do
      of "CenterZ" do
	dom::extslot(obj, slotName, newval);
	for t in hold([[angle, "Angle"],
		       [x,     "AxisX"],
		       [y,     "AxisY"],
		       [z,     "AxisZ"],
		       [cx,    "CenterX"],
		       [cy,    "CenterY"],
		       [cz,    "CenterZ"]]) do
	  if slot(obj, t[2]) = FAIL then
	    evalassign(t[1], plot::getDefault(slot(dom, t[2])));
	  else
	    evalassign(t[1], slot(obj, t[2]));
	  end_if;
	end_for;
	
	n := specfunc::sqrt(x^2+y^2+z^2);
	if iszero(n) then
	  return("3D rotation requires a non-zero axis");
	end_if;
	
	// norm [x,y,z]
	x := x/n; y := y/n; z := z/n;
	
	// rescale angle to radians
	// angle := angle*PI/180;
	
	// rotation matrix, as found in graphics gems, p. 466:
	s := specfunc::sin(angle);
	c := specfunc::cos(angle);
	t := 1-c;
	
	obj::Matrix3d := [
                           t*x^2+c,   t*x*y-s*z, t*x*z+s*y,
                           t*x*y+s*z, t*y^2+c,   t*y*z-s*x,
		           t*x*z-s*y, t*y*z+s*x, t*z^2+c  
                         ];
        // MC := numeric::matlinsolve(M, [cx, cy, cz], NoWarning);
        // obj::Shift:= [MC[1]-cx, MC[2]-cy, MC[3]-cz];
	obj::Shift := [ cy*(s*z - t*x*y) - cz*(s*y + t*x*z) - cx*t*(x^2 - 1),
		       -cx*(s*z + t*x*y) + cz*(s*x - t*y*z) - cy*t*(y^2 - 1),
		        cx*(s*y - t*x*z) - cy*(s*x + t*y*z) - cz*t*(z^2 - 1)];
	return(FALSE); // everything is done
    end_case;
    return(TRUE); // go ahead
  end_proc:

plot::Rotate3d::print :=
  proc(obj)
  begin
    if obj::Center <> FAIL and obj::Center <> [0,0,0] then
      hold(plot::Rotate3d)(obj::Angle, Center=obj::Center,
                           Axis=obj::Axis, op(obj::children),
                           dom::printAttributes(obj, {Angle, Axis, Center})):
    else
      hold(plot::Rotate3d)(obj::Angle,
                           Axis=obj::Axis, op(obj::children),
                           dom::printAttributes(obj, {Angle, Axis, Center})):
    end_if;
  end_proc:
