/**
 *   MuPlotML:  Domain which writes MuPAD plot primitives into a
 *              file;  the output is XML code conform with the
 *              muplotml_{23}d.dtd 
 */

alias(MuPlotML = plot::MuPlotML):
alias(EA = plot::ExprAttribute):

MuPlotML := newDomain("MuPlotML"):

// poor man's inheritance
map([op(plot::GenericOutput)],
  eq -> if not contains(MuPlotML, op(eq, 1)) then
          slot(MuPlotML, op(eq, 1), prog::bless(op(eq, 2), MuPlotML))
        end_if):

MuPlotML::create_dom:=hold(MuPlotML):
MuPlotML::OutputFile := 0:

MuPlotML::interface := {hold(pr), hold(prCdata),
                        hold(prP2), hold(prP3), hold(prColorArray2d),
                        hold(beginElem), hold(endElem)}:




/**
 *   MuPlotML::openOutputFile :  this function opens an output file.
 *
 *     All following calls of MuPlotML::new will write into this file
 *     until it is closed with MuPlotML::closeOutputFile.
 *
 *   Argument :     file : DOM_STRING  - the name of the file 
 *                  If the string is empty an temporary file is opened.

 *   Return value : TRUE   iff the file was successfully opened
 *                  FALSE  iff the file could not be opened
 */
MuPlotML::openOutputFile :=
proc(file : DOM_STRING) : DOM_BOOL
  local fd;
begin
  if file = "" then
    fd := xmlprint::fopen();
  else
    fd := xmlprint::fopen(file);
  end;
  if fd <> FAIL then
    dom::OutputFile := fd;
    return(TRUE)
  end;
  return(FALSE)
end:


/**
 *   MuPlotML::closeOutputFile :  closes the file opened by
 *                                MuPlotML::openOutputFile.
 *
 *   Argument :      -
 *
 *   Return value :  (0   -- (which is the new value of MuPlotML::OutputFile))
 *                   Testwise: the name of the just closed file.
 */
MuPlotML::closeOutputFile :=
proc()
  local a;
begin
  if dom::OutputFile <> 0 then
    a := xmlprint::fname(dom::OutputFile);
    xmlprint::fclose(dom::OutputFile)
  else
    a:=0
  end;
  dom::OutputFile := 0:
  if a = 0 then
    ""
  else
    a
  end_if;
end:

//-------------------------------------------------
/***   Utilities   ***/


/////////////////////////////////////////////////////////////////////////////
// Utils for 'new' MuPlotML (kd, 20.6.02)

//-------------------------------------------------

MuPlotML::header := () -> dom::pr("<?xml version=\"1.0\"?>\n<!DOCTYPE Canvas SYSTEM \"muplotml.dtd\">\n"):

MuPlotML::prCdata :=     () -> xmlprint::data(dom::OutputFile, args()):

MuPlotML::pr :=          () -> xmlprint::raw(dom::OutputFile, args()):

//-------------------------------------------------

//-------------------------------------------------
//  print a 2D-Point
MuPlotML::prP2 := () -> (xmlprint::prP2(dom::OutputFile, op(float([args()]))):
                         xmlprint::raw(dom::OutputFile, "\n")):
//-------------------------------------------------

//-------------------------------------------------
//  print a 3D-Point
MuPlotML::prP3 :=() -> (xmlprint::prP3(dom::OutputFile, op(float([args()]))):
                        xmlprint::raw(dom::OutputFile, "\n")):
//-------------------------------------------------

//-------------------------------------------------
//  print a hfarray color array
MuPlotML::prColorArray2d :=() -> xmlprint::prColorArray2d(dom::OutputFile, args()):
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::beginElem :=
   () -> xmlprint::beginElement(dom::OutputFile, args()):
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::endElem :=
   () -> (xmlprint::endElement(dom::OutputFile, args());
          xmlprint::raw(dom::OutputFile, "\n")):
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::startAnimatedObject :=
proc(obj, objName, nonExprAttrib, exprAttrib)
begin
  plot::MuPlotML::beginElem(objName,
                            "Type"=obj::dom::objType,
                            "id"=(extop(obj,1))::id,
                            "Consistent"=TRUE,
                            op(map(nonExprAttrib, float)));
  
  // now print the expression attributes of obj
  // which have to be of type plot::ExprAttribute
  map(exprAttrib, plot::ExprAttribute::MuPlotML);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::endAnimatedObject :=
(obj, objName, nonExprAttrib, exprAttrib) ->
  plot::MuPlotML::endElem(objName):
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::startStaticObject :=
proc(obj, objName, nonExprAttrib, exprAttrib)
begin
  plot::MuPlotML::beginElem(objName,
                            "Type"=obj::dom::objType,
                            "id"=(extop(obj,1))::id,
                            "Consistent"=TRUE,
                            op(map(nonExprAttrib, float)));
  
  // now print the expression attributes of obj
  // which have to be of type plot::ExprAttribute
  map(exprAttrib, plot::ExprAttribute::MuPlotML);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::endStaticObject :=
proc(obj, objName, nonExprAttrib, exprAttrib)
begin
  plot::MuPlotML::endElem(objName);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::startSpecialObject :=
proc(obj, nonExprAttrib, exprAttrib, structDefs)
begin
  plot::MuPlotML::beginElem(obj::dom::objType,
                      if not contains({"View2d", "View3d"},
                                      obj::dom::objType) then
                        "id"=(extop(obj,1))::id,
                        "Consistent"=TRUE
                      else null() end,
                      op(map(nonExprAttrib, float)));
  
  // now print the expression attributes of obj (optionals)
  // which have to be of type plot::ExprAttribute
  map(exprAttrib, plot::ExprAttribute::MuPlotML);
  map(structDefs, plot::ExprAttribute::MuPlotML);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::endSpecialObject :=
proc(obj, nonExprAttrib, exprAttrib, structDefs)
begin
  plot::MuPlotML::endElem(obj::dom::objType);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::printViewingBox :=
proc(obj, VB)
begin
  // conversion to plot::ExprAttribute
  VB := table(map([op(VB)],
                  x -> op(x,1)=plot::ExprAttribute::namedVExpr("".op(x,1),
                                                               op(x,2))));
  
  // print the computed viewingbox
  map(VB, plot::ExprAttribute::MuPlotML);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::printHints :=
() -> plot::MuPlotML::beginElem("Hints", args(), Empty):
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::printTimeRange :=
range -> plot::MuPlotML::beginElem("TimeRange",
                    "TimeBegin"=float(op(range, 1)),
                    "TimeEnd"  =float(op(range, 2)),
                    Empty):
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::startAnimationFrame :=
proc(obj, imgName, nonExprAttrib, exprAttrib, optionals, transf=table(), static=FALSE)
begin
  // print Img{23}d tag with Param, Time, TitleXYZ
  plot::MuPlotML::beginElem(imgName, 
    if not static then "Param"=optionals[ParameterValue] end,
    if contains(optionals, TitlePositionX) = TRUE then
      "TitlePositionX"=float(optionals[TitlePositionX]);
    end_if,
    if contains(optionals, TitlePositionY) = TRUE then
      "TitlePositionY"=float(optionals[TitlePositionY]);
    end_if,
    if contains(optionals, TitlePositionZ) = TRUE then
      "TitlePositionZ"=float(optionals[TitlePositionZ]);
    end_if,
    op(transf));
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::endAnimationFrame :=
proc(obj, imgName, nonExprAttrib, exprAttrib, optionals, transf=table(), static=FALSE)
begin
  plot::MuPlotML::endElem(imgName);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
MuPlotML::printDefaultStyles :=
proc(current, styleSheets, dimensions)
  local i;
begin
  if current::dom::objType = "Canvas" then
    plot::MuPlotML::pr("\n");
    if contains(dimensions, 2) then
      plot::MuPlotML::beginElem("Scene2dStyle",
                          op(map(styleSheets["Scene2dStyle"], float@plot::ExprAttribute::getExpr)), Empty);
      plot::MuPlotML::pr("\n");
    end_if;
    if contains(dimensions, 3) then
      plot::MuPlotML::beginElem("Scene3dStyle",
                          op(map(styleSheets["Scene3dStyle"], float@plot::ExprAttribute::getExpr)), Empty);
      plot::MuPlotML::pr("\n");
    end_if;
  end_if;
  delete styleSheets["Scene2dStyle"];
  delete styleSheets["Scene3dStyle"];

  if contains({"Canvas", "Scene2d", "Scene3d"}, current::dom::objType) then
    // print CoordinateSystemStyle
    if contains(dimensions, 2) then
      plot::MuPlotML::beginElem("CoordinateSystem2dStyle",
                          op(map(styleSheets["CoordinateSystem2dStyle"], float@plot::ExprAttribute::getExpr)), Empty);
      plot::MuPlotML::pr("\n");
    end_if;
    if contains(dimensions, 3) then
      plot::MuPlotML::beginElem("CoordinateSystem3dStyle",
                          op(map(styleSheets["CoordinateSystem3dStyle"], float@plot::ExprAttribute::getExpr)), Empty);
      plot::MuPlotML::pr("\n");
    end_if;
  end_if;  
  delete styleSheets["CoordinateSystem2dStyle"];
  delete styleSheets["CoordinateSystem3dStyle"];

  // print general ObjStyle
  if current::dom::objType = "Canvas" then
    plot::MuPlotML::beginElem("ObjStyle", "Type"="Default",
                        op(map(styleSheets["Default"], float@plot::ExprAttribute::getExpr)), Empty);
    plot::MuPlotML::pr("\n");
  end_if;

  delete styleSheets["Default"];
  
  for i in styleSheets do
    plot::MuPlotML::beginElem("ObjStyle", "Type"=op(i, 1),
                        if (slot(plot, op(i, 1)))::legendColor <> FAIL then
                          LegendColor = (slot(plot, op(i, 1)))::legendColor
                        else
                          null()
                        end,
                        op(map(op(i, 2), float@plot::ExprAttribute::getExpr)), Empty);
    plot::MuPlotML::pr("\n");
  end_for;
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// Low level output functions

// 2D
  
MuPlotML::writeArc2d :=
proc(attributes, explicit_attribs, centerX, centerY, axisX, axisY,
     startAngle=0.0, endAngle=float(2*PI))
  local xRange, yRange;
begin
  plot::MuPlotML::beginElem("Arc2d",
                            "Center" = (centerX, centerY),
                            "SemiAxisX" = axisX,
                            "SemiAxisY" = axisY,
                            "StartAngle" = startAngle,
                            "EndAngle" = endAngle,
                            op(explicit_attribs),
                            Empty);

  xRange := cos(startAngle...endAngle);
  xRange := axisX*xRange + centerX;
  yRange := sin(startAngle...endAngle);
  yRange := axisY*yRange + centerY;

  xRange := min(op(xRange, 1), centerX)..max(op(xRange, 2), centerX);
  yRange := min(op(yRange, 1), centerY)..max(op(yRange, 2), centerY);
  
  return([xRange, yRange]);
end_proc:

MuPlotML::writeArrow2d :=
proc(attributes, explicit_attribs, fx, fy, tx, ty)
begin
  plot::MuPlotML::beginElem("Arrow2d",
                            "From" = (fx, fy),
                            "To"   = (tx, ty),
                            op(explicit_attribs),
                            Empty);
   return([min(fx,tx) .. max(fx,tx), min(fy,ty) .. max(fy,ty)]);
end_proc:

MuPlotML::writeColorArray2d :=
proc(attributes, explicit_attribs, data, xmesh, ymesh, xmin, xmax, ymin, ymax)
  local i;
begin
  if domtype(data[1]) = DOM_HFARRAY then
    //--------------------------------------------
    // Write the color values into a ColorArray2d:
    //--------------------------------------------
    plot::MuPlotML::beginElem("ColorArray2d",
                              "XMesh" = xmesh,
                              "YMesh" = ymesh,
                              "XMin" = xmin,
                              "XMax" = xmax,
                              "YMin" = ymin,
                              "YMax" = ymax):
    plot::MuPlotML::prColorArray2d(data[1]);
    plot::MuPlotML::endElem("ColorArray2d"):
  else
    if domtype(data[1]) = DOM_LIST then
        // need to flatten a list of lists
        data:= map(data, op):
    end_if;
    //--------------------------------------------
    // Only check the first and the last entry of ColorData: these
    // must be RGB values. Systematic Checking of all color data
    // would be too expensive:
    //--------------------------------------------
    if domtype(data[1]) <> DOM_LIST or
        nops(data[1]) <> 3 or
        {op(map(data[1], domtype@float))} <> {DOM_FLOAT} or
        domtype(data[nops(data)]) <> DOM_LIST or
        nops(data[nops(data)]) <> 3 or
        {op(map(data[nops(data)], domtype@float))} <> {DOM_FLOAT}
    then
        error("expecting RGB values");
    end_if;
    //--------------------------------------------
    // Write the color values into a ColorArray2d:
    //--------------------------------------------
    plot::MuPlotML::beginElem("ColorArray2d",
                              "XMesh" = xmesh,
                              "YMesh" = ymesh,
                              "XMin" = xmin,
                              "XMax" = xmax,
                              "YMin" = ymin,
                              "YMax" = ymax):
    plot::MuPlotML::prCdata((op(data, i), " ") $ i=1..nops(data));
    plot::MuPlotML::endElem("ColorArray2d"):
  end_if;
  //-----------------------
  // return the viewing box
  //-----------------------
  return([xmin .. xmax, ymin .. ymax]);
end_proc:

MuPlotML::writeDensityArray2d :=
proc(attributes, explicit_attribs, data:Type::Union(DOM_ARRAY, DOM_HFARRAY), xmin, xmax, ymin, ymax)
  local xmesh, ymesh,
    i, j;
begin
  xmesh := op(data,[0,2,2]);
  ymesh := op(data,[0,3,2]);
  plot::MuPlotML::beginElem("DensityArray2d",
                            "XMesh" = xmesh,
                            "YMesh" = ymesh,
                            "XMin" = xmin,
                            "XMax" = xmax,
                            "YMin" = ymin,
                            "YMax" = ymax):
  for i from 1 to ymesh do
    plot::MuPlotML::prCdata((data[j, i], " ") $ j=1 .. xmesh);
  end_for;
  plot::MuPlotML::endElem("DensityArray2d"):
end_proc:

MuPlotML::writeField2d :=
proc(attributes, explicit_attribs, points, colorfunction,
      umin, vmin, stepU, stepV, umesh, vmesh, ops=1..2)
  local pt;
begin
    plot::MuPlotML::beginElem("Field2d" ,
                              "Anchor" = (float(umin),
                                          float(vmin)),
                              "Delta"   =(float(stepU),
                                          float(stepV)),
                              "XMesh"   = umesh,
                              "YMesh"   = vmesh,
                              op(explicit_attribs)
                              ):

    for pt in points do
      plot::MuPlotML::prP2(op(pt, ops),
                             colorfunction(op(pt)));
    end_for;
    plot::MuPlotML::endElem("Field2d");
    null();
end_proc:

MuPlotML::writeLine2d :=
proc(attributes, explicit_attribs, fromX, fromY, toX, toY)
begin
  plot::MuPlotML::beginElem("Line2d",
                            "From" = (fromX, fromY),
                            "To"   = (toX, toY),
                            op(explicit_attribs),
                            Empty);

  return([min(fromX,toX) .. max(fromX,toX), min(fromY,toY) .. max(fromY,toY)]);
end_proc:

MuPlotML::writePoint2d :=
proc(attributes, explicit_attribs, posX, posY)
begin
  plot::MuPlotML::beginElem("Pt2d",
                            "Position" = (posX, posY),
                            op(explicit_attribs),
                            Empty):  /* end of method MuPlotML */
  [posX..posX, posY..posY]
end_proc:

MuPlotML::writePoints2d :=
proc(attributes, explicit_attribs, data,
  linecolorfunction=(()->if args(0) > 2 then args(3) end),
  xyops=1..2)
  local pt, i;
begin
  plot::MuPlotML::beginElem("Pts2d", op(explicit_attribs));
  for pt in data do
     plot::MuPlotML::prP2(op(pt, xyops), linecolorfunction(op(pt)));
  end_for;
  plot::MuPlotML::endElem("Pts2d"):
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=1..2]);
end_proc:

MuPlotML::writePoly2d :=
proc(attributes, explicit_attribs, points, linecolorfunction=(()->null()), xyops=1..2, tops=FAIL)
  local coords, xmin, xmax, ymin, ymax, i;
begin
  if nops(points)=0 then return( null() ); end_if;
  plot::MuPlotML::beginElem("Poly2d", op(explicit_attribs));
  plot::MuPlotML::prP2(op(points[i], xyops), linecolorfunction(op(points[i]))) $ i = 1 .. nops(points);
  plot::MuPlotML::endElem("Poly2d");

  coords := map(points, op, 1);
  xmin := min(op(coords)):
  xmax := max(op(coords)):
  coords := map(points, op, 2);
  ymin := min(op(coords)):
  ymax := max(op(coords)):

  [xmin..xmax, ymin..ymax]
end_proc:

MuPlotML::writeText2d :=
proc(attributes, explicit_attribs, posX, posY, text, textRotation=0.0)
begin
    plot::MuPlotML::beginElem("Text2d",
                              "Position"     = (posX, posY),
                              "TextRotation" =  float(textRotation),
                              op(explicit_attribs));
    if domtype(text) <> DOM_STRING then
      text := expr2text(text);
    end_if;
    plot::MuPlotML::prCdata(text);
    plot::MuPlotML::endElem("Text2d");
    [posX..posX, posY..posY];
end_proc:

MuPlotML::writeVerticalAsymptote :=
proc(attributes, explicit_attribs, x)
begin
  plot::MuPlotML::beginElem("VerticalAsymptote", "Position"=x,
                              op(explicit_attribs), Empty);
  null();
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// 3D

MuPlotML::writeArrow3d :=
proc(attributes, explicit_attribs, fx, fy, fz, tx, ty, tz)
begin
   plot::MuPlotML::beginElem("Arrow3d",
                              "From" = (fx, fy, fz),
                              "To"=    (tx, ty, tz),
                              op(explicit_attribs),
                              Empty);
 return([min(fx,tx) .. max(fx,tx), 
            min(fy,ty) .. max(fy,ty),
            min(fz,tz) .. max(fz,tz)]);
end_proc:

MuPlotML::writeBox :=
proc(attributes, explicit_attribs, xmin, xmax, ymin, ymax, zmin, zmax)
begin
  plot::MuPlotML::beginElem("Box",
                            "XMin" = xmin,
                            "XMax" = xmax,
                            "YMin" = ymin,
                            "YMax" = ymax,
                            "ZMin" = zmin,
                            "ZMax" = zmax,
                            op(explicit_attribs),
                            Empty);
  return([xmin..xmax, ymin..ymax, zmin..zmax]);
end_proc:

MuPlotML::writeCircle3d :=
proc(attributes, explicit_attribs, centerX, centerY, centerZ, normalX, normalY, normalZ, radius)
  local nn, tmp, xmin, xmax, ymin, ymax, zmin, zmax;
begin
  plot::MuPlotML::beginElem("Circle3d",
                            "Center" = (centerX, centerY, centerZ),
                            "Normal" = (normalX, normalY, normalZ),
                            "Radius" =  radius, op(explicit_attribs),
                            Empty);
  nn:= normalX^2+normalY^2+normalZ^2;
  if iszero(nn) then
    [normalX, normalY, normalZ] := float([0, 0, 1]);
    nn := 1;
  end_if;
  tmp:= radius*(((normalY^2+normalZ^2)/nn)^(1/2));
  xmin:= (centerX- tmp);
  xmax:= (centerX+ tmp);
  tmp:= radius*(((normalX^2+normalZ^2)/nn)^(1/2));
  ymin:= (centerY- tmp);
  ymax:= (centerY+ tmp);
  tmp:= radius*(((normalX^2+normalY^2)/nn)^(1/2));
  zmin:= (centerZ- tmp);
  zmax:= (centerZ+ tmp);
  return(float([xmin .. xmax, ymin .. ymax, zmin .. zmax]));
end_proc:

MuPlotML::writeCone :=
proc(attributes, explicit_attribs, BaseX, BaseY, BaseZ,
    TopX, TopY, TopZ, BaseRadius, TopRadius)
  local n, nn, tmp, xmax, xmin, ymax, ymin, zmin, zmax;
begin
  plot::MuPlotML::beginElem("Cone",
      "Base" = (BaseX, BaseY, BaseZ), "Top" = (TopX, TopY, TopZ),
      "BaseRadius" = BaseRadius, "TopRadius" = TopRadius,
      op(explicit_attribs),
      Empty);
  //--------------------------------------------------------------
  // Compute the viewing box. The disc of radius r centered at the
  // vector c with normal vector n has the viewing box components
  // vbox[k] = c[k] - r*sqrt(1-n[k]^2/|n|^2) ..
  //           c[k] + r*sqrt(1-n[k]^2/|n|^2)
  //--------------------------------------------------------------
  n:= [TopX- BaseX, TopY- BaseY, TopZ- BaseZ]; // the direction of the cone
  nn:= n[1]^2 + n[2]^2 + n[3]^2;
  // If Base = Top, the normal of the disc is not defined.
  // Choose a default direction:
  if iszero(nn) then
      n := float([0, 0, 1]);
      nn:= float(1);
  end_if;
  tmp:= sqrt((n[2]^2+n[3]^2)/nn);
  xmin:= min(BaseX-BaseRadius*tmp, TopX-TopRadius*tmp);
  xmax:= max(BaseX+BaseRadius*tmp, TopX+TopRadius*tmp);
  tmp:= sqrt((n[1]^2+n[3]^2)/nn);
  ymin:= min(BaseY-BaseRadius*tmp, TopY-TopRadius*tmp);
  ymax:= max(BaseY+BaseRadius*tmp, TopY+TopRadius*tmp);
  tmp:= sqrt((n[1]^2+n[2]^2)/nn);
  zmin:= min(BaseZ-BaseRadius*tmp, TopZ-TopRadius*tmp);
  zmax:= max(BaseZ+BaseRadius*tmp, TopZ+TopRadius*tmp);
  return([xmin .. xmax, ymin .. ymax, zmin .. zmax]);
end_proc:

MuPlotML::writeEllipsoid :=
proc(attributes, explicit_attribs, CenterX, CenterY, CenterZ,
    SemiAxisX, SemiAxisY, SemiAxisZ)
begin
    plot::MuPlotML::beginElem("Ellipsoid",
                              "SemiAxisX" = SemiAxisX,
                              "SemiAxisY" = SemiAxisY,
                              "SemiAxisZ" = SemiAxisZ,
                              "Center" = (CenterX,
                                          CenterY,
                                          CenterZ),
                              op(explicit_attribs),
                              Empty);
  return([ CenterX-SemiAxisX ..CenterX+SemiAxisX,
           CenterY-SemiAxisY ..CenterY+SemiAxisY,
           CenterZ-SemiAxisZ ..CenterZ+SemiAxisZ
         ]);
end_proc:

MuPlotML::writeField3d :=
proc(attributes, explicit_attribs, points,
      xmin, ymin, zmin, stepX, stepY, stepZ, xmesh, ymesh, zmesh,
      colorfunction=(()->null()), xyops=1..3, uvops=FAIL)
  local pt;
begin
    plot::MuPlotML::beginElem("Field3d" ,
                              "Anchor" = (float(xmin),
                                          float(ymin),
                                          float(zmin)),
                              "Delta"   =(float(stepX),
                                          float(stepY),
                                          float(stepZ)),
                              "XMesh"   = xmesh,
                              "YMesh"   = ymesh,
                              "ZMesh"   = zmesh,
                              op(explicit_attribs)
                              ):

    for pt in points do
      plot::MuPlotML::prP3(op(pt, xyops),
                             colorfunction(op(pt)));
    end_for;
    plot::MuPlotML::endElem("Field3d");
    null(); /* ToDo: Viewing-Box */
end_proc:

MuPlotML::writeLine3d :=
proc(attributes, explicit_attribs, fromX, fromY, fromZ, toX, toY, toZ)
begin
  plot::MuPlotML::beginElem("Line3d",
                            "From" = (fromX, fromY, fromZ),
                            "To"   = (toX, toY, toZ),
                            op(explicit_attribs),
                            Empty);

  return([min(fromX,toX) .. max(fromX,toX), min(fromY,toY) .. max(fromY,toY), min(fromZ,toZ) .. max(fromZ,toZ)]);
end_proc:

MuPlotML::writeLines3d :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), xyops=1..3, tops=FAIL)
  local pt, i;
begin
  plot::MuPlotML::beginElem("Lines3d", "Type"="Lines",
                            op(explicit_attribs));
  assert(nops(data) mod 2 = 0);
  for pt in data do
     plot::MuPlotML::prP3(op(pt, xyops), linecolorfunction(op(pt)));
  end_for;
  plot::MuPlotML::endElem("Lines3d"):
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
end_proc:

MuPlotML::writeLineLoop3d :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), xyops=1..3, tops=FAIL)
  local pt, i;
begin
  plot::MuPlotML::beginElem("Lines3d", "Type"="LineLoop",
                            op(explicit_attribs));
  for pt in data do
     plot::MuPlotML::prP3(op(pt,xyops), linecolorfunction(op(pt)));
  end_for;
  plot::MuPlotML::endElem("Lines3d"):
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
end_proc:

MuPlotML::writePoint3d :=
proc(attributes, explicit_attribs, posX, posY, posZ)
begin
  explicit_attribs["PointsVisible"]:=TRUE;
  MuPlotML::writePoints3d(attributes, explicit_attribs, [[posX, posY, posZ]], ()->null(), 1..3);
end_proc:

MuPlotML::writePlane :=
proc(attributes, explicit_attribs, posX, posY, posZ, normalX, normalY, normalZ)
begin
   plot::MuPlotML::beginElem("Plane",
                             "Position" = (posX, posY, posZ),
                             "Normal" =   (normalX, normalY, normalZ),
                            op(explicit_attribs),
                             Empty);
   return([posX..posX, posY..posY, posZ..posZ]);
end_proc:

// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
// umesh, vmesh: Number of evaluation points with lines
// usubmesh, vsubmesh: Number of intermediate evaluation points
MuPlotML::writeRectangularMesh :=
proc(attributes, explicit_attribs, data, linecolorfunction, fillcolorfunction,
  umesh, vmesh, usubmesh, vsubmesh, usexylinesvisible=FALSE, xyops=FAIL, normalops=FAIL, uvops=FAIL)
  local pt, x, y, z, nx, ny, nz, u, v, i;
begin
  // For a non-adaptive Surface, the low-level primitive is a Mesh3d
  plot::MuPlotML::beginElem("Mesh3d", 
                            "UMesh"=expr2text(umesh), 
                            "VMesh"=expr2text(vmesh),
                            "USubmesh"=expr2text(usubmesh), 
                            "VSubmesh"=expr2text(vsubmesh),
                            "UseXYLinesVisible"=usexylinesvisible,
                            "HasLineColors"=bool(linecolorfunction(op(data[1]))<>null()),
                            op(explicit_attribs)
                           ):
  // data = [ [u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]:
  for pt in data do
    if xyops<>FAIL then
      if normalops<>FAIL then
        plot::MuPlotML::prP3(op(pt,xyops),
                             op(pt,normalops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      else
        plot::MuPlotML::prP3(op(pt,xyops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      end_if;
    elif nops(pt) = 8 then // normals are provided
       [u, v, x, y, z, nx, ny, nz]:= pt:
       plot::MuPlotML::prP3(x,  y,  z, // the point
                            nx, ny, nz, // the normal
                            linecolorfunction(u, v, x, y, z),
                            fillcolorfunction(u, v, x, y, z) 
                           );
    else // normals are not provided
       [u, v, x, y, z]:= float(pt):
       plot::MuPlotML::prP3(x, y, z, // the point
                            linecolorfunction(u, v, x, y, z),
                            fillcolorfunction(u, v, x, y, z) 
                           );
    end_if:
  end_for:
  plot::MuPlotML::endElem("Mesh3d"): // end of Mesh3d definition
  if xyops=FAIL then
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=3..5]);
  else
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
  end_if;
end_proc:
//-------------------------------------------------

MuPlotML::writeText3d :=
proc(attributes, explicit_attribs, posX, posY, posZ, baseDirX, baseDirY, baseDirZ,
  faceDirX, faceDirY, faceDirZ, text)
begin
    plot::MuPlotML::beginElem("Text3d",
                              "Position" = (posX, posY, posZ),
                              "BaseDir" = (baseDirX, baseDirY, baseDirZ),
                              "FaceDir" = (faceDirX, faceDirY, faceDirZ),
                              op(explicit_attribs));
    if domtype(text) <> DOM_STRING then
      text := expr2text(text);
    end_if;
    plot::MuPlotML::prCdata(text);
    plot::MuPlotML::endElem("Text3d");
    [posX..posX, posY..posY, posZ..posZ]
end_proc:

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
MuPlotML::writeTriangles :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=FAIL, normalops=FAIL, uvops=FAIL)
  local pt, u, v, x, y, z, nx, ny, nz, i;
begin
  if nops(data) = 0 then return(); end_if;
  if not contains(explicit_attribs, "HasLineColors") then
    explicit_attribs["HasLineColors"] := bool(linecolorfunction(op(data[1]))<>null());
  end_if;
  plot::MuPlotML::beginElem("Surf3d", 
                            "Type"="Triangles",
                            op(explicit_attribs));
  // For an adaptive Surface, the low-level primitive is a Surf3d 
  // data = [ [u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]:
  for pt in data do
    if xyops<>FAIL then
      if normalops<>FAIL then
        plot::MuPlotML::prP3(op(pt,xyops),
                             op(pt,normalops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      else
        plot::MuPlotML::prP3(op(pt,xyops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      end_if;
     elif nops(pt) = 8 then // normals are provided
        [u, v, x, y, z, nx, ny, nz]:= [op(pt, 1..2), map(op(pt, 3..8), float)]:
        plot::MuPlotML::prP3( x,  y,  z, // the point
                             nx, ny, nz, // the normal
                             linecolorfunction(u, v, x, y, z),
                             fillcolorfunction(u, v, x, y, z) 
                            );
     else // normals are not provided
        [u, v, x, y, z]:= [op(pt, 1..2), map(op(pt, 3..5), float)]:
        plot::MuPlotML::prP3(x, y, z, // the point
                             linecolorfunction(u, v, x, y, z),
                             fillcolorfunction(u, v, x, y, z) 
                            );
     end_if:
  end_for:
  plot::MuPlotML::endElem("Surf3d"): // end of Surf3d definition
  if xyops=FAIL then
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=3..5]);
  else
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
  end_if;
end_proc:
//-------------------------------------------------

MuPlotML::writeQuadStrip :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()), xyops=1..3,
    normalops=FAIL, uvops=FAIL)
  local i, p, hasLineColors;
begin
  hasLineColors := bool(linecolorfunction(op(data[1]))<>null());
  plot::MuPlotML::beginElem("Surf3d",
                            "Type"          = "QuadStrip",
                            "HasLineColors" = hasLineColors,
                            op(explicit_attribs));
  for p in data do
    plot::MuPlotML::prP3(op(p, xyops), linecolorfunction(op(p)), fillcolorfunction(op(p)));
  end_for;
  plot::MuPlotML::endElem("Surf3d"):
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
end_proc:

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
MuPlotML::writeTriangleFan :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=FAIL, normalops=FAIL, uvops=FAIL)
  local pt, u, v, x, y, z, nx, ny, nz, i;
begin
  if nops(data) = 0 then return(); end_if;
  if not contains(explicit_attribs, "HasLineColors") then
    explicit_attribs["HasLineColors"] := bool(linecolorfunction(op(data[1]))<>null());
  end_if;
  plot::MuPlotML::beginElem("Surf3d", 
                            "Type"="TriangleFan",
                            op(explicit_attribs));
  for pt in data do
    if xyops<>FAIL then
      if normalops<>FAIL then
        plot::MuPlotML::prP3(op(pt,xyops),
                             op(pt,normalops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      else
        plot::MuPlotML::prP3(op(pt,xyops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      end_if;
     elif nops(pt) = 8 then // normals are provided
        [u, v, x, y, z, nx, ny, nz]:= [op(pt, 1..2), map(op(pt, 3..8), float)]:
        plot::MuPlotML::prP3( x,  y,  z, // the point
                             nx, ny, nz, // the normal
                             linecolorfunction(u, v, x, y, z),
                             fillcolorfunction(u, v, x, y, z) 
                            );
     else // normals are not provided
        [u, v, x, y, z]:= [op(pt, 1..2), map(op(pt, 3..5), float)]:
        plot::MuPlotML::prP3(x, y, z, // the point
                             linecolorfunction(u, v, x, y, z),
                             fillcolorfunction(u, v, x, y, z) 
                            );
     end_if:
  end_for:
  plot::MuPlotML::endElem("Surf3d"): // end of Surf3d definition
  if xyops=FAIL then
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=3..5]);
  else
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
  end_if;
end_proc:

MuPlotML::writeTriangleStrip :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=FAIL, normalops=FAIL, uvops=FAIL)
  local pt, u, v, x, y, z, nx, ny, nz, i;
begin
  if nops(data) = 0 then return(); end_if;
  if not contains(explicit_attribs, "HasLineColors") then
    explicit_attribs["HasLineColors"] := bool(linecolorfunction(op(data[1]))<>null());
  end_if;
  plot::MuPlotML::beginElem("Surf3d",
                            "Type"="TriangleStrip",
                            op(explicit_attribs));
  for pt in data do
    if xyops<>FAIL then
      if normalops<>FAIL then
        plot::MuPlotML::prP3(op(pt,xyops),
                             op(pt,normalops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      else
        plot::MuPlotML::prP3(op(pt,xyops),
                            linecolorfunction(op(pt)),
                            fillcolorfunction(op(pt)) 
                           );
      end_if;
     elif nops(pt) = 8 then // normals are provided
        [u, v, x, y, z, nx, ny, nz]:= [op(pt, 1..2), map(op(pt, 3..8), float)]:
        plot::MuPlotML::prP3( x,  y,  z, // the point
                             nx, ny, nz, // the normal
                             linecolorfunction(u, v, x, y, z),
                             fillcolorfunction(u, v, x, y, z) 
                            );
     else // normals are not provided
        [u, v, x, y, z]:= [op(pt, 1..2), map(op(pt, 3..5), float)]:
        plot::MuPlotML::prP3(x, y, z, // the point
                             linecolorfunction(u, v, x, y, z),
                             fillcolorfunction(u, v, x, y, z) 
                            );
     end_if:
  end_for:
  plot::MuPlotML::endElem("Surf3d"): // end of Surf3d definition
  if xyops=FAIL then
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=3..5]);
  else
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
  end_if;
end_proc:

MuPlotML::writeSphere :=
proc(attributes, explicit_attribs, centerX, centerY, centerZ, radius)
begin
  plot::MuPlotML::beginElem("Sphere",
                            "Center" = (centerX,
                                        centerY,
                                        centerZ),
                            "Radius" =  radius,
                            op(explicit_attribs),
                            Empty);
  return([ centerX-radius..centerX+radius,
            centerY-radius..centerY+radius,
            centerZ-radius..centerZ+radius]);
end_proc:

MuPlotML::writeQuads :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=1..3, normalops=FAIL, uvops=FAIL)
  local tr, i;
begin
    plot::MuPlotML::beginElem("Surf3d", "Type" = "Quads", op(explicit_attribs));
    for tr in data do
      plot::MuPlotML::prP3(op(tr,xyops),
                           fillcolorfunction(op(tr)) );
    end_for;
    plot::MuPlotML::endElem("Surf3d"):
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=xyops]);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction: Function. If no function
//   is desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
// UV = TRUE if writing U or X, FALSE if writing V or Y
// XU = TRUE if writing X or Y, FALSE if writing U or V
MuPlotML::writeUVXYLine :=
proc(attributes, explicit_attribs, line, linecolorfunction, UV, XU)
  local pt, x, y, z, nx, ny, nz, u, v, i;
begin
  if nops(line) = 0 then return(); end_if;
	plot::MuPlotML::beginElem("Mesh3d",
	        if UV then
  				  "VLinesVisible"=FALSE,
  				  "VMesh"=1,
  				  "UMesh"=nops(line)
  				else
  				  "ULinesVisible"=FALSE,
  				  "UMesh"=1,
  				  "VMesh"=nops(line)
  				end_if,
				  "USubmesh"=0,
				  "VSubmesh"=0,
          "UseXYLinesVisible"=XU,
          "PointsVisible"=FALSE, // otherwise drawn twice
				  "Filled"=FALSE,
				  "HasLineColors"=bool(linecolorfunction(op(line[1]))<>null()),
				  op(explicit_attribs));
	for pt in line do
	  if nops(pt) = 8 then // normals are provided
	    [u, v, x, y, z, nx, ny, nz]:= pt:
	    plot::MuPlotML::prP3(x,  y,  z, // the point
				 nx, ny, nz, // the normal
				 linecolorfunction(u, v, x, y, z) 
				);
	  else // normals are not provided
	    [u, v, x, y, z]:= float(pt):
	    plot::MuPlotML::prP3(x, y, z, // the point
				 linecolorfunction(u, v, x, y, z) 
				);
	  end_if:
	end_for;
	plot::MuPlotML::endElem("Mesh3d");
  return([min(map(line,op,i))..max(map(line,op,i)) $ i=3..5]);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
//  writeMeshContours: plot contour lines
// for an irregular triangular mesh
// attributes: A table as given into doPlotStatic or MuPlotML
// data = a list of points
// ops = a range indicating in which position in the points in data x, y, z are found
// contdir = the index in the points where the direction is found
//    for which to plot contour lines
// contours = a list of values for where to write contour lines
//    or the list [Automatic, n] with n being the number of lines.
// linecolorfunction: Function mapping a point to a color. If no function
//   is desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
// _min, _max = viewing box extension in the direction currently considered.  
MuPlotML::writeMeshContours :=
proc(attributes, explicit_attribs, data, ops, contdir, contours,
     linecolorfunction, _min, _max)
  local writeCLine, linearZero, pt, c, p1, p2, p3,
	t, interpol, i;
begin
  contours := float(contours);
  if contours = [] then return(); end_if;

  writeCLine :=
  proc(p1, p2)
  begin
    plot::MuPlotML::prP3(op(p1, ops), // the point
			 linecolorfunction(p1));
    plot::MuPlotML::prP3(op(p2, ops), // the point
			 linecolorfunction(p2));
  end_proc;

  // linear interpolation over all components
  interpol := (a,b)->t*a+(1-t)*b;
  linearZero :=
  proc(p1, p2)
    local z1, z2;
  begin
    z1 := op(p1, contdir);
    z2 := op(p2, contdir);
    if iszero(z1-z2) then
      p1;
    else
      t := (c-z2)/(z1-z2);
      zip(p1, p2, interpol);
    end_if;
  end_proc;

  if contours[1] = hold(Automatic) then
    if nops(contours) > 1 and
       domtype(float(contours[2])) = DOM_FLOAT then
      contours := float(contours[2]);
    else
      contours := 15;
    end_if;
    if contours < 2 then contours := 2; end_if;
    _max := _max - frac(contours)/floor(contours) * (_max - _min);
    contours := floor(contours);
    contours := [_min + i*(_max-_min)/(contours-1)
		 $ i = 0..contours-1];
  end_if; // Automatic

  plot::MuPlotML::beginElem("Lines3d",
			    "Type"="Lines",
			    "PointsVisible"=FALSE,
			    op(explicit_attribs));

  for c in contours do
    pt := 1;
    while pt < nops(data) do // three consecutive points form a triangle
      p1 := data[pt  ];
      p2 := data[pt+1];
      p3 := data[pt+2];
      case [specfunc::sign(c-op(p1, contdir)),
  	    specfunc::sign(c-op(p2, contdir)),
  	    specfunc::sign(c-op(p3, contdir))]
    	of [ 0, 0, 0] do
    	  writeCLine(p1, p2);
    	  writeCLine(p2, p3);
    	  writeCLine(p3, p1); break;
    	of [ 0, 0, 1] do
    	of [ 0, 0,-1] do writeCLine(p1, p2); break;
    	of [ 0, 1, 0] do
    	of [ 0,-1, 0] do writeCLine(p1, p3); break;
    	of [ 1, 0, 0] do
    	of [-1, 0, 0] do writeCLine(p2, p3); break;
    	of [ 1,-1, 0] do
    	of [-1, 1, 0] do writeCLine(linearZero(p1, p2), p3); break
    	of [ 1, 0,-1] do
    	of [-1, 0, 1] do writeCLine(linearZero(p1, p3), p2); break
    	of [ 0, 1,-1] do
    	of [ 0,-1, 1] do writeCLine(linearZero(p2, p3), p1); break
    	of [ 1,-1,-1] do
    	of [-1, 1, 1] do writeCLine(linearZero(p1, p2), linearZero(p1, p3)); break;
    	of [-1, 1,-1] do
    	of [ 1,-1, 1] do writeCLine(linearZero(p1, p2), linearZero(p2, p3)); break;
    	of [-1,-1, 1] do
    	of [ 1, 1,-1] do writeCLine(linearZero(p2, p3), linearZero(p1, p3)); break;
      end_case;
      pt := pt+3;
    end_while;
  end_for;

  plot::MuPlotML::endElem("Lines3d");
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=ops]);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
//  writeGridContours: plot contour lines
// for a rectangular mesh
// attributes: A table as given into doPlotStatic or MuPlotML
// data = a list of points
// umesh, vmesh = dimension of data
// ops = a range indicating in which position in the points in data x, y, z are found
// contdir = the index in the points where the direction is found
//    for which to plot contour lines
// contours = a list of values for where to write contour lines
//    or the list [Automatic, n] with n being the number of lines.
// linecolorfunction: Function mapping a point to a color. If no function
//   is desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
// _min, _max = viewing box extension in the direction currently considered.  
MuPlotML::writeGridContours :=
proc(attributes, explicit_attribs, data, umesh, vmesh, ops, contdir, contours,
     linecolorfunction, _min, _max)
  local writeCLine, linearZero,
	c, i, j, ij,
	p1, p2, p3, p4,
	p12, p24, p13, p34,
	t, interpol;
begin
  contours := float(contours);
  if contours = [] then return(); end_if;

  writeCLine :=
  proc(p1, p2)
  begin
    plot::MuPlotML::prP3(op(p1, ops), // the point
			 linecolorfunction(p1));
    plot::MuPlotML::prP3(op(p2, ops), // the point
			 linecolorfunction(p2));
  end_proc;

  // linear interpolation over all components
  interpol := (a,b)->t*a+(1-t)*b;
  linearZero :=
  proc(p1, p2)
    local z1, z2;
  begin
    z1 := op(p1, contdir);
    z2 := op(p2, contdir);
    if iszero(z1-z2) then
      p1;
    else
      t := (c-z2)/(z1-z2);
      zip(p1, p2, interpol);
    end_if;
  end_proc;

  if contours[1] = hold(Automatic) then
    if nops(contours) > 1 and
       domtype(float(contours[2])) = DOM_FLOAT then
      contours := float(contours[2]);
    else
      contours := 15;
    end_if;
    if contours < 2 then contours := 2; end_if;
    _max := _max - frac(contours)/floor(contours) * (_max - _min);
    contours := floor(contours);
    contours := [_min + i*(_max-_min)/(contours-1)
		 $ i = 0..contours-1];
  end_if; // Automatic

  plot::MuPlotML::beginElem("Lines3d",
			    "Type"="Lines",
			    "PointsVisible"=FALSE,
			    op(explicit_attribs));

  for c in contours do
    for i from 1 to vmesh-1 do
      for j from 1 to umesh-1 do
      	ij := (i-1)*umesh+j;
      	p1 := data[ij];
      	p2 := data[ij+1];
      	p3 := data[ij+umesh];
      	p4 := data[ij+umesh+1];
      	case [specfunc::sign(c-op(p1, contdir)),
      	      specfunc::sign(c-op(p2, contdir)),
      	      specfunc::sign(c-op(p3, contdir)),
      	      specfunc::sign(c-op(p4, contdir))]
      	// we never draw lines from p3 to p4 or from p2 to p4,
      	// to avoid duplicates.  Those at the left and
      	// the upper corner will be handled after the loop.
      	  of [ 0, 0, 0, 0] do
      	    writeCLine(p1, p2);
      	    writeCLine(p2, p3);
      	    writeCLine(p4, p1);
      	    writeCLine(p1, p3); break;
      	  of [ 0, 0, 0, 1] do
      	  of [ 0, 0, 0,-1] do
      	    writeCLine(p1, p2);
      	    writeCLine(p2, p3);
      	    writeCLine(p1, p3); break;
      	  of [ 0, 0, 1, 0] do
      	  of [ 0, 0,-1, 0] do
      	    writeCLine(p1, p2);
      	    writeCLine(p4, p1); break;
      	  of [ 0, 1, 0, 0] do
      	  of [ 0,-1, 0, 0] do
      	    writeCLine(p4, p1);
      	    writeCLine(p1, p3); break;
      	  of [ 1, 0, 0, 0] do
      	  of [-1, 0, 0, 0] do writeCLine(p2, p3); break;
      	  of [ 0, 0, 1, 1] do
      	  of [ 0, 0,-1,-1] do writeCLine(p1, p2); break;
      	  of [ 0, 1, 0, 1] do
      	  of [ 0,-1, 0,-1] do writeCLine(p1, p3); break;
      	  of [ 1, 0, 0, 1] do
      	  of [-1, 0, 0, 1] do
      	  of [ 1, 0, 0,-1] do
      	  of [-1, 0, 0,-1] do writeCLine(p2, p3); break;
      	  of [ 0, 1, 1, 0] do
      	  of [ 0,-1, 1, 0] do
      	  of [ 0, 1,-1, 0] do
      	  of [ 0,-1,-1, 0] do writeCLine(p1, p4); break;
    
      	  of [ 1, 1, 1,-1] do
      	  of [-1,-1,-1, 1] do
      	    writeCLine(linearZero(p2,p4), linearZero(p3,p4)); break;
    
      	  of [ 0, 1, 1,-1] do
      	  of [ 0,-1,-1, 1] do
      	    p24 := linearZero(p2, p4);
      	    p34 := linearZero(p3, p4);
      	    writeCLine(p1, p24);
      	    writeCLine(p1, p34);
      	    writeCLine(p24, p34);
      	    break;
    
      	  of [-1, 1, 1, 1] do
      	  of [ 1,-1,-1,-1] do
      	    writeCLine(linearZero(p1, p3), linearZero(p1, p2)); break;
    
      	  of [-1, 1, 1, 0] do
      	  of [ 1,-1,-1, 0] do
      	    p12 := linearZero(p1, p2);
      	    p13 := linearZero(p1, p3);
      	    writeCLine(p12, p4);
      	    writeCLine(p13, p4);
      	    writeCLine(p12, p13);
      	    break;
    
      	  of [-1, 1, 1,-1] do
      	    p12 := linearZero(p1, p2);
      	    p24 := linearZero(p2, p4);
      	    p13 := linearZero(p1, p3);
      	    p34 := linearZero(p3, p4);
      	    writeCLine(p12, p24); writeCLine(p24, p34);
      	    writeCLine(p34, p13); writeCLine(p13, p12);
      	    writeCLine(p12, p34); writeCLine(p24, p13);
      	    break;
    
      	  of [ 1, 0, 1,-1] do
      	  of [-1, 0,-1, 1] do
      	    writeCLine(p2, linearZero(p3, p4)); break;
    
      	  of [ 0, 0, 1,-1] do
      	  of [ 0, 0,-1, 1] do
      	    p34 := linearZero(p3, p4);
      	    writeCLine(p2, p34);
      	    writeCLine(p1, p34);
      	    writeCLine(p1, p2);
      	    break;
    
      	  of [-1, 0, 1, 1] do
      	  of [ 1, 0,-1,-1] do
      	    writeCLine(linearZero(p1, p3), p2); break;
    
      	  of [-1, 0, 1, 0] do
      	  of [ 1, 0,-1, 0] do
      	    p13 := linearZero(p1, p3);
      	    writeCLine(p13, p4);
      	    writeCLine(p13, p2);
      	    break;
    
      	  of [-1, 0, 1,-1] do
      	  of [ 1, 0,-1, 1] do
      	    p13 := linearZero(p1, p3);
      	    p34 := linearZero(p3, p4);
      	    writeCLine(p2, p34);
      	    writeCLine(p2, p13);
      	    writeCLine(p13, p34);
      	    break;
    
      	  of [ 1,-1, 1, 1] do
      	  of [-1, 1,-1,-1] do
      	    writeCLine(linearZero(p1, p2), linearZero(p2, p4));
      	    break;
    
      	  of [ 1,-1, 1, 0] do
      	  of [-1, 1,-1, 0] do
      	    writeCLine(linearZero(p1, p2), p4); break;
    
      	  of [ 1,-1, 1,-1] do
      	  of [-1, 1,-1, 1] do
      	    writeCLine(linearZero(p1, p2), linearZero(p3, p4));
      	    break;
    
      	  of [ 0,-1, 1, 1] do
      	  of [ 0, 1,-1,-1] do
      	    writeCLine(p1, linearZero(p2, p4)); break;
    
      	  of [ 0,-1, 1,-1] do
      	  of [ 0, 1,-1, 1] do
      	    writeCLine(p1, linearZero(p3, p4)); break;
    
      	  of [-1,-1, 1, 1] do
      	  of [ 1, 1,-1,-1] do
      	    writeCLine(linearZero(p1, p3), linearZero(p2, p4)); break;
    
      	  of [-1,-1, 1, 0] do
      	  of [ 1, 1,-1, 0] do
      	    writeCLine(linearZero(p1, p3), p4); break;
    
      	  of [-1,-1, 1,-1] do
      	  of [ 1, 1,-1, 1] do
      	    writeCLine(linearZero(p1, p3), linearZero(p3, p4)); break;
    
      	  of [ 1, 1, 0,-1] do
      	  of [-1,-1, 0, 1] do
      	    writeCLine(linearZero(p2, p4), p3); break;
    
      	  of [ 0, 1, 0,-1] do
      	  of [ 0,-1, 0, 1] do
      	    p24 := linearZero(p2, p4);
      	    writeCLine(p1, p3);
      	    writeCLine(p1, p24);
      	    writeCLine(p3, p24);
      	    break;
    
      	  of [-1, 1, 0, 1] do
      	  of [ 1,-1, 0,-1] do
      	    writeCLine(linearZero(p1, p2), p3); break;
    
      	  of [-1, 1, 0, 0] do
      	  of [ 1,-1, 0, 0] do
      	    p12 := linearZero(p1, p2);
      	    writeCLine(p3, p12);
      	    writeCLine(p4, p12);
      	    break;
    
      	  of [-1, 1, 0,-1] do
      	  of [ 1,-1, 0, 1] do
      	    p12 := linearZero(p1, p2);
      	    p24 := linearZero(p2, p4);
      	    writeCLine(p3, p12);
      	    writeCLine(p3, p24);
      	    writeCLine(p12, p24);
      	    break;
  
      	end_case;
      end_for; // j
      // "right" border
      if iszero(c-op(p2, contdir)) and
        iszero(c-op(p4, contdir)) then
      	writeCLine(p2, p4);
      end_if;
    end_for; // i
    // "upper" border
    for j from 1 to umesh-1 do
      p1 := data[umesh*(vmesh-1)+j];
      p2 := data[umesh*(vmesh-1)+j+1];
      if iszero(c-op(p1, contdir)) and
        iszero(c-op(p2, contdir)) then
      	writeCLine(p1, p2);
      end_if;
    end_for;
  end_for; // c

  plot::MuPlotML::endElem("Lines3d");
  return([min(map(data,op,i))..max(map(data,op,i)) $ i=ops]);
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
MuPlotML::writePoly3d :=
proc(attributes, explicit_attribs, data,
  linecolorfunction=(()->null()), ops=3..5)
  local pt, i;
begin
  plot::MuPlotML::beginElem("Poly3d", op(explicit_attribs));
  for pt in data do
     plot::MuPlotML::prP3(op(pt,ops), linecolorfunction(op(pt)));
  end_for;
  plot::MuPlotML::endElem("Poly3d"):
  if data = [] then
    return(null());
  else
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=ops]);
  end_if;
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
MuPlotML::writePoints3d :=
proc(attributes, explicit_attribs, data,
  colorfunction=(()->null()), ops=3..5)
  local pt, i;
begin
  plot::MuPlotML::beginElem("Pts3d", op(explicit_attribs));
  for pt in data do
     plot::MuPlotML::prP3(op(pt,ops), colorfunction(op(pt)));
  end_for;
  plot::MuPlotML::endElem("Pts3d"):
  if data = [] then
    return(null());
  else
    return([min(map(data,op,i))..max(map(data,op,i)) $ i=ops]);
  end_if;
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// Auxiliary
MuPlotML::writeAmbientLight := proc(attributes, explicit_attribs, lightIntensity)
begin
  plot::MuPlotML::beginElem("AmbientLight",
                            "LightIntensity" = float(lightIntensity),
                            op(explicit_attribs),
                            Empty);
end_proc:

MuPlotML::writeCamera := proc(attributes, explicit_attribs, posX, posY, posZ,
  focalPointX, focalPointY, focalPointZ, currentUpVector, keepUpVector, viewingAngle)
begin
  plot::MuPlotML::beginElem("Camera",
                            "Position"     =(posX, posY, posZ),
                            "FocalPoint"   =(focalPointX, focalPointY, focalPointZ),
                            "UpVector"     =(op(currentUpVector)),
                            "KeepUpVector" = keepUpVector,
                            "ViewingAngle" = viewingAngle,
                            op(explicit_attribs),
                            Empty);
end_proc:

MuPlotML::writeClippingBox := proc(attributes, explicit_attribs, xmin, xmax, ymin, ymax, zmin, zmax)
begin
  plot::MuPlotML::beginElem("ClippingBox",
                            "XMin" = float(xmin),
                            "XMax" = float(xmax),
                            "YMin" = float(ymin),
                            "YMax" = float(ymax),
                            "ZMin" = float(zmin),
                            "ZMax" = float(zmax),
                            op(explicit_attribs),
                            Empty);
end_proc:

MuPlotML::writeDistantLight := proc(attributes, explicit_attribs, posX, posY, posZ, targetX, targetY, targetZ, lightIntensity)
begin
  plot::MuPlotML::beginElem("DistantLight",
                            "Position"       =(posX, posY, posZ),
                            "Target"         =(targetX, targetY, targetZ),
                            "LightIntensity" = lightIntensity,
                            op(explicit_attribs),
                            Empty);
end_proc:

MuPlotML::writePointLight := proc(attributes, explicit_attribs, posX, posY, posZ, lightIntensity)
begin
  plot::MuPlotML::beginElem("PointLight",
                            "Position"       =(posX, posY, posZ),
                            "LightIntensity" = lightIntensity,
                            op(explicit_attribs),
                            Empty);
end_proc:

MuPlotML::writeSpotLight := proc(attributes, explicit_attribs, posX, posY, posZ,
    targetX, targetY, targetZ, spotAngle, lightIntensity)
begin
  plot::MuPlotML::beginElem("SpotLight",
                            "Position"       =(posX, posY, posZ),
                            "Target"         =(targetX, targetY, targetZ),
                            "SpotAngle"      = spotAngle,
                            "LightIntensity" = lightIntensity,
                            op(explicit_attribs),
                            Empty);
end_proc:

// colStr and utilities used in plot::checkFont
// Zahl d in {0,1, .. , 15} --> Hex
MuPlotML::xdigit:=
proc(d)
begin     
  case d
    of 10 do return("a");
    of 11 do return("b");
    of 12 do return("c");
    of 13 do return("d");
    of 14 do return("e");
    of 15 do return("f");
  end_case;
  expr2text(d)
end_proc:

//-------------------------------------------------
// return sequence (d1,d2) for number 0 <= d < 256 where
// d1, d2 are strings with the hex digits of d
MuPlotML::hex2str :=
proc(d)
begin
  dom::xdigit(d div 16), dom::xdigit(d mod 16)
end:

//-------------------------------------------------
// return sequence of strings for printing the HTML color rep of
// the MuPAD color c (a list with three numbers in [0,1])
// still used for colors in font encodings
MuPlotML::colStr :=
proc(c)
begin
  "#",
  dom::hex2str(floor(op(c,1)*255.0)),
  dom::hex2str(floor(op(c,2)*255.0)),
  dom::hex2str(floor(op(c,3)*255.0))
end_proc:

