//      

/* -----------------------------------------------------------
    Arc3d -- The graphical primitive for 3D arcs

    Syntax:
    Arc3d(radius<, center><, normal><, angleRange><, op1, op2, ...>)

    radius                : expression; radius of circle
                        OR  list of 2 expressions; 2 semi-axes of ellipse
    center                : list of 3 expressions
    normal                : list of 3 expressions
    angleRange            : range of 2 expressions
    op1, op2, ...         : options of the form 'option = value'

    Example:
    >> plot::Arc3d(2)

    >> plot::Arc3d(2, [1, 2, 3])

    >> plot::Arc3d([2, 3], [1, 1, 1])

    >> plot::Arc3d(2, -PI/4..PI/3)

    >> plot::Arc3d(2, [1, 2, 3], -PI/4..PI/3)

    >> plot::Arc3d([2, 3], [1, 2, 3], -PI/4..PI/3)

    >> plot::Arc3d([2, 3], [1, 2, 3], [1, 1, 1], -PI/4..PI/3)

    >> plot::Arc3d([2, 3], [1, 2, 3], [1, 1, 1], -PI/4..PI/3, Filled)

    >> plot::Arc3d([2, 3], [1, 2, 3], [1, 1, 1], Angle=0..2*PI, -PI/4..PI/3, Filled)

    >> Arc:= [1, 1], [0, 0, 0], [0, -1, 0], PI/4..PI/2, Filled, Closed:
       plot(plot::Arc3d(Arc, Angle=0,      FillColor=RGB::Red),
            plot::Arc3d(Arc, Angle=1/2*PI, FillColor=RGB::Green),
            plot::Arc3d(Arc, Angle=PI,     FillColor=RGB::Yellow),
            plot::Arc3d(Arc, Angle=3/2*PI, FillColor=RGB::Blue))

----------------------------------------------------------------*/ 

plot::createPlotDomain("Arc3d",
                       "graphical primitive for 3D arcs",
                       3,
       [CenterX, CenterY, CenterZ, Center, 
        SemiAxisX, SemiAxisY, SemiAxes,
        NormalX, NormalY, NormalZ, Normal,
        Angle,
        [AngleBegin, ["Optional", 0],
         ["Definition", "Expr", FAIL, "Beginning of the angle", FALSE]],
        [AngleEnd,   ["Optional", 2*PI],
         ["Definition", "Expr", FAIL, "End of the angle", FALSE]],
        // library interface for AngleBegin, AngleEnd
        [AngleRange,   ["Library", NIL,
                        plot::libRangeOfOptExpr("Angle"),
                        plot::readRangeOfOptExpr("Angle")],
         [[AngleBegin..AngleEnd]]],
        LineColor, LineColor2, LineColorType, LineWidth, LineStyle, LinesVisible, Closed,
        LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ, LineColorDirection,
        Filled, FillColor, FillColor2, FillColorType,
        FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ, FillColorDirection]
):
//-----------------------------------------------------------------------------------
plot::Arc3d::styleSheet := table(
      CenterX = 0,
      CenterY = 0,
      CenterZ = 0,
      SemiAxisX = 1,
      SemiAxisY = 1,
      NormalX = 0,
      NormalY = 0,
      NormalZ = 1,
      Angle   = 0,
      AngleBegin = 0,
      AngleEnd   = PI/2,
      Filled  = FALSE,
      FillColorType = Flat,
   // LineColor = RGB::Black.[0.25],  // primarily, this is a line object!!
      FillColor = RGB::LightBlue,
      LineColorDirectionY=0
):
plot::Arc3d::hints := {Scaling = Constrained}:
plot::Arc3d::setPrimaryColor(LineColor):
//-----------------------------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//-----------------------------------------------------------------------------------
plot::Arc3d::new:=
  proc()
    local object, other, i, L, isCenter;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;

    if testargs() then
      if nops(other) > 4 then
        error("unexpected argument: ".expr2text(other[5]))
      end_if;

      if nops(other) > 0 then
        if not testtype(other[1], Type::Arithmetical) and
           not testtype(other[1], Type::ListOf(Type::Arithmetical, 2, 2)) and
           not (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
           error("1st argument: expecting an expression (the radius of a spherical arc) ".
                 "or a list of 2 expressions (the semi-axes of an elliptical arc)")
        end_if;

        for i from 2 to nops(other) do
          L:= op(other,i);
          if not testtype(L, Type::ListOf(Type::Arithmetical, 3, 3)) and
             not testtype(L, "_range")  and
             not L::dom::hasProp(Cat::Matrix)=TRUE then
             error(output::ordinal(i)." argument: expecting a list of 3 expressions or a range")
          end_if;
        end_for;
      end_if;
    end_if;
    
    if nops(other) <> 0 then
      if testtype(other[1], Type::ListOf(Type::Arithmetical, 2, 2)) then
        object::SemiAxisX := other[1][1];
        object::SemiAxisY := other[1][2];
      elif testtype(other[1], Type::Arithmetical) then
        object::SemiAxisX := other[1];
        object::SemiAxisY := other[1];
      else
        object::SemiAxes := other[1]
      end_if;

      isCenter := TRUE;
      if nops(other) > 1 then
        for i in [op(other, 2..nops(other))] do
          if testtype(i, Type::ListOf(Type::Arithmetical, 3, 3)) or
            i::dom::hasProp(Cat::Matrix)=TRUE then
            if isCenter then
              object::Center := i;
              isCenter := FALSE;
            else
              object::Normal := i;
            end_if;
          elif testtype(i, "_range") then
            object::AngleRange := i;
          end_if;
        end_for;
      end_if;      
    end_if;
      
    // semantically check for validity
    dom::checkObject(object);
  end_proc:

//-----------------------------------------------------------------------------------
plot::Arc3d::print :=
  obj -> if obj::Normal = FAIL and obj::AngleRange = FAIL then
           hold(plot::Arc3d)(obj::SemiAxes, obj::Center):
         elif obj::Normal <> FAIL and obj::AngleRange <> FAIL then
           hold(plot::Arc3d)(obj::SemiAxes, obj::Center, obj::Normal, obj::AngleRange):
         elif obj::AngleRange = FAIL then
           hold(plot::Arc3d)(obj::SemiAxes, obj::Center, obj::Normal):
         else
           hold(plot::Arc3d)(obj::SemiAxes, obj::Center, obj::AngleRange):
         end_if:
//-----------------------------------------------------------------------------------
plot::Arc3d::doPlotStatic:=
  proc(out, obj, attrib, inherited)
    local axisX, axisY, cx, cy, cz, nx, ny, nz, angleA, angleB, angle,
          fillcolorfunction, linecolorfunction,
          edges, points, c, k, n, r, ra, s, t, x, y, z;
  begin
    axisX := float(attrib[SemiAxisX]);
    axisY := float(attrib[SemiAxisY]);
    cx := float(attrib[CenterX]);
    cy := float(attrib[CenterY]);
    cz := float(attrib[CenterZ]);
    nx := float(attrib[NormalX]);
    ny := float(attrib[NormalY]);
    nz := float(attrib[NormalZ]);
    angle  :=  float(attrib[Angle]);
    angleA :=  float(attrib[AngleBegin]);
    angleB :=  float(attrib[AngleEnd]);

    angleA:= angleA + angle;
    angleB:= angleB + angle;

    // check for fill-color function
    if contains(attrib, FillColorFunction) then
      fillcolorfunction:= float@(attrib[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attrib, LineColorFunction) then
      linecolorfunction:= float@(attrib[LineColorFunction]); // procedure
    else linecolorfunction:= () -> null();
    end_if;

   [x,y,z]:= float([op(linalg::crossProduct(matrix([0,0,1]), matrix([nx,ny,nz])))]);
   n:= specfunc::sqrt(x^2+y^2+z^2);
   
   if iszero(n) then // no rotation is needed
      r:= 1.0;
   else
      [x,y,z]:= [x/n,y/n,z/n];
      ra:= linalg::angle(matrix([0,0,1]), matrix([nx,ny,nz]));
      // rotation matrix, as found in graphics gems, p. 466:
      s:= specfunc::sin(ra);
      c:= specfunc::cos(ra);
      t:= 1-c;
      r:= matrix([[t*x^2+c,   t*x*y-s*z, t*x*z+s*y],
                  [t*x*y+s*z, t*y^2+c,   t*y*z-s*x],
                  [t*x*z-s*y, t*y*z+s*x, t*z^2+c  ]]);
   end_if;
  
   edges:= 41;
   points:= float([[0, k+1,
                    op(r*matrix([axisX*cos(k/edges*(angleB-angleA)+angleA), 
                                 axisY*sin(k/edges*(angleB-angleA)+angleA), 
                                 0.0]) + matrix([cx,cy,cz]))] 
                    $ k=0..edges]);

   out::writeTriangleFan(attrib, table("MeshVisible" = FALSE),
     [[0, 0, cx, cy, cz]].points, 
     (()->null()), // linecolorfunction
     ((i, j, x, y, z) -> fillcolorfunction(j, x, y, z)));

   // TODO TODO TODO:
   // Reacting to attrib[Closed] at the library level is BROKEN!
   // Closed can be changed interactively and NEVER requires recalculation!
   if attrib[Closed] then
      points:= [[0, 0, cx, cy, cz]].points;
   end_if;

   return(out::writePoly3d(attrib, table("Filled" = FALSE), points,
     ((i, j, x, y, z) -> linecolorfunction(j, x, y, z))));
  end_proc:
//-----------------------------------------------------------------------------------
