// 

/*
 plot::Bars2d -- plot statistical data, comparing groups

*/

plot::createPlotDomain("Bars2d",
                       "graphical primitive for bar plots",
                       2,
             	       [LinesVisible, LineStyle, LineWidth, LineColor, Color,
                        PointSize, PointStyle, Filled, AntiAliased,
                        Colors,
                        [FillPatterns, ["Optional", NIL],
                          [["Style", "Surface"], "ExprSeq", FAIL,
                            "Fill patterns for the bars.", FALSE]],
                        [Data, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                         "Data to plot.", TRUE]],
                        [DrawMode, ["Optional", Vertical],
                          ["Style", "Prop", "TitleOrientation",
                           "Orientation of bars/boxes", TRUE]],
                        [BarStyle, ["Optional", Boxes],
                          [["Style", "Surface"], "Prop", 
                          {Boxes, Lines, Points, LinesPoints},
                          "Type of bars.", TRUE]],
                        [BarWidths, ["Optional", NIL],
                          [["Style", "Surface"], "ExprSeq", FAIL,
                          "List of widths for the bars", TRUE]],
                        [BarCenters, ["Optional", NIL],
                          [["Style", "Surface"], "ExprSeq", FAIL,
                          "List of centers for the bars", TRUE]],
                        [Shadows, ["Optional", FALSE],
                          [["Style", "Surface"], "Prop", "Bool",
                          "Shadows behind bars?", TRUE]],
                        [GroupStyle, ["Optional", MultipleBars],
			 ["Style", "Prop", 
			  {MultipleBars, SingleBars},
			  "Style of grouped bars.", TRUE]]
                       ]):
//--------------------------------------------------------
plot::Bars2d::styleSheet := table(LinesVisible = TRUE,
                                  LineColor = RGB::Black,
                                  AntiAliased = FALSE,
                                  FillPatterns = [Solid],
                                  Colors = RGB::ColorList[1..20],
                                  BarWidths = [[1.0]]
                              //  BarCenters = [[0.0]] // Do not(!) set a default
                                 ):
//--------------------------------------------------------
plot::Bars2d::hints := {AxesInFront = TRUE, Axes = Frame, XAxisVisible = FALSE}:
//--------------------------------------------------------
plot::Bars2d::new :=
  proc()
    local object, other, check;
  begin
    object := dom::checkArgs([], args());
    
    other := object::other;
    if nops(other) > 0 then
      if testtype(other[1], Type::ListOf(Type::Arithmetical)) = TRUE then
         other[1] := [other[1]];
         while nops(other) > 1 and
               testtype(other[2], Type::ListOf(Type::Arithmetical)) = TRUE do
           other[1] := other[1].[other[2]];
           delete other[2];
         end_while;
      end_if;
      if nops(other) > 1 then
         error("unexpected argument ".expr2text(other[2]));
      end_if; 
      check := dom::changeNotifier(object, "Data" = other[1]);
      if domtype(check) = DOM_STRING then
         error(check);
      end_if;
    end_if;
    
    if object::FillPatterns <> FAIL then
      check := dom::changeNotifier(object, "FillPatterns"=object::FillPatterns);
      if domtype(check) = DOM_STRING then
         error(check);
      end_if;
    end_if;
    if object::BarCenters <> FAIL then
      check := dom::changeNotifier(object, "BarCenters"=object::BarCenters);
      if domtype(check) = DOM_STRING then
         error(check);
      end_if;
    end_if;
    if object::BarWidths <> FAIL then
      check := dom::changeNotifier(object, "BarWidths"=object::BarWidths);
      if domtype(check) = DOM_STRING then
         error(check);
      end_if;
    end_if;
    if object::Data <> FAIL then
      check := dom::changeNotifier(object, "Data"=object::Data);
      if domtype(check) = DOM_STRING then
         error(check);
      end_if;
    end_if;
    
    dom::checkObject(object);
  end_proc:
//--------------------------------------------------------
plot::Bars2d::print := b -> hold(plot::Bars2d)(b::Data):
//--------------------------------------------------------
plot::Bars2d::changeNotifier :=
  proc(b: plot::Bars2d, newval)
    local slotname, value, v, c, i, j;
  begin
    [slotname, value] := [op(newval)];
    if value <> NIL then
      case slotname
         of "FillPatterns" do
           if domtype(value) <> DOM_LIST or nops(value)<1 then
             return("attribute FillPatterns requires a non-empty list");
           end_if;
           if not {op(value)} subset plot::attributes[FillPattern][3] then
             return("invalid FillPatterns: ".expr2text(op({op(value)} minus plot::attributes[FillPattern][3])));
           end_if;
           break;
         of "BarCenters" do
           if domtype(float(value)) = DOM_FLOAT then
              value := [[value]];
           end_if;
           if domtype(value) <> DOM_LIST or nops(value)<1 then
              return("attribute BarCenters requires a non-empty list"):
           end_if;
           if domtype(value[1]) <> DOM_LIST then
              value:= [value]:
           end_if;
           dom::extslot(b, "BarCenters", value);
           return(FALSE);
         of "BarWidths" do
           if domtype(float(value)) = DOM_FLOAT then
              value := [[value]];
           end_if;
           if domtype(value) <> DOM_LIST or nops(value)<1 then
              return("attribute BarWidths requires a a non-empty list"):
           end_if;
           if domtype(value[1]) <> DOM_LIST then
              value:= [value]:
           end_if;
           dom::extslot(b, "BarWidths", value);
           return(FALSE);
         of "Data" do
           if value::dom::hasProp(Cat::Matrix) = TRUE then
             value := coerce(value, DOM_ARRAY);
           end_if;
           if domtype(value) = DOM_ARRAY then
             case op(value, [0,1])
               of 1 do
                  value := [coerce(value, DOM_LIST)];
                  break;
               of 2 do
                  value := [[value[i,j]$j=op(value, [0,3])]$i=op(value, [0,2])];
                  break;
               otherwise
                  return("bad array dimension for Data");
             end_case;
           end_if;
           if testtype(value, Type::ListOf(Type::Arithmetical))=TRUE then
             value := [value];
           end_if;
           if domtype(value) <> DOM_LIST then
             return("wrong type of value for Data");
           end_if;
           if nops(map({op(value)}, nops)) > 1 then
             return("Data must be list of lists of same length");
           end_if;
           for v in value do
             c := plot::checkList(dom, "Value", v);
             if c::dom = DOM_STRING then
               return("Invalid entry in Data: ".expr2text(v));
             end_if;
           end_for;
           dom::extslot(b, "Data", value);
           // hint
           return(FALSE);
      end_case;
    end_if;
    return(TRUE);
  end_proc:
//--------------------------------------------------------
plot::Bars2d::doPlotStatic :=
  proc(out, b: plot::Bars2d, attributes, inheritedAttributes)
    local groups, colors, patterns,
           i, j, k, n, pattern, color, values, swap, 
           doDraw, barwidth,
           shadowwidth, shadowheight, allvalues,
           barsep, groupsep, shadowcolor,
           bbs, centers, widths, lastcenters, xmin, xmax;
  begin
    groups   := attributes[Data];
    patterns := attributes[FillPatterns];
    colors   := attributes[Colors];
    shadowcolor := RGB::LightGrey;
    barwidth  := attributes[BarWidths][1][1];

    if attributes[Shadows] = TRUE and 
       attributes[BarStyle] = Boxes then
         allvalues := {op(map(groups, op))};
         shadowwidth  := 0.10 * barwidth;
         if attributes[GroupStyle] = SingleBars then
            shadowheight := 0.10 * 
                         (float(max(_plus(groups[k][j] $ k=1..nops(groups)) $ j = 1..nops(groups[1])))
                         / nops(groups[1]))
         else
            shadowheight := 0.10 * (max(0, op(allvalues)) - min(0, op(allvalues))) / 
                            (nops(groups) * nops(groups[1]))
         end_if;
         barsep      := 0.05 * barwidth;
         // we usually have an aspect ratio of approx 2/3
         if attributes[DrawMode] = Horizontal then
            shadowwidth := 1.5*shadowwidth;
            shadowheight := 0.66*shadowheight;
         end_if;
    else
         shadowwidth  := 0.0;
         shadowheight := 0.0;
         barsep       := 0.0;
    end_if;
    if nops(groups) = 1 then
         groupsep:= 0.0:
    else groupsep:= 0.3*barwidth;
    end_if:
    
    if attributes[DrawMode] = Horizontal then
         swap := (x, y) -> (float(y), float(x));
    else swap:= (x, y) -> (float(x), float(y));
    end_if;

    bbs := barwidth + barsep + shadowwidth:
    //---------------------------------
    //---------------------------------
    // Widths
    //---------------------------------
    //---------------------------------
    if contains(attributes, BarWidths) then
         widths:= float(attributes[BarWidths]);
    else widths:= [[]]; // default values are inserted below
    end_if;
    //--------------------------------
    // set default widths if necessary
    //--------------------------------
    for i from 1 to nops(groups) do
        if i > nops(widths) then
           widths:= widths.[[barwidth $ nops(groups[i])]];
        end_if;
        if nops(widths[i]) < nops(groups[i]) then
           widths[i]:= widths[i].[barwidth $ nops(groups[i]) - nops(widths[i])];
        end_if;
    end_for:
    //---------------------------------
    //---------------------------------
    // Centers
    //---------------------------------
    //---------------------------------
    if contains(attributes, BarCenters) then
         centers:= float(attributes[BarCenters]);
    else centers:= [[]]; // default values are inserted below
    end_if:
    //---------------------------------
    // set default centers if necessary
    //---------------------------------
    if attributes[GroupStyle] = SingleBars then
       n := 1
    else
       n := nops(groups)
    end;
    if nops(centers) = 1 and nops(centers[1]) = 0 then
       centers := [ [(j-1)*(n*bbs + groupsep)+(i-1)*bbs 
                    $ j = 1..nops(groups[i])
                  ] $ i = 1..n];
    end_if;
    for i from 1 to n do
        if i > nops(centers) then
           if i = 1 then
              lastcenters:= [(j-1)*(n*bbs + groupsep) $ $ j = 1..nops(groups[i])]:
           else
              lastcenters:= centers[i - 1]
           end_if;
           centers:= centers.[map(lastcenters, _plus, bbs)];
        end_if;
        if nops(centers[i]) < nops(groups[i]) then
           centers[i]:= centers[i].[centers[i][nops(centers[i])] + j*(n*bbs + groupsep)
                         $ j = 1 .. nops(groups[i]) - nops(centers[i])];
        end_if;
    end_for:
    //---------------------------------
    // find ViewingBox
    //---------------------------------
    xmin:= centers[1][1]:
    xmax:= centers[1][1]:
    for i from 1 to n do
     for j from 1 to nops(groups[i]) do
         xmin:= min(xmin, centers[i][j] - widths[i][j]/2):
         xmax:= max(xmax, centers[i][j] + widths[i][j]/2 + shadowwidth):
     end_for;
    end_for:
    //---------------------------------
    // prepare plotting
    //---------------------------------
    
    // doDraw(<nr of group>, <nr in group>, <height>, <pattern>, <color>)

    if attributes[GroupStyle] = SingleBars then

       if has(map(map(groups,op), bool@_less, 0), TRUE) then
          error("GroupStyle = SingleBars: All data must be >= 0")
       end_if;

    doDraw := proc(i, j, v, pattern, color)
                  local c, k, r, l, vi;
               begin
                  c := centers[1][j];
                  l := c - widths[i][j]/2;
                  r := c + widths[i][j]/2;
                  vi := _plus(groups[k][j] $ k=1..i-1);
                  if attributes[Shadows] then
                    if v > 0 then
                      out::writePoly2d( attributes, table( "Closed"=TRUE,
                                              "FillPattern"=Solid,
                                              "FillColor"=shadowcolor,
                                              "LinesVisible" = FALSE ),
                                              map([[r,vi],[r,vi+v],[l+shadowwidth, vi+v],[l+shadowwidth, vi+v+shadowheight],[r+shadowwidth, vi+v+shadowheight],[r+shadowwidth, vi]],X->[swap(op(X))]) );
                    else
                      out::writePoly2d( attributes, table( "Closed"=TRUE,
                                              "FillPattern"=Solid,
                                              "FillColor"=shadowcolor,
                                              "LinesVisible" = FALSE ), map([[l+shadowwidth, shadowheight], [r+shadowwidth, shadowheight], [r+shadowwidth, min(0, vi+v+shadowheight)], [r, min(0, vi+v+shadowheight)], [r, vi], [l+shadowwidth, vi]],X->[swap(op(X))]) );
                    end_if;
                  end_if;
                    out::writePoly2d( attributes, table( "Closed"=TRUE,
                                            "FillPattern"=pattern,
                                            "FillColor"=color ), map([[l,vi],[l,vi+v],[r,vi+v],[r,vi]],X->[swap(op(X))]) ):
               end_proc:
    
    if attributes[BarStyle] = Lines then
      doDraw := proc(i, j, v, pattern, color)
                    local c,k, vi;
                  begin
                    c := centers[1][j];
                    out::writePoly2d( attributes, table("Closed"=FALSE,
                                              "LinesVisible"=TRUE,
                                              "LineColor"=color ), map([[c,vi],[c,vi+v]],X->[swap(op(X))]) ):
                  end_proc:
    elif attributes[BarStyle] = Points then
      doDraw := proc(i, j, v, pattern, color)
                    local c, k, vi;
                  begin
                    c := centers[1][j];
                    vi := _plus(groups[k][j] $ k=1..i-1);
                    out::writePoint2d( attributes, table("PointsVisible"=TRUE,
                                              "PointColor"=color), swap(float(c),float(vi+v)) ):
                  end_proc:
    elif attributes[BarStyle] = LinesPoints then
      doDraw := proc(i, j, v, pattern, color)
                    local c, vi;
                  begin
                    c := centers[1][j];
                    vi := _plus(groups[k][j] $ k=1..i-1);
                    out::writePoly2d( attributes, table("Closed"=FALSE,
                                              "LinesVisible"=TRUE,
                                              "LineColor"=color ), map([[c,vi],[c,vi+v]],X->[swap(op(X))]) ):
                    out::writePoint2d( attributes, table("PointsVisible"=TRUE,
                                              "PointColor"=color), swap(float(c),float(vi+v)) ):
                  end_proc:
    end_if;

    else // GroupStyle = MultipleBars

    doDraw := proc(i, j, v, pattern, color)
                  local c, r, l;
               begin
                  c := centers[i][j];
                  l := c - widths[i][j]/2;
                  r := c + widths[i][j]/2;
                  if attributes[Shadows] then
                    if v > 0 then
                      out::writePoly2d( attributes, table( "Closed"=TRUE,
                                              "FillPattern"=Solid,
                                              "FillColor"=shadowcolor,
                                              "LinesVisible" = FALSE ),
                                              map([[r,0],[r,v],[l+shadowwidth, v],[l+shadowwidth, v+shadowheight],[r+shadowwidth, v+shadowheight],[r+shadowwidth, 0]],X->[swap(op(X))]) );
                    else
                      out::writePoly2d( attributes, table( "Closed"=TRUE,
                                              "FillPattern"=Solid,
                                              "FillColor"=shadowcolor,
                                              "LinesVisible" = FALSE ), map([[l+shadowwidth, shadowheight], [r+shadowwidth, shadowheight], [r+shadowwidth, min(0, v+shadowheight)], [r, min(0, v+shadowheight)], [r, 0], [l+shadowwidth, 0]],X->[swap(op(X))]) );
                    end_if;
                  end_if;
                    out::writePoly2d( attributes, table( "Closed"=TRUE,
                                            "FillPattern"=pattern,
                                            "FillColor"=color ), map([[l,0],[l,v],[r,v],[r,0]],X->[swap(op(X))]) ):
               end_proc:
    
    if attributes[BarStyle] = Lines then
      doDraw := proc(i, j, v, pattern, color)
                    local c;
                  begin
                    c := centers[i][j];
                    out::writePoly2d( attributes, table("Closed"=FALSE,
                                              "LinesVisible"=TRUE,
                                              "LineColor"=color ), map([[c,0],[c,v]],X->[swap(op(X))]) ):
                  end_proc:
    elif attributes[BarStyle] = Points then
      doDraw := proc(i, j, v, pattern, color)
                    local c;
                  begin
                    c := centers[i][j];
                    out::writePoint2d( attributes, table("PointsVisible"=TRUE,
                                              "PointColor"=color), swap(float(c),float(v)) ):
                  end_proc:
    elif attributes[BarStyle] = LinesPoints then
      doDraw := proc(i, j, v, pattern, color)
                    local c;
                  begin
                    c := centers[i][j];
                    out::writePoly2d( attributes, table("Closed"=FALSE,
                                              "LinesVisible"=TRUE,
                                              "LineColor"=color ), map([[c,0],[c,v]],X->[swap(op(X))]) ):
                    out::writePoint2d( attributes, table("PointsVisible"=TRUE,
                                              "PointColor"=color), swap(float(c),float(v)) ):
                  end_proc:
    end_if; // BarStyle

    end_if; // GroupStyle
    
    //--------------------------------
    // doDraw
    //--------------------------------
    for i from 1 to nops(groups) do
      pattern := patterns[modp(i-1, nops(patterns))+1];
      color := float(colors[modp(i-1, nops(colors))+1]);
      values := float(groups[i]);
      for j from 1 to nops(values) do
        doDraw(i, j, values[j], pattern, color);
      end_for;
    end_for;
    //--------------------------------
    // return ViewingBox
    //--------------------------------
    if attributes[GroupStyle] = SingleBars then

    [swap(xmin..xmax,
          0 .. float(max(_plus(groups[k][j] $ k=1..nops(groups)) $ j = 1..nops(groups[1])))
               +shadowheight)]
    else

    [swap(xmin..xmax,
          float(min(0, op(map(map(groups, op), `+`, -shadowheight))))
       .. float(max(0, op(map(map(groups, op), `+`,  shadowheight)))))];

    end_if;

  end_proc:
