//      

/* -----------------------------------------------------------
    Circle3d -- The graphical primitive for 3D circles

    Syntax:
    Circle3d(radius, <center, normal> <, op1, op2, ...>)

    radius                : expression
    normal, center        : list of 3 expressions
                            The defaults are center = [0, 0, 0], normal = [0,0,1]
    op1, op2, ...         : options of the form 'option = value'

    Example:
    >> plot::Circle3d(2, [0, 0, 1])
    >> plot::Circle3d(2, [0, 0, 1], [1, 1, 0])
----------------------------------------------------------------*/ 

plot::createPlotDomain("Circle3d",
                       "graphical primitive for 3D circles"):

plot::Circle3d::styleSheet := table(
      Radius  = 1,
      CenterX = 0, CenterY = 0, CenterZ = 0,
      NormalX = 0, NormalY = 0, NormalZ = 1,
      Filled  = FALSE,
   // LineColor = RGB::Black.[0.25],  // primarily, this is a line object!!
      FillColor = RGB::LightBlue,
      LineColorDirectionY=0
     ):

plot::Circle3d::setPrimaryColor(LineColor):
//------------------------------------------------------------------------
// Hints for parents:
plot::Circle3d::hints := {Scaling = Constrained}:
//-------------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//-------------------------------------------------------------------
plot::Circle3d::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;

    if testargs() then
      if nops(other) >= 1 and not testtype(other[1], Type::Arithmetical) then
        error("1st argument: expecting an expression (the radius)")
      end_if;
      if nops(other) >= 2 and not testtype(other[2],
                      Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[2])::dom::hasProp(Cat::Matrix)=TRUE  then
        error("2nd argument: expecting a list of 3 expressions (the center)")
      end_if;
      if nops(other) >= 3 and not testtype(other[3],
                     Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[3])::dom::hasProp(Cat::Matrix)=TRUE  then
        error("3rd argument: expecting a list of 3 expressions (the normal)")
      end_if;
      if nops(other) > 3 then
        error("unexpected argument: ".expr2text(other[4]))
      end_if;
    end_if;
   
    if nops(other) <> 0 then
      object::Radius := other[1];
      if nops(other) > 1 then
         object::Center := other[2];
      end_if;
      if nops(other) = 3 then
         object::Normal := other[3];
      end_if;
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

//-------------------------------------------------------------------
plot::Circle3d::print :=
  obj -> hold(plot::Circle3d)(obj::Radius, obj::Center, obj::Normal):

//-------------------------------------------------------------------
plot::Circle3d::doPlotStatic:=
  proc(out, obj, attrib, inherited)
    local radius, b, n;
  begin
    radius := float(attrib[Radius]);
    b:=  float([attrib[CenterX], attrib[CenterY], attrib[CenterZ]]);
    n:=  float([attrib[NormalX], attrib[NormalY], attrib[NormalZ]]);
    // If the normal of the circle is the 0 vector, the circle
    // is not properly defined. Choose a default direction:
    if iszero(n[1]^2 + n[2]^2 + n[3]^2) then
       n := float([0, 0, 1]);
    end_if;
    return(out::writeCircle3d(attrib, table(),
      b[1], b[2], b[3], n[1], n[2], n[3], radius));
  end_proc:
