/* -----------------------------------------------------------
    Cone -- The graphical primitive for 3D cones

    Syntax:

    Cone(baseRadius, baseCenter, <, topRadius = 0>, topCenter <, op1, op2, ...>)

    baseRadius, topRadius : expressions
    baseCenter, topCenter : list of 3 expressions
    op1, op2, ...         : options of the form 'option = value'

    Example:
    >> plot::Cone(1, [0, 0, 0], [0, 0, 3])
    >> plot::Cone(1, [0, 0, 0], 0.2, [0, 0, 3])
----------------------------------------------------------------*/ 

plot::createPlotDomain("Cone",
                       "graphical primitive for 3D cones"):

//-----------------------------------------------------------------------------
plot::Cone::styleSheet:= table(
      LinesVisible = TRUE,
      LineColor    = RGB::Black.[0.25],
      FillColor = RGB::LightBlue,
      BaseRadius   = 1,
      BaseX        = 0,
      BaseY        = 0,
      BaseZ        = 0,
      TopRadius    = 0,
      TopX         = 0,
      TopY         = 0,
      TopZ         = 1,
      LineColorDirectionY=0
):

//------------------------------------------------------------------------
// Hints for parents:
plot::Cone::hints := {
      Scaling = Constrained
}:

// methods yet to be implemented:  convert, convert_to, expr

//-----------------------------------------------------------------------------
plot::Cone::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;
    
    if testargs() and nops(other) > 0 then      
      if nops(other) >=1 and not testtype(other[1], Type::Arithmetical) then
        error("1st argument: expecting an expression (the base radius)")
      end_if;
      
      if nops(other) >= 2 and
        not testtype(other[2],
                     Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
        error("2nd argument: expecting a list of 3 expressions (the base center)")
      end_if;

      if nops(other) >= 3 and
        (not testtype(other[3], Type::Arithmetical)) and
        (not testtype(other[3], Type::ListOf(Type::Arithmetical, 3, 3))) and
        not (other[3])::dom::hasProp(Cat::Matrix)=TRUE then
        error("3rd argument: expecting an expression (the top radius) ".
              "or a list of 3 expressions (the top center)")
      end_if;
      if nops(other) >= 4 and
        (not testtype(other[4], Type::ListOf(Type::Arithmetical, 3, 3))) and
        not (other[4])::dom::hasProp(Cat::Matrix)=TRUE then
        error("4th argument: expecting a list of 3 expressions (the top center)")
      end_if;
      if nops(other) > 4 then
        error("unexpected argument: ".expr2text(other[5]))
      end_if;
    end_if;
    
    if nops(other) > 0 then
      object::BaseRadius   := other[1];
      if nops(other) > 1 then
        object::Base       := other[2];
        if nops(other) = 3 then
          object::Top      := other[3];
        elif nops(other) > 2 then
          object::TopRadius:= other[3];
		  if nops(other) = 4 then
            object::Top    := other[4];
		  end_if;
		end_if;
      end_if;
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

plot::Cone::print :=
  obj -> hold(plot::Cone)(
                          obj::BaseRadius, 
                          obj::Base, 
                          obj::TopRadius,
                          obj::Top
                         ):

//--------------------------------------------------------------
plot::Cone::doPlotStatic:= proc(out, obj, attrib, inherited)
begin
  return(out::writeCone(attrib, table(), float(attrib[BaseX]), float(attrib[BaseY]), float(attrib[BaseZ]),
    float(attrib[TopX]), float(attrib[TopY]), float(attrib[TopZ]),
    float(attrib[BaseRadius]), float(attrib[TopRadius])));
end_proc:
