/* -----------------------------------------------------------
    Cylinder -- The graphical primitive for 3D Cylinders

Call(s): plot::Cylinder(radius, baseCenter, topCenter <, op1, op2, ...>)

Parameters:
    radius      - expression
    baseCenter  - list of 3 expressions
    topCenter : - list of 3 expressions

Example:
    >> plot::Cylinder(1, [0, 0, 0], [0, 0, 3])
----------------------------------------------------------------*/ 

plot::createPlotDomain("Cylinder",
                       "graphical primitive for 3D cylinders",
                       3,
                       [Radius,
                        Base, BaseX, BaseY, BaseZ, 
                        Top, TopX, TopY, TopZ,
                        LineColor, LineWidth, LineStyle, LinesVisible,
                        Filled, FillColor
                       ]):

//------------------------------------------------------------------------
plot::Cylinder::styleSheet:= table(
        LinesVisible = TRUE,
        LineColor    = RGB::Black.[0.25],
        FillColor = RGB::LightBlue,
        Radius       = 1,
        BaseX        = 0,
        BaseY        = 0,
        BaseZ        = 0,
        TopX         = 0,
        TopY         = 0,
        TopZ         = 1
):
//------------------------------------------------------------------------
// Hints for parents:
plot::Cylinder::hints := {Scaling = Constrained}:
//------------------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//------------------------------------------------------------------------
plot::Cylinder::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;
    
    if nops(other) > 0 then
      if testargs() then
        if nops(other) >=1 and not testtype(other[1], Type::Arithmetical) then
          error("1st argument: expecting an expression (the radius)")
        end_if;
        if nops(other) >= 2 and not testtype(other[2],
                       Type::ListOf(Type::Arithmetical, 3, 3)) then 
          if not (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
            error("2nd argument: expecting a list of 3 expressions or a matrix (the base center)")             
          else 
            other[2]:= [op(other[2])];
          end_if;
        end_if;
        if nops(other) >= 3 and not testtype(other[3],
                       Type::ListOf(Type::Arithmetical, 3, 3)) then 
          if not (other[3])::dom::hasProp(Cat::Matrix)=TRUE then
            error("3rd argument: expecting a list of 3 expressions or a matrix (the top center)")
          else 
            other[3]:= [op(other[3])];
          end_if;
        end_if;
        if nops(other) >= 4 then
          error("unexpected argument: ".expr2text(other[4]))
        end_if;
      end_if;
      object::Radius := other[1];
      if nops(other) > 1 then
        object::Base   := other[2];
      end_if;
      if nops(other) > 2 then
        object::Top    := other[3];
      end_if;
    end_if;
    // semantically check for validity
    dom::checkObject(object);
end_proc:

//--------------------------------------------------------------
plot::Cylinder::print :=
  obj -> hold(plot::Cylinder)(obj::Radius, obj::Base, obj::Top):

//--------------------------------------------------------------
plot::Cylinder::doPlotStatic:= proc(out, obj, attrib, inheritedAttributes)
local r, b, t, n, nn, tmp,
      xmin, xmax, ymin, ymax, zmin, zmax;
begin
    r:= float(attrib[Radius]);
    b:= float([attrib[BaseX], attrib[BaseY], attrib[BaseZ]]);
    t:= float([attrib[TopX], attrib[TopY], attrib[TopZ]]);
    n:= [t[1]- b[1], t[2]- b[2], t[3]- b[3]]; // the direction of the cylinder
    nn:= n[1]^2 + n[2]^2 + n[3]^2;
    // If Base = Top, the normal of the disc is not defined.
    // Choose a default direction:
    if iszero(nn) then 
       n := float([0, 0, 1]);
       nn:= float(1); 
    end_if;
    out::writeCone(attrib, table(), b[1], b[2], b[3], t[1], t[2], t[3], r, r);
    out::writeCircle3d(attrib, table(), b[1], b[2], b[3], -n[1], -n[2], -n[3], r);
    out::writeCircle3d(attrib, table(), t[1], t[2], t[3], n[1], n[2], n[3], r);
    //--------------------------------------------------------------
    // Compute the viewing box. The disc of radius r centered at the
    // vector c with normal vector n has the viewing box components
    // vbox[k] = c[k]-r*sqrt(1-n[k]^2/|n|^2) .. c[k]+r*sqrt(1-n[k]^2/|n|^2)
    //--------------------------------------------------------------
    tmp:= sqrt((n[2]^2+n[3]^2)/nn); 
    xmin:= min(b[1]-r*tmp, t[1]-r*tmp); 
    xmax:= max(b[1]+r*tmp, t[1]+r*tmp);
    tmp:= sqrt((n[1]^2+n[3]^2)/nn);
    ymin:= min(b[2]-r*tmp, t[2]-r*tmp); 
    ymax:= max(b[2]+r*tmp, t[2]+r*tmp);
    tmp:= sqrt((n[1]^2+n[2]^2)/nn);
    zmin:= min(b[3]-r*tmp, t[3]-r*tmp); 
    zmax:= max(b[3]+r*tmp, t[3]+r*tmp);
    return([xmin .. xmax, ymin .. ymax, zmin .. zmax]);
end_proc:
