/*-----------------------------------------------------------------------
plot::Density - 2D density plot

Call:  
  plot::Density(f(x, y <, a>), 
                x = x0(a)..x1(a),
                y = y0(a)..y1(a), 
               <a = a0..a1>, 
               <XMesh = n>, 
               <YMesh = m>, 
               <attributes> 
                )
  plot::Density(densitylist,  < x = x0..x1, y = y0..y1, a = a0..a1>, < attributes>)
  plot::Density(densityarray, < x = x0..x1, y = y0..y1, a = a0..a1>,  <attributes>)

Parameters:
    f(x, y, a)  : a function expression or a procedure
    densitylist : a list of lists of n*m real numerical values
    densityarray: an array(1..n, 1..m) of real numerical values
    x, y        : the coordinates: identifiers or indexed identifiers
    x0(a),x1(a),  borders of the coordinate range: univariate
    y0(a),y1(a) : expressions of the animation parameter
    a           : the animation parameter: identifier or indexed identifier
    n           : number of 'density pixels' in x direction: pos integer
    m           : number of 'density pixels' in y direction: pos integer
    attributes  : further attributes such as FillColor, FillColor2 etc.

Details:
  - The object consists of n x m rectangles. 
  - The rectangle with lower left corner (x, y) has a flat color 
             (fmax - f(x, y))/(fmax - fmin) * FillColor
           + (f(x, y) - fmin)/(fmax - fmin) * FillColor2
    (this is a 'Height' colouring scheme from FillColor to FillColor2)
    If a FillColorFunction = mycolor is given, the flat color is
    determined by mycolor(x, y, f(x, y, <a>), <a>).

Examples:
 >> plot::Density(sin(x^2 + y^2), x = -2..2, y = -2..2, Mesh = [100, 100])
--------------------------------------------------------------------------*/

plot::createPlotDomain(
   "Density",
   "graphical primitive for 2D density plots",
   2,  // Dimension
   [ // declare the attributes visible in the object inspector
    [DensityFunction,  // name of the new attribute
     ["Optional", NIL],// [type, default value]
     ["Definition",    // where to be found in the inspector
      "Expr",          // type according to DTD 
      plot::elementOptFunctionExpr, // activate automatic conversions
                                    // Expr -> Procedure, add floats etc
      "The density function.",  // inspector info entry
      FALSE            // recalculation flag
     ] 
    ],
    [DensityData,      // name of the new attribute
     ["Optional", NIL],// [type, default value]
     ["Definition",    // where to be found in the inspector
      "ExprSeq",       // type according to DTD 
      FAIL,            // no automatic conversions
      "The density data.", // inspector info entry
      FALSE            // recalculation flag
     ]
    ],
    XName, XMin, XMax, XMesh, Mesh,
    YName, YMin, YMax, YMesh, 
    LinesVisible, LineWidth, LineStyle, AntiAliased,
    LineColor, Color,
    FillColor, FillColor2, FillColorFunction, FillColorType
    ]):

plot::Density::styleSheet:= table(
                                  LinesVisible = FALSE,
                                  AntiAliased = FALSE,
                                  XMesh = 25,
                                  YMesh = 25
):
plot::Density::hints := {XAxisTitle, YAxisTitle, 
                         AxesInFront = TRUE,
                         GridInFront = TRUE}:
plot::Density::setHint:=
/*
  object -> plot::setHint(object,
                          XAxisTitle = expr2text(object::XName),
                          YAxisTitle = expr2text(object::YName)
                         ):
*/
  object -> (if object::XName <> FAIL and object::XAxisTitle = FAIL then
              plot::setHint(object, XAxisTitle = expr2text(object::XName)):
             end_if:
             if object::YName <> FAIL and object::YAxisTitle = FAIL then
              plot::setHint(object, YAxisTitle = expr2text(object::YName)):
             end_if):

//--------------------------------------------------------
plot::Density::print := proc(obj)
begin
  if obj::DensityFunction <> FAIL then
     return(hold(plot::Density)(
               obj::DensityFunction,
               obj::XName = obj::XMin .. obj::XMax,
               obj::YName = obj::YMin .. obj::YMax 
        //     XMesh = obj::XMesh,
        //     YMesh = obj::YMesh
           ));
  end_if:
  return(hold(plot::Density)(
            ["..."],
            obj::XName = obj::XMin .. obj::XMax,
            obj::YName = obj::YMin .. obj::YMax
       //   XMesh = obj::XMesh,
       //   YMesh = obj::YMesh
           ));
end_proc:

//----------------------------------------------------------------
// changeNotifier: eq is "AttributeName" (as string) = value.
// This method is called whenever the object is changed via
// a slot call.
// Here, the relevant attribute to be changed is DensityData
//----------------------------------------------------------------
plot::Density::changeNotifier := proc(object, eq)
local attributename, data, s, i, j;
begin
    attributename:= op(eq, 1);
    if attributename = "DensityFunction" then
       if op(eq, 2) = FAIL then // allow destruction of DensityFunction
          object::dom::extslot(object, "DensityFunction", FAIL);
          return(FALSE);
       end_if:
       object::dom::extslot(object, "DensityFunction", op(eq, 2));
       object::dom::extslot(object, "DensityData", FAIL);
       return(FALSE); // notify extslot that the assignment was already
                      // done inside the changeNotifier method
    elif attributename = "DensityData" then
      data := op(eq, 2);
      if data = FAIL then // allow destruction of DensityData
         object::dom::extslot(object, "DensityData", FAIL);
         return(FALSE);
      end_if:
      if data::dom::hasProp(Cat::Matrix)=TRUE then
         data:= expr(data);
      end_if;
      case domtype(data) 
      of DOM_LIST do // input via a list of lists
         object::dom::extslot(object, "YMesh", nops(data));
         s:= {op(data)}:
         if map(s, domtype) <> {DOM_LIST} then
           error("expecting a list of lists of density values");
         end_if;
         s:= map(s, nops):
         if nops(s) <> 1 then
            error("all sublists (rows of the density image) must have the same length");
         end_if:
         object::dom::extslot(object, "XMesh", op(s));

         if object::XMin = FAIL then
            object::dom::extslot(object, "XMin", 0):
         end_if;
         if object::XMax = FAIL then
            object::dom::extslot(object, "XMax", object::XMesh);
         end_if;
         if object::YMin = FAIL then
            object::dom::extslot(object, "YMin", 0):
         end_if;
         if object::YMax = FAIL then
            object::dom::extslot(object, "YMax", object::YMesh);
         end_if;

         // Quick check: just look at the first data element
         data:= map(data, float):
         if not testtype(data[1][1], Type::Arithmetical) then
              error("expecting a list of lists of density values. Got the ".
                    "following first entry in the first sublist: ".expr2text(data[1][1]));
         end_if;

         // DensityData are the 'official' data seen in the inspector
         object::dom::extslot(object, "DensityData", data);
         object::dom::extslot(object, "DensityFunction", FAIL);
         break;
      //---------------------------------------------------------------------
      of DOM_ARRAY do // input via an array
         if op(data, [0, 1]) <> 2 then
             error("First argument: expecting a two dimensional array ".
                   "of density values")
         end_if;
         if object::XMin = FAIL then
            object::dom::extslot(object, "XMin", op(data, [0, 3, 1]) - 1);
         end_if;
         if object::XMax = FAIL then
            object::dom::extslot(object, "XMax", op(data, [0, 3, 2]));
         end_if;
         if object::YMin = FAIL then
            object::dom::extslot(object, "YMin", op(data, [0, 2, 1]) - 1);
         end_if;
         if object::YMax = FAIL then
            object::dom::extslot(object, "YMax", op(data, [0, 2, 2]));
         end_if;
         object::dom::extslot(object, "XMesh", 
                              1 + op(data, [0, 3, 2]) - op(data, [0, 3, 1]));
         object::dom::extslot(object, "YMesh", 
                              1 + op(data, [0, 2, 2]) - op(data, [0, 2, 1]));
         // Quick check: just look at the first data element
         if not testtype(op(data,1), Type::Arithmetical) then
               return("Expecting an array of density values. ".
                      "Got as first density value: ".expr2text(op(data, 1))):
         end_if:
         object::dom::extslot(object, "DensityData",
                 [[float(data[i,j]) $ j = op(data, [0, 3, 1]) .. op(data, [0, 3, 2])]
                                    $ i = op(data, [0, 2, 1]) .. op(data, [0, 2, 2])]); 
         object::dom::extslot(object, "DensityFunction", FAIL);
         break;
      //---------------------------------------------------------------------
      otherwise
         return("Expecting a list of lists or an array of density values");
      end_case;
      return(FALSE); // notify extslot that the assignment was already
                     // done inside the changeNotifier method
    else 
      // any attributes other than DensityData are OK. 
      // Let extop do the assignment
      return(TRUE);
    end_if;
end_proc:

//----------------------------------------------------------------
plot::Density::new:= proc()
local object, other;
begin
  object := dom::checkArgs([["X"], ["Y"]], args());
  if (object::XRange = FAIL and object::YRange <> FAIL)  or
     (object::YRange = FAIL and object::XRange <> FAIL)  then
     error("expecting ranges 'x = xmin .. xmax' and 'y = ymin .. ymax'");
  end_if;
  other:= object::other;
  if nops(other) <> 0 then
     if nops(other) > 1 then
        error("Unknown attribute: ".expr2text(other[2]));
     end_if;
     if object::XName = FAIL then
        object::XName:= hold(x); // ouch!
     end_if;
     if object::YName = FAIL then
        object::YName:= hold(y); // ouch!
     end_if;
     if (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
        other[1]:= expr(other[1]);
     end_if;
     case domtype(other[1])
     of DOM_LIST do
     of DOM_ARRAY do
        // changeNotifier does further type checking
        plot::Density::changeNotifier(object, "DensityData" = other[1]);
        break;
     otherwise // a density function is specified
        object::DensityFunction:= other[1];
     end_case;
  end_if;
  //--------------------------------
  // check consistency of the object
  //--------------------------------
  dom::setHint(object);
  dom::checkObject(object);
end_proc:

//----------------------------------------------------------------
plot::Density::doPlotStatic := proc(out, object, attributes, inherited)
local x, y, xmesh, ymesh, xmin, xmax, ymin, ymax, 
      hasfillcolorfunction, hasdensityfunction, hasdensitydata, 
      fillcolorfunction, fillcolorvalue,
      data, f, fvalue, i0, i, j0, j, densdata, colordata;
begin
    //---------------------------------
    // get the attributes
    //---------------------------------
    xmesh:= attributes[XMesh];
    ymesh:= attributes[YMesh];
    xmin := float(attributes[XMin]);
    xmax := float(attributes[XMax]);
    ymin := float(attributes[YMin]);
    ymax := float(attributes[YMax]);
    //---------------------------------
    // get the data
    //---------------------------------
    hasdensityfunction:= FALSE;
    hasdensitydata:= FALSE;
    if contains(attributes, DensityFunction) then
       f:= attributes[DensityFunction];
       hasdensityfunction:= TRUE;
    else
       data:= float(attributes[DensityData]);
       hasdensitydata:= TRUE;
    end_if:
    if contains(attributes, FillColorFunction) then
         fillcolorfunction := attributes[FillColorFunction];
         hasfillcolorfunction:= TRUE;
    else fillcolorfunction := null();
         hasfillcolorfunction:= FALSE;
    end_if;
    //---------------------------------
    // do the computation
    //---------------------------------
    // The following x, y values are the centers of the squares indexed
    // from i=1 to i=ymesh and j=1 to j=xmesh:
    x := array(1..xmesh, [((j-1/2)/xmesh)*xmax + ((xmesh-j+1/2)/xmesh)*xmin $ j=1..xmesh]);
    x:= map(x, float);
    y := array(1..ymesh, [((i-1/2)/ymesh)*ymax + ((ymesh-i+1/2)/ymesh)*ymin $ i=1..ymesh]);
    y:= map(y, float);
    if hasdensityfunction then 
         fvalue:= (i, j) -> f(x[j], y[i]);
    elif hasdensitydata and domtype(data) = DOM_LIST then
         fvalue:= (i, j) -> data[i][j]:
    elif hasdensitydata and domtype(data) = DOM_ARRAY then
         j0:= op(data, [0, 2, 1]) - 1;
         i0:= op(data, [0, 3, 1]) - 1;
         if j0 = 0 and i0 = 0 then
              fvalue:= (i, j) -> data[i, j];
         else fvalue:= (i, j) -> data[i0 + i, j0 + j];
         end_if;
    else error("expecting either a function expression for the density function, ".
               "or density values as a list of lists, an array or a matrix");
    end_if;
    //-------------------------------------------------------------
    // If no FillColorFunction = fillcolorfunction is specified,
    // write the density values into a DensityArray2d:
    //-------------------------------------------------------------
    if not hasfillcolorfunction then
      densdata := array(1..xmesh, 1..ymesh);
      (densdata[j, i] := fvalue(i, j)) $ j=1..xmesh $ i = 1..ymesh;
      out::writeDensityArray2d(attributes, table(), densdata, xmin, xmax, ymin, ymax);
    end_if;
    //-------------------------------------------------------------
    // If FillColorFunction = fillcolorfunction is specified,
    // write the color values into a ColorArray2d:
    //-------------------------------------------------------------
    if hasfillcolorfunction then
      colordata := array(1..ymesh, 1..xmesh);
      for i from 1 to ymesh do
        for j from 1 to xmesh do
          fillcolorvalue:= fillcolorfunction(x[j], y[i], fvalue(i, j));
          if not (domtype(fillcolorvalue) = DOM_LIST and 
                 (nops(fillcolorvalue) = 3 or 
                  nops(fillcolorvalue) = 4)) then
             error("The fillcolorfunction does not return an RGB or RGBa value. ".
                  "When called with the 3 arguments ".
                  "x = ".expr2text(x[j]).", ".
                  "y = ".expr2text(y[i]).", ".
                  "density function value = ".expr2text(fvalue(i, j)).
                  ", it produced ".expr2text(fillcolorvalue));
          end_if;
          colordata[i, j] := [fillcolorvalue];
        end_for;
      end_for;
      out::writeColorArray2d(attributes, table(), [op(colordata)], xmesh, ymesh, xmin, xmax, ymin, ymax);
    end_if;
    //-----------------------
    // return the viewing box
    //-----------------------
    return([xmin .. xmax, ymin .. ymax]);
end_proc:

//------------------------------------------------------------
