/* ----------------------------------------------------------------------------
plot::Dodecahedron -- graphical primitive of a dodecahedron

Call(s):
    plot::Dodecahedron( <args, ...> )

Parameters:
    -

Options:
    attr, ...   : plot options, i.e., equations Attribute = value, where
                  Attribute is one of the attributes listed below or a hint.

Related Domains:
    plot::Tetrhedron, plot::Hexahedron, plot::Octahedron, plot::Icosahedron,
    plot::Box, plot::Cone, plot::Cylinder, plot::Ellipsoid, plot::Sphere, plot::Rotate3d,

Example:
    >> plot( plot::Dodecahedron() );

Further Reading:
    Refer to Tetrahedron.mu

---------------------------------------------------------------------------- */

plot::createPlotDomain("Dodecahedron",
                       "graphical primitive for dodecahedrons",
                       3,
   [
    CenterX, CenterY, CenterZ, Center, Radius,
    LinesVisible, LineStyle, LineWidth,
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::Dodecahedron::styleSheet:= table(
      LinesVisible  = TRUE,
      LineColor     = RGB::Black.[0.25],
      Radius        = 1,
      CenterX       = 0,
      CenterY       = 0,
      CenterZ       = 0,
      LineColorDirectionY=0
):

//-----------------------------------------------------------------------------
plot::Dodecahedron::hints:= {
      Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::Dodecahedron::new:= proc()
local  object, other;
option escape;
begin
    // filter out all generic arguments
    object:= dom::checkArgs([],args());

    // process the special attributes in object::other
    other:= object::other;
    if nops(other) > 0 then
      error("unexpected argument: ".expr2text(other[1]));
    end_if;

    // check all attributes
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::Dodecahedron::print:= obj -> "plot::Dodecahedron(...)":

//-----------------------------------------------------------------------------
plot::Dodecahedron::doPlotStatic:= proc(out, object, attributes, inheritedAttributes)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local
    cx, cy, cz, r, p, q,
    t, triangles,
    fillcolorfunction,
    linecolorfunction, haslinecolorfunction,
    tn, tr, pe,
    entries;

begin
    // check for fill-color function
    if contains(attributes, FillColorFunction) then
      fillcolorfunction := float@(attributes[FillColorFunction]); // procedure
    else
      fillcolorfunction := () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else
      linecolorfunction:= () -> null();
      haslinecolorfunction:= FALSE;
    end_if;

    cx:= float(attributes[CenterX]);
    cy:= float(attributes[CenterY]);
    cz:= float(attributes[CenterZ]);
    r := float(attributes[Radius]);
    q := r*0.611; // hack: scale to embed into hexahedron
    p := float(q*(sqrt(5)-1)/2);

    t:= [
      [cx  +p,cy  +0,cz+q+p], [cx  -p,cy  +0,cz+q+p],
      [cx  +p,cy  +0,cz-q-p], [cx  -p,cy  +0,cz-q-p],
      [cx+q+p,cy  -p,cz  +0], [cx+q+p,cy  +p,cz  +0],
      [cx-q-p,cy  -p,cz  +0], [cx-q-p,cy  +p,cz  +0],
      [cx  +0,cy+q+p,cz  +p], [cx  +0,cy+q+p,cz  -p],
      [cx  +0,cy-q-p,cz  +p], [cx  +0,cy-q-p,cz  -p],
      [cx  +q,cy  +q,cz  +q], [cx  +q,cy  +q,cz  -q],
      [cx  +q,cy  -q,cz  +q], [cx  +q,cy  -q,cz  -q],
      [cx  -q,cy  +q,cz  +q], [cx  -q,cy  +q,cz  -q],
      [cx  -q,cy  -q,cz  +q], [cx  -q,cy  -q,cz  -q]
    ];

    triangles:= [
      [t[ 2], t[17], t[ 8],  t[ 7], t[19]],  //1
      [t[ 1], t[13], t[ 9],  t[17], t[ 2]],  //2
      [t[17], t[ 9], t[10],  t[18], t[ 8]],  //3
      [t[ 8], t[18], t[ 4],  t[20], t[ 7]],  //4
      [t[ 7], t[20], t[12],  t[11], t[19]],  //5
      [t[19], t[11], t[15],  t[ 1], t[ 2]],  //6
      [t[ 6], t[ 5], t[15],  t[ 1], t[13]],  //7
      [t[ 6], t[13], t[ 9],  t[10], t[14]],  //8
      [t[14], t[10], t[18],  t[ 4], t[ 3]],  //9
      [t[ 3], t[ 4], t[20],  t[12], t[16]],  //10
      [t[16], t[12], t[11],  t[15], t[ 5]],  //11
      [t[ 5], t[ 6], t[14],  t[ 3], t[16]]   //12
    ];

    tn:= 1;
    for pe in triangles do
      tr := zip([([ 0,0])$i=1..nops(pe)], pe, _concat);
      out::writeTriangleFan(attributes, table("MeshVisible"   = FALSE,
                                "PointsVisible" = FALSE,
                                "HasLineColors" = FALSE), tr,
        (u,v,x,y,z)->fillcolorfunction(tn, x, y, z));
      out::writePoly3d(attributes, table("Filled" = FALSE, "Closed" = TRUE), tr,
        (u,v,x,y,z)->linecolorfunction(tn, x, y, z));
      tn:= tn + 1;
    end_for;

    // finally, return the viewing box
    entries:= (i) -> map(t,op,i);
    return( [min(entries(1))..max(entries(1)),
             min(entries(2))..max(entries(2)),
             min(entries(3))..max(entries(3))] );
  end_proc:
//------------------------------------------------------------------------