//      

/* -----------------------------------------------------------
    Hatch -- The graphical primitive for 2D hatches

    Syntax:
    Hatch(f<, g><, xmin..xmax><, op1, op2, ...>)

    f            : a plot::Function2d
    g            : a plot::Function2d or an arithmetical expression
    xmin, xmax   : arithmetical expressions defining the hatch range
    op1, op2, ...: plot options of the form 'option = value'

    Examples:
    // Hatch between sin(x) and y=0 over the full range of f
    >> f := plot::Function2d(sin(x), x = -PI..PI):
       plot::Hatch(f)

    // Hatch between sin(x) and y=1 over the full range of f
    >> plot::Hatch(f, 1)

    // Hatch between sin(x) and y=1 from -PI/2 to PI/2
    >> plot::Hatch(f, 1, -PI/2..PI/2)

    // Hatch between sin(x) and cos(x) over the common range 0..PI
    >> g := plot::Function2d(cos(x), x = 0..2*PI):
       plot::Hatch(f, g)

    // Hatch between sin(x) and cos(x) over the  range 0..PI/2
    >> plot::Hatch(f, g, 0..PI/2)

    // ... with an additional attribute
    >> plot::Hatch(f, g, FillPattern = VerticalLines)
----------------------------------------------------------------*/ 

// creates the domain plot::Hatch and sets generic entries like:
// dimension, op, nops, subsop, slot, _index, _set_index, print, ...
plot::createPlotDomain("Hatch",
                       "graphical primitive for 2D hatches"):

plot::Hatch::styleSheet :=
   table(XMin = -infinity,
         XMax = infinity):

plot::Hatch::hints := {}:

// methods yet to be implemented:  convert, convert_to, expr

// here we have to do something
plot::Hatch::new:=
  proc()
    local children, object, other, f;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;
    // get children which are one or 2 Function2Ds
    children := object::children;
    
    // if one or two functions are given, they are in children
    if nops(children) <> 0 then
      f := children[1];
      if domtype(f) = plot::Curve2d then
        if f::Name = FAIL then
          // implicitly set a name;  use the ID, since it is uniq
          f::Name := stringlib::remove(expr2text(f), "plot::");
        end_if;
        object::Function1 := f::Name;
        dom::extslot(object, "Function2d1", f);
      else
        if domtype(f) = plot::Function2d then
          if f::Name = FAIL then
            // implicitly set a name;  use the ID, since it is uniq
            f::Name := stringlib::remove(expr2text(f), "plot::");
          end_if;
          object::Function1 := f::Name;
          dom::extslot(object, "Function2d1", f);
        else
          error("expecting a plot::Function2d as 1st argument")
        end_if;
        if nops(children) > 1 then
          f := children[2];
          if domtype(f) = plot::Function2d then
            if f::Name = FAIL then
              // implicitly set a name;  use the ID, since it is uniq
              f::Name := stringlib::remove(expr2text(f), "plot::");
            end_if;
            object::Function2 := f::Name;
            dom::extslot(object, "Function2d2", f);
          else
            error("expecting a plot::Function2d as 2nd argument")
          end_if;
        end_if;
        object::children := [];
      end_if;
      
      // now process other which can contain a Baseline and/or a XRange
      if nops(other) <> 0 then
        f := other[1];
        if testtype(f, Type::Arithmetical) then
          object::Baseline := f;
        elif testtype(f, "_range") then
          object::XRange := f;
        else
          error("unexpected argument: ".expr2text(f))
        end_if;
        if nops(other) = 2 then
          f := other[2];
          if testtype(f, "_range") then
            object::XRange := f;
          else
            error("unexpected argument: ".expr2text(f))
          end_if;
        else
          if nops(other) > 2 then
            error("unexpexted argument: ".expr2text(other[3]))
          end_if;
        end_if;
      end_if;
    end_if;

    if object::ParameterRange = FAIL and object::Function2d1 <> FAIL and
      object::Function2d1::ParameterRange <> FAIL then
      // 1. function is animated, Hatch not:  use its parameter range
      object::ParameterRange := object::Function2d1::ParameterRange;
    end_if;
    if object::ParameterRange = FAIL and object::Function2d2 <> FAIL and
      object::Function2d2::ParameterRange <> FAIL then
      // 2. function is animated, Hatch not:  use its parameter range
      object::ParameterRange := object::Function2d2::ParameterRange;
    end_if;
    
    dom::extslot(object, "myXMin", object::XMin);
    dom::extslot(object, "myXMax", object::XMax);

    // check if Function2d2 or Baseline is there
    if object::Function2d2 = FAIL and object::Baseline = FAIL then
      object::Baseline := 0;
    end_if;

    if object::Function1 = FAIL then
      error("no plot::Function2d or plot::Curve2d given")
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:                          /* end of method new */

plot::Hatch::changeNotifier :=
proc(object, eq)
  local attributename, data;
begin
  attributename:= op(eq, 1);
  if attributename = "XMin" then
    data := op(eq, 2);
    dom::extslot(object, "myXMin", data);
  end_if;
  if attributename = "XMax" then
    data := op(eq, 2);
    dom::extslot(object, "myXMax", data);
  end_if;
  if attributename = "XRange" then
    data := op(eq, 2);
    dom::extslot(object, "myXMin", op(data, 1));
    dom::extslot(object, "myXMax", op(data, 2));
  end_if;
  TRUE
end_proc:


plot::Hatch::print :=
  obj -> hold(plot::Hatch)(obj::Function1,
                           if domtype(obj::Function2d1) = plot::Function2d then
                             if obj::Function2 = FAIL then
                               obj::Baseline
                             else
                               obj::Function2
                             end_if
                           end_if,
                           obj::XRange):

plot::Hatch::doPlotStatic:=
  proc(out, obj, attrib, inherited)
    local f1, f2, func1, func2, sing1, sing2, i, j, curveplot,
          setopts, prepareFunction, fattrib, xmin, xmax, baseline,
          findFunction, VB, VB1, VB2, a, points;
  begin
    // prepare attributes, as done in plot::MuPlotML::new
    setopts :=
    proc(func, inherited)
      local attr, optionals, res, paramVal, timeVal, styleSheet,
            parBegin, parEnd, parDiff, timeBegin, timeEnd;
    begin
      // get inheritable attributes of this object
      attr := plot::getInherited(func);
      // merge inhertited with settings in attrib
      inherited := table(inherited, attr);
      // get optional attributes for func from attribute table
      // combined with optional attributes given in func
      optionals := plot::getOptionals(func);
      if contains(attrib, func::dom::objType) then
        styleSheet :=attrib[func::dom::objType];
      else
        styleSheet := null();
      end_if;
      res := table(inherited, func::dom::styleSheet, styleSheet,
                   optionals[1], attr,
                   map(optionals[2], plot::ExprAttribute::getExpr),
                   optionals[3]);
      
      paramVal := null():
      if contains(attrib, TimeValue) then
        timeVal := attrib[TimeValue];
        if func::ParameterRange <> FAIL then
          timeBegin := attrib[TimeBegin];
          timeEnd := attrib[TimeEnd];
          if timeBegin-timeEnd = 0.0 then
            error("time range has length 0")
          end_if;
          parBegin := func::ParameterBegin;
          parEnd :=   func::ParameterEnd;
          parDiff :=  parEnd-parBegin;
          paramVal := parBegin+parDiff*(timeVal-timeBegin)/(timeEnd-timeBegin);
          attrib[ParameterValue] := paramVal;
          res := subs(res, func::ParameterName=paramVal);
        end_if;
      end_if;
      out::makeFuncs(res, paramVal)
    end_proc:

    prepareFunction :=
    proc(branches, singularities)
    begin
      // insert lines at singularities

      xmin := max(xmin, branches[1][1][1]);
      xmax := min(xmax, branches[-1][-1][1]);

      branches
    end_proc:
    
    f1 :=  obj::Function2d1;
    f2 :=  obj::Function2d2;
    xmin := float(attrib[hold(myXMin)]);
    xmax := float(attrib[hold(myXMax)]);
    baseline := float(attrib[Baseline]);

    findFunction :=
    proc(str)
      local xxx, i;
    begin
      xxx := context(hold(context(hold(context(hold(args(1)))))));
      for i in xxx::children do
        if i::Name = str then
          return(i)
        end_if;
      end_for;
      error("could not find function ".str)
    end_proc:
    
    // get Functions when object::Function2d{1,2} are missing
    if f1 = FAIL then
      f1 := findFunction(obj::Function1);
    end_if;
    if obj::Function2 <> FAIL and f2 = FAIL then
      f2 := findFunction(obj::Function2);
    end_if;

    fattrib := setopts(f1, inherited);
    [VB1, func1, sing1] := f1::dom::doPlotStatic(out, f1, fattrib,
                                        inherited, TRUE);

    
    if domtype(f1) = plot::Function2d then
      func1 := prepareFunction(func1, sing1);
    end_if;
    
    if domtype(f1) = plot::Function2d then
      curveplot := FALSE;
      if f2 = FAIL then
        // there has to be a baseline
        VB2  := [xmin..xmax,baseline..baseline];
        func2 := [[[xmin, baseline], [xmax, baseline]]];
        sing2 := {};
      else
        // we have a second function
        fattrib := setopts(f2, inherited);
        [VB2, func2, sing2] := plot::Function2d::doPlotStatic(out, f2, fattrib,
                                                     inherited, TRUE);
        func2 := prepareFunction(func2, sing2);
      end_if;
    else
      curveplot := TRUE;
    end_if;

    points := [];
    if curveplot = FALSE then
      for i from 1 to nops(func1) do
        // loop over branches of the 1. Function or curve
        for j from 1 to nops(func1[i]) do
          // skip points below xmin
          if func1[i][j][1] < xmin then
            if j < nops(func1[i]) and func1[i][j+1][1] > xmin then
              // interpolate on point inbetween
              a := (xmin - func1[i][j][1])/(func1[i][j+1][1]-func1[i][j][1]);
              points := points.[[float(xmin),
                                   float((func1[i][j][2]+a*
                                          (func1[i][j+1][2]-func1[i][j][2])))]];
            end_if;
            next;
          end:
          // break if we are over xmax
          if func1[i][j][1] > xmax then
            if j > 1 and func1[i][j-1][1] < xmax then
              // interpolate on point inbetween
              a := (xmax-func1[i][j-1][1])/(func1[i][j][1]-func1[i][j-1][1]);
              points := points.[[float(xmax),
                                   float((func1[i][j-1][2]+a*
                                          (func1[i][j][2]-func1[i][j-1][2])))]];
            end_if;
            break;
          end:
          points := points.[[op(float(func1[i][j]))]];
        end_for;
      end_for;

      for i from nops(func2) downto 1 do
        // reverse loop over branches of the 2. Function
        for j from nops(func2[i]) downto 1 do
          // skip points above xmax
          if func2[i][j][1] > xmax then
            if j > 1 and func2[i][j-1][1] < xmax then
              // interpolate on point inbetween
              a := (xmax - func2[i][j-1][1])/(func2[i][j][1]-func2[i][j-1][1]);
              points := points.[[float(xmax),
                                   float((func2[i][j-1][2]+a*
                                          (func2[i][j][2]-func2[i][j-1][2])))]];
            end_if;            
            next;
          end:
          // break if we are over xmax
          if func2[i][j][1] < xmin then
            if j < nops(func2[i]) and func2[i][j+1][1] > xmin then
              // interpolate on point inbetween
              a := (xmin-func2[i][j][1])/(func2[i][j+1][1]-func2[i][j][1]);
              points := points.[[float(xmin),
                                   float((func2[i][j][2]+a*
                                          (func2[i][j+1][2]-func2[i][j][2])))]];
            end_if;            
            break;
          end:
          points := points.[[op(float(func2[i][j]))]];
        end_for;
      end_for;
      // a equence of viewingboxes of the 2 functions      
      VB := VB1, VB2;
    else
      // hatch under curve
      for i from 1 to nops(func1) do
        // loop over branches of the 1. Function or curve
        for j from 1 to nops(func1[i]) do
          if func1[i][j][2] < xmin then
            func1[i][j][2] := xmin;
          end:
          if func1[i][j][2] > xmax then
            func1[i][j][2] := xmax;
          end:
          points := points.[[op(float(func1[i][j]), 2..3)]];
        end_for;
      end_for;
      VB := VB1;
    end_if;
    out::writePoly2d(attrib, table("FillStyle"=Winding,"LinesVisible"=FALSE, "Closed"=TRUE),
      points);
    VB
  end_proc:
