// 

/*
 plot::Histogram2d

  renders bar plots of frequency of data

  plot::Histogram2d is pretty similar to plot::Bars2d.  The major
  difference is that Histogram2d preprocesses the data by counting the
  number of entries in each of predefined cells.

*/

plot::createPlotDomain("Histogram2d",
		       "graphical primitive for 2D histogram plots",
		       2,
		       [LinesVisible, LineStyle, LineWidth, LineColor, Color,
			PointSize, PointStyle, Filled, AntiAliased,
			FillColor,
			FillPattern,
			// Cells may be a single number (in a list)
			// or a list of ranges or two-element lists
			[Cells, ["Mandatory", NIL],
			 ["Definition", "ExprSeq", FAIL,
			  "Cell specifications.", TRUE]],
			[Data, ["Mandatory", NIL],
			 ["Definition", "ExprSeq", FAIL,
			  "Data to plot.", TRUE]],
                        [CellsClosed, ["Optional", Right],
                         ["Definition", "Prop", {Left, Right},
                          "Cell a..b interpreted as (a, b] or as [a, b)?", TRUE
                         ]
                        ],
              [ClassesClosed, ["Library", NIL,
                  plot::libAlias([ClassesClosed, CellsClosed]),
                  plot::readAlias([ClassesClosed, CellsClosed])],
               [[[CellsClosed]]]
               ],
			[Area, ["Optional", 0],
			 ["Definition", "Prop", "NonNegFloat",
			  "If nonzero, the total area of the bars.", TRUE]],
                        [DrawMode, ["Optional", Vertical],
			 ["Style", "Prop", "TitleOrientation",
			  "Orientation of bars/boxes", TRUE]]
		       ]):

  
plot::Histogram2d::styleSheet := table(Filled=TRUE,
				       FillPattern=Solid,
				       Cells = [7],
                                       CellsClosed = Right,
				       FillColor=RGB::GeraniumLake,
				       LinesVisible=TRUE,
                                       LineColor = RGB::Black):

plot::Histogram2d::hints := {AxesInFront = TRUE, GridInFront = TRUE}:

plot::Histogram2d::new :=
  proc()
    local object, other, check;
  begin
    object := dom::checkArgs([], args());
    
    if object::Cells <> FAIL then
      check := dom::changeNotifier(object, "Cells"=object::Cells);
      if domtype(check) = DOM_STRING then
	error(check);
      end_if;
    end_if;
    
    other := object::other;
    if nops(other) > 0 then
      check := dom::changeNotifier(object, "Data" = other[1]);
      if domtype(check) = DOM_STRING then
	error(check);
      end_if;
    elif object::Data <> FAIL then
      check := dom::changeNotifier(object, "Data" = object::Data);
      if domtype(check) = DOM_STRING then
	error(check);
      end_if;
    end_if;
    if nops(other) > 1 then
      error("unexpected argument ".expr2text(other[2]));
    end_if;
        
    dom::checkObject(object);
  end_proc:

plot::Histogram2d::print := h -> "plot::Histogram2d(...)":

plot::Histogram2d::changeNotifier :=
  proc(object, eq)
    local idx, value;
  begin
    [idx, value] := [op(eq)];
    case idx
      of "Data" do
	dom::extslot(object, "_count", FAIL);
        if value::dom::hasProp(Cat::Matrix) = TRUE then
           value:= [op(value)];
        end_if;
	if domtype(value) <> DOM_LIST then
	  return("wrong type of value for Data");
	end_if;
	dom::extslot(object, "Data", /* sort */ (map(value, float)));
        return(FALSE);
	break;	
      of "Cells" do
	if domtype(value) <> DOM_LIST then
	  return("wrong type of value for Cells");
	end_if;
	if not ((nops(value) = 1 and domtype(value[1])<> DOM_LIST) or
		map({op(value)}, type) subset {DOM_LIST, "_range"} and
		map({op(value)}, nops) = {2}) then
	  return("wrong type of value for Cells");
	end_if;
	if domtype(value[1])=DOM_LIST then
	  if not _lazy_and(op(map(value, x->op(x,1)<op(x,2))),
			   op(zip(value, value[2..-1],
				  (x,y) -> op(x,2)<=op(y,1)))) then
	    return("cells must be given in increasing order");
	  end_if;
	end_if;
	// FALL THROUGH
      of "Area" do
	dom::extslot(object, "_count", FAIL);
	break;
    end_case;
    TRUE;
  end_proc:
  
plot::Histogram2d::reCount :=
  proc(h: plot::Histogram2d, cells, closed_left, parameter)
    local values, maxval, minval, bucketsort,
	  i, counts, av_width, area, cur_area,
	  area_factor, areas;
  begin
    if h::_count <> FAIL then return(); end_if;
    
    assert(domtype(cells) = DOM_LIST);
    
    values := float(h::Data);
    
    if args(0) > 3 then
      values := (float@eval)(subs(values, h::ParameterName = parameter));
    elif h::ParameterName <> FAIL then
      values := (float@eval)(subs(values, h::ParameterName = h::ParameterBegin));
    end_if;
    
    if not(testtype(values, Type::ListOf(DOM_FLOAT))) then
      context((error)("data must be a list of real values"));
    end_if;
    
    //-------------------------------------------------------
    // automatic generation of equidistant cells
    //-------------------------------------------------------
    if nops(cells) = 1 and 
       domtype(cells[1]) <> DOM_LIST and
       type(cells[1]) <> "_range" then
      // allow animation
      if domtype(float(cells[1])) = DOM_FLOAT then
         cells[1]:= round(cells[1]);
      end_if;
      // a value is tallied into cell[i] by bucketsort if
      //      cell[i][1] < value <= cell[i][2].
      // Unfortunately, this does not count the leftmost value.
      // Hence, decrease the minimal value a tiny bit to the left.
      // Similarly, increase the right value a bit, if
      // CellsClosed = Left (passed by closed_left);
      minval := min(values);
      maxval := max(values);
      if closed_left then
         if iszero(maxval) then
            maxval:= maxval + 0.4*10^(-DIGITS);
         else 
            maxval:= maxval + specfunc::abs(maxval)* 0.4*10^(-DIGITS);
         end_if;
      else
        if iszero(minval) then
           minval:= minval - 0.4*10^(-DIGITS);
        else
           minval:= minval - specfunc::abs(minval)* 0.4*10^(-DIGITS);
        end_if;
      end_if;
      cells := [((cells[1]-i)/cells[1]*minval+i/cells[1]*maxval)
		$i=0..cells[1]];
      cells := zip(cells, cells[2..-1], DOM_LIST);
    end_if;
    
    //-------------------------------------------------------
    // tally data into cells
    //-------------------------------------------------------
    if closed_left then
      bucketsort:= 
       proc(data, locellindex, hicellindex)
         local midcellindex, midvalue, lodata, dummy;
       begin
         if nops(data) = 0 then
           return()
         end_if;
           if locellindex = hicellindex then
             data:= select(data, _less, op(cells[locellindex], 2));
             data:= select(data, _not@_less, op(cells[locellindex], 1));
             counts[hicellindex]:= nops(data);
             return():
           end_if;
           midcellindex:= (locellindex + hicellindex) div 2;
           midvalue:= op(cells[midcellindex], 2):
           [lodata, data, dummy]:= split(data, _less, midvalue);
           bucketsort(lodata, locellindex, midcellindex);
           bucketsort(data, midcellindex+1, hicellindex);
       end_proc:
    else
      bucketsort:= 
       proc(data, locellindex, hicellindex)
         local midcellindex, midvalue, lodata, dummy;
       begin
         if nops(data) = 0 then
           return()
         end_if;
         if locellindex = hicellindex then
           data:= select(data, _leequal, op(cells[locellindex], 2));
           data:= select(data, _not@_leequal, op(cells[locellindex], 1));
           counts[hicellindex]:= nops(data);
           return():
         end_if;
         midcellindex:= (locellindex + hicellindex) div 2;
         midvalue:= op(cells[midcellindex], 2):
         [lodata, data, dummy]:= split(data, _leequal, midvalue);
         bucketsort(lodata, locellindex, midcellindex);
         bucketsort(data, midcellindex+1, hicellindex);
       end_proc:
    end_if;
    
    counts := [0$nops(cells)];
    bucketsort(values, 1, nops(cells));
    
    //-------------------------------------------------------
    // for displaying, crop infinite boundaries
    //-------------------------------------------------------
    if op(cells, [1,1]) = -infinity or
       op(cells, [1,1]) = RD_NINF then
      if nops(cells) > 2 then
	av_width := _plus(op(map(cells[2..-2], _subtract@op)))
		   /(2-nops(cells));
	cells := subsop(cells, [1,1] = op(cells, [1,2])
				      - 1.5*av_width);
      elif nops(cells) = 2 then
	if op(cells, [2,2]) = infinity or
	   op(cells, [2,2]) = RD_INF then
	  cells := subsop(cells, 
			  [1,1] = op(cells, [1,2]) - 1,
			  [2,2] = op(cells, [2,1]) + 1);
	else
	  cells := subsop(cells,
			  [1,1] = op(cells, [1,2]) 
				 - 1.5 * (op(cells, [2,2]) - op(cells, [2,1])));
	end_if;
      else // only one cell
	cells := subsop(cells,
			[1,1] = min(op(cells, [1,2] - 1, 0)));
      end_if;
    end_if;
    
    if op(cells[-1], 2) = infinity or
       op(cells[-1], 2) = RD_INF then
      if nops(cells) > 2 then
	av_width := _plus(op(map(cells[2..-2], _subtract@op)))
		   /(2-nops(cells));
	cells[-1] := subsop(cells[-1], 2 = op(cells[-1], 1)
					    + 1.5*av_width);
      elif nops(cells) = 2 then
	cells[-1] := subsop(cells[-1], 2 = op(cells[-1], 1)
					    + 1.5*(op(cells, [1,2])
						   -op(cells, [1,1])));
      else // only one cell
	cells := subsop(cells,
			[1,2] = max(op(cells, [1,1] + 1, 0)));
      end_if;
    end_if;
    
    area := h::Area;
    if area=FAIL then
      area := plot::getDefault(Area);
    end_if;
    
    if area = 0 then
      dom::extslot(h, "_count", zip(cells, counts, DOM_LIST));
    else
//      areas := zip(cells, counts,
//		   (x, y) -> (x[2]-x[1])*y);
      areas := float(counts);
      cur_area := _plus(op(areas));
      area_factor := area/cur_area;
      dom::extslot(h, "_count",
		   zip(cells, counts,
		       (x, y) -> [x, y*area_factor/(op(x,2)-op(x,1))]));
    end_if;
  end_proc:

plot::Histogram2d::doPlotStatic :=
  proc(out, h: plot::Histogram2d, attributes, inheritedAttributes)
    local values, swap, val, i, closed_left;
  begin
    closed_left:= bool(attributes[CellsClosed] = Left);
    if contains(attributes, ParameterValue) then
      dom::extslot(h, "_count", FAIL);
      dom::reCount(h, attributes[Cells], closed_left, attributes[ParameterValue]);
    else
      dom::reCount(h, attributes[Cells], closed_left);
    end_if;
    
    values := h::_count; // nested list, each entry is [range, count]
    
    swap := (X) -> (float(X));
    if attributes[DrawMode] = Horizontal then
      swap := (X) -> (float([X[1],X[2]]));
    end_if;
    
    // TODO: BarStyles auswerten, vgl. plot::Bars2d.
    for i from 1 to nops(values) do
      val := values[i];
      out::writePoly2d(attributes, table("Closed"=TRUE),
        [swap([op(val, [1,2]), 0]),
         swap([op(val, [1,2]), op(val, 2)]),
         swap([op(val, [1,1]), op(val, 2)]),
         swap([op(val, [1,1]), 0])]);
    end_for;
    
    swap([op(values[1], [1,1]) .. op(values[-1], [1,2]),
	  0 .. max(op(map(values, op, 2)))]);
  end_proc:
