// 

/* ----------------------------------------------------------------------
   Implicit2d -- The graphical primitive for 2D implicit plots

   Syntax:
   Implicit2d(f, x=xl..xr, y=yl..yr, <,op...>)

   f              : arithemetical expression
   x, y           : identifiers
   xl, xr, yl, yr : arithemetical expressions

---------------------------------------------------------------------- */

plot::createPlotDomain("Implicit2d",
                "graphical primitive for 2D implicit functions"):
//------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//------------------------------------------------------------
plot::Implicit2d::styleSheet := table(Contours = [0], 
                                      LegendEntry = TRUE,
                                      XMesh = 11,
                                      YMesh = 11):
//------------------------------------------------------------

plot::Implicit2d::new := 
  proc()
    local object, fn;
  begin
    // check all known options from the argument list
    object := dom::checkArgs(["X", "Y"], args());
    
    // get arguments which are not yet processed
    fn := object::other;
    
    if nops(fn) > 1 then
      error("unexpected argument ".expr2text(fn[2]));
    end_if;
    
    if nops(fn) = 1 then
      if type(fn[1])="_equal" then
        object::Function := subsop(fn[1],0=hold(_subtract));
      else
        object::Function := fn[1];
      end_if;
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

plot::Implicit2d::print :=
  obj -> hold(plot::Implicit2d)(obj::Function,
                                obj::XName=obj::XRange,
                                obj::YName=obj::YRange):

plot::Implicit2d::iv_eval :=
proc(f, xr)
  local iv;
begin
  if traperror((iv := interval::evalSimple(args()))) = 0 then
    iv
  else
    0...RD_INF;
  end_if;
end_proc:

alias(perturb(x) =
      ((userinfo(20,"Perturbing...");
        while traperror((x := x + 1/(100*(stdlib::frandom()-0.5))))=260
        do end_while;))):


plot::Implicit2d::fsolve :=
proc(f, x, y, Df, x0, y0, prec)
  local i, fxy, Df_square, Dfxy, hf;
begin
  return(numeric::fsolve(f, [x=x0, y=y0]));

  i := 0;
  prec := min(prec, 10.0^(4-DIGITS));
  while traperror((fxy := float(subs(f,[x=x0,y=y0],EvalChanges)))) <> 0
  do
    perturb(x0);
    perturb(y0);
  end_while;
  while traperror((
                    Dfxy := [float(subs(Df[1],[x=x0,y=y0],EvalChanges)),
                             float(subs(Df[2],[x=x0,y=y0],EvalChanges))];
                   )) <> 0 do
    perturb(x0);
    perturb(y0);
    while traperror((fxy := float(subs(f,[x=x0,y=y0],EvalChanges)))) <> 0 do
      perturb(x0); 
      perturb(y0);
    end_while;
  end_while;
  Dfxy := map(Dfxy, float@eval);
  if map({op(Dfxy)}, domtype) = {DOM_FLOAT} then
    Df_square := rdmult(Dfxy[1],Dfxy[1],1)+rdmult(Dfxy[2],Dfxy[2],1);
    while
      specfunc::abs(fxy)>prec
      /*
      not iszero(fxy)
      and (iszero(Df_square)
      or specfunc::abs(fxy)/specfunc::sqrt(Df_square) > prec)
      */
      and i<50 do
      while traperror((
                        Dfxy := [float(subs(Df[1],[x=x0,y=y0],EvalChanges)),
                                 float(subs(Df[2],[x=x0,y=y0],EvalChanges))];
                       )) <> 0 do
        perturb(x0);
        perturb(y0);
        while traperror((fxy := float(subs(f,[x=x0,y=y0],EvalChanges)))) <> 0 do
          perturb(x0);
          perturb(y0);
        end_while;
      end_while;
      if iszero(fxy) then break; end_if;
      if map({op(Dfxy)}, domtype) <> {DOM_FLOAT} then
        break;
      end_if;
      Df_square := rdmult(abs(Dfxy[1])$2,1)+rdmult(abs(Dfxy[2])$2,1);
      if specfunc::abs(Df_square/fxy)<1e-15 then
        // Rang(Df) not maximal. Make a small perturbation instead.
        perturb(x0);
        perturb(y0);
      else
        hf := fxy/Df_square;
        [x0, y0] := [x0 - hf*Dfxy[1], y0 - hf*Dfxy[2]];
      end_if:
      while traperror((fxy := float(subs(f,[x=x0,y=y0],EvalChanges)))) <> 0 do
        perturb(x0);
        perturb(y0);
      end_while;
      if iszero(fxy) then break; end_if;
      i := i+1;
    end_while:
  else
    i := 51;
  end_if;
  [x0, y0] := [x=x0, y=y0];
  if i >= 50 then
    x0 := numeric::fsolve(rewrite(f, piecewise), [x0, y0]);
    if x0=FAIL then return(FAIL); end_if;
    [x0, y0] := x0;
  end_if;
  [x0, y0];
end_proc:


plot::Implicit2d::doPlotStatic :=
  proc(out, object, attributes, inheritedAttributes)
    // the first and third arguments are currently ignored
    local f,c,x,y,xmin,xmax,ymin,ymax,plotfactor,
          Lstart,clean_startpoints,
          fxy, colorfunction,
          minstepsize, maxstepsize, startingstepsize, precision,
          points,
          rescale, Lf, i;
  begin
    f := attributes[Function];
    
    xmin := float(attributes[XMin]);
    xmax := float(attributes[XMax]);
    ymin := float(attributes[YMin]);
    ymax := float(attributes[YMax]);
    
    x := `#x`;
    while has(f, x) do
      x := x."1";
    end_while;
    
    y := `#y`;
    while has(f, y) do
      y := y."1";
    end_while;
    
    // rescale the whole thing to 0..1 x 0..1
    rescale := (x, y) -> (x*xmax + (1-x)*xmin,
                          y*ymax + (1-y)*ymin);
    f := f(rescale(x, y));
    
    minstepsize      := 1.0e-3;
    maxstepsize      := 5.0e-2;
    startingstepsize := 1.0e-2;
    
    fxy := Re(hull(abs(plot::Implicit2d::iv_eval(f, [x=0...1, y=0...1]))));
    if rhs(fxy) > 1 then fxy := 0...1; end_if;

    precision := 10.0^(2-DIGITS) * rhs(fxy);
   
    if contains(attributes, LineColorFunction) then
      colorfunction := attributes[LineColorFunction];
    else
      colorfunction := () -> null();
    end_if;
      
    if attributes[XMesh] < 2 or attributes[YMesh] < 2 then
      warning("Mesh must be at least 2 in each direction!");
      attributes[XMesh] := max(attributes[XMesh], 2);
      attributes[YMesh] := max(attributes[YMesh], 2);
    end_if;
  
    clean_startpoints :=
    proc(last_x, last_y, cur_x, cur_y, epsilon)
      name plot::Implicit2d::clean_startpoints;
      local min_x, max_x, min_y, max_y;
      option escape;
    begin
      min_x := cur_x - epsilon;
      max_x := cur_x + epsilon;
      min_y := cur_y - epsilon;
      max_y := cur_y + epsilon;
      
      Lstart := select(Lstart,
                       p -> (p[1] < min_x or
                             p[1] > max_x or
                             p[2] < min_y or
                             p[2] > max_y));
      
      // more exact, but much slower:
//      min_x := min(last_x, cur_x) - epsilon;
//      max_x := max(last_x, cur_x) + epsilon;
//      min_y := min(last_y, cur_y) - epsilon;
//      max_y := max(last_y, cur_y) + epsilon;
//      for i from 1 to nops(Lstart) do
//        if(min_x <= Lstart[i][1]) then
//          break;
//        else
//          if Lstart[i][1]>max_x then return(); end_if;
//        end_if;
//      end_for;
//      
//      a := last_y - cur_y;
//      b := cur_x - last_x;
//      dist := sqrt(rdmult(a,a,1)+rdmult(b,b,1));
//      if iszero(dist) then return() end_if;
//      a := a/dist;
//      b := b/dist;
//      
//      while(i<=nops(Lstart) and Lstart[i][1]<max_x) do
//        if (Lstart[i][2]>min_y and Lstart[i][2]<max_y and 
//            specfunc::abs(a*(Lstart[i][1]-cur_x)+b*(Lstart[i][2]-cur_y))
//            <epsilon )
//          then
//          userinfo(25,"Deleting ".expr2text(Lstart[i]));
//          delete Lstart[i];
//        else
//          i := i+1;
//        end_if;
//      end_while;
    end_proc;
  
    plotfactor :=
    proc(f, c)
      name plot::Implicit2d::plotfactor;
      local Df, plotsingle, startx, starty, cur_x, x_i, y_i;
    begin
      plotsingle :=
      proc(cur_x, cur_y, c)
        name plot::Implicit2d::plotfactor::plotsingle;
        local direction, h, Df_x, Df_y, Df_xy, dx, dy,
              res, old_nres, old_Dx, old_Dy,
              last_x, last_y, t, minmax, no_of_points_written;
      begin
        minmax := proc()
                  begin 
                    [min(args()),max(args())]; 
                  end_proc;
        res := [[startx,starty,0.0]];
        for direction from -1 to 1 step 2 do
          points := [];
          
          // to enable good breaks:
          repeat
                 old_nres := nops(res);
                 cur_x := startx;
                 last_x := startx;
                 last_y := starty;
                 cur_y := starty;
                 h := startingstepsize;
                 t := 0.0;
                 old_Dx:=0;
                 old_Dy:=0;
                 no_of_points_written := 0;
          traperror ((
          // xxx Schutz vor periodischen Lsungen verbessern!
                   while 0 <= cur_x and cur_x <= 1 and 
                         0 <= cur_y and cur_y <= 1 and
                         (nops(res)-old_nres<5 or
                          ((rdmult(abs(cur_x-startx)$2,1)
                            +rdmult(abs(cur_y-starty)$2,1))>h^2))
                         and no_of_points_written < 500 do
                     Df_x := float(subs(Df[1],x=cur_x,y=cur_y,EvalChanges));
                     Df_y := float(subs(Df[2],x=cur_x,y=cur_y,EvalChanges));
                     points := points.[[rescale(cur_x, cur_y),
                                          [map(colorfunction(rescale(cur_x, cur_y),
                                                            Df_x*(xmax-xmin),
                                                            Df_y*(ymax-ymin), c),
                                             float)]]];
                     no_of_points_written := no_of_points_written + 1;
// Mit der QR-Zerlegung von numeric::factorQR ist det(Q)=
//
//                 3                            2
//                b                            a  b
//  - --------------------------- - ---------------------------,
//                 /    2    \1/2                /    2    \1/2
//      2    2 3/2 |   b     |        2    2 3/2 |   b     |
//    (a  + b )    | ------- |      (a  + b )    | ------- |
//                 |  2    2 |                   |  2    2 |
//                 \ a  + b  /                   \ a  + b  /
//
// also sign(det(Q))=-sign(b), falls (a,b)<>(0,0).  Weiterhin gilt
//      R = ((Df_xy[1]^2+Df_xy[2]^2)^(1/2)),
// also det(R)>=0.  Somit ist
//     ( diff(f,x)  diff(f,y) )
// sign(   dx          dy     ) = direction
//
// erfllt fr

                     if traperror((Df_xy := sqrt(rdmult(abs(Df_x)$2,1)
                                                +rdmult(abs(Df_y)$2,1)))) <> 0 or
                       iszero(Df_xy) then
                       // small random perturbation instead
                       Df_x := frandom()*1e-2;
                       Df_y := frandom()*1e-2;
                       Df_xy := sqrt(rdmult(abs(Df_x)$2,1)+rdmult(abs(Df_y)$2,1));
                     end_if;
                     dx    := direction * -Df_y/Df_xy;
                     dy    := direction * Df_x/Df_xy;
                     
                     old_Dx := old_Dx * Df_x;
                     old_Dy := old_Dy * Df_y;
                     if old_Dx <= 0 and old_Dy <=0 and
                        not(iszero(old_Dx) and iszero(old_Dy))
                       then
            // possible bifurcation point
                       break;
                     end_if;
                     old_Dx := Df_x;
                     old_Dy := Df_y;
                     
                     last_x := cur_x;
                     last_y := cur_y;
                     
                     cur_x := float(cur_x + h*dx);
                     cur_y := float(cur_y + h*dy);

                     cur_x := dom::fsolve(f, x, y, Df, cur_x, cur_y, precision);
                     if cur_x = FAIL then
                       break;
                     end_if;
                     
                     [cur_x, cur_y] := map(cur_x, op, 2);

          //        t := t + direction * h * Df_xy;
          // Avoid jumps by prohibiting over-long steps:
                     if rdmult(abs(cur_x-last_x)$2,1)
                        +rdmult(abs(cur_y-last_y)$2,1) > 8*h^2 then
                       if h > minstepsize then
                         h := max(h/2, minstepsize);
                         next;
                       else
                         break;
                       end_if
                     end_if;
                     t := t + direction * sqrt(rdmult(abs(cur_x-last_x)$2,1)
                                               +rdmult(abs(cur_y-last_y)$2,1));
                     clean_startpoints(last_x,last_y,cur_x,cur_y,2*h);
//                     h := h*5/(n+1);
//          traperror((h := float(100000 * h * specfunc::abs(
//                ((Dfxy_n2d[1] + Dfxy_n2d[2]) * fxy_n2d * Df_xy^2)
//                /((Df_x + Df_y)*subs(f, x=last_x+h*dx, y=last_y+h*dy) * Df_square_n2d)))));
                     h := min(max(h,minstepsize),maxstepsize);
                   end_while;
                  ));
          until TRUE end_repeat;
          out::writePoly2d(attributes, table("Closed"=FALSE, "Filled"=FALSE), points, (A,B,C)->(op(C)));

        end_for; // direction
        if nops(res) = 2 then
          cur_x := dom::fsolve(f, x, y, Df, 
            (res[1][1]+res[2][1])/2.0, (res[1][2]+res[2][2])/2.0, precision);
          if cur_x = FAIL then
            res := [];
          else
            [cur_x, cur_y] := map(cur_x, op, 2);
            res := res.[[cur_x,cur_y,(res[1][3]+res[2][3])/2.0]];
          end_if;
        
        end_if;
        res := prog::sort(res, (a)->op(a,3));
      end_proc:

// back to plotfactor

      while op(f, 0) = hold(_power) and is(op(f,2)>0) do
        f := op(f,1);
      end_while;

      fxy := rhs(Re(hull(abs(plot::Implicit2d::iv_eval(f, [x=0...1, y=0...1])))));
      if fxy = RD_INF then
        fxy := 0;
        misc::maprec(f, {DOM_FLOAT} = 
               (x -> (fxy := max(fxy, specfunc::abs(x)))));
        fxy := fxy/5;
      end_if;
      if fxy > 1 and fxy < RD_INF then f := f/fxy; end_if;

      Df:=[diff(f,x), diff(f,y)];

      if op(Df[1], 0)=hold(diff) or op(Df[2], 0)=hold(diff) then
        context(hold(error("Function must be differentiable.")));
      end_if;

      Lstart := {};
      userinfo(10,"Calculating startpoints");

      for x_i from 0 to 1 step 1/(attributes[XMesh]-1) do
        for y_i from 0 to 1 step 1/(attributes[YMesh]-1) do
          cur_x := dom::fsolve(f, x, y, Df, float(x_i), float(y_i), 
            precision);
          if cur_x <> FAIL then
            Lstart := Lstart union {map(cur_x, op, 2)};
          end_if;
        end_for;
      end_for;
      
      Lstart := select(Lstart, a->testtype(a,Type::ListOf(DOM_FLOAT,2,2)));
      Lstart := select(Lstart,
                       proc(a)
                         local fa;
                       begin
                         if a[1] < 0 or a[1] > 1 or
                            a[2] < 0 or a[2] > 1 then
                           return(FALSE);
                         end_if;
                         fa := specfunc::abs(float(eval
                                    (subs(f, [x=a[1], y=a[2]]))));
                         if domtype(fa) <> DOM_FLOAT or fa > precision then
                           return(FALSE);
                         end_if;
                         TRUE;
                       end_proc);
// print(map(Lstart, a-> [a[1], a[2], abs(float(subs(f, [x=a[1], y=a[2]])))]));
      userinfo(15,"Calculated ".expr2text(nops(Lstart))." points");
      while nops(Lstart)>0 do
        [startx,starty] := op(Lstart, 1);
        Lstart := Lstart minus {[startx, starty]};
        userinfo(20,"Starting at (".
                 expr2text(startx).",".expr2text(starty).")");
        plotsingle(startx, starty, c);
      end_while;
    end_proc:

    userinfo(5..9,"Plotting ".expr2text(f));

    // care for abs, by rewriting:
    proc()
     local xx, yy;
    begin
     xx := genident("impl");
     yy := genident("impl");
     save xx; save yy;
     assume(xmin <= xx <= xmax);
     assume(ymin <= yy <= ymax);

     f := subs(f, [x = xx, y = yy,
            hold(abs) = (x -> sqrt(x*conjugate(x)))],EvalChanges);
     f := subs(f, [xx=x, yy=y]);
    end_proc();

    for c in attributes[Contours] do
      if traperror((Lf := factor(f-c);
                    Lf := Factored::convert_to(Lf,DOM_LIST);
                    Lf := [Lf[2*i]$i=1..(nops(Lf)-1)/2])) = 0 then
        userinfo(10,"Now plotting ".expr2text(Lf));
        map(Lf, plotfactor, c);
      else
        plotfactor(f-c, c);
      end_if;
    end_for;
    
    return([xmin..xmax, ymin..ymax]);
  end_proc:
