// 

/*
 plot::Integral -- plot numeric quadrature

*/

alias(IntMethods = [hold(Exact),
                    hold(Simpson), hold(Trapezoid),
                    hold(RiemannMiddle), hold(RiemannLeft), hold(RiemannLower),
                    hold(RiemannRight), hold(RiemannUpper),
                    hold(RiemannLowerAbs), hold(RiemannUpperAbs)]):

plot::createPlotDomain("Integral",
                       "graphical primitive for numeric quadrature",
                       2,
             	       [Color,
                        // function ==================================
                        Function1,
                        LineColor, LineColor2, LineWidth, LineStyle,
                        LinesVisible, AntiAliased,
                        LineColorType, LineColorFunction, PointSize,
                        PointStyle, PointsVisible,
                        [Nodes, ["Optional", NIL],
                         ["Definition", "ExprSeq", FAIL,
                         "Number of rectangles or list of nodes.", TRUE]],
                        // different methods
                        [IntMethod, ["Optional", hold(Exact)],
                          ["Definition", "Prop",
                          {op(IntMethods)},
                          "Method of approximation.", TRUE]],
                        // rectangles ================================
                        FillPattern, FillColor, Filled,
                        // text ======================================
                        [ShowInfo, ["Optional", NIL],
                         ["Annotation", "ExprSeq", FAIL,
                          "Format of displayed information.", TRUE]],
                        
                        TextFont,
                        TextRotation,
                        HorizontalAlignment, VerticalAlignment
                       ]):
//--------------------------------------------------------
plot::Integral::styleSheet := table(LinesVisible = TRUE,
                                    LineColor = RGB::Black,
                                    LineColor2 = RGB::Gray,
                                    FillPattern = Solid,
                                    FillColor = RGB::PaleBlue,
                                    Nodes = [10],
                                    IntMethod = hold(Exact),
                                    //XMesh = plot::getDefault(plot::Function2d::XMesh),
                                    //Mesh = plot::getDefault(plot::Function2d::Mesh),
                                    // text
                                    HorizontalAlignment = Left,
                                    VerticalAlignment = Bottom,
                                    ShowInfo = [2, // number of digits
                                                   // !! default value is set in line 409 too !!
                                                IntMethod,
                                                Integral
                                                //Nodes = "", // not displayed
                                                //Position = [] // auto
                                                ]
                                    ):
//--------------------------------------------------------
plot::Integral::hints := {AxesInFront = TRUE, Axes = Origin}:
//--------------------------------------------------------
plot::Integral::new :=
  proc()
    local children, object, other, f;
  begin
    object := dom::checkArgs([], args());
    
    // get children which is a 2 Function2D
    // and non optional arguments
    children := object::children;
    other := object::other;
    if nops(children) = 0 and nops(other) > 0 then
      error("plot::Function2d object expected as first argument")
    elif nops(children) > 1 then
      error("only one function expected")
    elif nops(children) = 1 then
      // one Function2d given
      if domtype(children[1]) <> plot::Function2d then
        error("illegal plot object given; plot::Function2d expected")
      end_if;
      if nops(other) > 0 then
        if testtype(other[1], Type::Arithmetical) or
          testtype(other[1], Type::ListOf(Type::Arithmetical)) then
          object::Nodes := other[1];
        else
          error("3rd argument: integer or list expected")
        end_if
      end_if;
      f := children[1]; // function object
      if f::Name = FAIL then
        // implicitly set a name;  use the ID, since it is uniq
        f::Name := stringlib::remove(expr2text(f), "plot::");
      end_if;
      object::Function1 := f::Name;
      dom::extslot(object, "Function2d1", f);
    elif nops(other) > 1 then
      if testtype(other[2], Type::Arithmetical) or
        testtype(other[2], Type::ListOf(Type::Arithmetical)) then
        object::Nodes := other[2];
      else
        error("3rd argument: integer or list expected")
      end_if
    end_if;
    if object::ParameterRange = FAIL and object::Function2d1 <> FAIL and
      object::Function2d1::ParameterRange <> FAIL then
      // 1. function is animated, Hatch not:  use its parameter range
      object::ParameterRange := object::Function2d1::ParameterRange;
    end_if;
    dom::checkObject(object);
  end_proc:
//--------------------------------------------------------
plot::Integral::print := b -> hold(plot::Integral)(text2expr(b::Function1),
                                                   //(if nops(b::Nodes) = 1 then
                                                   //   Nodes = op(b::Nodes)
                                                   // else
                                                   //   Nodes = [...]
                                                   // end_if),
                                                   (if b::IntMethod <> Exact and b::IntMethod <> FAIL then
                                                      IntMethod = b::IntMethod
                                                    else
                                                      null()
                                                    end_if)):
//--------------------------------------------------------
plot::Integral::doPlotStatic :=
  proc(out, obj: plot::Integral, attributes, inheritedAttributes)
    local f, fterm, z, a, b, N, x, tmp, i, k, area, approximation,
          pvalues, values, f1, findFunction, prepareAnimation, vb, fX,
          areat, fattrib, fu, parabel, y, integral, fMinMax,
          vbmax, vbmin, //vbmaxl, // maximal y values
          // info attributes
          showInfo,     // list attributes[ShowInfo]
          infoText,     // info text string
          infoDigits,   // number of digits
          infoPosition; // position of info
    save DIGITS;
  begin

    // prepare attributes, as done in plot::MuPlotML::new
    prepareAnimation :=
    proc(func, inherited)
      local attr, optionals, res, paramVal, timeVal, styleSheet,
            parBegin, parEnd, parDiff, timeBegin, timeEnd;
    begin
      // get inheritable attributes of this object
      attr := plot::getInherited(func);
      // merge inhertited with settings in attributes
      inherited := table(inherited, attr);
      // get optional attributes for func from attribute table
      // combined with optional attributes given in func
      optionals := plot::getOptionals(func);
      if contains(attributes, func::dom::objType) then
        styleSheet :=attributes[func::dom::objType];
      else
        styleSheet := null();
      end_if;
      res := table(inherited, func::dom::styleSheet, styleSheet,
                   optionals[1], attr,
                   map(optionals[2], plot::ExprAttribute::getExpr),
                   optionals[3]);
      
      paramVal := null():
      if contains(attributes, TimeValue) then
        timeVal := attributes[TimeValue];
        if func::ParameterRange <> FAIL then
          timeBegin := attributes[TimeBegin];
          timeEnd := attributes[TimeEnd];
          if timeBegin-timeEnd = 0.0 then
            error("time range has length 0")
          end_if;
          parBegin := func::ParameterBegin;
          parEnd :=   func::ParameterEnd;
          parDiff :=  parEnd-parBegin;
          paramVal := parBegin+parDiff*(timeVal-timeBegin)/(timeEnd-timeBegin);
          attributes[ParameterValue] := paramVal;
          res := subs(res, func::ParameterName=paramVal);
        end_if;
      end_if;
      out::makeFuncs(res, paramVal)
    end_proc:

    findFunction :=
    proc(str)
      local xxx, i;
    begin
      xxx := context(hold(context(hold(context(hold(args(1)))))));
      for i in xxx::children do
        if i::Name = str then
          return(i)
        end_if;
      end_for;
      error("could not find function ".str)
    end_proc:

    f1 := obj::Function2d1; // saved reference
    // get Function when object::Function2d1 is missing
    if f1 = FAIL then
      f1 := findFunction(obj::Function1); // the same function given to plot together with this
    end_if;

    fattrib := prepareAnimation(f1, inheritedAttributes);
    
    //
    //f := f1::Function://attributes[hold(Function)];
    f := fattrib[Function];
    z := f1::XName://attributes[hold(XName)];
    a := f1::XMin://attributes[hold(XMin)];
    b := f1::XMax://attributes[hold(XMax)];

    //
    if not testtype(f, Type::Function) then
      fterm := f;
      //f := X -> eval(subs(fterm, z = X));
      f := fp::unapply(f, z)
    end_if;
    
    N := attributes[hold(Nodes)];
    if nops(N) = 1 then // only one node given
      N := op(N)
    end_if;
    x:= table();
    
    // x-ranges for rectangles
    if domtype(N) = DOM_LIST then
      N := sort(float([op({op(N.[a, b])})])); // remove double values, append begin and end
      //if float(a) > N[1] or float(b) < op(N, nops(N)) then
      //  error("range must contain all elements of the node set");
      //end_if;
      
      tmp := select(N, X -> X >= float(a) and X <= float(b)); // remove values out of the definition interval
      N := nops(tmp) - 1;
      (x[i] := tmp[i+1]) $ i=0..N
    else
      N := ceil(N); // if N is animated
      (x[i] := a + i * (b-a)/N) $ i=0..N
    end_if:

    vbmin := 0;
    vbmax := 0;
    //vbmaxl := -infinity;
    
    // function value without errors
    fX :=
      proc(x)
        local ret;
      begin
        while traperror((ret := f(float(x)))) <> 0 do
          x := x + 10.0^-DIGITS
        end_while;
        if domtype(ret) <> DOM_FLOAT then
          ret := 0.0
        end_if;
        if ret < vbmin then
          vbmin := ret
        end_if;
        if ret > vbmax then
          vbmax := ret;
          //if x < 0 then
          //  vbmaxl := ret
          //end_if
        end_if;
        ret
      end_proc;

    // some function values in interval
    fMinMax :=
      proc(a, b)
        local s, i, steps, r, ymin, ymax, ymin1, ymax1, maxerror, lerror;
        save DIGITS;
      begin
        steps := 10;              // number of intervals

        ymin := RD_INF;
        ymax := RD_NINF;
        
        maxerror := abs(fX(b) - fX(a))/400;
        
        s := (b - a)/steps;     // one step
        r := {fX((a+2*i*s)) $ i = 0..steps/2}; // all values in interval
        
        repeat
          s := (b - a)/steps;     // one step
          ymin1 := ymin; // keep old values
          ymax1 := ymax;

          if traperror((r := r union {fX((a+(2*i+1)*s)) $ i = 0..(steps/2)-1})) <> 0 then
            return([ymin, ymax])
          end_if;
          ymin := min(r);
          ymax := max(r);
          
          steps := 2*steps;
          //DIGITS := DIGITS + 2;
          if traperror((lerror := abs(ymin - ymin1) + abs(ymax - ymax1))) <> 0 then
            return([ymin, ymax])
          end_if;
        until lerror < maxerror or steps > 10000 end_repeat;
        userinfo(Text, 10, "used steps: ", steps);
        [ymin, ymax]
      end_proc;    

    // area of rectanlges
    area := proc(values)
              local i, k, ret;
            begin
              ret := 0;
              for i from 1 to N do
                k := x[i]-x[i-1];
                k := k*_plus(op(values[i]));
                ret := ret + k
              end_for;
              ret
            end_proc;
    
    areat := values -> _plus((x[i]-x[i-1])/2*_plus(op(values[i]), op(values[i+1])) $ i = 1..N);

    pvalues :=
      proc(opt)
        local i;
        name plot::Integral::values;
      begin
        case opt
          of hold(RiemannUpper) do
            [[0, fMinMax(x[i-1], x[i])[2]]$i=1..N];
            break
          of hold(RiemannUpperAbs) do
            map([fMinMax(x[i-1], x[i]) $ i=1..N],
                mima -> if mima[1] >= 0 then // positive interval
                       [0, mima[2]]
                     elif  mima[2] <= 0 then // negative interval
                       [0, mima[1]]
                     else // both
                       mima
                     end_if);
            break
          of hold(RiemannMiddle) do
            [[0, fX((x[i] + x[i-1])/2)]$i=1..N];
            break
          of hold(RiemannRight) do
            [[0, fX(x[i-1])]$i=2..N+1];
            break
          of hold(Trapezoid) do
            [[0, fX(x[i-1])]$i=1..N+1];
            break
          of hold(RiemannLeft) do
            [[0, fX(x[i-1])]$i=1..N];
            break
          of hold(RiemannLower) do
            [[0, fMinMax(x[i-1], x[i])[1]]$i=1..N];
            break
          of hold(RiemannLowerAbs) do
            map([fMinMax(x[i-1], x[i]) $ i=1..N],
                mima -> if mima[1] >= 0 then // positive interval
                       [0, mima[1]]
                     elif  mima[2] <= 0 then // negative interval
                       [0, mima[2]]
                     else // both
                       [0, 0]
                     end_if);
            break
          otherwise
            error("unknown approximation method '".expr2text(opt)."'")
        end_case
      end_proc;

    // DRAW and return common viewing box
    out::commonViewingBox
    ({
      // rectangles/trapezoids
      //////////////////////////////////////////////////////////////////////
      (k := attributes[hold(IntMethod)];
       if k = hold(Exact) then
         // print hatch
         approximation := 0.0;
         // get viewing box YRange
         tmp := out(plot::Hatch(plot::Function2d(f, z = a..b,
                                                            op(out::fixAttributes(fattrib,
                                                                                             plot::Function2d))
                                                            ),
                                           op(out::fixAttributes(attributes,
                                                                            plot::Hatch))),
                               inheritedAttributes, Raw);
         [vbmin, vbmax] := [op(tmp["ViewingBox"][2])];
         tmp
       elif k = hold(Simpson) then
         // Simpson
         //
         y := table((x[i]=fX(x[i]))$i=0..N,
	            ((x[i]+x[i+1])/2 = fX((x[i]+x[i+1])/2))$i=0..N-1);
         parabel := i -> (1/(x[i + 1] - x[i])^2*
                          (2*`#t`^2*y[x[i + 1]] + y[x[i + 1]]*x[i]^2 +
                           y[x[i]]*x[i + 1]^2 + 2*`#t`^2*y[x[i]] -
                           4*`#t`^2*y[(x[i + 1] + x[i])/2] +
                           y[x[i + 1]]*x[i + 1]*x[i] - 3*`#t`*y[x[i + 1]]*x[i] -
                           3*`#t`*y[x[i]]*x[i + 1] +
                           4*`#t`*y[(x[i + 1] + x[i])/2]*x[i + 1] +
                           y[x[i]]*x[i + 1]*x[i] - `#t`*y[x[i + 1]]*x[i + 1] -
                           `#t`*y[x[i]]*x[i] -
                           4*y[(x[i + 1] + x[i])/2]*x[i + 1]*x[i] +
                           4*`#t`*y[(x[i + 1] + x[i])/2]*x[i]));
         
         vb := {}; // used for simpson, empty set otherwise
         approximation := 0;
         (fu := plot::Function2d(parabel(i), `#t`=x[i]..x[i+1],
                                 Name = expr2text(x[i]..x[i+1]),
                                 op(out::fixAttributes(attributes, plot::Function2d)));
          approximation := approximation + numeric::int(parabel(i), `#t`=x[i]..x[i+1]);
          //approximation := approximation + numeric::int(fu::Function, fu::XName = fu::XMin..fu::XMax);
          out(plot::Hatch(fu,
                                     op(out::fixAttributes(attributes, plot::Hatch))),
                         inheritedAttributes, Raw);
          vb := vb union {out(fu, inheritedAttributes, Raw)};
          out(plot::Line2d([x[i], 0], [x[i], subs(parabel(i), `#t` = x[i])],
                                      op(out::fixAttributes(attributes, plot::Line2d))),
                         inheritedAttributes, Raw);
          out(plot::Line2d([x[i+1], 0], [x[i+1], subs(parabel(i), `#t` = x[i+1])],
                                      op(out::fixAttributes(attributes, plot::Line2d))),
                         inheritedAttributes, Raw);
          // text position
          //tmp := f(x[i]...x[i+1]);
          //[vbmin, vbmax] := [min(vbmin, min(tmp)), max(vbmax, max(tmp))];
          ) $ i = 0..N-1;
         
         // viewing box
         out::commonViewingBox(vb)["ViewingBox"]
       else
         values := pvalues(k);
         if k = hold(Trapezoid) then
           tmp := areat(values)
         else
           tmp := area(values)
         end_if;
         approximation := float(tmp);
         
         //xmin := RD_INF; xmax := RD_NINF;
         //ymin := RD_INF; ymax := RD_NINF;
         
         //prP2 := proc(x, y)
         //        begin
         //          [x, y] := float([x, y]);
         //          xmin := min(xmin, x);
         //          xmax := max(xmax, x);
         //          ymin := min(ymin, y);
         //          ymax := max(ymax, y);
         //          plot::MuPlotML::prP2(x, y, args(3..args(0)));
         //        end_proc:

         // rectangles by hand - options problems
         (
          if k = hold(Trapezoid) then
            out::writePoly2d(attributes, table("Closed"=TRUE, "FillStyle" = "Winding"),
              [[float(x[i-1]), float(op(values[i], 1))],
               [float(x[i-1]), float(op(values[i], 2))],
               [float(x[i]), float(op(values[i+1], 2))],
               [float(x[i]), float(op(values[i+1], 1))],
               [float(x[i-1]), float(op(values[i], 1))]]);
          else
            out::writePoly2d(attributes, table("Closed"=TRUE, "FillStyle" = "Winding"),
              [[float(x[i-1]), float(op(values[i], 1))],
               [float(x[i-1]), float(op(values[i], 2))],
               [float(x[i]), float(op(values[i], 2))],
               [float(x[i]), float(op(values[i], 1))],
               [float(x[i-1]), float(op(values[i], 1))]]);
          end_if;
          )
         $ i = 1..N;
         
         [a..b, vbmin..vbmax] // viewing box
       end_if
       ),
      // function
      
      // text
      (
       if attributes[hold(ShowInfo)] = FALSE or attributes[hold(ShowInfo)] = [] then
         // don't display information
         null()
       else
         showInfo := attributes[hold(ShowInfo)];
         
         // automatic text position
         if not has(showInfo, hold(Position)) // not given or empty list
            or op(select(showInfo, has, hold(Position)), [1, 2]) = [] then
           // x
           if float(a) <= 0 and float(b) > 0 and float(-a/b) < 0.6 then // prevent collision with y axis
             tmp := (b - a)/20;
           else
             tmp := a; //[(19*a+b)/20, tmp];
             // only left part of function
             //vbmax := vbmaxl
           end_if;
           // y
           if vbmax <= 0 then // prevent collision with x axis
             tmp := [tmp, (vbmax - vbmin)/20];
           else
             tmp := [tmp, (22*vbmax - vbmin)/20];
           end_if;
           infoPosition := tmp
         else
           infoPosition := op(select(showInfo, has, hold(Position)), [1, 2])
         end_if;

         infoText := "";
         infoDigits := 2; // default value
  
         integral := numeric::int(f(z), z = a..b);
         if domtype(integral) <> DOM_FLOAT then
           integral := "undefined";
           //warning("numeric integration failed")
         end_if;
         if attributes[hold(IntMethod)] = hold(Exact) then
           approximation := integral
         end_if;
  
         for k in showInfo do
           // build the text string
           if k = "" then // empty line
             infoText := infoText."\n "
           //elif type(k) = "_equal" and op(k, 1) = hold(Position) then // absolute Positionierung
           //  infoPosition := op(k, 2)
           elif domtype(k) = DOM_INT then // number of digits from here
             infoDigits := k
           elif domtype(k) = DOM_STRING then // user string
             infoText := infoText.k
           elif (tmp := contains([hold(IntMethod), hold(Integral),
                                  hold(Error), hold(Nodes), hold(Function)], k)) <> 0
                or (type(k) = "_equal"
                    and (tmp := contains([hold(IntMethod), hold(Integral),
                                          hold(Error), hold(Nodes), hold(Function)], op(k, 1))) <> 0) then
             infoText := infoText
                     . (if type(k) = "_equal" then
                          if op(k, 2) = "" then
                            op(k, 2)
                          else
                            "\n".op(k, 2)
                          end_if
                        elif k = hold(IntMethod) then
                          "\n".prog::getname(attributes[hold(IntMethod)])
                        else
                          "\n".prog::getname(k)
                        end_if)
                     . (if type(k) = "_equal" and op(k, 2) = "" then "" else ": " end_if)
                     . op([(if approximation = "undefined" then approximation else
                              stringlib::formatf(approximation, infoDigits)
                            end_if),
                           (if integral = "undefined" then integral else
                              stringlib::formatf(integral, infoDigits)
                            end_if),
                           (if integral = "undefined" then integral else
                              stringlib::formatf(abs(approximation - integral), infoDigits)
                            end_if),
                           (if testtype(attributes[hold(Nodes)][1], Type::PosInt) then
                              expr2text(attributes[hold(Nodes)][1])
                            else
                              expr2text(nops(attributes[hold(Nodes)]))
                            end_if
                              ),
                           expr2text(f1::Function)
                           ], tmp)
           else
             // error??
           end_if
         end_for;
         
         out(plot::Text2d(// text definition
                                     infoText,
                                     // default position
                                     infoPosition,
                                     op(out::fixAttributes(attributes, plot::Text2d))
                                     ),
                        inheritedAttributes, Raw)
       end_if
       )})
    
  end_proc:
