/*--------------------------------------------------------------------------  

TODO: auch bei InterpolationStyle = Cubic sollten komplexe/symbolische
      Matrixwerte zulaessig sein.
------------------------------------------------------------------------

Calls: plot::Matrixplot(data, 
                        <x = xmin .. xmax, y = ymin .. ymax >
                        <InterpolationStyle = Linear/Cubic>
                       )  
Parameters:
    data         : List, Sequence of Lists, List of Lists, 
                   Matrix or Array (1-dim or 2-dim)
    x, y         : identifiers or indexed identifiers
    xmin .. xmax : range in x direction
                   If not set, xmin = 1 and 
                   xmax = number of data columns
    ymin .. ymax : range in y direction
                   If not set, ymin = 1 and 
                   ymax = number of data columns

Details:
  * A column of the matrix extends in x direction,
    a row in y direction
  * x, y serve as hints for the axes titles
  * The default is InterpolationStyle = Linear
  * With InterpolationStyle = Cubic, cubic spline
    interpolation is used with XSubmesh, YSubmesh 
    plot points between the data points
  * the data can be animated

Examples: (This is a little test suite that should work)
 >> // non-degenerate surfaces
    L := [ [ 1,PI, 3,PI,  2 ],
           [ 1, 1, 1, 1,  1 ],
           [ 1, 1, 3, 1,  1 ] ];
    plot(plot::Matrixplot(op(L)));
    plot(plot::Matrixplot(L));
    plot(plot::Matrixplot(L, InterpolationStyle = Cubic));
    plot(plot::Matrixplot(L, x = 0 .. 10, y = -5 .. 5, InterpolationStyle = Linear));
    plot(plot::Matrixplot(L, x = 0 .. 10, y = -5 .. 5, InterpolationStyle = Cubic));
    plot(plot::Matrixplot(L, x = 0 .. 10, y = -5 .. 5, InterpolationStyle = Cubic, XSubmesh = 5));
    plot(plot::Matrixplot(L, x = 0 .. 10, y = -5 .. 5, InterpolationStyle = Cubic, YSubmesh = 5));
    plot(plot::Matrixplot(L, x = 0 .. 10, y = -5 .. 5, InterpolationStyle = Cubic, Submesh = [5, 5]));
    plot(plot::Matrixplot(array(1..3, 1..5, L)));
    plot(plot::Matrixplot(matrix(L))):

    // degenerate surfaces
    plot(plot::Matrixplot(array(1..6, [1, 2, 2, 1, 1, 1]))):
    plot(plot::Matrixplot(array(1..6, [1, 2, 2, 1, 1, 1]), InterpolationStyle=Cubic)):
    plot(plot::Matrixplot(matrix(1, 6, [1, 2, 2, 1, 1, 1]))):
    plot(plot::Matrixplot(matrix(1, 6, [1, 2, 2, 1, 1, 1]), InterpolationStyle = Cubic)):
    plot(plot::Matrixplot(matrix(6, 1, [1, 2, 2, 1, 1, 1]))):
    plot(plot::Matrixplot(matrix(6, 1, [1, 2, 2, 1, 1, 1]), InterpolationStyle = Cubic)):

    // animated surfaces
    L := [ [ 1, 2, 1 + a, PI,  2 ],
           [ 1, 1,   1  ,  a,  1 ],
           [ 1, 1, 3 - a,  1,  1 ] ];
    plot(plot::Matrixplot(op(L), x = 0..5, y = 0..3, a = 0..1));
    plot(plot::Matrixplot(  L  , x = 0..1, y = 0..1, a = 0..1));
    plot(plot::Matrixplot(  L  , x = 0..1, y = 0..1, a = 0..1, InterpolationStyle = Cubic));
    plot(plot::Matrixplot(L, x = 0 .. 10, y = -5 .. 5, a = 0..1, InterpolationStyle = Linear));
    plot(plot::Matrixplot(L, x = 0 .. a,  y = -a .. a, a = 0..1, InterpolationStyle = Cubic));
    Umbrella:= [ [0, 0, 0],
                 [0, 1, 0],
                 [0, 0, 0] ]:
    plot(plot::Matrixplot(Umbrella, XCoord = -a .. a,  YCoord = -a .. a, 
                          a = 0..1, InterpolationStyle = Cubic, Submesh = [5, 5]));
--------------------------------------------------------------------------*/

plot::createPlotDomain("Matrixplot",
                       "graphical primitive for matrixplots",
                       3,  // Dimension
                       [[Data, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                          "Data to plot.", TRUE]],
                        XName, XMin, XMax, XSubmesh,
                        YName, YMin, YMax, YSubmesh, 
                        Submesh,
                        [InterpolationStyle, ["Optional", Linear],
                         ["Style", "Prop", {Linear, Cubic},
                          "Linear or cubic interpolation?", TRUE
                         ]
                        ],
                        PointSize, PointStyle, PointsVisible, PointColor,
                        XLinesVisible, YLinesVisible, LineWidth, LineStyle,
                        LineColor, LineColor2, LineColorFunction, LineColorType,
                        Filled, Shading, Color,
                        FillColor, FillColor2, FillColorFunction, FillColorType,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ]):
//------------------------------------------------
plot::Matrixplot::styleSheet := table(
                      PointsVisible = TRUE,
                      LineColor = RGB::Black.[0.25], // consistent with Function3d and Surface
                      XLinesVisible = TRUE, // consistent with Function3d and Surface
                      YLinesVisible = TRUE, // consistent with Function3d and Surface
                      XSubmesh = 2,         // only relevant for spline interpolation
                      YSubmesh = 2,         // only relevant for spline interpolation
                      LineColorDirectionY=0 
                     ):
//------------------------------------------------
// TODO: wirklich XName, YName per Hint als Achsentitle?
plot::Matrixplot::hints:= {XAxisTitle, YAxisTitle,
                           Axes = Boxed}:
plot::Matrixplot::setHint:= 
/*
  object -> plot::setHint(object,
                          XAxisTitle = expr2text(object::XName),
                          YAxisTitle = expr2text(object::YName)
                         ):
*/
  object -> (if object::XName <> FAIL and object::XAxisTitle = FAIL then
              plot::setHint(object, XAxisTitle = expr2text(object::XName)):
             end_if:
             if object::YName <> FAIL and object::YAxisTitle = FAIL then
              plot::setHint(object, YAxisTitle = expr2text(object::YName)):
             end_if):

//-----------------------------------------------------------------------
plot::Matrixplot::new:=
  proc()
    local object, i, columnIndices, dim, j, l, numberOfColumns, m, n;
  begin
    object := dom::checkArgs([["X"], ["Y"]], args());
    
    if nops(object::other) = 0 then
      return(dom::checkObject(object));
    end_if;

    l := object::other[1];
    
    if not (
      testtype(l, Type::ListOf(DOM_LIST)) or
      domtype(l) = DOM_ARRAY or
      domtype(l) = stats::sample or
      domtype(l) = DOM_LIST  or
      l::dom::hasProp(Cat::Matrix) = TRUE
      ) then
       error("1st argument: list or list of lists, array, matrix, or stats::sample expected")
   end_if;

   //--------------------------------------------
   // process matrixplot(List1, options): 
   // convert to matrixplot([List1], options)
   //--------------------------------------------
   if domtype(l) = DOM_LIST and domtype(l[1])<>DOM_LIST then 
      l:=[l];
      for i from 2 to nops(object::other) do
      	if domtype( object::other[i] ) = DOM_LIST then
             l := append( l, object::other[i] );
        else error("unexpected argument: ".expr2text(object::other[i]));
      	end_if;
      end_for;
   end_if;

   //--------------------------------------------
   // Matrix, Array --> List of Lists :
   //--------------------------------------------
   if l::dom::hasProp( Cat::Matrix ) = TRUE then
      l:= expr(l): // Matrix -> Array
      if nops(object::other) > 1 then
         error("unexpected argument: ".expr2text(object::other[2]));
      end_if:
   end_if;

   //--------------------------------------------
   // process data array: convert to list of lists
   //--------------------------------------------
   if domtype(l) = DOM_ARRAY then
      // Array -> List of Lists
      dim := op(l, [0,1]): // Dimension of Array
      if dim = 2 then
         l:=[ [l[i,j] $ j = op(l, [0,3,1])..op(l, [0,3,2])] 
                      $ i = op(l, [0,2,1])..op(l, [0,2,2])];
      elif dim = 1 then 
         // 1-dimensionaler Array in eine ListOfList
         l:= [ [op(l)] ]:
      else
          error("1st argument: the data array must be one or two-dimensional")
      end_if;
      if nops(object::other) > 1 then
         error("unexpected argument: ".expr2text(object::other[2]));
      end_if:
   end_if;

   // ---------------------------
   // process statistical samples
   // ---------------------------
   if domtype(l) = stats::sample then
      columnIndices:= []:
      // check if column indices are provided by the user:
      if nops(object::other) > 1 and domtype(object::other[2]) = DOM_LIST 
      then
        // the column indices are specified by a list
        columnIndices:= object::other[2];
        numberOfColumns:= nops(columnIndices);         
      else
        i:= 1; // initialize: if args(0) = 1, the
               // following loop will not be entered
        for i from 2 to nops(object::other) do
          if domtype(object::other[i]) = DOM_INT then
             columnIndices:= append(columnIndices, object::other[i]);
          else
             break;
          end_if;
        end_for;
        numberOfColumns:= nops(columnIndices);
      end_if;
      if numberOfColumns = 0 then
         // No columns were specified by the
         // user. Extract *all* columns of l.
         // Check the number of elements in the
         // first row extop(l, 1)[1]:
         numberOfColumns:= nops(extop(l, 1)[1]);
         columnIndices:= [i $ i = 1 .. numberOfColumns];
      end_if;
      l:= [stats::getdata(
            testargs(),     // double check the data
            "all_data",     // accept all sorts of numerical data
                            // as well as strings
            numberOfColumns,// number of data columns to be read
            l,              // the sample
            columnIndices   // the indices of the columns to be read
           )];
      // a string returned by stats::getdata indicates an error:
      if domtype(l) = DOM_STRING then 
         error(l);
      end_if;
   
      // check for string columns in l:
      for i from 1 to nops(l) do
        if domtype(l[i][1]) = DOM_STRING then
           l[i]:= NIL;
        end_if;
      end_for:
      if has(l, NIL) then 
         l:= subs(l, NIL = null);
      end_if;
      if nops(l) = 0 then
         error("No numerical data found");
      end_if;
   end_if; // of domtype(l) = stats::sample

   //---------------------------------------------
   // ---> Now: the data l is a list of lists ----
   //---------------------------------------------
   l := float(l);

   // do not check for floats here, because there might
   // be an animation parameter
   /* ----------------------------------------------------------
   if (_union (map({op(l[i])},domtype )$i=1..nops(l)) minus {DOM_FLOAT}) <> {} then 
       error("Some data elements are not numerical"); 
   end_if;  
   ----------------------------------------------------------*/

   dom::extslot(object, "Data", l);

   //---------------------------------------------
   // Set XMin .. YMax. They must be set, or the
   // consistency check via dom::checkObject fails
   //---------------------------------------------
   m := nops(l); 
   n := nops(l[1]); 
   if object::XName = FAIL then
      object::XName:= hold(x); // ouch!
   end_if;
   if object::XMin = FAIL then
      object::dom::extslot(object, "XMin", 1):
   end_if;
   if object::XMax = FAIL then
      object::dom::extslot(object, "XMax", n);
   end_if;
   if object::YName = FAIL then
      object::YName:= hold(y); // ouch!
   end_if;
   if object::YMin = FAIL then
      object::dom::extslot(object, "YMin", 1):
   end_if;
   if object::YMax = FAIL then
      object::dom::extslot(object, "YMax", m);
   end_if;
   //------------------
   // set hints
   //------------------
   plot::Matrixplot::setHint(object):
   //------------------
   // consistency check
   //------------------
   dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------
plot::Matrixplot::print:= () -> hold(plot::Matrixplot)("..."):

//-----------------------------------------------------------------------
plot::Matrixplot::doPlotStatic := 
proc(out, object, attributes, inherited)
  local i, j, k, l, m, n, scale, 
        xmin, xmax, ymin, ymax, zmin, zmax,
        X, Y, K, interpoly, L, style, numerical,
        isRealAndFinite,
        viewingbox, linecolorfunction, fillcolorfunction,
        nx, ny, nz, p, pt, 
        u, v, x, y, z, x1, y1, z1, x2, y2, z2,
        points;
  begin    
    if contains(attributes, LineColorFunction) then
       linecolorfunction:=  attributes[LineColorFunction] // 1 procedure,
                                                  // 3 expressions (RGB) or 
                                                  // 4 expressions (RGBa)
    else linecolorfunction:= () -> null();
    end_if;

    if contains(attributes, FillColorFunction) then
       fillcolorfunction:=  attributes[FillColorFunction] // 1 procedure,
                                                  // 3 expressions (RGB) or 
                                                  // 4 expressions (RGBa)
    else fillcolorfunction:= () -> null();
    end_if;
    
    viewingbox:= NIL:
    l:= float(attributes[Data]);  
    if domtype(l) <> DOM_LIST or
       domtype(l[1]) <> DOM_LIST then
       error("Data was lost. Bind the Matrixplot object to an identifier before plotting!");
    end_if; 
    if (_union(map({op(l[i])},domtype)$i=1..nops(l)) minus {DOM_FLOAT}) <> {} then 
      numerical:= FALSE;
    elif _lazy_or(has({op(l[i])}, {RD_INF, RD_NINF, RD_NAN}) $ i = 1..nops(l)) then
      numerical:= FALSE;
    else
      numerical:= TRUE;
    end_if; 
    isRealAndFinite:= x -> _lazy_and(domtype(x) = DOM_FLOAT,
                                     x <> RD_INF, 
                                     x <> RD_NINF, 
                                     x <> RD_NAN);
    m := nops(l); 
    n := nops(l[1]); 
    xmin:= float(attributes[XMin]);
    xmax:= float(attributes[XMax]);
    ymin:= float(attributes[YMin]);
    ymax:= float(attributes[YMax]);
    scale:= FALSE;
    if iszero(xmin - 1) and iszero(xmax - n) then 
         X:= id;
    elif n = 1 then
         X:= x -> xmin;
         scale:= TRUE;
    else X:= x -> xmin + (x - 1) / (n - 1) * (xmax - xmin);
         scale:= TRUE;
    end_if;
    if iszero(ymin - 1) and iszero(ymax - m) then 
         Y:= id;
    elif m = 1 then
         Y:= y -> ymin;
         scale:= TRUE;
    else Y:= y -> ymin + (y - 1) / (m - 1) * (ymax - ymin);
         scale:= TRUE;
    end_if;
    //--------------------------------------------------------
    // Write the surface but not the points (they may have a
    // different color and thus have to be written separately). 
    // (Note that Mesh3d plots the lines with the PointColor!)
    // The Mesh3d should react to XLinesVisible, YLinesVisible
    //--------------------------------------------------------
    style:= attributes[InterpolationStyle];
    if not contains({Linear, Cubic}, style) then
      error("unknown InterpolationStyle '".expr2text(style).
            "'. Expecting 'Linear' or 'Cubic'"):
    end_if;
    //----------------------------------------------------------------------
    // Style = Linear, the matrix contains only real numbers
    //----------------------------------------------------------------------
    if style = Linear and numerical then
      points := [];
       for j from 1 to n do
         for i from 1 to m do
            [x, y, z]:= [j, i, l[i][j]];
            if scale then
               [x, y]:= [X(x), Y(y)];
            end_if;
            points := points.[[[linecolorfunction(x, y, z)], [fillcolorfunction(x, y, z)], x, y, z]];
         end_for;
       end_for;
      viewingbox := out::writeRectangularMesh(attributes, table("PointsVisible" = FALSE), points,
        ()->op(args(1)), ()->op(args(2)),
        m, n, 0, 0, TRUE);
    end_if:
    //----------------------------------------------------------------------
    // Style = Linear, the matrix contains symbolic objects or complex stuff
    //----------------------------------------------------------------------
    if style = Linear and not numerical then
      points := [];
       zmin:= RD_INF;
       zmax:= RD_NINF;
       for j from 1 to n-1 do
         for i from 1 to m-1 do
            p:= [0 $ 4]:
            p[1]:= [ j ,  i , l[ i ][ j ]];
            p[2]:= [j+1,  i , l[ i ][j+1]];
            p[3]:= [j+1, i+1, l[i+1][j+1]];
            p[4]:= [ j , i+1, l[i+1][ j ]];
            for k from 1 to 4 do
               z:= p[k][3]:
               if isRealAndFinite(z) then
                  if z < zmin then zmin:= z: end_if;
                  if z > zmax then zmax:= z: end_if;
               else
                  p[k]:= FAIL;
               end_if:
            end_for:

            p:= select([p[1], p[2], p[3], p[4]], _not@_equal, FAIL):
            case nops(p)
            // Beware: choose a good ordering of the triangle vertices!
            // Otherwise, the color of the triangles will differ 
            // significantly from the color of neighboring triangles.
        //  of 4 do K:= [1,2,4,2,3,4]; break;
            of 4 do K:= [4,2,1,4,3,2]; break;
        //  of 3 do K:= [1,2,3]; break;
            of 3 do K:= [1,3,2]; break;
            otherwise K:= [];
            end_case;
            for k in K do
                [x, y, z]:= p[k]:
                if scale then [x, y]:= [X(x), Y(y)]; end_if;
                points := points.[[[linecolorfunction(x, y, z)], [fillcolorfunction(x, y, z)], x, y, z]];
            end_for:
         end_for;
       end_for;
      out::writeTriangles(attributes, table("HasLineColors" = bool(attributes[LineColorType] = Functional)), points,
        ()->op(args(1)), ()->op(args(2)));

       // plot the mesh lines
      points := [];
       for j from 1 to n do
         for i from 1 to m-1 do
           p:= [0 $ 2]:
           [x1, y1, z1]:= [j,  i , l[ i ][j]];
           [x2, y2, z2]:= [j, i+1, l[i+1][j]];
           if isRealAndFinite(z1) and
              isRealAndFinite(z2) then
              if scale then
               [x1, y1]:= [X(x1), Y(y1)];
               [x2, y2]:= [X(x2), Y(y2)];
              end_if;
              points := points.[[x1,y1,z1],[x2,y2,z2]];
           end_if;
         end_for;
       end_for:
       for i from 1 to m do
         for j from 1 to n-1 do
           [x1, y1, z1]:= [ j , i, l[i][ j ]];
           [x2, y2, z2]:= [j+1, i, l[i][j+1]];
           if isRealAndFinite(z1) and
              isRealAndFinite(z2) then
              if scale then
               [x1, y1]:= [X(x1), Y(y1)];
               [x2, y2]:= [X(x2), Y(y2)];
              end_if;
              points := points.[[x1,y1,z1],[x2,y2,z2]];
           end_if;
         end_for;
       end_for:
      out::writeLines3d(attributes, table(), points);

       viewingbox:= [xmin .. xmax, ymin .. ymax, zmin .. zmax];
    end_if:


    //------------------------------------
    // Cubic splines
    //------------------------------------
    if style = Cubic then
      if not numerical then
         error("Some data elements are not real numerical values. ".
               "Use Style = Linear to plot matrices containing such data.");
      end_if;
      interpoly := numeric::cubicSpline2d([$1..m], [$1..n],
                                          array(0..m-1, 0..n-1, l)); 
      [viewingbox, L] := plot::surfaceEval((x, y)->y, (x,y)->x, interpoly,
                                           m, attributes[YSubmesh], 1..m,
                                           n, attributes[XSubmesh], 1..n,
                                           1..m, 1..n, Automatic..Automatic,
                                           AdaptiveMesh = 0,
                                           "NoSingularities");
      viewingbox[1]:= xmin .. xmax; // if scale then 1..n -> xmin .. xmax
      viewingbox[2]:= ymin .. ymax; // if scale then 1..m -> ymin .. ymax
      points := [];
      for pt in L do
        if nops(pt) = 8 then // normals are provided
             [u, v, x, y, z, nx, ny, nz]:= float(pt):
        else // normals are not provided
             [u, v, x, y, z]:= float(pt):
        end_if:
        if scale then
           [x, y]:= [X(x), Y(y)];
        end_if;
        points := points.[[[linecolorfunction(x, y, z)], [fillcolorfunction(x, y, z)], x, y, z]];
      end_for:
      out::writeRectangularMesh(attributes, table("PointsVisible" = FALSE), points,
        ()->op(args(1)), ()->op(args(2)),
        m, n, attributes[YSubmesh], attributes[XSubmesh], TRUE);
    end_if;

    points := [];
    for j from 1 to n do
      for i from 1 to m do
        [x, y, z]:= [j, i, l[i][j]];
        if isRealAndFinite(z) then
          if scale then
            [x, y]:= [X(x), Y(y)];
          end_if;
          points := points.[[x,y,z]];
        end_if;
      end_for;
    end_for;
    out::writePoints3d(attributes, table(), points, ()->null(), 1..3);
   
    //--------------------------------------------
    // done
    //--------------------------------------------
    if viewingbox = NIL then
       return();
    end_if;
    return(viewingbox);
end_proc:
