/*-----------------------------------------------------------------------
plot::Raster - 2D raster plot

Call:  
  plot::Raster(colorlist,  <x = x0(a)..x1(a), y = y0(a)..y1(a), a = a0..a1>, <attributes>)
  plot::Raster(colorarray, <x = x0(a)..x1(a), y = y0(a)..y1(a), a = a0..a1>, <attributes>)
  plot::Raster(colormatrix,<x = x0(a)..x1(a), y = y0(a)..y1(a), a = a0..a1>, <attributes>)

Parameters:
    colorlist   : a list of lists of RGB or RGBa values
    colorarray  : an array(1..n, 1..m) or hfarray(1..n, 1..m) of RGB or RGBa values
    colormatrix : a matrix(1..n, 1..m) of RGB or RGBa values
    x, y        : identifiers or indexed identifiers (dummys)  
    x0(a),x1(a),  borders of the coordinate range: univariate
    y0(a),y1(a) : expressions of the animation parameter
    a           : the animation parameter: identifier or indexed identifier
    attributes  : further attributes such as FillColor, FillColor2 etc.

Details:
  - The object consists of n x m rectangles, where m, n is the
    dimension of the colorarray. Respectively, colorlist is
    a list of m sublists each having n elements (each element
    being an RGB/RGBa value).

Examples:
 >> data:= import::readBitmap("filename.jpg"):
 >> plot::Raster(data, <x = 0..1, y = 0..1>)
--------------------------------------------------------------------------*/

plot::createPlotDomain(
   "Raster",
   "graphical primitive for 2D raster plots",
   2,  // Dimension
   [ // declare the attributes visible in the object inspector
    [ColorData,        // name of the new attribute
     ["Mandatory", NIL],// [type, default value]
     ["Definition",    // where to be found in the inspector
      "ExprSeq",       // type according to DTD 
      FAIL,            // no automatic conversions
      "The color data.", // inspector info entry
      FALSE            // recalculation flag
     ] 
    ],
    XName, XMin, XMax, XMesh, Mesh,
    YName, YMin, YMax, YMesh, 
    LinesVisible, LineWidth, LineStyle, AntiAliased, Color,
    LineColor
   ]):

plot::Raster::styleSheet:= table(
                                 AntiAliased = FALSE,
                                 LinesVisible = FALSE
):
plot::Raster::hints := {AxesInFront = TRUE,
                        GridInFront = TRUE,
                        Scaling = Constrained,
                        Axes = None}:

//--------------------------------------------------------
plot::Raster::print :=
proc(obj)
begin
  if length(obj::ColorData) < plot::pointsInObjectInspector() then
    hold(plot::Raster)(obj::ColorData,
                       dom::printAttributes(obj, {ColorData}));
  else
    hold(plot::Raster)(hold([`...`]),
                       dom::printAttributes(obj, {ColorData}));
  end_if;
end_proc:

//----------------------------------------------------------------
// changeNotifier: eq is "AttributeName" (as string) = value.
// This method is called whenever the object is initialized or
//  changed via a slot call to an attribute that is not 'official' 
// (an attribute is 'official', if it was was declared above). 
// Here, the relevant attribute is ColorData 
//----------------------------------------------------------------
plot::Raster::changeNotifier := proc(object, eq)
local attributename, data, s, i, j;
begin
    attributename:= op(eq, 1);
    if attributename = "ColorData" then
       data := op(eq, 2);
       if data::dom::hasProp(Cat::Matrix)=TRUE then
          data:= expr(data);
       end_if;
       case domtype(data) 
       of DOM_LIST do // input via a list of lists
          object::dom::extslot(object, "YMesh", nops(data));
          s:= {op(data)}:
          if map(s, domtype) <> {DOM_LIST} then
             error("expecting a list of lists of RGB or RGBa values");
          elif domtype(s[1][1]) <> DOM_LIST or 
               not contains({3, 4}, nops(s[1][1])) then
             error("expecting a list of lists of RGB or RGBa values");
          end_if;
          s:= map(s, nops):
          if nops(s) <> 1 then
             error("all sublists (rows of the raster image) must have the same length");
          end_if:
          object::dom::extslot(object, "XMesh", nops(data[1]));

          if object::XMin = FAIL then
             object::dom::extslot(object, "XMin", 0):
          end_if;
          if object::XMax = FAIL then
             object::dom::extslot(object, "XMax", object::XMesh);
          end_if; 
          if object::YMin = FAIL then
             object::dom::extslot(object, "YMin", 0):
          end_if;
          if object::YMax = FAIL then
             object::dom::extslot(object, "YMax", object::YMesh);
          end_if;

           // ColorData are the 'official' data seen in the inspector
           object::dom::extslot(object, "ColorData", data);
           break;
       //-------------------------------------------------------------------
       of DOM_ARRAY do
          if op(data, [0, 1]) <> 2 then 
             error("First argument: expecting a two dimensional array ".
                   "of RGB or RGBa values")
          end_if;
          if object::XMin = FAIL then
             object::dom::extslot(object, "XMin", op(data, [0, 3, 1]) - 1);
          end_if;
          if object::XMax = FAIL then
             object::dom::extslot(object, "XMax", op(data, [0, 3, 2]));
          end_if;
          if object::YMin = FAIL then
             object::dom::extslot(object, "YMin", op(data, [0, 2, 1]) - 1);
          end_if;
          if object::YMax = FAIL then
             object::dom::extslot(object, "YMax", op(data, [0, 2, 2]));
          end_if;
          object::dom::extslot(object, "XMesh",
                              1 + op(data, [0, 3, 2]) - op(data, [0, 3, 1]));
          object::dom::extslot(object, "YMesh",
                              1 + op(data, [0, 2, 2]) - op(data, [0, 2, 1]));

          object::dom::extslot(object, "ColorData", 
            [[float(data[i,j]) $ j = op(data, [0, 3, 1]) .. op(data, [0, 3, 2])]
                               $ i = op(data, [0, 2, 1]) .. op(data, [0, 2, 2])]);
          break;
       //-------------------------------------------------------------------
       of DOM_HFARRAY do
          if op(data, [0, 1]) <> 3 or op(data, [0,4,2])-op(data, [0,4,1]) <> 2 then 
             error("First argument: expecting a three dimensional hf-array ".
                   "of size m*n*3")
          end_if;
          if object::XMin = FAIL then
             object::dom::extslot(object, "XMin", op(data, [0, 3, 1]) - 1);
          end_if;
          if object::XMax = FAIL then
             object::dom::extslot(object, "XMax", op(data, [0, 3, 2]));
          end_if;
          if object::YMin = FAIL then
             object::dom::extslot(object, "YMin", op(data, [0, 2, 1]) - 1);
          end_if;
          if object::YMax = FAIL then
             object::dom::extslot(object, "YMax", op(data, [0, 2, 2]));
          end_if;
          object::dom::extslot(object, "XMesh",
                              1 + op(data, [0, 3, 2]) - op(data, [0, 3, 1]));
          object::dom::extslot(object, "YMesh",
                              1 + op(data, [0, 2, 2]) - op(data, [0, 2, 1]));

          object::dom::extslot(object, "ColorData", [data]);
          break;
       //-------------------------------------------------------------------
       otherwise
            return("Expecting a list of lists or an array of color values");
       end_case;
       return(FALSE); // notify extslot that the assignment was already
                      // done inside the changeNotifier method
    else
       // any attributes other than ColorData are OK.
       // Let extop do the assignment
       return(TRUE);
    end_if;
end_proc:

//----------------------------------------------------------------
plot::Raster::new := proc()
local object, other;
begin
  object := dom::checkArgs([["X"], ["Y"]], args());
  if (object::XRange = FAIL and object::YRange <> FAIL)  or
     (object::YRange = FAIL and object::XRange <> FAIL)  then
     error("expecting ranges 'x = xmin .. xmax' and 'y = ymin .. ymax'");
  end_if;
  other:= object::other;
  if nops(other) <> 0 then
     if nops(other) > 1 then
        error("Unknown attribute: ".expr2text(other[2]));
     end_if;
     if object::XName = FAIL then
        object::XName:= hold(x); // ouch!
     end_if;
     if object::YName = FAIL then
        object::YName:= hold(y); // ouch!
     end_if;
     if (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
        other[1]:= expr(other[1]);
     end_if;
     case domtype(other[1])
     of DOM_LIST do
     of DOM_HFARRAY do
     of DOM_ARRAY do
        // changeNotifier does further type checking
        plot::Raster::changeNotifier(object, "ColorData" = other[1]);
        break;
     otherwise 
       error("expecting a list of list, an array, or a matrix of color data");
     end_case;
  end_if;
  //--------------------------------
  // check consistency of the object
  //--------------------------------
  dom::checkObject(object);
end_proc:

//----------------------------------------------------------------
plot::Raster::doPlotStatic := proc(out, object, attributes, inherited)
begin
  out::writeColorArray2d(attributes, table(), float(attributes[ColorData]), attributes[XMesh], attributes[YMesh],
    float(attributes[XMin]),float(attributes[XMax]),
    float(attributes[YMin]), float(attributes[YMax]));
end_proc:

//----------------------------------------------------------------------------
