/*----------------------------------------------------------------------------------

plot::Rootlocus(f(x, u), u = u0..u1)

Parameters:
   f(x, u) -- a bivariate expression depending on 2 symbolic variables
              (identifiers or indexed identifiers). It must be a rational
              function of x; the dependence on the parameter u is arbitrary.
   x, u    --  identifiers or indexed identifiers
   u0, u1  --  finite real numerical values: the range for the parameter u

Reference: googlen nach "Wurzelortskurve", z.B.:
         http://www.user.fh-stralsund.de/~emasch/1024x768/Dokumentenframe/Kompendium/Fachvorlesungen/Regelungstechnik/kapitel6.pdf


----------------------------------------------------------------------------------*/
plot::createPlotDomain("Rootlocus",
                       "curves of roots of a rational expression",
                       2,
                       [[RationalExpression, 
                         ["Mandatory", NIL],
                         ["Definition", "Expr", FAIL, "The function whose roots are to be plotted", TRUE]
                        ],
                        UName, UMin, UMax, URange, UMesh,
                        Mesh, AdaptiveMesh,
                        LineColor, LineWidth, LineStyle, LinesVisible, 
                        LineColorFunction, LineColorType,
                        AntiAliased,
                        PointSize, PointStyle, PointsVisible
                       ]
):
//-----------------------------------------------------------------------------------
plot::Rootlocus::styleSheet := table(UMesh = 51,
                                     AdaptiveMesh = 4,
                                     PointsVisible = FALSE,
                                     PointSize = 1.0,
                                     LinesVisible = TRUE):
plot::Rootlocus::hints := {Axes = Boxed}:
plot::Rootlocus::setPrimaryColor(LineColor):
//-----------------------------------------------------------------------------------
plot::Rootlocus::new:=
  proc()
    local object, other, x, ratexpr;
  begin
    // check all known options from the argument list
    object := dom::checkArgs(["U"], args());

    // get arguments which are not yet processed
    other := object::other;

    if testargs() then
      if nops(other) > 1 then
        error("unexpected argument: ".expr2text(other[2]))
      end_if;
    end_if;

    if nops(other) > 0 then
       ratexpr:= op(other, 1);
       if type(ratexpr) = "_equal" then
          ratexpr:= op(ratexpr, 1) - op(ratexpr, 2);
       end_if;
       // if ratexpr is of type DOM_POLY, we need to convert 
       // it to an expression
       if not testtype(ratexpr, Type::Arithmetical) then
          ratexpr:= expr(ratexpr);
       end_if;
       object::RationalExpression := ratexpr;
       x:= numeric::indets(object::RationalExpression) minus
           {object::UName}:
       if object::ParameterName <> FAIL then
          // there is an animation parameter
          x:= x minus {object::ParameterName};
       end_if;
       if nops(x) > 1 then
          error("first argument: expecting a bivariate expression. ".
                "Found the indeterminates ".expr2text(op(x)));
       end_if;
       if nops(x) > 0 then
          x:= op(x):
       else
          x:= hold(z):
       end_if;
       dom::extslot(object, "Z_Name", x):
    end_if;

    // identify x as a legal identifier (without the following call,
    // dom::checkObject would throw an error because of an unbound 
    // identifier x)
    dom::extslot(object, "legalIdents", {x}):

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

//-----------------------------------------------------------------------------------
plot::Rootlocus::print :=
  obj -> hold(plot::Rootlocus)(obj::RationalExpression, obj::UName = obj::URange):

//-----------------------------------------------------------------------------------
plot::Rootlocus::doPlotStatic:=
  proc(out, obj, attributes, inherited)
    local maxlevel, maxbend, subseq, ptcount, rationalized,
          u, z, umin, umax, umesh,
          umin_, umax_, umesh_, umid, urange, du,
          distribution, f, fprime, i, k, L,
          branchEqu, branch, branches, nbp,
          colorfunction, 
          xmin, xmax, ymin, ymax, K, dK, n,
          getroots, getdata, roots, data, points;
  begin
    maxbend:= float(PI)/18; // 10 degrees = the max angle between consecutive 
                            // curve segments. For larger angles, additional 
                            // points are inserted adaptively.
    maxlevel:= attributes[AdaptiveMesh]; // do up to maxlevel refinements via
                                         // bisectioning steps of the parameter
    u    := attributes[UName];
    z    := attributes[Z_Name];
    umin := float(attributes[UMin]);
    umax := float(attributes[UMax]);
    [umin, umax]:= [min(umin, umax), max(umin, umax)];
    umesh:= attributes[UMesh];
    f:= attributes[RationalExpression]:
    if not testtype(f, Type::PolyExpr([z, u])) then
      // Beware of non-rational parameter input such as (x - exp(k))^2.
      // Via normal, this is rewritten as x^2 - 2*x*exp(k) + exp(2*k)
      // and cannot be processed properly by polylib::sqrfree!
      // Hence: do rationalize:
      rationalized:= TRUE;
      [f, subseq]:= [rationalize(f)]:
      f:= numer(normal(f));
    else
      rationalized:= FALSE;
    end_if;
    if not testtype(f, Type::PolyExpr(z)) then
       error("expecting a polynomial expression");
    end_if;

    // do a squarefree factorization to avoid unnecessary multiplicities
    L:= Factored::convert_to(polylib::sqrfree(f), DOM_LIST):

    // initialize the bounding box
    xmin:= RD_INF; xmax:= RD_NINF;
    ymin:= RD_INF; ymax:= RD_NINF;

    if contains(attributes, LineColorFunction) then
         colorfunction:= attributes[LineColorFunction];
    else colorfunction:= () -> null();
    end_if;

    //----------------------------------------------------
    // getdata = adaptive scheme for computing the roots 
    // and identifying the curves on which each root lies:
    //----------------------------------------------------
    // evaluate new roots at point K, bisect
    // the interval K0 .. K if necessary
    getdata:= proc(Krange, branch, level)
    local K0, K, dK, roots, root, i, j, mindistance,
          d, r, c, r_indices, c_indices, 
          pt1, pt2, match, k, ndk;
    begin
        K0:= op(Krange, 1):
        K:= op(Krange, 2):
        dK:= K - K0;
        roots:= getroots(K);
        if nops(roots) <> n then
           return();
        end_if;

        //-----------------------------------------------------
        // Try to match the new roots and the existing curves.
        // First, pick the root and the curve with the smallest
        // distance. Regard them as a match. Take this root and
        // this curve away and repeat the process.
        //-----------------------------------------------------
        match:= table(): // store the matches: match[root] = curve index
        d:= array(1..n, 1..n):
        for r from 1 to n do
          for c from 1 to n do
            // compute the distance beetween the root r and the last
            // point on the existing curve c. Note that the points on
            // curve c are sorted w.r.t. increasing values of the 
            // parameter 
            pt1:= roots[r]:
            pt2:= data[branch][c][nops(data[branch][c])][2]:
            d[r,c]:= specfunc::abs(pt1 - pt2):
  /* ------------------------------
     // Experimental code to improve the quality of the matching:
            if nops(data[branch][c]) > 1 then
               // compare the new line segment pt1 - pt2 to the
               // last line segment pt2 - pt3 of the curve. 
               // If there is a large bend, increase the distance
               // by a factor increasing with the angle:
               pt3:= data[branch][c][nops(data[branch][c]) - 1][2]:
               if not iszero(pt2 - pt3) then
                  d[r,c]:= d[r,c]*(1 + 0.3*specfunc::abs(arg((pt1-pt2)/(pt2-pt3)))):
               end_if;
            end_if;
    ------------------------------- */
          end_for;
        end_for;
        // the roots r and the curves c that need to be matched
        r_indices:= {k $ k = 1..n}; // all roots to be matched
        c_indices:= {k $ k = 1..n}; // all curves to be matched
        for k from 1 to n do
          // find the indices [r,c] of root and curve with the
          // smallest distance
          mindistance:= RD_INF;
          for i in r_indices do
           for j in c_indices do
               if d[i,j] < mindistance then
                  mindistance:= d[i,j];
                  [r, c]:= [i, j];
               end_if;
           end_for:
          end_for:
          match[r]:= c;
          r_indices:= r_indices minus {r};
          c_indices:= c_indices minus {c};
        end_for:
        //-------------------------
        // The matching is finished. Now, decide 
        // whether bisectioning is necessary:
        //-------------------------
        for r from 1 to n do
            root:= roots[r];
            k:= match[r];
            ndk:= nops(data[branch][k]);
            if ndk = 1 and
               level < maxlevel then
               // For the 2nd point, there is no 'maxbend' criterion.
               // Insert a 2nd point very close to the starting point
               // of the branch to make this criterion available
               getdata(K0 .. K0 + dK/2^maxlevel, branch, maxlevel);
               // investigate the remaining part of the branch:
               getdata(K0 + dK/2^maxlevel .. K, branch, level);
               return();
            end_if;
            // refine by bisectioning
            ndk:= nops(data[branch][k]);
            if ndk > 1 and
               level < maxlevel and
               (not iszero(data[branch][k][ndk-1][2]-data[branch][k][ndk][2])) and
               // the 'maxbend' criterion:
               specfunc::abs(arg(
                   (root-data[branch][k][ndk][2]) /
                   (data[branch][k][ndk][2]-data[branch][k][ndk-1][2])
               )) > maxbend then
               assert(K - dK < K - dK/2);
               getdata(K - dK .. K - dK/2, branch, level + 1);
               assert(K - dK/2 < K);
               getdata(K - dK/2 .. K, branch, level + 1);
               return();
            end_if;

            // If no refinement is required: store the data.
            // Watch out: the data need to be sorted w.r.t.
            // the parameter values at all times!

            if data[branch][k][ndk][1] < K then
               data[branch][k]:= data[branch][k] . [[K, root]];
            end_if;
        end_for;
    end_proc;

    //-------------------------------
    // main
    //-------------------------------

    // treat each squarefree factor separately:
    ptcount:= 0;
    for f in [L[2*k] $ k = 1..(nops(L)-1)/2] do ;
      // ignore the leading unit and the powers of the factors:
      // f:= _mult(L[2*k] $ k = 1..(nops(L)-1)/2);
      if rationalized then
          // undo the rationalization w.r.t. the parameter
          f:= subs(f, subseq, EvalChanges):
      end_if;
      fprime:= diff(f, z);
      n:= degree(f, [z]);
      if iszero(n) then
         next;
      end_if;

      // compute the branchpoints of the root curves:
      // f(z) and f'(z) must have zeroes simultaneously
      branchEqu:= polylib::resultant(f, fprime, z):
      if iszero(branchEqu) then 
         branches:= [];
      else
         branches:= numeric::realroots(branchEqu, u = umin .. umax, 
                                       max(10^(-DIGITS), 10.0^(-5)*(umax - umin)));
      end_if;
      if nops(branches) = 0 then
         branches:= [[umin, umax]];
      else
         nbp:= nops(branches):
         branches:= [ [umin, branches[1][1]],
                      [branches[i][2], branches[i+1][1]] $ i = 1..nbp - 1,
                      [branches[nbp][2], umax] 
                  ];
      end_if;
/*
      // delete branches of length zero:
      branches:= map(branches, 
                     branch -> if iszero(branch[2] - branch[1]) then
                                    null()
                               else branch; end_if);
*/
  
      //-----------------------------------------------------
      // Utility for computing the roots with option remember.
      // Redefine getroots in this loop over all factors f to
      // get a fresh remember table for each f:
      //-----------------------------------------------------
      getroots:= proc(K) option remember;
      local p;
      begin 
         p:= subs(f, u = K, EvalChanges);
         // numeric::polyroots aborts with an error
         // if the polynomial is 0:
         if iszero(p) then return([]); end_if;
         numeric::polyroots(p, NoWarning);
      end_proc;

      //--------------------------------------
      // The work starts: Fill the data table
      //--------------------------------------
      data:= table();
      for branch from 1 to nops(branches) do
         [umin_, umax_]:= branches[branch];
         //--------------------------------------
         // initialize data table
         //--------------------------------------
         du:= umax_ - umin_;
         if iszero(du) then
            // exception: branch has length 0:
             roots:= [op(getroots(umin_))];
             data[branch]:= table(k = [[umin_,roots[k]]] $ k = 1..nops(roots));
         end_if;
         repeat
           roots:= [op(getroots(umin_))];
           if nops(roots) <> n then 
              umin_:= umin_ + du/100;
           end_if;
         until nops(roots) = n or umin_ > umax_ 
         end_repeat;

         // initialize the data table
         data[branch]:= table(k = [[umin_,roots[k]]] $ k = 1..n);

         if umin_ >= umax_ then
            next;
         end_if;
         //------------------------------------------
         // adapt the mesh size between branch points
         // by interpolation of the total mesh size:
         //------------------------------------------
         if iszero(umax - umin) then
            umesh_:= 2;
         else
            umesh_:= round((umax_ - umin_)/(umax - umin)*(umesh - 1));
            if umesh_ < 10 then umesh_:= 10 end_if;
         end_if;
         //-----------------
         // Compute the data.
         // Do not use an equidistant mesh for the parameter,
         // but let them accumulate close to the branch points
         // (the beginning and the end of the branch)
         //-----------------
         umid:= (umin_ + umax_)/2;
         urange:= (umax_ - umin_)/2;
         //-----------------
         // distribution maps equidistant points on the intervall
         // umin_ .. umax_ (represented by umind and urange) to
         // non-equidistand points accumulating at the endpoints
         // of the interval (we expect branch points at the ends,
         // that some of the curves pass with infinite speed, i.e.,
         // closer 
         distribution:= x -> if x > umid then
                               //umid + urange*2/float(PI)*arctan(1000*(x - umid)/urange)
                                 umid + ((x - umid)*urange)^(0.5)
                              else
                               //umid - urange*2/float(PI)*arctan(1000*(umid - x)/urange)
                                 umid - ((umid - x)*urange)^(0.5)
                              end_if;
         dK:= (umax_ - umin_)/(umesh_ - 1);
         K:= [distribution(umin_ + (i-1)*dK) $ i = 1 .. umesh_];
         for i from 1 to umesh_-1 do
            getdata(K[i] .. K[i+1], branch, 0);
         end_for:
    end_for:

   /*----------------------------------------
   // for debugging: verify the ordering of the plot data
   for branch from 1 to nops(branches) do
    for k from 1 to n do
      for j from 1 to nops(data[branch][k]) - 1 do
         if data[branch][k][j][1] >= data[branch][k][j + 1][1] then
            warning("unsorted branch"):
         end_if;
      end_for:
     end_for:
    end_for:
   //---------------------------------------- */

    //--------------------------------------
    // write the plot data
    //--------------------------------------
      // branchpts:= [];
      for branch from 1 to nops(branches) do

/* -------- 
        //----------------------------------
        // Other systems display the branch points of the curves.
        // If MuPAD is to do this, too, we should now determine 
        // the branch points that shall be displayed:
        //----------------------------------
        b:= [data[branch][k][1] $ k = 1..n]:
        fprimevalues:= map(b, pt -> specfunc::abs(eval(subs(fprime, [u = pt[1], z = pt[2]])))):
        maxfprimevalue:= max(fprimevalues);
        for k from 1 to n do
           if fprimevalues[k] <= maxfprimevalue/10 then
              branchpts:= branchpts. [b[k][2]];
           end_if;
        end_for:

        //--------------------------------------------------
        // plot the branch points
        //--------------------------------------------------
        for pt in branchpts do
          plot::MuPlotML::beginElem("Pt2d",
                                    "Position" = (Re(pt), Im(pt)),
                                    "PointStyle"=Diamonds,
                                    "PointSize" = 2,
                                    "PointsVisible" = TRUE,
                                    "PointColor" = RGB::Red,
                                    Empty):  /* end of method MuPlotML */
        end_for; // for branchpts
---------------  */

        //------------------------------
        // plot the lines
        //------------------------------
        for k from 1 to n do
          points := map(data[branch][k], X->[Re(X[2]), Im(X[2]), X[1]]);
          xmin := min(xmin, op(map(points, op, 1)));
          xmax := max(xmax, op(map(points, op, 1)));
          ymin := min(ymin, op(map(points, op, 2)));
          ymax := max(ymax, op(map(points, op, 2)));
          out::writePoly2d(attributes, table("Filled"=FALSE, "Closed"=FALSE),
            points, (x,y,c)->colorfunction(c, x, y));
        end_for:
      end_for:
  
      //--------------------------------------------------
      // count the points (for output via userinfo)
      //--------------------------------------------------
      for branch from 1 to nops(branches) do
       for k from 1 to n do
          ptcount:= ptcount + nops(data[branch][k])
       end_for;
      end_for:
  
    end_for; // for f in sqrfree
    //--------------------------------------------------

    userinfo(1, "total number of points  = ".expr2text(ptcount));

    //-----------------------------------
    // return the bounding box
    //-----------------------------------
    if xmin = RD_INF then
       // no point was generated: empty plot
       return([0 .. 0, 0 .. 0]);
    end_if;
    return([xmin .. xmax, ymin .. ymax]);
  end_proc:
//-----------------------------------------------------------------------------------
