/*--------------------------------------------------------------------------  
TODOs: Unschoen, dass die PointSize nicht in Modellkoordinaten
       angebbar ist und die Punkte demnach nicht genau einen
       graphischen Matrixeintrag ausfuellen.
       Das waere per ColorArray2d zu machen, aber das sprengt
       bei grossen Matrizen Zeit und Speicher.
------------------------------------------------------------------------
Calls: plot::SparseMatrixplot(A, <x = xmin .. xmax, y = ymin .. ymax >)  
            
Parameters:
    A            : Matrix or 2-dim Array
    x, y         : identifiers or indexed identifiers
    xmin .. xmax : range in x direction
                   If not set, xmin = 1 and 
                   xmax = number of data columns
    ymin .. ymax : range in y direction
                   If not set, ymin = 1 and 
                   ymax = number of data columns

Details:
  * A point is plotted for each non-zero entry 
    in the matrix
  * A row of the matrix extends in x direction,
    a row in y direction.
  * x, y serve as hints for the axes titles
  * the data can be animated

Examples: 
    plot(plot::SparseMatrixplot(matrix::random(30, 30, 100, frandom)))
--------------------------------------------------------------------------*/

plot::createPlotDomain("SparseMatrixplot",
                       "graphical primitive for sparse matrix plots",
                       2,  // Dimension
                       [[Data, ["Mandatory", NIL],
                        ["Definition", "ExprSeq", FAIL,
                         "Data to plot.", TRUE]],
                        PointSize, 
                        PointStyle, AntiAliased,
                        Color, 
                        [PointColorType, ["Optional", Flat],
                         [["Style", "Points"], "Prop", {Flat, Dichromatic},
                          "Color: flat or indicating size of matrix entries?", TRUE]],
                        PointColor,
                        [PointColor2, ["Optional", RGB::Red],
                         [["Style", "Points"], "Prop", "Color",
                          "Secondary point color.", TRUE]],
                        PointsVisible, 
                        XName, XMin, XMax, 
                        YName, YMin, YMax 
                       ]):
//------------------------------------------------
plot::SparseMatrixplot::styleSheet := table(
                      PointsVisible = TRUE,
                      PointColorType = Flat,
                      PointColor = RGB::MidnightBlue,
                      PointColor2 = RGB::Red,
                      PointStyle = Diamonds,
                      PointSize = 1.0 // unit::mm
                     ):
//------------------------------------------------
// TODO: wirklich XName, YName per Hint als Achsentitle?
plot::SparseMatrixplot::hints:= {XAxisTitle, YAxisTitle,
                                 Axes = Boxed}:
plot::SparseMatrixplot::setHint:= 
/*
  object -> plot::setHint(object,
                          XAxisTitle = expr2text(object::XName),
                          YAxisTitle = expr2text(object::YName)
                         ):
*/
  object -> (if object::XName <> FAIL and object::XAxisTitle = FAIL then
              plot::setHint(object, XAxisTitle = expr2text(object::XName)):
             end_if:
             if object::YName <> FAIL and object::YAxisTitle = FAIL then
              plot::setHint(object, YAxisTitle = expr2text(object::YName)):
             end_if):

//-----------------------------------------------------------------------
plot::SparseMatrixplot::setPrimaryColor(PointColor):
//-----------------------------------------------------------------------
plot::SparseMatrixplot::changeNotifier:=
  proc(object : plot::SparseMatrixplot, newvalue)
    local slot_name;
  begin
    slot_name:= lhs(newvalue):
    newvalue:= rhs(newvalue);
    if slot_name = "Data" then
       if newvalue::dom::hasProp(Cat::Matrix) <> TRUE then
          newvalue:= matrix(newvalue);
       end_if;
       if newvalue::dom::hasProp(Cat::Matrix) <> TRUE then
          return("expecting a matrix or an array")
       end_if;
       newvalue:= newvalue::dom::convert_to(newvalue, DOM_TABLE);
       newvalue:= [op(newvalue)];
       dom::extslot(object, "Data", newvalue);
       return(FALSE);
    end_if;
    return(TRUE);
  end_proc:
//-----------------------------------------------------------------------
plot::SparseMatrixplot::new:=
  proc()
    local object, A, tmp, m, n, other;
  begin
    object := dom::checkArgs([["X"], ["Y"]], args());

    other := object::other;
    n := 0; m := 0;
    if nops(other) > 0 then
      A := other[1];
      if A::dom::hasProp(Cat::Matrix) <> TRUE then
        A:= matrix(A);
      end_if;
      if A::dom::hasProp(Cat::Matrix) <> TRUE then
        error("first argument: expecting a matrix or an array")
      end_if;
      [m, n]:= A::dom::matdim(A);
      dom::extslot(object, "matdim", [m, n]);
      tmp:= plot::SparseMatrixplot::changeNotifier(object, "Data" = A);
      if domtype(tmp) = DOM_STRING then
        error(tmp);
      end_if:
    end_if;
   //---------------------------------------------
   // Set XMin .. YMax. They must be set, or the
   // consistency check via dom::checkObject fails
   //---------------------------------------------
   if object::XName = FAIL then
      object::XName:= hold(x); // ouch!
   end_if;
   if object::XMin = FAIL then
      object::dom::extslot(object, "XMin", 1):
   end_if;
   if object::XMax = FAIL then
      object::dom::extslot(object, "XMax", n);
   end_if;
   if object::YName = FAIL then
      object::YName:= hold(y); // ouch!
   end_if;
   if object::YMin = FAIL then
      object::dom::extslot(object, "YMin", 1):
   end_if;
   if object::YMax = FAIL then
      object::dom::extslot(object, "YMax", m);
   end_if;
   //------------------
   // set hints
   //------------------
   plot::SparseMatrixplot::setHint(object):
   //------------------
   // consistency check
   //------------------
   dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------
plot::SparseMatrixplot::print:= () -> "plot::SparseMatrixplot(...)":
//-----------------------------------------------------------------------
plot::SparseMatrixplot::doPlotStatic := 
proc(out, object, attributes, inherited)
  local A, m, n, xmin, xmax, ymin, ymax, 
        X, Y, fmin, fmax, fvalue;
  begin    
    A:= attributes[Data];  
    [m, n] := object::matdim:
    xmin:= float(attributes[XMin]);
    xmax:= float(attributes[XMax]);
    ymin:= float(attributes[YMin]);
    ymax:= float(attributes[YMax]);
    if iszero(xmin - 1) and iszero(xmax - n) then 
         X:= id;
    elif n = 1 then
         X:= x -> xmin;
    else X:= x -> xmin + (x - 1) / (n - 1) * (xmax - xmin);
    end_if;
    if iszero(ymin - 1) and iszero(ymax - m) then 
         Y:= id;
    elif m = 1 then
         Y:= y -> ymin;
    else Y:= y -> ymin + (y - 1) / (m - 1) * (ymax - ymin);
    end_if;

    //---------------------
    // For the present value of the animation parameter,
    // some elements may have become zero.
    //---------------------
    A := select(A, _not@iszero@op, 2);

    //---------------------
    // prepare color values
    //---------------------
    if attributes[PointColorType] = Flat then
       fvalue:= () -> null(): // use PointColor
    else
       A:= map(A, eq -> op(eq, 1) =  float(expr(op(eq, 2)))): // make sure the matrix entries are numerical
       fmin:= min(op(map(A, op, 2)));
       fmax:= max(op(map(A, op, 2)));
       if iszero(fmax - fmin) then
          fvalue:= () -> null(): // use PointColor
       else
          fvalue:= (i, j, f) -> zip(attributes[PointColor],
                                    attributes[PointColor2],
                                    proc(x, y) local tmp; begin
                                       tmp:= (f-fmin)/(fmax - fmin);
                                       (1 - tmp)*x + tmp*y
                                    end_proc);
       end_if;
    end_if;

    //---------------
    // write XML data
    //---------------
    if attributes[PointColorType] = Flat then
      out::writePoints2d(attributes, table(), map(A, x->[X(op(x, [1,2])), Y(op(x, [1,1]))]));
    else
      out::writePoints2d(attributes, table(),
        map(A, x->[X(op(x, [1,2])), Y(op(x, [1,1])), [fvalue(op(x,1), op(x,2))]]),
        ()->op(args(3)));
    end_if;

/*  -----------------------------------------------
    // Experimental: plot a small ColorArray2d instead of a point.
    // For large matrices, this pushes time and memory to a limit.  

    // Beware: Make sure that fvalue is defined !!!
    for x in A do  // x =  ( (i, j) = value )
        [i, j]:= [op(x, 1)]; 
        plot::MuPlotML::beginElem("ColorArray2d",
                                  "XMesh" = 1,
                                  "YMesh" = 1,
                                  "XMin" =  xmin + (j - 1)/n*(xmax - xmin),
                                  "XMax" =  xmin + j/n*(xmax - xmin),
                                  "YMin" =  ymin + (i - 1)/m*(ymax - ymin),
                                  "YMax" =  ymin + i/m*(ymax - ymin),
                                  "LinesVisible" = FALSE):
        plot::MuPlotML::prCdata(fvalue(i, j, op(x, 2)), " ");
        plot::MuPlotML::endElem("ColorArray2d"):
    end_for:
----------------------------------------------------  */

    //--------------------------------------------
    // done
    //--------------------------------------------
    return([xmin..xmax, ymin..ymax]);
end_proc:
