// 

/*
 plot::Streamlines2d -- visualize vector fields by
                        evenly distributed streamlines.

 Call:  plot::Streamlines2d([v1, v2], x=xmin..xmax,
                            y=ymin..ymax,
                           < MinimumDistance = dmin>)

        plot::Streamlines2d( v1, v2,  x=a..b, y=c..d <.options>)
        plot::Streamlines2d([v1, v2], x=a..b, y=c..d <.options>)
        plot::Streamlines2d(V, x=a..b, y=c..d <.options>)

 Parameters: v1, v2 - expressions in x, y and the animation parameter
             V      - a matrix with 2 entries providing v1, v2

 Examples:

 plot::Streamlines2d([1, sin(x)+cos(y)], x=0..3, y=1..2)

 plot(plot::Streamlines2d([sin(x^2+y^2), cos(x^2+y^2)], x=-3..3, y=-3..3,
                          MinimumDistance=0.05),
      plot::VectorField2d([sin(x^2+y^2), cos(x^2+y^2)], x=-3..3, y=-3..3,
                          Mesh = [61, 61])):

 plot(plot::Streamlines2d([sin(x^2+y^2), cos(x^2-y^2)], x=-3..3, y=-3..3,
                          MinimumDistance=0.05),
      plot::VectorField2d([sin(x^2+y^2), cos(x^2-y^2)], x=-3..3, y=-3..3,
                          Mesh = [61, 61])):

 plot(plot::Streamlines2d([x, y^2], x = -2..2, y = -2..2,
                          LineColorFunction = ((x,y,vx,vy) ->
                                               [abs(vx)/sqrt(vx^2 + vy^2),
                                                0,
                                                abs(vy)/sqrt(vx^2 + vy^2)]),
                          MinimumDistance=0.1, TipLength=3))

 plot(plot::Streamlines2d(y, -x, x=-2..2, y=-2..2))

 */
 /* TODO TODO TODO TODO TODO TODO TODO TODO TODO TODO TODO TODO
    
    Fix:
    
    TODO TODO TODO TODO TODO TODO TODO TODO TODO TODO TODO TODO */
//----------------------------------------------------------------
plot::createPlotDomain("Streamlines2d",
                       "graphical primitive for 2D streamlines of vector fields",
                       2,
                       [XFunction, YFunction,
                        XName, XMin, XMax,
                        YName, YMin, YMax,
                        [MinimumDistance, ["Optional", NIL],
                         ["Calculation", "Expr", FAIL,
                          "Minimum distance between stream lines.", TRUE]],
                        [ODEMethod, ["Optional", DOPRI78],
                         ["Calculation", "Expr", //"Prop",
                          FAIL,
                      /*
                        {EULER1,    RKF43,      xRKF43,
                         RKF34,     xRKF34,     RK4,
                         RKF54a,    RKF54b,     DOPRI54,
                         xDOPRI54,  CK54,       xRKF54a,
                         xRKF54b,   xCK54,      RKF45a,
                         RKF45b,    DOPRI45,    CK45,
                         xRKF45a,   xRKF45b,    xDOPRI45,
                         xCK45,     BUTCHER6,   RKF87,
                         xRKF87,    RKF78,      xRKF78,
                         DOPRI65, xDOPRI65, DOPRI56, xDOPRI56,
                         DOPRI87, xDOPRI87, DOPRI78, xDOPRI78,
                         GAUSS(s),  ABM4
                        },
                      */
                          "Numerical evaluation scheme.", TRUE]],
                        [Stepsize, ["Optional", NIL],
                         ["Calculation", "Expr", FAIL,
                          "Manual step size control.", TRUE]],
                        [RelativeError, ["Optional", NIL],
                         ["Calculation", "Expr", FAIL,
                          "Tolerance for relative errors.", TRUE]],
                        [AbsoluteError, ["Optional", NIL],
                         ["Calculation", "Expr", FAIL,
                          "Tolerance for absolute errors.", TRUE]],
                        LineStyle, LineWidth, LineColor, AntiAliased,
                        LineColorType, LineColor2, LineColorFunction,
                        TipAngle, TipLength, TipStyle
                       ]):
//----------------------------------------------------------------
plot::Streamlines2d::styleSheet := table(LineColorType = Flat,
                                         LineColor = RGB::Black,
                                         LineWidth = 0.35*unit::mm,
                                         ODEMethod = ABM4,
                                         RelativeError = 10^(-5),
                                         TipLength = 0):

//----------------------------------------------------------------
plot::Streamlines2d::new :=
proc()
  local object, other;
begin
  object := dom::checkArgs(["X", "Y"], args());
  other := object::other;
  
  if nops(other) > 0 then
    if (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
      // convert a matrix to a list
      other[1]:= [op(other[1])];
    end_if;

    if domtype(other[1]) = DOM_LIST then
      object::XFunction := other[1][1];
      object::YFunction := other[1][2];
      
      delete other[1];
    elif nops(other) > 1 then
      // XFunction and YFunction must be the first two args
      object::XFunction := other[1];
      object::YFunction := other[2];
      
      delete other[1];
      delete other[1];
    end_if;

    if nops(other) > 0 then
      if has(other[1],
             {EULER1,    RKF43,      xRKF43,
              RKF34,     xRKF34,     RK4,
              RKF54a,    RKF54b,     DOPRI54,
              xDOPRI54,  CK54,       xRKF54a,
              xRKF54b,   xCK54,      RKF45a,
              RKF45b,    DOPRI45,    CK45,
              xRKF45a,   xRKF45b,    xDOPRI45,
              xCK45,     BUTCHER6,   RKF87,
              xRKF87,    RKF78,      xRKF78,
              DOPRI65,   xDOPRI65, DOPRI56, xDOPRI56,
              DOPRI87,   xDOPRI87, DOPRI78, xDOPRI78,
              GAUSS,     ABM4
             }) then
        object::ODEMethod := other[1];
        delete other[1];
      end_if;
    end_if;
    
    if nops(other) > 0 then
      error("Unexpected argument: ".expr2text(other[1]));
    end_if;
  end_if;
  
  dom::checkObject(object);
end_proc:

plot::Streamlines2d::print :=
obj -> hold(plot::Streamlines2d)([obj::XFunction, obj::YFunction],
                                 obj::XName = obj::XRange,
                                 obj::YName = obj::YRange):
                                 
plot::Streamlines2d::doPlotStatic :=
proc(out, obj, attributes, inheritedAttributes)
  local f, fx, fy, xmin, xmax, ymin, ymax, Q,
        currline, line, dmin, dstart, invdmin,
        grid, insert, traceline, i, j, curve,
        odesolveoptions, plotline, xy, dxy, nxy,
        goodstartpoint, linecolorfunction, atol,
        rtol, eukl_dist, rnd, dstartSquare, dminSquare,
        done, k, points,
        // shared between traceline and odesolve:
        odesolve, lastfxvals, lastfyvals
        //
        ;
begin
  fx := attributes[XFunction];
  fy := attributes[YFunction];
  f  := subs((t, Y)->[`#xf`(Y[1], Y[2]), `#yf`(Y[1], Y[2])],
             [`#xf`=fx, `#yf`=fy]):
  
  /*
  f := subs(proc(t, Y)
              local xf, yf, dist;
            begin
              xf := `#xf`(Y[1], Y[2]);
              yf := `#yf`(Y[1], Y[2]);
              dist := specfunc::sqrt(xf^2+yf^2);
              if iszero(dist) then
                [0.0, 0.0];
              else
                [xf/dist, yf/dist];
              end_if
            end_proc,
            [`#xf`=attributes[XFunction], `#yf`=attributes[YFunction]]):
   */
  
  xmin := attributes[XMin];
  xmax := attributes[XMax];
  ymin := attributes[YMin];
  ymax := attributes[YMax];

  rtol:= 10.0^(-4):
  atol:= min(xmax  - xmin, ymax - ymin)/10.0^4:

  odesolveoptions := attributes[ODEMethod];
  if contains(attributes, Stepsize) then
    odesolveoptions := odesolveoptions, Stepsize = attributes[Stepsize];
  end_if;
  if contains(attributes, RelativeError) then
    odesolveoptions := odesolveoptions,
    RelativeError = attributes[RelativeError];
  else
    odesolveoptions := odesolveoptions,
    RelativeError = rtol;
  end_if;
  if contains(attributes, AbsoluteError) then
    odesolveoptions := odesolveoptions,
    AbsoluteError = attributes[AbsoluteError];
  else
    odesolveoptions := odesolveoptions,
    AbsoluteError = atol;
  end_if;
  
  if contains(attributes, LineColorFunction) then
    linecolorfunction := attributes[LineColorFunction];
  else
    linecolorfunction := null();
  end_if;
  
  if contains(attributes, MinimumDistance) then
    dmin := attributes[MinimumDistance];
  else
    dmin := max(ymax-ymin, xmax-xmin)*0.02;
  end_if;
  invdmin := 1/dmin;
  
  grid := table();
  for i from -1 to ceil((xmax-xmin)*invdmin)+1 do
    for j from -1 to ceil((ymax-ymin)*invdmin)+1 do
      grid[i,j] := table();
    end_for;
  end_for;
  
  // Is (x, y) a proper starting point for a new curve?
  dstart := 2*dmin;
  dstartSquare := (0.75*dstart)^2;
  goodstartpoint := 
    proc(x, y)
      local i, j, z, cell;
    begin
      i := specfunc::floor((x-xmin)*invdmin);
      j := specfunc::floor((y-ymin)*invdmin);

      for cell in
/*
 [grid[i-2,j-2], grid[i-2,j-1], grid[i-2,j], grid[i-2,j+1], grid[i-2,j+2],
  grid[i-1,j-2], grid[i-1,j-1], grid[i-1,j], grid[i-1,j+1], grid[i-1,j+2],
  grid[i,  j-2], grid[i,  j-1], grid[i,  j], grid[i,  j+1], grid[i,  j+2],
  grid[i+1,j-2], grid[i+1,j-1], grid[i+1,j], grid[i+1,j+1], grid[i+1,j+2],
  grid[i+2,j-2], grid[i+2,j-1], grid[i+2,j], grid[i+2,j+1], grid[i+2,j+2]]
 */
          // search from inside out, simply faster
          [grid[i, j],
           grid[i-1, j], grid[i+1, j],
           grid[i, j-1], grid[i, j+1],
           grid[i-1, j-1], grid[i+1, j-1],
           grid[i-1, j+1], grid[i+1, j+1],
           grid[i-2, j], grid[i+2, j],
           grid[i, j-2], grid[i, j+2],
           grid[i-2, j-1], grid[i-2, j+1],
           grid[i+2, j-1], grid[i+2, j+1],
           grid[i-1, j-2], grid[i-1, j+2],
           grid[i+1, j-2], grid[i+1, j+2]
//         ,grid[i-2, j-2], grid[i-2, j+2],
//         grid[i+2, j-2], grid[i+2, j+2]
          ]
        do
        if not testtype(cell, DOM_TABLE) then
          next;
        end_if;
        for z in cell do
          if op(z,[2,1]) = -1 then next; end_if; // pseudo-curve
          if (x-op(z, [1,1]))^2 + (y-op(z, [1,2]))^2
             < dstartSquare then
            return(FALSE);
          end_if;
        end_for;
      end_for;
      TRUE;
    end_proc:
  
  // insert a new point unless its distance
  // to the set of previously computed points
  // is less than dmin.  return TRUE if point
  // accepted & inserted, FALSE otherwise.
  dminSquare := dmin^2;
  insert := proc(x, y, curvenumber, t)
              local i, j, z, cell;
            begin
              i := specfunc::floor((x-xmin)*invdmin);
              j := specfunc::floor((y-ymin)*invdmin);
              
              for cell in [grid[i-1,j-1], grid[i-1,j], grid[i-1,j+1],
                           grid[i,  j-1], grid[i,  j], grid[i,  j+1],
                           grid[i+1,j-1], grid[i+1,j], grid[i+1,j+1]] do
                if not testtype(cell, DOM_TABLE) then
                  next;
                end_if;
                for z in cell do
                  // ignore pseudo-curve
                  if op(z, [2,1]) = -1 then next; end_if;
                  // ignore points on the same curve,
                  // unless the Euclidean distance travelled
                  // since that point is larger than 2*dmin:
                  if op(z, [2,1]) = curvenumber and
                     specfunc::abs(t-op(z, [2,2])) < 2*dmin
                    then next;
                  end_if;
                  if (x-op(z, [1,1]))^2 + (y-op(z, [1,2]))^2
                     < dminSquare then
                    return(FALSE);
                  end_if;
                end_for;
              end_for;
              grid[i,j][[x, y]] := [curvenumber, t];
              TRUE;
            end_proc:
  
  // Euclidean distance
  eukl_dist := (x1, y1, x2, y2) -> specfunc::abs((x1-x2)+(y1-y2)*I);
  
  if attributes[ODEMethod] = ABM4 then
    // ode solver using Adams-Bashford-Moulton predictor/corrector
    // method with constant step size, seeded by DOPRI78
    lastfxvals := lastfyvals := [];
    odesolve := proc(x, y, h)
                  local xp, yp, fxy;
                begin
                  if nops(lastfxvals) < 4 then
                  // still seeding
                    [x, y] := numeric::odesolve(f, 0..h, [x, y],
                                                Stepsize = h/2,
                                                subs(odesolveoptions,
                                                     ABM4=DOPRI45));
                    lastfxvals := lastfxvals.[fx(x, y)];
                    lastfyvals := lastfyvals.[fy(x, y)];
                  else
                  // predictor
                    xp := x + h/24 * (+ 55*lastfxvals[4]
                                      - 59*lastfxvals[3]
                                      + 37*lastfxvals[2]
                                      -  9*lastfxvals[1]);
                    yp := y + h/24 * (+ 55*lastfyvals[4]
                                      - 59*lastfyvals[3]
                                      + 37*lastfyvals[2]
                                      -  9*lastfyvals[1]);
                  // corrector
                    x := x + h/24 * (+  9*fx(xp, yp)
                                     + 19*lastfxvals[4]
                                     -  5*lastfxvals[3]
                                     +    lastfxvals[2]);
                    y := y + h/24 * (+  9*fy(xp, yp)
                                     + 19*lastfyvals[4]
                                     -  5*lastfyvals[3]
                                     +    lastfyvals[2]);
                    lastfxvals := lastfxvals[2..4].[fx(x, y)];
                    lastfyvals := lastfyvals[2..4].[fy(x, y)];
                  end_if;
                  [[x, y]];
                end_proc:
  else
    odesolve := (x, y, h) -> map(numeric::odesolve(f, 0..h, [x, y],
                                                   odesolveoptions,
                                                   Alldata)[2..-1],
                                 op, 2);
  end_if;
  
  
  // routine tracing one line in both directions
  curve := 0;
  traceline := proc(x0, y0)
                 local x, y, t, tstep, prevx, prevy, pprevx, pprevy,
                       lxy, i, l, dist, done, tl, dir, tx, ty, lsol, xy,
                       prevdist, dx, dy, prevdx, prevdy;
               begin
                 l := table(1 = table(0=[x0, y0, 0.0, 0.0]),
                            -1 = table());
                 tstep := dmin/2;
                 if has(attributes, Stepsize) then
                   tstep := attributes[Stepsize];
                 end_if;
                 
                 insert(x0, y0, curve, 0);
                 for dir in [1, -1] do
                   lastfxvals := lastfyvals := [];
                   t := 0.0;
                   tl := 0.0;
                   [x, y] := [x0, y0];
                   [dx, dy] := [0.0, 0.0];
                   dist := max(xmax-xmin, ymax-ymin)/300.0;
                   if contains(l[1], 1) then
                     [prevx, prevy] := l[1][1][1..2];
                   else
                     [prevx, prevy] := [x0, y0];
                   end_if;
                   done := FALSE;
                   repeat
                     if traperror((lsol :=
                                   odesolve(x, y, dir*tstep)))
                        <> 0 or
                        hastype(lsol, DOM_COMPLEX) then
                       done := TRUE;
                       break;
                     end_if;
                     for xy in lsol do
                       [pprevx, pprevy] := [prevx, prevy];
                       [prevx,  prevy]  := [x, y];
                       [prevdx, prevdy] := [dx, dy];
                       [x, y] := xy;
                       prevdist := dist;
                       [dx, dy] := [x - prevx, y - prevy];
                       dist := eukl_dist(x, y, prevx, prevy);
                       if dist < max(xmax-xmin, ymax-ymin)*1e-7 or
                         dist > 4*dmin or
                         (dist > 5*prevdist and
                           // check angle > 10°, ignoring ratios
                           (dx*prevdx + dy*prevdy)^2 < 
                            0.984^2*(dx^2 + dy^2)*(prevdx^2 + prevdy^2))
                       then
                         done := TRUE;
                       elif dist > tstep then
                         tx := prevx;
                         ty := prevy;
                         for i from 0 to 1 step dmin/(3*dist) do
                           if iszero(i) then next; end_if;
                           lxy := [prevx + i*(1/2*x - 1/2*pprevx +
                                              i*(1/2*x + 1/2*pprevx - prevx)),
                                   prevy + i*(1/2*y - 1/2*pprevy +
                                              i*(1/2*y + 1/2*pprevy - prevy)),
                                   t + dir*i*tstep];
                           tl := tl + dir * eukl_dist(lxy[1], lxy[2], tx, ty);
                           [tx, ty] := lxy[1..2];
                           if not insert(lxy[1], lxy[2], curve, tl) then
                             done := TRUE;
                             break;
                           end_if;
                           l[dir][nops(l[dir])] := lxy.[tl];
                         end_for;
                         if not done and
                            eukl_dist(tx, ty, x, y) > dmin/4 then
                           tl := tl + dir * eukl_dist(tx, ty, x, y);
                           done := not insert(x, y, curve, tl);
                           if not done then
                             t := t + dir*tstep;
                             l[dir][nops(l[dir])] := [x, y, t, tl];
                           end_if;
                         end_if;
                       else // dist <= tstep
                         tl := tl + dir*eukl_dist(x, y, prevx, prevy);
                         done := not insert(x, y, curve, tl);
                         if not done then
                           t := t + dir*tstep;
                           l[dir][nops(l[dir])] := [x, y, t, tl];
                         end_if;
                       end_if;
                       if not done then
                     // left image boundaries?
                         if x < xmin-dmin/2 or
                            x > xmax+dmin/2 or
                            y < ymin-dmin/2 or
                            y > ymax+dmin/2 then
                           done := TRUE;
                         elif nops(l[dir]) > 3000 then
                           done := TRUE;
                         end_if;
                       end_if;
                       if done then break; end_if;
                     end_for;
                   until done
                   end_repeat;
                 end_for; // dir
                 
                 if nops(l[1]) + nops(l[-1]) < 5 then
                   // empirically, these short snippets
                   // do more harm than good
                   grid := map(grid, select, eq -> op(eq,[2,1]) <> curve);
                   return([]);
                 end_if;
                 
//               print("curve ".expr2text(curve)." done");
                 curve := curve + 1;
                 [l[-1][i] $ i = nops(l[-1])-1..0 step -1,
                  l[1][i] $ i = 0..nops(l[1])-1];
               end_proc:
  
  // do plot the line
  plotline := proc(line)
                local xy, vx, vy, tmin, lmin, mid, col;
              begin
                if line = [] then return(); end_if;

                points := [];
                if contains(attributes, LineColorFunction) then
                  tmin := line[1][3];
                  lmin := line[1][4];
                  for xy in line do
                    xy := float(xy);
                    // `#t` is just a dummy
                    if traperror(([vx, vy]:= f(`#t`, xy))) = 0 then
                      points := points.[[xy[1], xy[2],
                                            [linecolorfunction(xy[1], xy[2],
                                                              vx, vy,
                                                              xy[3]-tmin,
                                                              xy[4]-lmin,
                                                              curve)]]];
                    else
                       points := points.[[xy[1], xy[2], []]];
                    end_if;
                  end_for;
                else
                  for xy in line do
                    points := points.[[xy[1], xy[2], []]];
                  end_for;
                end_if;
                out::writePoly2d(attributes, table("Closed" = FALSE, "Filled" = FALSE, "PointsVisible" = FALSE),
                  points, ()->op(args(3)));

                if nops(line) > 10 then
                  mid := specfunc::floor(nops(line)/2);
                  xy := float(line[mid]);
                  if xy[1] >= xmin and
                     xy[1] <= xmax and
                     xy[2] >= ymin and
                     xy[2] <= ymax then
                    
                    col := null();
                    if contains(attributes, LineColorFunction) then
                      if traperror(([vx, vy] := f(`#t`,
                                                float(xy))))
                         = 0 then
                        col := ("LineColor" = linecolorfunction(xy[1],
                                                                xy[2],
                                                                vx, vy,
                                                                xy[3]-tmin,
                                                                xy[4]-lmin,
                                                                curve));
                      end_if;
                    end_if;

                    out::writeArrow2d(attributes, table(), op(xy[1..2]), op(line[mid+1][1..2]));
                  end_if;
                end_if;
              end_proc:
  
  // start in the middle initially
  line := traceline(float(xmin+xmax)/2, float(ymin+ymax)/2);
  
  // unless that didn't work, of course
  rnd := frandom(1745328):
  while nops(line) < 5 do
    line := traceline(xmin + rnd()*(xmax-xmin),
                      ymin + rnd()*(ymax-ymin));
  end_while;
  
  plotline(line);
  
  // queue of lines to trace for further starting points
  Q := adt::Queue(line);

  repeat
    while not Q::empty() do
      currline := Q::dequeue();
      for i from 2 to nops(currline) do
        xy := currline[i];
        dxy := zip(xy, currline[i-1], _subtract);
        nxy := [-dxy[2]/specfunc::sqrt(dxy[1]^2+dxy[2]^2)*dstart,
              +dxy[1]/specfunc::sqrt(dxy[1]^2+dxy[2]^2)*dstart];
        for dxy in [[xy[1]+nxy[1], xy[2]+nxy[2]],
                    [xy[1]-nxy[1], xy[2]-nxy[2]]] do
          if dxy[1] < xmin - dmin or
             dxy[1] > xmax + dmin or
             dxy[2] < ymin - dmin or
             dxy[2] > ymax + dmin then
            next;
          end_if;
          if goodstartpoint(dxy[1], dxy[2]) then
            line := traceline(dxy[1], dxy[2]);
            Q::enqueue(line);
            plotline(line);
          end_if;
        end_for;
      end_for;
    end_while;
    done := TRUE;
    for i from -1 to ceil((xmax-xmin)*invdmin)+1 do
      for j from -1 to ceil((ymax-ymin)*invdmin)+1 do
        if nops(grid[i,j]) = 0 then
          for k from 1 to 2 do
            line := traceline(xmin + (i+rnd())*dmin,
                              ymin + (j+rnd())*dmin);
            if nops(line) > 4 then
              done := FALSE;
              plotline(line);
              Q::enqueue(line);
            end_if;
            if not done then break; end_if;
          end_for;
          // no curve? then never try again
          if nops(grid[i,j]) = 0 then
            grid[i,j][FAIL] := [-1];
          end_if;
        end_if;
//        if not done then break; end_if;
      end_for;
//      if not done then break; end_if;
    end_for;
  until done end_repeat;
  
  [xmin..xmax, ymin..ymax];
end_proc:
