/* -------------------------------------------------------------
plot::Surface -- The graphical primitive for three-dimensional surfaces

Call(s):
    Surface([x(u,v,<t>), y(u,v,<t>), z(u,v,<t>)], 
            u = umin .. umax,
            v = vmin .. vmax,
            t = tmin .. tmax, // t = the animation parameter
            <Color = c>,
            <Mesh = [nu, nv]>,
            <Submesh = [mu, mv]>,
            <AdaptiveMesh = l>
            <opt1, opt2, ...>)
    Surface(xyz,
            u = umin .. umax,
            v = vmin .. vmax,
            t = tmin .. tmax, // t = the animation parameter
            <Color = c>,
            <Mesh = [nu, nv]>,
            <Submesh = [mu, mv]>,
            <AdaptiveMesh = l>
            <opt1, opt2, ...>)

Parameters:
    x(u,v),   : arithmetical expressions depending on the 
    y(u,v)    : surface parameters u and v and, possibly, 
    z(u,v)    : on the animation parameter.
    xyz       : a procedure such that xyz(u, v) returns a list
                [x(u, v), y(u, v), z(u, v)]
    u, v      : DOM_IDENTs or indexed identifiers
    umin..umax: numerical real values or univariate expressions
    vmin..vmax: of the animation  parameter
    c         : an RGB value, an RGBa value or a procedure
    nu,nv     : the number of mesh points: integers > 0
    mu,mv     : the number of 'inner mesh points': integers >= 0
    l         : the number of recursive refinements in the
                adaptive mode: an integer >= 0

Options:
    opt1, opt2, ... : general 3d plot options

Example:
>> plot::Surface([r*cos(phi), r*sin(phi), r^2*sin(phi)*cos(phi)],
                    r = 0 .. 2, phi = 0..2*PI )

----------------------------------------------------------------*/
plot::createPlotDomain("Surface", "graphical primitive for surfaces"):
//------------------------------------------------
plot::Surface::styleSheet:= table(
   // ULinesVisible = TRUE; // 'global' default
   // VLinesVisible = TRUE; // 'global' default
      LegendEntry = TRUE,
      MeshVisible = FALSE,
      LineColor = RGB::Black.[0.25], //global default is RGB::Red
      LineColorDirectionY=0
):
//------------------------------------------------
plot::Surface::print :=
  obj -> hold(plot::Surface)([obj::XFunction, obj::YFunction, obj::ZFunction],
                             obj::UName=obj::URange,
                             obj::VName=obj::VRange):
//------------------------------------------------
plot::Surface::new:= proc()
local object, other, xyz, u, v, a; 
option escape;
begin
    // 1st step:   call dom::checkArgs to filter out all generic arguments
    //             that are allowed for arbitrary primitives/objects. Let
    //             'object' be the return value of dom::checkArgs
    // object: a domain element of plot::Surface
    // (see model.mu). The special input parameters for this
    // object are in object::other.
    object:= dom::checkArgs(["U", "V"], args());

    // process the special attributes in object::other. In the
    // case of Surface, this is just the list [[x(u,v), y(u,v), z(u,v)]]
    other:= object::other;

    if nops(other) > 1 then
      error("unexpected argument: ".expr2text(other[2]));
    end_if;

    // In addition to other = [[x, y, z]], we also allow [matrix(x, y, z)]
    if nops(other) > 0 and (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
       other[1] := [op(other[1])];
    end_if;

    // Convert the input parameters special to this plot object into
    // the generic form "AttributeName = AttributeValue" as definied in
    // the DTD. In this case, the form must be Function = f(x), where
    // 'Function' is a (protected) attribute name.

    if nops(other) > 0 then
      if nops(other) > 1 then
           error("unexpected argument: ".expr2text(other[2]));
      end_if;
      other:= op(other): // now, other = [x(u,v),y(u,v),z(u,v)]
      //------------------
      // type checking: the paramtrization may be given either
      // as a list [x(u,v), y(u,v), z(u,v)] or as a procedure
      // returning such a list 
      //------------------
      case type(other) 
      of DOM_LIST do
         if  nops(other) <> 3 then
            error("First argument: expecting a list of 3 function expressions"):
         end_if;
         object::XFunction:= other[1];
         object::YFunction:= other[2];
         object::ZFunction:= other[3];
         break;
      of DOM_PROC do
      of DOM_FUNC_ENV do
      of DOM_EXEC do
      of "_fconcat" do
         // try to evaluate the procedure at a random point in
         // the parameter range and inspect the return value:
         u:= float(object::UMin + frandom()*(object::UMax - object::UMin));
         v:= float(object::VMin + frandom()*(object::VMax - object::VMin));
         // If the surface is animated we also have to provide a value for
         // the animation parameter (in case the user defined function expects
         // three parameters. If not, it doesn't harm to provide this value)
         if object::ParameterRange <> NIL then
            a:= object::ParameterBegin + frandom()*(object::ParameterEnd - object::ParameterBegin);
         else
            a:= null();
         end_if;
         if traperror((xyz:= other(u, v, a))) = 0 and
            domtype(xyz) = DOM_LIST and
            nops(xyz) = 3 then
            // everything seems fine
            // the body of the following functions may become visible in
            // the inspector of the MuPAD graphics. So, do not use
            // 'other', but a better name:
            xyz:= other;
            /* --------------------------------------------------
               With the following version, the user must pass the
               animation parameter to his function xyz! Using a global
               variable does not work (because substitution by a
               numerical value does not work)
            object::XFunction:= () -> xyz(args())[1];
            object::YFunction:= () -> xyz(args())[2];
            object::ZFunction:= () -> xyz(args())[3];
            -------------------------------------------------- */
            // With the following version, the user may pass the
            // animation parameter to his procedure xyz or he may
            // also use a global variable for the animation parameter
            // (substitution by a numerical value will work)
            object::XFunction:= (xyz -> xyz[1])@xyz;
            object::YFunction:= (xyz -> xyz[2])@xyz;
            object::ZFunction:= (xyz -> xyz[3])@xyz;
         else
            error("First argument: the procedure supplying the parametrization ".
                  "does not return a list of 3 numerical coordinate values"):
         end_if;
         break;
      otherwise
         error("First argument: expecting a list of 3 function expressions/procedures ".
               "or a procedure returning such a list");
       end_case;
      //-------------------
      // type checking done
      //-------------------
    end_if; // nops(other) > 0
  
    dom::checkObject(object);
end_proc:


//--------------------------------------------------------------------
//--------------------------------------------------------------------
//--------------------------------------------------------------------
// generic routine for plotting a non-animated object or a single frame
// of an animated one, called from the parent objects.
// Here, the computation of the data is invoked.
//--------------------------------------------------------------------
plot::Surface::doPlotStatic:= proc(out, object, attributes, inheritedAttributes)
// out:        output object, providing the specifics for MuPlotML, POV, etc.
// object:     the object that was created by the new method above.
//             Usually, we do not need it here.
// attributes: a table of all attributes (inherited attributes as well as
//             those created from the mandatory parameters in the new method).
//             Zusaetzlich sind enthalten:
//             attributes[TimeValue] (float)
//             attributes[ParameterValue] (float)
//             However, hints are not included in 'attributes'.
// inheritedAttributes:
//             the MuPlotML will be called with 3 arguments.
//             However, you do no need to process the 3rd
//             argument 'inheritedAttributes': just ignore it.
local fX, fY, fZ, 
      u, umin, umax, umesh, usubmesh,
      v, vmin, vmax, vmesh, vsubmesh,
      adaptivemesh,
      fillcolortype, fillcolorfunction,
      linecolortype, linecolorfunction, haslinecolorfunction,
      xmin, xmax, 
      ymin, ymax, 
      zmin, zmax, 
      viewingbox, data, i, ulines, vlines,
      _min, _max, _range, clipFactor, clipBranch, line;
begin
    fX   := attributes[XFunction]; // a procedure
    fY   := attributes[YFunction]; // a procedure
    fZ   := attributes[ZFunction]; // a procedure
    u    :=        attributes[UName]:
    umin :=        attributes[UMin]:
    umax :=        attributes[UMax]:
    umesh:=        attributes[UMesh];
    usubmesh:=     attributes[USubmesh];
    v    :=        attributes[VName]:
    vmin :=        attributes[VMin]:
    vmax :=        attributes[VMax]:
    vmesh:=        attributes[VMesh];
    vsubmesh:=     attributes[VSubmesh];
    adaptivemesh:= attributes[AdaptiveMesh];

    if iszero(float(umax - umin)) and umesh > 2 then
       umesh:= 2;    // for speed
       usubmesh:= 0: // for speed
    end_if;
    if iszero(float(vmax - vmin)) and vmesh > 2 then
       vmesh:= 2;   // for speed
       vsubmesh:= 0:// for speed
    end_if;

    fillcolortype:=attributes[FillColorType]; // either Flat, Monochrome(=Height),
                                              // Dichromatic, Rainbow or Functional.
    linecolortype:=attributes[LineColorType]; // either Flat, Monochrome(=Height), 
                                              // Dichromatic, Rainbow or Functional.
    // If one of the color types above is Functional, we need to compute
    // and write color values. Otherwise, no action is required here.
    [xmin,xmax,ymin,ymax,zmin,zmax]:= [Automatic $ 6]:
    if contains(attributes, ViewingBoxXMin) then xmin:= attributes[ViewingBoxXMin]: end_if;
    if contains(attributes, ViewingBoxXMax) then xmax:= attributes[ViewingBoxXMax]: end_if;
    if contains(attributes, ViewingBoxYMin) then ymin:= attributes[ViewingBoxYMin]: end_if;
    if contains(attributes, ViewingBoxYMax) then ymax:= attributes[ViewingBoxYMax]: end_if;
    if contains(attributes, ViewingBoxZMin) then zmin:= attributes[ViewingBoxZMin]: end_if;
    if contains(attributes, ViewingBoxZMax) then zmax:= attributes[ViewingBoxZMax]: end_if;
    if contains(attributes, FillColorFunction) then 
      fillcolorfunction:= float@(attributes[FillColorFunction]); // a procedure
    else fillcolorfunction:= () -> null();
    end_if;
    if contains(attributes, LineColorFunction) then 
      linecolorfunction:= float@(attributes[LineColorFunction]); // a procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;
    //--------------------------------------------
    // let the library compute the graphical data:
    //--------------------------------------------
    //-------------------------------------------------------------
    // Set the parameters:
    //-------------------------------------------------------------
    [umin, umax, vmin, vmax, xmin, xmax, ymin, ymax, zmin, zmax]:= 
           float([umin, umax, vmin, vmax, xmin, xmax, ymin, ymax, zmin, zmax]);
    //-------------------------------------------------------------
    // Modifications for logarithmic plots
    //  !!! This still needs to be tested !!!
    //-------------------------------------------------------------
/*                             
    if attributes[CoordinateType] in 
       {LogLinLin, LogLogLin, LogLinLog, LogLogLog} then
       if (domtype(xmin) = DOM_FLOAT and xmin <= 0) then
          error("expecting a positive X range for the viewing box of a logarithmic X axis. ".
                "Got ViewingBoxXmin = ".expr2text(xmin));
       end_if:
       if (domtype(xmax) = DOM_FLOAT and xmax <= 0) then
          error("expecting a positive X range for the viewing box of a logarithmic X axis. ".
                "Got ViewingBoxXmax = ".expr2text(xmax));
       end_if: 
       if xmin <> Automatic then xmin:= ln(xmin); end_if:
       if xmax <> Automatic then xmax:= ln(xmax); end_if:
       fX:= ln@fX:
    end_if;
    if attributes[CoordinateType] in 
       {LinLogLin, LogLogLin, LinLogLog, LogLogLog} then
       if (domtype(ymin) = DOM_FLOAT and ymin <= 0) then
          error("expecting a positive Y range for the viewing box of a logarithmic Y axis. ".
                "Got ViewingBoxYmin = ".expr2text(ymin));
       end_if:
       if (domtype(ymax) = DOM_FLOAT and ymax <= 0) then
          error("expecting a positive Y range for the viewing box of a logarithmic Y axis. ".
                "Got ViewingBoxYmax = ".expr2text(ymax));
       end_if: 
       if ymin <> Automatic then ymin:= ln(ymin); end_if:
       if ymax <> Automatic then ymax:= ln(ymax); end_if:
       fY:= ln@fY;
    end_if;
    if attributes[CoordinateType] in 
       {LinLinLog, LogLinLog, LinLogLog, LogLogLog} then
       if (domtype(zmin) = DOM_FLOAT and zmin <= 0) then
          error("expecting a positive Z range for the viewing box of a logarithmic Z axis. ".
                "Got ViewingBoxZmin = ".expr2text(zmin));
       end_if:
       if (domtype(zmax) = DOM_FLOAT and zmax <= 0) then
          error("expecting a positive Z range for the viewing box of a logarithmic Z axis. ".
                "Got ViewingBoxYmax = ".expr2text(zmax));
       end_if: 
       if zmin <> Automatic then zmin:= ln(zmin); end_if:
       if zmax <> Automatic then zmax:= ln(zmax); end_if:
       fZ:= ln@fZ;
    end_if;
*/
    //-------------------------------------------------------------
    // call the plot::surfaceEval utility
    //-------------------------------------------------------------
  data := FAIL;
  if adaptivemesh = 0 then
    traperror(([viewingbox, data]:=
	       plot::surfaceEval(fX, fY, fZ, 
				 umesh, usubmesh, umin .. umax,
				 vmesh, vsubmesh, vmin .. vmax,
				 xmin .. xmax,
				 ymin .. ymax,
				 zmin .. zmax,
				 AdaptiveMesh = adaptivemesh
				)));
  end_if;
  if data=FAIL then
    [viewingbox, data, ulines, vlines] :=
    plot::adaptiveSurfaceEval(fX, fY, fZ, 
			      umesh, usubmesh, umin .. umax,
			      vmesh, vsubmesh, vmin .. vmax,
			      xmin .. xmax,
			      ymin .. ymax,
			      zmin .. zmax,
			      AdaptiveMesh = adaptivemesh):
    if adaptivemesh=0 then adaptivemesh:=1; end_if; // for below
  end_if;
    //-------------------------------------------------------------
    // Important: we need to undo the modification of fX(u, v) etc.
    // before writing to the XML file. The plot data are assumed to
    // be:   data = [ [u1,v1,x1,y1,z1], [u2,v2,x2,y2,z2], ... ]
    //-------------------------------------------------------------
/*
    if attributes[CoordinateType] in 
       {LogLinLin, LogLogLin, LogLinLog, LogLogLog} then
       // x -> exp(x)
       viewingbox[1]:= map(viewingbox[1], exp);
       data:= map(data, pt -> (pt[3]:= exp(pt[3]);pt)):
    end_if;
    if attributes[CoordinateType] in 
       {LinLogLin, LogLogLin, LinLogLog, LogLogLog} then
       // y -> exp(y)
       viewingbox[2]:= map(viewingbox[2], exp);
       data:= map(data, pt -> (pt[4]:= exp(pt[4]);pt)):
    end_if;
    if attributes[CoordinateType] in 
       {LinLinLog, LogLinLog, LinLogLog, LogLogLog} then
       // z -> exp(z)
       viewingbox[3]:= map(viewingbox[3], exp);
       data:= map(data, pt -> (pt[5]:= exp(pt[5]);pt)):
    end_if;
*/
    // ------------------------------------------------------------
    // Special: clip out points that are really far away.
    // This seems necessary for Windoof, where plotting
    // raises an error if there are points way out of the
    // viewing area (even if they are clipped).
    // ------------------------------------------------------------
    if viewingbox <> NIL then
      clipFactor:= 1000:
      clipBranch:= proc(t, i)
                   begin
                     if t[i] < _min then t[i]:= _min; end_if;
                     if t[i] > _max then t[i]:= _max; end_if;
                     t;
                   end_proc:
      for i from 1 to 3 do // i = 1 = x, i = 2 = y, i = 3 = z
        _min:= op(viewingbox, [i, 1]);
        _max:= op(viewingbox, [i, 2]);
        _range:= _max - _min;
        if iszero(_range) then  
          _range:= _plus(op(viewingbox, [i, 2]) - op(viewingbox, [i, 1]) $ i=1..3);
           if iszero(_range) then
              _range:= 1;
           end_if;
        end_if;
        _min:= _min - clipFactor*_range;
        _max:= _max + clipFactor*_range;
        data:= map(data, clipBranch, 2 + i);
      end_for:
    end_if:
    //-----------------------------------------
    // write the graphical data to the XML file
    //-----------------------------------------
    // We have to create the low-level primitives of the muplotml.dtd.

    if adaptivemesh > 0 then
       out::writeTriangles(attributes, table(), data,
         linecolorfunction, fillcolorfunction);

      // now for the U/V lines:
      for line in ulines do
      	if nops(line) = 0 then next; end_if;
      	line := sort([op(line)],
      		     (a,b) -> a[2] < b[2]);
      	out::writeUVXYLine(attributes, table(), line, linecolorfunction, TRUE, FALSE);
      end_for;
      
      for line in vlines do
      	if nops(line) = 0 then next; end_if;
      	line := sort([op(line)],
      		     (a,b) -> a[1] < b[1]); 
      	out::writeUVXYLine(attributes, table(), line, linecolorfunction, FALSE, FALSE);
      end_for;
      
      // write contours if asked for
      if viewingbox <> NIL then
      	out::writeMeshContours(attributes, table(), data, 3..5, 3, attributes[XContours],
      				p -> linecolorfunction(op(p, 1..5)),
      				op(viewingbox, [1, 1]), op(viewingbox, [1, 2]));
      	out::writeMeshContours(attributes, table(), data, 3..5, 4, attributes[YContours],
      				p -> linecolorfunction(op(p, 1..5)),
      				op(viewingbox, [2, 1]), op(viewingbox, [2, 2]));
      	out::writeMeshContours(attributes, table(), data, 3..5, 5, attributes[ZContours],
      				p -> linecolorfunction(op(p, 1..5)),
      				op(viewingbox, [3, 1]), op(viewingbox, [3, 2]));
      end_if;
	

    else // adaptivemesh = 0
      out::writeRectangularMesh(attributes, table(), data,
        linecolorfunction, fillcolorfunction,
        umesh, vmesh, usubmesh, vsubmesh, FALSE, 3..5);
      // write contour lines
      if viewingbox <> NIL then
      	out::writeGridContours(attributes, table(), data,
      				umesh + (umesh - 1) * usubmesh,
      				vmesh + (vmesh - 1) * vsubmesh,
      				3..5, 3, attributes[XContours],
      				p -> linecolorfunction(op(p, 1..5)),
      				op(viewingbox, [1, 1]), op(viewingbox, [1, 2]));
      	out::writeGridContours(attributes, table(), data,
      				umesh + (umesh - 1) * usubmesh,
      				vmesh + (vmesh - 1) * vsubmesh,
      				3..5, 4, attributes[YContours],
      				p -> linecolorfunction(op(p, 1..5)),
      				op(viewingbox, [2, 1]), op(viewingbox, [2, 2]));
      	out::writeGridContours(attributes, table(), data,
      				umesh + (umesh - 1) * usubmesh,
      				vmesh + (vmesh - 1) * vsubmesh,
      				3..5, 5, attributes[ZContours],
      				p -> linecolorfunction(op(p, 1..5)),
      				op(viewingbox, [3, 1]), op(viewingbox, [3, 2]));
      end_if;
    end_if;
    //-------------------------------------------------------------
    // Finally, return the viewing box
    //-------------------------------------------------------------
    if viewingbox = NIL then
       return();
    end_if;
    return(viewingbox);
  end_proc:
//------------------------------------------------------------------------
