/* ----------------------------------------------------------------------------
plot::SurfaceSTL -- import of STL graphics files 

plot::SurfaceSTL(InputFile) creates a 3D graphical object from the 
data of a given STL graphics file.

Call(s):
    plot::SurfaceSTL( InputFile <, UseNormals = b> <, args, ...> )

Parameters:
    InputFile    : the file name: a character string of type DOM_STRING

Options:
    UseNormals=b : b may be TRUE or FALSE. This attribute specifies 
                   whether the normals defined in the STL file are 
                   used for the mupad plot.

    args, ...    : plot options, i.e., equations Attribute = value,
                   where Attribute is one of the attributes listed
                   below or a hint. 

Related Domains:
    plot::Surface, plot::SurfaceSet, plot::Rotate3d, plot::Scale3d,
    plot::Translate3d

Details:
    * An STL file consists of a list of facet data. Each facet is uniquely
      identified by a unit normal (a line perpendicular to the triangle and
      with a length of 1.0) and by three vertices (corners). The normal and
      each vertex are specified by three coordinates each, so there is a
      total of 12 numbers stored for each facet. Read Section Background
      for further details.

    * When setting the attribute UseNormals to FALSE the normals defined 
      in the STL graphics file are ignored when plotting the object in 
      mupad. This reduces the data volume of the graphics object and the 
      computing time as well. However, it leads to a less brilliant image.

    * User-defined color functions LineColorFunction and FillColorFunction
      will be called with the index of the current facet as its first 
      parameter, followed by the x, y, and z coordinate of the current point.

    * Use plot::SurfaceSTL::boundingBox(S) or S::boundingBox() for computing 
      the bounding box of an STL object S. Note that the STL graphics file 
      must be read completely for computing this action. Also note that once 
      the bounding box is computed by S::boundingBox(), it will not be 
      recomputed by this function even if the STL object S has changed. Use 
      plot::SurfaceSTL::boundingBox(S) is this case.

    * Use plot::SurfaceSTL::center(S) or S::center() for computing the center 
      of the bounding box of an STL object S. Note that the STL graphics file 
      must be read completely for computing this action for the first time.
      Also note that the computed bounding box is cached and will not be
      recomputed even if the STL object S has changed.

Examples:
    We create an STL graphics file which defines a tetrahedron and plot
    this object afterwards:

    >> if sysname() = "MSDOS" then
          myFile:= "C:\\sample.stl";
       else
          myFile:= "/tmp/sample.stl";
       end_if

    >> fprint( Unquoted, Text, myFile, 
               "SOLID TRI 
                FACET NORMAL 0.0 0.0 -1.0 
                  OUTER LOOP 
                    VERTEX -1.5 -1.5 1.4 
                    VERTEX 0.0 1.7 1.4 
                    VERTEX 1.5 -1.5 1.4 
                  ENDLOOP 
                ENDFACET 
                FACET NORMAL 0.0 0.88148 0.472221 
                  OUTER LOOP 
                    VERTEX -1.5 -1.5 1.4 
                    VERTEX 1.5 -1.5 1.4 
                    VERTEX 0.0 0.0 -1.4 
                  ENDLOOP 
                ENDFACET 
                FACET NORMAL -0.876814 -0.411007 0.24954 
                  OUTER LOOP 
                    VERTEX 1.5 -1.5 1.4 
                    VERTEX 0.0 1.7 1.4 
                    VERTEX 0.0 0.0 -1.4 
                  ENDLOOP 
                ENDFACET 
                FACET NORMAL 0.876814 -0.411007 0.24954 
                  OUTER LOOP 
                    VERTEX 0.0 1.7 1.4 
                    VERTEX -1.5 -1.5 1.4
                    VERTEX 0.0 0.0 -1.4 
                  ENDLOOP 
                ENDFACET 
                ENDSOLID TRI"
             );

    >> plot( plot::SurfaceSTL(myFile, MeshVisible=TRUE) );

    We plot the object with its bounding box:

    >> S:= plot::SurfaceSTL(myFile);
       B:= plot::SurfaceSTL::boundingBox(S);
       plot( S, plot::Box(op(B), Color=[0.1,0.1,0.9,0.1]) );
       delete S, B:

Background:
    There are two storage formats available for STL files, which are ASCII
    and BINARY. ASCII files are human-readable while BINARY files are smaller
    and faster to process. A typical ASCII STL file looks like this:

    solid sample
     facet normal -4.470293E-02 7.003503E-01 -7.123981E-01
      outer loop
       vertex -2.812284E+00 2.298693E+01 0.000000E+00
       vertex -2.812284E+00 2.296699E+01 -1.960784E-02
       vertex -3.124760E+00 2.296699E+01 0.000000E+00
      endloop
     endfacet
     ...
    endsolid sample

    STL BINARY files have the following format:

    Bytes  Type   Description
    80	   ASCII  header, no data significance
    4	   uint   number of facets in file

    4	   float  normal  x - start of facet
    4      float  normal  y
    4      float  normal  z
    4      float  vertex1 x
    4      float  vertex1 y
    4      float  vertex1 z
    4      float  vertex2 x
    4      float  vertex2 y
    4      float  vertex2 z
    4      float  vertex3 x
    4      float  vertex3 y
    4      float  vertex3 z
    2      byte   not used  - end of facet
           ...

    Facet orientation: The facets define the surface of a 3-dimensional object.
    As such, each facet is part of the boundary between the interior and exterior
    of the object. The orientation of the facets (which way is "out" and which
    way is "in") is specified redundantly in two ways which should be consistent.
    First, the direction of the normal is outward. Second, which is most commonly
    used now-a-day, the facet vertices are listed in counter-clockwise order when
    looking at the object from the outside (right-hand rule).  

    Vertex-to-vertex rule: Each triangle must share two vertices with each of its
    adjacent triangles. In other words, a vertex of one triangle cannot lie on the
    side of another.

    Axes: The format specifies that all vertex coordinates must be positive-definite
    (nonnegative and nonzero) numbers. However, with a few exceptions it seems that
    most software used today (MuPAD included) allow negative coordinates as well.

    Units: The STL file does not contain any scale information; the coordinates 
    are in arbitrary units.

    Further details about the STL file format are available in the web, e.g., at:
      - www.ennex.com/fabbers/StL.asp,
      - www.math.iastate.edu/burkardt/data/stl/stl.html and
      - rpdrc.ic.polyu.edu.hk/content/stl/stl\_introduction.htm.

    Collections of STL sample files can be found in the web, e.g., at:
      - www.wohlersassociates.com/Software-for-Rapid-Prototyping.html.

    MuPAD only accepts the following notations for the keywords '\Mex{facet}' and 
    '\Mex{vertex}' in STL ASCII files: \Mex{facet}, \Mex{FACET}, \Mex{Facet} and 
    \Mex{vertex}, \Mex{VERTEX}, \Mex{Vertex}, respectively.

    The normal of a facet defined in an STL file is used for all its vertices when 
    plotting this object. Due to the fact that some facets (triangles) share points 
    with other ones, these points are plotted with different normals.

    Depending on your hardware we recommend to plot STL objects with no more than 
    50.000 to 150.000 facets. You should activate the option 'Accelerate OpenGL' 
    in the VCam options menu.

---------------------------------------------------------------------------- */

plot::createPlotDomain("SurfaceSTL", 
                       "graphical primitive for 3D STL surfaces", 
                       3,
   [
	[InputFile,  ["Mandatory", NIL], ["Definition", "Prop", "String",
                      "STL file name.", TRUE]],

	[UseNormals, ["Optional", TRUE], [["Style","Surface"], "Prop", "Bool",
                      "Use predefined normal vectors.", TRUE]],

    MeshVisible, LineStyle, LineWidth, 
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::SurfaceSTL::styleSheet:= table(
      MeshVisible = FALSE,
      LineColor   = RGB::Black.[0.25],
      LineColorDirectionY=0
):

//-----------------------------------------------------------------------------
plot::SurfaceSTL::hints:= {
      Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::SurfaceSTL::new:= proc()
local  object, other; 
option escape;
begin
    // filter out all generic arguments
    object:= dom::checkArgs([],args());

    // process the special attributes in object::other
    other:= object::other;
    if nops(other) > 0 then
      if nops(other) > 1 then
           error("unexpected argument: ".expr2text(other[2]));
      end_if;

      case type(other[1]) 
        of DOM_STRING do
           object::InputFile:= other[1];
           break;
        otherwise
           error("First argument: expecting an STL file name");
       end_case;
    end_if;

    dom::extslot(object, "boundingBox", 
         proc() 
           option remember; 
         begin
           plot::SurfaceSTL::boundingBox(object)
         end
    );
    dom::extslot(object, "center",
         proc() 
           option remember; 
           local  bbox;
         begin
           bbox:= object::boundingBox();
           map( bbox, (`#r`) -> (rhs(`#r`)+lhs(`#r`)) / 2 );
         end
    );
 
    // check all attributes
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::SurfaceSTL::print:= obj -> "plot::SurfaceSTL(\"".obj::InputFile."\")":

//-----------------------------------------------------------------------------
plot::SurfaceSTL::doPlotStatic:= proc(out, object, attributes, inheritedAttributes, doPlot=TRUE)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local 
    path, fp, firstline, line, list, lnum, isAscii, byteorder,
    fillcolorfunction,
    linecolorfunction, haslinecolorfunction,
    facet, facets, nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z, 
    xmin, xmax, ymin, ymax, zmin, zmax,
    useNormals, points;

save BigEndian, LittleEndian,
     Word, Byte;

begin
  // Walter 16.10.06: do not redefine min, max: we use the
  // kernel versions with more than 2 arguments below!
  //min:= (a,b) -> if a<b then a else b end_if; // faster version for two elements
  //max:= (a,b) -> if a<b then b else a end_if; // faster version for two elements

    // at first try to open with direct path
    path:= "";
    userinfo(9,"Trying to open '".path.attributes[InputFile]."' [plot::SurfaceSTL::doPlotStatic]");
    fp:=fopen(attributes[InputFile],Read,Text);

    if fp = FAIL then
      if InputFile[1] = stdlib::PathSep or domtype(READPATH) = DOM_IDENT then
        error("Cannot open STL file");
      end_if;

      // otherwise try to open with READPATH
      for path in READPATH do
        if path[-1] <> stdlib::PathSep then
          path:= path.stdlib::PathSep;
        end_if;
        userinfo(9,"Trying to open '".path.attributes[InputFile]."' [plot::SurfaceSTL::doPlotStatic]");
        fp:= fopen(path.attributes[InputFile],Read,Text);
        if fp <> FAIL then
          break;
        end_if;
      end_for;
      if fp = FAIL then
        error("Cannot open STL file");
      end_if;
    end_if;

    // ASCII files start with a 'solid' keyword followed by a 'facet' line
    isAscii:= FALSE;
    if domtype(ftextinput(fp,line)) = DOM_STRING then 
      userinfo(9,"First line: '".line."' [plot::SurfaceSTL::doPlotStatic]");
      firstline:= line;
      list:= subs(text2list(stringlib::lower(stringlib::collapseWhitespace(line)),[" "]),""=null()," "=null());
      if nops(list) > 0 and list[1] = "solid" then
        if domtype(ftextinput(fp,line)) = DOM_STRING then
          list:= subs(text2list(stringlib::lower(stringlib::collapseWhitespace(line)),[" "]),""=null()," "=null());
          if nops(list) > 0 and list[1] = "facet" then
            userinfo(1,"File info: '".firstline."' [plot::SurfaceSTL::doPlotStatic]");
            userinfo(1,"STL ASCII format detected [plot::SurfaceSTL::doPlotStatic]");
            isAscii:= TRUE;
          else
            userinfo(9,"Reading keyword 'facet' failed [plot::SurfaceSTL::doPlotStatic]");
          end_if;
        else
          userinfo(9,"Reading second text line failed [plot::SurfaceSTL::doPlotStatic]");
        end_if;
      else
        userinfo(9,"Reading keyword 'solid' failed [plot::SurfaceSTL::doPlotStatic]");
      end_if;
    else
      userinfo(9,"Reading first text line failed [plot::SurfaceSTL::doPlotStatic]");
    end_if;

    byteorder:= LittleEndian; // BigEndian;

    // BINARY files start with 80 characters followed by the number of facets
    if not isAscii then
      fclose(fp);
      fp:= fopen(path.attributes[InputFile], Read, Raw):
      list:= readbytes(fp, 80);                    // header, no data significance
      if list = null() or list = [] then
        fclose(fp);
        error("Not a valid STL file");
      end_if;
      userinfo(1,"File info: '".numlib::fromAscii(list)."' [plot::SurfaceSTL::doPlotStatic]");
      facets:= readbytes(fp, byteorder, Word, 1);  // unsigned number of facets
      if facets = null() or facets = [] then
        fclose(fp);
        error("Not a valid STL file");
      end_if;
      facets:= facets[1];
      userinfo(1,"STL BINARY format detected [plot::SurfaceSTL::doPlotStatic]");
      if facets > 1000*1000 then
        fclose(fp);
        error("Invalid number of facets: ".expr2text(facets));
      end_if;
    end_if;

    // check if STL normal vectors are to be used
    useNormals:= attributes[UseNormals];

    // check for fill-color function
    if contains(attributes, FillColorFunction) then 
      fillcolorfunction:= float@(attributes[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then 
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;

    // initial viewing box of this object
    [xmin,xmax,ymin,ymax,zmin,zmax]:= [infinity,-infinity,infinity,-infinity,infinity,-infinity]:

    // start to write triangle mesh to XML stream 
    if doPlot then
      points := [];
    end_if;

    // read and write triangles of the STL file
    if isAscii then
  
      // ASCII STL file
      userinfo(1,"Reading STL ASCII file [plot::SurfaceSTL::doPlotStatic]");
      [facet, lnum]:= [0, 1];
      repeat
        lnum:= lnum + 1;

        // skip empty lines
        if line = "" then
          next;
        end_if;

        // create list of tokens
        list:= subs(text2list(stringlib::collapseWhitespace(line),[" "]),""=null()," "=null());
        userinfo(9,"Line #".expr2text(lnum).": '".expr2text(list)."' [plot::SurfaceSTL::doPlotStatic]");

        // skip empty lines
        if nops(list) = 0 then
          next;
        end_if;

        // check for 'facet' and store its normal vector
        if contains({"facet","FACET","Facet"},list[1]) then
          userinfo(5,"Line #".expr2text(lnum).": Facet #".expr2text(facet)." [plot::SurfaceSTL::doPlotStatic]");
          facet:= facet + 1;
          if useNormals then
            //[nx,ny,nz]:= map(list[3..5], float@text2expr@stringlib::subs, "E"="e"); // ltere MuPAD Kerne
            [nx,ny,nz]:= float(map(list[3..5], text2expr));
            if [domtype(nx),domtype(ny),domtype(nz)] <> [DOM_FLOAT,DOM_FLOAT,DOM_FLOAT] then
              fclose(fp);
              error("float expected in normal of facet ".expr2text(facet)." in line ".expr2text(lnum));
            end_if;
          end_if;
          next;
        end_if;

        // check for 'vertex' and write it out using the current normal vector
        if contains({"vertex","VERTEX","Vertex"},list[1]) then
          [px,py,pz]:= float(map(list[2..4], text2expr));
          if [domtype(px),domtype(py),domtype(pz)] <> [DOM_FLOAT,DOM_FLOAT,DOM_FLOAT] then
            fclose(fp);
            error("float expected in vertex of facet ".expr2text(facet)." in line ".expr2text(lnum));
          end_if;

          if doPlot then
            if useNormals then
              points := points.[[facet, 0, px, py, pz, nx, ny, nz]];
            else
              points := points.[[facet, 0, px, py, pz]];
            end_if;
          end_if;
       
          // compute viewing box
          [xmin,xmax,ymin,ymax,zmin,zmax]:= [min(xmin,px), max(xmax,px), 
                                             min(ymin,py), max(ymax,py), 
                                             min(zmin,pz), max(zmax,pz)];
        end_if;
      until ftextinput(fp,line) = null() end_repeat;
      userinfo(1,"".expr2text(facet)." facets in ".expr2text(lnum)." lines [plot::SurfaceSTL::doPlotStatic]");

    else

      // BINARY STL file
      userinfo(1,"".expr2text(facets)." facets to read [plot::SurfaceSTL::doPlotStatic]");
      facet:= 0;

      points := array(1..facets*3);
      // read facet list [nx,ny,nz, p1x,p1y,p1z, p2x,p2y,p2z, p3x,p3y,p3z]
      while(facet < facets and (list:=readbytes(fp,byteorder,Float,12)) <> null()) do
        facet:= facet + 1;
        userinfo(5,"Facet #".expr2text(facet)."' [plot::SurfaceSTL::doPlotStatic]");
        userinfo(9,"Data: ".expr2text(list)."' [plot::SurfaceSTL::doPlotStatic]");
        if nops(list) < 12 then
          fclose(fp);
          error("12 values expected in facet ".expr2text(facet)." of ".expr2text(facets));
        end_if;

        [nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z]:= list;

        if doPlot then
          if useNormals then
            points[facet*3-2] := [facet, 0, px, py, pz, nx, ny, nz];
            points[facet*3-1] := [facet, 0, p2x, p2y, p2z, nx, ny, nz];
            points[facet*3] := [facet, 0, p3x, p3y, p3z, nx, ny, nz];
/*            points := points.[[facet, 0, px, py, pz, nx, ny, nz],
              [facet, 0, p2x, p2y, p2z, nx, ny, nz],
              [facet, 0, p3x, p3y, p3z, nx, ny, nz]];*/
          else
            points[facet*3-2] := [facet, 0, px, py, pz];
            points[facet*3-1] := [facet, 0, p2x, p2y, p2z];
            points[facet*3] := [facet, 0, p3x, p3y, p3z];
/*            points := points.[[facet, 0, px, py, pz],
              [facet, 0, p2x, p2y, p2z],
              [facet, 0, p3x, p3y, p3z]];*/
          end_if;
        end_if;

        // compute viewing box
        [xmin,xmax,ymin,ymax,zmin,zmax]:= [min(xmin, px, p2x, p3x), 
                                           max(xmax, px, p2x, p3x),
                                           min(ymin, py, p2y, p3y),
                                           max(ymax, py, p2y, p3y), 
                                           min(zmin, pz, p2z, p3z), 
                                           max(zmax, pz, p2z, p3z)];

        // skip padding bytes
        readbytes(fp, Byte, 2);
      end_while;
      userinfo(1,"".expr2text(facet)." facets read ok [plot::SurfaceSTL::doPlotStatic]");

    end_if;
    fclose(fp);

    // close definition of triangle mesh
    if doPlot then
      out::writeTriangles(attributes, table(), points, linecolorfunction, fillcolorfunction);
    end_if;

    // finally, return the viewing box
    return([xmin..xmax,ymin..ymax,zmin..zmax]);
  end_proc:

//------------------------------------------------------------------------
plot::SurfaceSTL::boundingBox:= proc(object)
begin
  // accept an STL file name as well
  if domtype(object) = DOM_STRING then
    object:= plot::SurfaceSTL(object);
  end_if;

  plot::SurfaceSTL::doPlotStatic(FAIL, object,
        table(InputFile=object::InputFile,UseNormals=FALSE), 
        [], FALSE
  );

/*
  if module::which("stl") = FAIL then 
    plot::SurfaceSTL::MuPlotML(object, 
          table(InputFile=object::InputFile,UseNormals=FALSE), 
          [], FALSE);
  else
    external("stl","boundingbox")(object::InputFile);
  end_if;
*/
end_proc:

plot::SurfaceSTL::center:= proc(object)
local bbox;
begin
  bbox:= plot::SurfaceSTL::boundingBox(object);
  map( bbox, (`#r`) -> (rhs(`#r`)+lhs(`#r`)) / 2 );
end_proc:
//------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// keep in sync with the code above;  writing native MuPlotML here is more
// efficient and far less memory hungry than using the generic mechanism
//-----------------------------------------------------------------------------
plot::SurfaceSTL::MuPlotML:= proc(object, attributes, inheritedAttributes, doPlot=TRUE)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local 
    path, fp, firstline, line, list, lnum, isAscii, byteorder,
    fillcolorfunction,
    linecolorfunction, haslinecolorfunction,
    facet, facets, nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z, 
    xmin, xmax, ymin, ymax, zmin, zmax,
    useNormals;

save BigEndian, LittleEndian,
     Word, Byte;

begin
  // Walter 16.10.06: do not redefine min, max: we use the
  // kernel versions with more than 2 arguments below!
  //min:= (a,b) -> if a<b then a else b end_if; // faster version for two elements
  //max:= (a,b) -> if a<b then b else a end_if; // faster version for two elements

    // at first try to open with direct path
    path:= "";
    userinfo(9,"Trying to open '".path.attributes[InputFile]."' [plot::SurfaceSTL::MuPlotML]");
    fp:=fopen(attributes[InputFile],Read,Text);

    if fp = FAIL then
      if InputFile[1] = stdlib::PathSep or domtype(READPATH) = DOM_IDENT then
        error("Cannot open STL file");
      end_if;

      // otherwise try to open with READPATH
      for path in READPATH do
        if path[-1] <> stdlib::PathSep then
          path:= path.stdlib::PathSep;
        end_if;
        userinfo(9,"Trying to open '".path.attributes[InputFile]."' [plot::SurfaceSTL::MuPlotML]");
        fp:= fopen(path.attributes[InputFile],Read,Text);
        if fp <> FAIL then
          break;
        end_if;
      end_for;
      if fp = FAIL then
        error("Cannot open STL file");
      end_if;
    end_if;

    // ASCII files start with a 'solid' keyword followed by a 'facet' line
    isAscii:= FALSE;
    if domtype(ftextinput(fp,line)) = DOM_STRING then 
      userinfo(9,"First line: '".line."' [plot::SurfaceSTL::MuPlotML]");
      firstline:= line;
      list:= subs(text2list(stringlib::lower(stringlib::collapseWhitespace(line)),[" "]),""=null()," "=null());
      if nops(list) > 0 and list[1] = "solid" then
        if domtype(ftextinput(fp,line)) = DOM_STRING then
          list:= subs(text2list(stringlib::lower(stringlib::collapseWhitespace(line)),[" "]),""=null()," "=null());
          if nops(list) > 0 and list[1] = "facet" then
            userinfo(1,"File info: '".firstline."' [plot::SurfaceSTL::MuPlotML]");
            userinfo(1,"STL ASCII format detected [plot::SurfaceSTL::MuPlotML]");
            isAscii:= TRUE;
          else
            userinfo(9,"Reading keyword 'facet' failed [plot::SurfaceSTL::MuPlotML]");
          end_if;
        else
          userinfo(9,"Reading second text line failed [plot::SurfaceSTL::MuPlotML]");
        end_if;
      else
        userinfo(9,"Reading keyword 'solid' failed [plot::SurfaceSTL::MuPlotML]");
      end_if;
    else
      userinfo(9,"Reading first text line failed [plot::SurfaceSTL::MuPlotML]");
    end_if;

    byteorder:= LittleEndian; // BigEndian;

    // BINARY files start with 80 characters followed by the number of facets
    if not isAscii then
      fclose(fp);
      fp:= fopen(path.attributes[InputFile], Read, Raw):
      list:= readbytes(fp, 80);                    // header, no data significance
      if list = null() or list = [] then
        fclose(fp);
        error("Not a valid STL file");
      end_if;
      userinfo(1,"File info: '".numlib::fromAscii(list)."' [plot::SurfaceSTL::MuPlotML]");
      facets:= readbytes(fp, byteorder, Word, 1);  // unsigned number of facets
      if facets = null() or facets = [] then
        fclose(fp);
        error("Not a valid STL file");
      end_if;
      facets:= facets[1];
      userinfo(1,"STL BINARY format detected [plot::SurfaceSTL::MuPlotML]");
      if facets > 1000*1000 then
        fclose(fp);
        error("Invalid number of facets: ".expr2text(facets));
      end_if;
    end_if;

    // check if STL normal vectors are to be used
    useNormals:= attributes[UseNormals];

    // check for fill-color function
    if contains(attributes, FillColorFunction) then 
      fillcolorfunction:= float@(attributes[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then 
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;

    // initial viewing box of this object
    [xmin,xmax,ymin,ymax,zmin,zmax]:= [infinity,-infinity,infinity,-infinity,infinity,-infinity]:

    // start to write triangle mesh to XML stream 
    if doPlot then
      plot::MuPlotML::beginElem("Surf3d", 
                                "Type"="Triangles",
                                "HasLineColors"=haslinecolorfunction);
    end_if;

    // read and write triangles of the STL file
    if isAscii then
  
      // ASCII STL file
      userinfo(1,"Reading STL ASCII file [plot::SurfaceSTL::MuPlotML]");
      [facet, lnum]:= [0, 1];
      repeat
        lnum:= lnum + 1;

        // skip empty lines
        if line = "" then
          next;
        end_if;

        // create list of tokens
        list:= subs(text2list(stringlib::collapseWhitespace(line),[" "]),""=null()," "=null());
        userinfo(9,"Line #".expr2text(lnum).": '".expr2text(list)."' [plot::SurfaceSTL::MuPlotML]");

        // skip empty lines
        if nops(list) = 0 then
          next;
        end_if;

        // check for 'facet' and store its normal vector
        if contains({"facet","FACET","Facet"},list[1]) then
          userinfo(5,"Line #".expr2text(lnum).": Facet #".expr2text(facet)." [plot::SurfaceSTL::MuPlotML]");
          facet:= facet + 1;
          if useNormals then
            //[nx,ny,nz]:= map(list[3..5], float@text2expr@stringlib::subs, "E"="e"); // ltere MuPAD Kerne
            [nx,ny,nz]:= float(map(list[3..5], text2expr));
            if [domtype(nx),domtype(ny),domtype(nz)] <> [DOM_FLOAT,DOM_FLOAT,DOM_FLOAT] then
              fclose(fp);
              error("float expected in normal of facet ".expr2text(facet)." in line ".expr2text(lnum));
            end_if;
          end_if;
          next;
        end_if;

        // check for 'vertex' and write it out using the current normal vector
        if contains({"vertex","VERTEX","Vertex"},list[1]) then
          [px,py,pz]:= float(map(list[2..4], text2expr));
          if [domtype(px),domtype(py),domtype(pz)] <> [DOM_FLOAT,DOM_FLOAT,DOM_FLOAT] then
            fclose(fp);
            error("float expected in vertex of facet ".expr2text(facet)." in line ".expr2text(lnum));
          end_if;

          if doPlot then
            if useNormals then
              plot::MuPlotML::prP3(px, py, pz, // the point
                                   nx, ny, nz, // the normal
                                   linecolorfunction(facet, px, py, pz),
                                   fillcolorfunction(facet, px, py, pz)
              );
            else
              plot::MuPlotML::prP3(px, py, pz, // the point
                                   linecolorfunction(facet, px, py, pz),
                                   fillcolorfunction(facet, px, py, pz)
              );
            end_if;
          end_if;
       
          // compute viewing box
          [xmin,xmax,ymin,ymax,zmin,zmax]:= [min(xmin,px), max(xmax,px), 
                                             min(ymin,py), max(ymax,py), 
                                             min(zmin,pz), max(zmax,pz)];
        end_if;
      until ftextinput(fp,line) = null() end_repeat;
      userinfo(1,"".expr2text(facet)." facets in ".expr2text(lnum)." lines [plot::SurfaceSTL::MuPlotML]");

    else

      // BINARY STL file
      userinfo(1,"".expr2text(facets)." facets to read [plot::SurfaceSTL::MuPlotML]");
      facet:= 0;

      // read facet list [nx,ny,nz, p1x,p1y,p1z, p2x,p2y,p2z, p3x,p3y,p3z]
      while(facet < facets and (list:=readbytes(fp,byteorder,Float,12)) <> null()) do
        facet:= facet + 1;
        userinfo(5,"Facet #".expr2text(facet)."' [plot::SurfaceSTL::MuPlotML]");
        userinfo(9,"Data: ".expr2text(list)."' [plot::SurfaceSTL::MuPlotML]");
        if nops(list) < 12 then
          fclose(fp);
          error("12 values expected in facet ".expr2text(facet)." of ".expr2text(facets));
        end_if;

        [nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z]:= list;

        if doPlot then
          if useNormals then
            plot::MuPlotML::prP3(px, py, pz,    // the point
                                 nx, ny, nz,    // the normal
                                 linecolorfunction(facet, px, py, pz),
                                 fillcolorfunction(facet, px, py, pz)
            );
            plot::MuPlotML::prP3(p2x, p2y, p2z, // the point
                                 nx, ny, nz,    // the normal
                                 linecolorfunction(facet, p2x, p2y, p2z),
                                 fillcolorfunction(facet, p2x, p2y, p2z)
            );
            plot::MuPlotML::prP3(p3x, p3y, p3z, // the point
                                 nx, ny, nz,    // the normal
                                 linecolorfunction(facet, p3x, p3y, p3z),
                                 fillcolorfunction(facet, p3x, p3y, p3z)
            );
          else
            plot::MuPlotML::prP3(px, py, pz,    // the point
                                 linecolorfunction(facet, px, py, pz),
                                 fillcolorfunction(facet, px, py, pz)
            );
            plot::MuPlotML::prP3(p2x, p2y, p2z, // the point
                                 linecolorfunction(facet, p2x, p2y, p2z),
                                 fillcolorfunction(facet, p2x, p2y, p2z)
            );
            plot::MuPlotML::prP3(p3x, p3y, p3z, // the point
                                 linecolorfunction(facet, p3x, p3y, p3z),
                                 fillcolorfunction(facet, p3x, p3y, p3z)
            );
          end_if;
        end_if;

        // compute viewing box
        [xmin,xmax,ymin,ymax,zmin,zmax]:= [min(xmin, px, p2x, p3x), 
                                           max(xmax, px, p2x, p3x),
                                           min(ymin, py, p2y, p3y),
                                           max(ymax, py, p2y, p3y), 
                                           min(zmin, pz, p2z, p3z), 
                                           max(zmax, pz, p2z, p3z)];

        // skip padding bytes
        readbytes(fp, Byte, 2);
      end_while;
      userinfo(1,"".expr2text(facet)." facets read ok [plot::SurfaceSTL::MuPlotML]");

    end_if;
    fclose(fp);

    // close definition of triangle mesh
    if doPlot then
      plot::MuPlotML::endElem("Surf3d"): 
    end_if;

    // finally, return the viewing box
    return([xmin..xmax,ymin..ymax,zmin..zmax]);
  end_proc:
