/* -------------------------------------------------------------
plot::Sweep -- The graphical primitive for 3D curves

Call(s): Sweep( [x1(t), y1(t), z1(t)], 
               <[x2(t), y2(t), z2(t)]>,
                 t = tmin .. tmax,
                <a = amin .. amax>
                <Bottom = b>,
                <Color = c>,
                <Mesh = n>,
                <Submesh = m>,
                <AdaptiveMesh = l>
                <DiscontinuitySearch = TRUE/FALSE>
                <opt1, opt2, ...>)

         Sweep( Curve1 <, Curve2> <, a = amin..amax>, ..) 
Parameters:
    x1,y1,z1,   arithmetical expressions (depending on t)
    x2,y2,z2  : and, possibly, on an animation parameter
    t         : the curve parameter: DOM_IDENT or indexed identifier
    tmin, tmax: numerical real values or univariate expressions
                of the animation  parameter
    a         : the animation parameter: DOM_IDENT or indexed identifier
    amin, amax: numerical real values
    c         : an RGB value, an RGBa value or a procedure
    n         : the number of mesh points: an integer > 0
    m         : the number of 'inner mesh points': an integer >= 0
    Curve1,   : objects of type plot::Curve3d
       Curve2

Options:
    opt1, opt2, ... : general 3d plot options

Details:  [x1(t),y1(t),z1(t)] defines the original curve
          [x2(t),y2(t),z2(t)] defines the final curve.
          A linear homotopy is used to deform the original curve
          to its final form. This defines a sweep surface that
          is visualized by plot::Sweep
          If no final curve is specified, x2(t) = x1(t), y2(t) = y1(t),
          z2(t) = b is used, where b is the value of the attribute
          Bottom = b (default is Bottom = 0).

Examples:
 >> plot::Sweep([t*sin(t), t*cos(t), t], [t*sin(t), t*cos(t), 0], t = 0..2*PI )
 >> plot::Sweep([sin(t)/t, cos(t)/t, 1/t], [sin(t)/t, 0, 1/t], t = 0..2*PI, 
                Color = RGB::Blue )
 >> plot( plot::Sweep([x*cos(x - a), x*sin(x - a), x - 5*a], [0, 0, x - 5*a], x = 0..10*PI,
                        Mesh = 50, a = 0..2*PI) )
----------------------------------------------------------------*/
plot::createPlotDomain("Sweep",
                       "graphical primitive for 3D curves 'with a curtain'",
                       3,
                       [// XFunction, YFunction, ZFunction,
                        [XFunction1, 
                          ["Mandatory", NIL],              // [type, default value],
                          ["Definition",                   // where to be found in the inspector
                           "Expr",                         // type according to DTD,
                           plot::elementOptFunctionExpr,   // activate automatic conversions
                                                           // Expr -> Procedure, add floats etc
                           "x-coordinate of the 1st curve",// inspector info entry,
                           FALSE                           // recalculation flag
                          ]
                         ],
                        [YFunction1, 
                          ["Mandatory", NIL],              // [type, default value],
                          ["Definition",                   // where to be found in the inspector
                           "Expr",                         // type according to DTD,
                           plot::elementOptFunctionExpr,   // activate automatic conversions
                                                           // Expr -> Procedure, add floats etc
                           "y-coordinate of the 1st curve",// inspector info entry,
                           FALSE                           // recalculation flag
                          ]
                         ],
                        [ZFunction1, 
                          ["Mandatory", NIL],              // [type, default value],
                          ["Definition",                   // where to be found in the inspector
                           "Expr",                         // type according to DTD,
                           plot::elementOptFunctionExpr,   // activate automatic conversions
                                                           // Expr -> Procedure, add floats etc
                           "z-coordinate of the 1st curve",// inspector info entry,
                           FALSE                           // recalculation flag
                          ]
                         ],
                        [XFunction2, 
                          ["Mandatory", NIL],              // [type, default value],
                          ["Definition",                   // where to be found in the inspector
                           "Expr",                         // type according to DTD,
                           plot::elementOptFunctionExpr,   // activate automatic conversions
                                                           // Expr -> Procedure, add floats etc
                           "x-coordinate of the 2nd curve",// inspector info entry,
                           FALSE                           // recalculation flag
                          ]
                         ],
                        [YFunction2, 
                          ["Mandatory", NIL],              // [type, default value],
                          ["Definition",                   // where to be found in the inspector
                           "Expr",                         // type according to DTD,
                           plot::elementOptFunctionExpr,   // activate automatic conversions
                                                           // Expr -> Procedure, add floats etc
                           "y-coordinate of the 2nd curve",// inspector info entry,
                           FALSE                           // recalculation flag
                          ]
                         ],
                        [ZFunction2, 
                          ["Mandatory", NIL],              // [type, default value],
                          ["Definition",                   // where to be found in the inspector
                           "Expr",                         // type according to DTD,
                           plot::elementOptFunctionExpr,   // activate automatic conversions
                                                           // Expr -> Procedure, add floats etc
                           "z-coordinate of the 2nd curve",// inspector info entry,
                           FALSE                           // recalculation flag
                          ]
                         ],
                         [Ground, ["Optional", NIL],
                               ["Definition", "Expr", FAIL, "The base value.", TRUE]
                         ],

                        UName, UMin, UMax, URange, UMesh, USubmesh, ULinesVisible, 
                    //  VName, VMin, VMax, VRange, VMesh, VSubmesh, 
                        VLinesVisible,
                        Mesh, Submesh, 
                        AdaptiveMesh,
                    //  MeshVisible,
                        Filled, FillColor, Color, FillColor2, FillColorType, FillColorFunction,
                        LineWidth, LineStyle, 
                        LineColor, LineColor2,
                        LineColorType, LineColorFunction,
                        PointSize, PointStyle, PointsVisible,
                     // XContours, 
                     // YContours, 
                     // ZContours,
                        DiscontinuitySearch,
                        LineColorDirection, 
                        LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
                        FillColorDirection,
                        FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
                       ]):

// methods yet to be implemented:  convert, convert_to, expr

//----------------------------------------------------------------------
plot::Sweep::styleSheet := table(
     LegendEntry = TRUE, 
     UMesh=25, 
     USubmesh = 4,
     Filled = TRUE,
     LineColor = plot::getDefault(plot::Surface::LineColor),
     FillColor = plot::getDefault(plot::Surface::FillColor),
     FillColor2= plot::getDefault(plot::Surface::FillColor2),
     Ground = 0,
     LineColorDirectionY=0):
//----------------------------------------------------------------------
plot::Sweep::setPrimaryColor(LineColor):
//----------------------------------------------------------------------
plot::Sweep::new:= proc()
local object, other, X, Y, Z,
      curve1, curve2, attr, tmp,
      u1, u2, umin1, umin2, umax1, umax2,
      a1, a2, amin1, amin2, amax1, amax2;
begin
    // check all known options from the argument list
    object := dom::checkArgs(["U"], args());
    // get arguments which are not yet processed
    other := object::other;

    if nops(other) > 0 and (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
      other[1] := [op(other[1])];
    end_if;
    if nops(other) > 1 and (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
      other[2] := [op(other[2])];
    end_if;

    if object::Ground = FAIL then
       object::Ground:= plot::getDefault(plot::Sweep::Ground);
    end_if;

    if nops(object::children) > 0 then
       curve1:= object::children[1];
       if nops(object::children) < 2 then
          curve2:= plot::Curve3d([curve1::XFunction,
                                  curve1::YFunction,
                                  object::Ground],
                                  curve1::UName = curve1::URange);
       else
          curve2:= object::children[2];
       end_if;
       if testtype(curve1, plot::Curve3d) and
          testtype(curve2, plot::Curve3d) then
          object::XFunction1:= curve1::XFunction;
          object::YFunction1:= curve1::YFunction;
          object::ZFunction1:= curve1::ZFunction;
          object::XFunction2:= curve2::XFunction;
          object::YFunction2:= curve2::YFunction;
          object::ZFunction2:= curve2::ZFunction;

          // copy all attributes of curve1
          for attr in 
               (plot::Curve3d::knownAttributes intersect // all attributes and hints
                plot::Sweep::knownAttributes) minus 
                plot::allLibraryAttributes
              do
                if slot(object, "".attr) = FAIL then
                   slot(object, "".attr):= slot(curve1, "".attr);
                end_if;
          end_for;
       end_if;

       //--------------------------
       // adapt parametrization of curve2
       // This will only work if {XYZ}Function2 are expressions!
       // This will not work if {XYZ}Function2 are procedures!
       //--------------------------
       u1:= curve1::UName;
       u2:= curve2::UName;
       umin1:= curve1::UMin;
       umin2:= curve2::UMin;
       umax1:= curve1::UMax;
       umax2:= curve2::UMax;
       if u1 <> FAIL and u2 <> FAIL and
         (u1 <> u2 or umin1 <> umin2 or umax1 <> umax2) then
          if iszero(umax1 - umin1) then
             if iszero(umax2 - umin2) then
                tmp:= 1;
             else
                error("the first curve has a zero parameter range, ".
                      "the second curve has a non-zero parameter range. ".
                      "Cannot adapt the parametrization of the second curve.");
             end_if;
          else 
             tmp:= (umax2 - umin2)/(umax1 - umin1);
          end_if;
          object::XFunction2:= subs(object::XFunction2, u2 = umin2 + tmp*(u1 - umin1));
          object::YFunction2:= subs(object::YFunction2, u2 = umin2 + tmp*(u1 - umin1));
          object::ZFunction2:= subs(object::ZFunction2, u2 = umin2 + tmp*(u1 - umin1));
       end_if;

       //--------------------------
       // adapt animation of curve2
       // This will only work if {XYZ}Function2 are expressions!
       // This will not work if {XYZ}Function2 are procedures!
       //--------------------------
       a1 := curve1::ParameterName;
       a2 := curve2::ParameterName;
       amin1 := curve1::ParameterBegin;
       amin2 := curve2::ParameterBegin;
       amax1 := curve1::ParameterEnd;
       amax2 := curve2::ParameterEnd;
     
       if a1 = FAIL and a2 <> FAIL then
          a1:= a2;
          amin1:= amin2:
          amax1:= amax2:
       end_if;
       if a1 <> FAIL and a2 = FAIL then
          a2:= a1;
          amin2:= amin1:
          amax2:= amax1:
       end_if;
       object::ParameterName:= a1;
       object::ParameterBegin:= amin1;
       object::ParameterEnd:= amax1;
       if a1 <> FAIL and a2 <> FAIL and 
         (a1 <> a2 or amin1 <> amin2 or amax1 <> amax2) then
          if iszero(amax1 - amin1) then
             if iszero(amax2 - amin2) then
                tmp:= 1;
             else
                error("the first curve has a zero animation range, ".
                      "the second curve has a non-zero animation range. ".
                      "Cannot adapt the animation of the second curve.");
             end_if;
          else 
             tmp:= (amax2 - amin2)/(amax1 - amin1);
          end_if;
          object::XFunction2:= subs(object::XFunction2, a2 = amin2 + tmp*(a1 - amin1));
          object::YFunction2:= subs(object::YFunction2, a2 = amin2 + tmp*(a1 - amin1));
          object::ZFunction2:= subs(object::ZFunction2, a2 = amin2 + tmp*(a1 - amin1));
       end_if;

       // semantically check for validity
       return(
          dom::checkObject(object)
       );
    end_if;

//------------------------------------------------------------------------------
// process input that does not consist of Curve3d objects
//------------------------------------------------------------------------------
    if testargs() and nops(other) > 0 then
      if domtype(other[1]) <> DOM_LIST or
        nops(other[1]) <> 3 then
        error("expecting a list of three function expressions for the 1st curve");
      end_if;
      if nops(other) = 2 then
         if domtype(other[2]) <> DOM_LIST or
            nops(other[2]) <> 3 then
            error("expecting a list of three function expressions for the 2nd curve");
         end_if;
      end_if;
      if nops(other) > 2 then
        error("unexpected argument: ".expr2text(other[3]));
      end_if;
    end_if;

    if nops(other) > 0 then
       [X, Y, Z]:= other[1];
       object::XFunction1:= X;
       object::YFunction1:= Y;
       object::ZFunction1:= Z;
    end_if;
    if nops(other) > 1 then
       [X, Y, Z]:= other[2];
       object::XFunction2:= X;
       object::YFunction2:= Y;
       object::ZFunction2:= Z;
    else
       object::XFunction2:= object::XFunction1;
       object::YFunction2:= object::YFunction1;
       object::ZFunction2:= object::Ground;
    end_if;
    
    // semantically check for validity
    dom::checkObject(object);
end_proc:
//----------------------------------------------------------------------
plot::Sweep::print :=
  obj -> hold(plot::Sweep)(map([obj::XFunction1, obj::YFunction1, obj::ZFunction1], generate::sortSums),
                           map([obj::XFunction2, obj::YFunction2, obj::ZFunction2], generate::sortSums),
                           obj::UName=obj::URange):

//----------------------------------------------------------------------
plot::Sweep::doPlotStatic := proc(out, object, attributes, inheritedAttributes,
                                flag = FALSE)
    local x1, y1, z1, x2, y2, z2, cleaned_att;
begin
    x1 := attributes[XFunction1];
    y1 := attributes[YFunction1];
    z1 := attributes[ZFunction1];
    x2 := attributes[XFunction2];
    y2 := attributes[YFunction2];
    z2 := attributes[ZFunction2];
    cleaned_att := out::fixAttributes(attributes, plot::Surface);
    cleaned_att[XFunction] := (u, v) -> v*x2(u) + (1-v)*x1(u):
    cleaned_att[YFunction] := (u, v) -> v*y2(u) + (1-v)*y1(u):
    cleaned_att[ZFunction] := (u, v) -> v*z2(u) + (1-v)*z1(u):
    cleaned_att[VName] := `#v`;
    cleaned_att[VMin] := 0;
    cleaned_att[VMax] := 1;
    cleaned_att[VMesh] := 2;
    cleaned_att[VSubmesh] := 0;
    cleaned_att[VLinesVisible] := TRUE;
    if contains(cleaned_att, LineColorFunction) then
      cleaned_att[LineColorFunction] :=
      subs(((u, x, y, z) -> `#f`(u, `#v`, args(2..args(0)))),
           [`#f` = cleaned_att[LineColorFunction]]);
    end_if;
    if contains(cleaned_att, FillColorFunction) then
      cleaned_att[FillColorFunction] :=
      subs(((u, x, y, z) -> `#f`(u, `#v`, args(2..args(0)))),
           [`#f` = cleaned_att[FillColorFunction]]);
    end_if;
    out(plot::Surface(op(cleaned_att)), inheritedAttributes, Raw):
end_proc:
