/* ----------------------------------------------------------------------------
plot::Tetrahedron -- graphical primitive of a tetrahedron

plot::Tetrahedron() creates a 3D graphical object: a
thetrahedron.

Call(s):
    plot::Tetrahedron( <args, ...> )

Parameters:
    -

Options:
    attr, ...   : plot options, i.e., equations Attribute = value, where
                  Attribute is one of the attributes listed below or a hint.

Related Domains:
    plot::Hexahedron, plot::Octahedron, plot::Icosahedron, plot::Dodecahedron,
    plot::Box, plot::Cone, plot::Cylinder, plot::Ellipsoid, plot::Sphere, plot::Rotate3d,
    plot::Scale3d, plot::Translate3d

Details:
    * The LineColorFunction and FillColorFunction will be called with the
      index of the current facet as the first parameter, followed by the
      x-, y- and z-coordinate of the current point.

    * For rotated, scaled or translated surface objects use plot::Rotate3d,
      plot::Scale3d or plot::Translate3d, respectively, see example ???.

Example:
    We plot all polyhedrons together in one canvas:

    >> plot( plot::Scene3d(plot::Tetrahedron()),
             plot::Scene3d(plot::Hexahedron()),
             plot::Scene3d(plot::Octahedron()),
             plot::Scene3d(plot::Icosahedron()),
             plot::Scene3d(plot::Dodecahedron()),
             PointsVisible = TRUE,
             MeshVisible  = TRUE,
             LineWidth     = 0.5,
             LineColor     = RGB::Grey,
             BackgroundStyle=Pyramid,
             Axes=None,
             Width=100, Height=100
       )

Example 2:
      Color function can be given as well. This expects the index
      of the current facet as its first parameter followed by the x-,
      y- and z-coordinate of the current point:

      Es kann ebenfalls eine Farbfunktion angegeben werden. Diese
      erwartet den Index der aktuellen Facette als erstes Argument,
      gefolgt von der x-, y- und z-Koordinate des aktuellen Punktes:

    >> plot( plot::Tetrahedron(
             LineColor = RGB::Black,
             FillColorFunction = ((facet,x,y,z)->
             [RGB::Red,RGB::Green,RGB::Blue][facet mod 3 + 1])
            )
       )

Example 3:
    Again we plot the object defined in example 1, but now we add a
    rotated, scaled and translated copy of it:

    Wieder zeichnen wir das Objekt aus Beispiel 1. Diesmal ergnzen
    wir eine skalierte, rotierte und verschobene Kopie des Objektes:

    >> plot( plot::Tetrahedron(), plot::Scale3d( [2,2,2],
             plot::Tetrahedron(Color=[.9,.9,.9,.1]))
       )

     >> plot( plot::Tetrahedron(), plot::Rotate3d( PI/4,
             plot::Tetrahedron(Color=[.9,.9,.9,.1]))
       )

     >> plot( plot::Tetrahedron(), plot::Translate3d( [3,0,0],
             plot::Tetrahedron(Color=[.9,.9,.9,.1]))
       )

Background:
    GERMAN:
    Definition: Ein Polyeder heit regulr, wenn alle seine Oberflchen
    aus demselben regelmigen Vieleck bestehen und in jeder Ecke gleich
    viele dieser Vielecke zusammenstoen.

    Sptestens seit Platon ist bekannt, da es nur genau fnf regulre
    konvexe Polyeder gibt:

    Tetraeder  aus  4 (grch. tetra)  Dreiecken
    Hexaeder   aus  6 (grch. hexa)   Quadraten
    Oktaeder   aus  8 (grch. okta)   Dreiecken
    Dodekaeder aus 12 (grch. dodeka) Fnfecken (grch. pentagon)
    Ikosaeder  aus 20 (grch. eikosi) Dreiecken

    Fr die Winkel in den Ecken des regelmen n-Ecks gilt nmlich

    n        3   4    5    6  ...          n
    Winkel  60  90  108  120  ...  180-360/n

    In jeder Ecke eines Polyeders mssen mindestens drei Vielecke zusammenstoen
    um eine rumliche Ecke zu bilden. Da andererseits das regulre Polyeder konvex
    ist, mu die gesamte Winkelsumme aller n-Ecke, die in jeder Krperecke
    zusammenstoen, stets echt kleiner als 360o sein. Es knnen also nur 3,4 oder
    5 regelmge Dreiecke, 3 Quadrate oder 3 regelme Fnfecke sein. Diese fnf
    mglichen Flle lassen sich aber nur durch die oben angegebenen Krper realisieren.

    Es gilt mit R ist der Radius der Auenkugel, r ist der Radius der Innenkugel,
    O ist die Oberflche und V das Volumen:

           Tetraeder     Hexaeder    Oktaeder     Dodekaeder                  Ikosaeder
    R/a    1/4*sqrt(6)   1/2*sqrt(3) 1/2*sqrt(2)  1/4*sqrt(3)*(1+sqrt(5))     1/4*sqrt(10+2*sqrt(5))
    r/a    1/12*sqrt(6)  1/2         1/6*sqrt(6)  1/20*sqrt(250+110*sqrt(5))  1/12*sqrt(3)(3+sqrt(5))
    O/a^2  sqrt(3)       6           2*sqrt(3)    3*sqrt(25+10*sqrt(5))       5*sqrt(3)
    V/a^3  1/12*sqrt(2)  1           1/3*sqrt(2)  1/4*(15+7*sqrt(5))          5/12*(3+sqrt(5))

---------------------------------------------------------------------------- */

plot::createPlotDomain("Tetrahedron",
                       "graphical primitive for tetrahedrons",
                       3,
   [
    CenterX, CenterY, CenterZ, Center, Radius,
    LinesVisible, LineStyle, LineWidth,
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::Tetrahedron::styleSheet:= table(
      LinesVisible  = TRUE,
      LineColor     = RGB::Black.[0.25],
      Radius        = 1,
      CenterX       = 0,
      CenterY       = 0,
      CenterZ       = 0,
      LineColorDirectionY=0
):

//-----------------------------------------------------------------------------
plot::Tetrahedron::hints:= {
      Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::Tetrahedron::new:= proc()
local  object, other;
option escape;
begin
    // filter out all generic arguments
    object:= dom::checkArgs([],args());

    // process the special attributes in object::other
    other:= object::other;
    if nops(other) > 0 then
      error("unexpected argument: ".expr2text(other[1]));
    end_if;

    // check all attributes
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::Tetrahedron::print:= obj -> "plot::Tetrahedron(...)":

//-----------------------------------------------------------------------------
plot::Tetrahedron::doPlotStatic:= proc(out, object, attributes, inheritedAttributes)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local
    cx, cy, cz, r,
    triangles,
    fillcolorfunction,
    linecolorfunction, haslinecolorfunction,
    i,
    entries;

begin
    // check for fill-color function
    if contains(attributes, FillColorFunction) then
      fillcolorfunction:= float@(attributes[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;

    cx:= float(attributes[CenterX]);
    cy:= float(attributes[CenterY]);
    cz:= float(attributes[CenterZ]);
    r := float(attributes[Radius]);

    triangles:= [
      [cx+r,cy+r,cz-r], [cx+r,cy-r,cz+r], [cx-r,cy+r,cz+r],
      [cx+r,cy+r,cz-r], [cx-r,cy-r,cz-r], [cx+r,cy-r,cz+r],
      [cx-r,cy-r,cz-r], [cx+r,cy+r,cz-r], [cx-r,cy+r,cz+r],
      [cx-r,cy-r,cz-r], [cx-r,cy+r,cz+r], [cx+r,cy-r,cz+r]
    ];

    triangles := zip([([ (i-1 div 3)+1,0])$i=1..nops(triangles)], triangles, _concat);
    out::writeTriangles(attributes, table("MeshVisible"   = FALSE,
                              "PointsVisible" = FALSE,
                              "HasLineColors" = FALSE), triangles,
      (u,v,x,y,z)->fillcolorfunction(u, x, y, z));
    out::writePoly3d(attributes, table("Filled" = FALSE, "Closed" = TRUE), triangles,
      (u,v,x,y,z)->linecolorfunction(u, x, y, z));

    // finally, return the viewing box
    entries:= (i) -> map(triangles,op,i);
    return( [min(entries(3))..max(entries(3)),
             min(entries(4))..max(entries(4)),
             min(entries(5))..max(entries(5))] );
  end_proc:
//------------------------------------------------------------------------