// 

// plot::Tube -- canal surfaces (generalized tube plots)

/*

 The idea currently implemented resembles that from
 http://www.ii.uj.edu.pl/EMIS/monographs/CGM/2c.html,
 with the principal normal replaced by the projection
 of the previous principal normal onto the plane
 perpendicular to the current curve direction.

*/

/*
 Fr Ralf:

plot(plot::Tube([sin(t), cos(t^2), t], 0.1, t=0..5))
plot(plot::Tube([sin(u), cos(u), (u*(1+0.1*sin(a)))/10], 0.1, u=0..50, a=0..2*PI))
plot(plot::Tube([sin(u), cos(u), 0], 0.4+0.3*sin(u), u=0..2*PI,
                   UMesh = 50, VSubmesh=2,
   FillColorFunction = ((u, v, x, y, z) -> RGB::fromHSV([(u+sin(4*v))*180/PI, 1, 1]))))

*/

/*

TODO:
 AdaptiveMesh  (Should use curveEval)
 a plot::Curve3d should be accepted as an input
 error checking in MuPlotML
 self-intersections of surface
 boundary cases in MuPlotML: does the code always find a starting vector?

*/

plot::createPlotDomain("Tube",
                       "generalized tube plots (canal surfaces)",
                       3,
                       [XFunction, YFunction, ZFunction,
                        [RadiusFunction, ["Mandatory", NIL],
                         ["Definition", "Expr", plot::elementOptFunctionExpr,
                          "The radius as a function of the parameter", TRUE]],
                        UName, UMin, UMax, URange,
                        [AngleBegin, ["Optional", 0],
                         ["Definition", "Expr", FAIL, "Beginning of the angle", FALSE]],
                        [AngleEnd,   ["Optional", 2*PI],
                         ["Definition", "Expr", FAIL, "End of the angle", FALSE]],
                        // library interface for AngleBegin, AngleEnd
                        [AngleRange,   ["Library", NIL,
                                        plot::libRangeOfOptExpr("Angle"),
                                        plot::readRangeOfOptExpr("Angle")],
                         [[AngleBegin..AngleEnd]]],
                        UMesh, VMesh, USubmesh, VSubmesh,
                        Mesh, Submesh,
                        ULinesVisible, VLinesVisible,
                        PointsVisible,
                        LineColor, LineColorType, LineColor2,
                        LineColorFunction,
                        LineWidth,
                        Filled, FillColor, FillColorType, FillColor2,
                        FillColorFunction,
                        Shading,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
                       ]):

plot::Tube::styleSheet := table(UMesh = 60, VMesh = 11, VSubmesh = 1,
                                LegendEntry = TRUE,
                                LineColor = RGB::Black.[0.25],
                                LineColorType = Flat,
                                RadiusFunction = 1/10,
            LineColorDirectionY=0):

plot::Tube::hints := {Scaling = Constrained}:

plot::Tube::new :=
  proc()
    local object, other;
  begin
    object := dom::checkArgs(["U"], args());
    
    other := object::other;
    
    case nops(other)
      of 0 do
        break;
      of 1 do
        if (other[1])::dom::hasProp(Cat::Matrix) = TRUE then
          other[1] := [op(other[1])];
        end_if;
        if domtype(other[1]) = DOM_LIST and nops(other[1]) = 3 then
          object::XFunction := other[1][1];
          object::YFunction := other[1][2];
          object::ZFunction := other[1][3];
        else
          object::RadiusFunction := other[1];
        end_if;
        break;
      of 2 do
        if (other[1])::dom::hasProp(Cat::Matrix) = TRUE then
          other[1] := [op(other[1])];
        end_if;
        if domtype(other[1]) = DOM_LIST and nops(other[1]) = 3 then
          object::XFunction := other[1][1];
          object::YFunction := other[1][2];
          object::ZFunction := other[1][3];
        else
          error("First argument should be a list of three expressions");
        end_if;
        object::RadiusFunction := other[2];
        break;
      otherwise
        error("unexpected argument ".expr2text(other[3]));
    end_case;
    
    dom::checkObject(object);
  end_proc:
  
plot::Tube::print :=
  obj -> hold(plot::Tube)([obj::XFunction, obj::YFunction, obj::ZFunction],
                          obj::RadiusFunction, obj::UName=obj::URange):

plot::Tube::doPlotStatic :=
  proc(out, obj, attributes, inherited)
    local x, y, z, P, B, tmp, uname, u, v, umin, umax, umesh,
          fr, fx, fy, fz, vmesh, dx, dy, dz, i, j, mesh, r, c, s,
          findStart, project, normalize, fillcolorfunction, linecolorfunction,
          haslinecolorfunction, oldB, oldP, ab, ae, angleBegin, angleEnd, rnd, vals,
          points;
  begin
    uname := attributes[UName];
    umin  := attributes[UMin];
    umax  := attributes[UMax];
    
    if iszero(umin-umax) then
      return(null());
    end_if;
    
    umesh := (attributes[UMesh] - 1)*(attributes[USubmesh] + 1);
    // vmesh is larger, because the last point is repeated!
    vmesh := (attributes[VMesh] - 1)*(attributes[VSubmesh] + 1);
    fr    := attributes[RadiusFunction];
    fx    := attributes[XFunction];
    fy    := attributes[YFunction];
    fz    := attributes[ZFunction];
    
    angleBegin := float@fp::unapply(attributes[AngleBegin], uname);
    angleEnd :=   float@fp::unapply(attributes[AngleEnd], uname);
    
    if umesh < 2 then
      warning("UMesh lower than 3.  Using 3 instead.");
      umesh := 2;
    end_if;
    
    if contains(attributes, FillColorFunction) then
      fillcolorfunction := attributes[FillColorFunction];
    else
      // option hold, to avoid a float(subs(...))
      fillcolorfunction := proc() option hold; begin null(); end_proc;
    end_if;
    if contains(attributes, LineColorFunction) then
      linecolorfunction := attributes[LineColorFunction];
      haslinecolorfunction := TRUE;
    else
      linecolorfunction := proc() option hold; begin null(); end_proc;
      haslinecolorfunction := FALSE;
    end_if;
    
    rnd := stats::normalRandom(0, abs(umax-umin)/(10*umesh)+1e-8, Seed=123456);
    u := [float(((umesh-i)*umin + i*umax)/(umesh)) $ i=0..umesh];
    vals := map(u, proc(uv)
                     local u, i, ret;
                   begin
                     i := 0;
                     u := uv;
                     while traperror((ret := [fr, fx, fy, fz,
                                              angleBegin,
                                              angleEnd](u))) <> 0 or
                       hastype(ret, DOM_COMPLEX) do
                       i := i+1;
                       if i > 20 then
                         context(hold(error)("Can't evaluate to real value near ".expr2text(u)));
                       end_if;
                       repeat
                         u := uv + rnd();
                       until u in hull(umin, umax) end_repeat;
                     end_while;
                     ret;
                   end_proc);
    
    r := map(vals, op, 1);
    x := map(vals, op, 2);
    y := map(vals, op, 3);
    z := map(vals, op, 4);
    ab := map(vals, op, 5);
    ae := map(vals, op, 6);
    
    v := proc(i, j)
           option remember;
         begin
           float(((j-1)/(vmesh)*(ae[i]-ab[i]))+ab[i]);
         end_proc:

    c := proc(i, j)
           option remember;
         begin
           specfunc::cos(v(args()));
         end_proc:
    s := proc(i, j)
           option remember;
         begin
           specfunc::sin(v(args()));
         end_proc:
    
    if map({op(r)}, domtype) <> {DOM_FLOAT} then
      error("radius function must return real values!");
    end_if;
    
    // first order derivative approximation
    dx := [x[2]-x[1], x[i+1]-x[i]$i=1..umesh];
    dy := [y[2]-y[1], y[i+1]-y[i]$i=1..umesh];
    dz := [z[2]-z[1], z[i+1]-z[i]$i=1..umesh];
    
    // the example
    // plot(plot::Tube([sin(u), cos(u), 0], 0.4+0.3*sin(u), u=0..2*PI, VSubmesh=2))
    // shows that the tangent vectors must be properly aligned
    // for closed curves:
    tmp := hull(x, y, z);
    if (x[1] - x[-1])^2 + (y[1] - y[-1])^2 + (z[1] - z[-1])^2 
       < 10^(3-DIGITS) * (op(tmp, 2) - op(tmp, 1)) then
      dx[1] := dx[-1];
      dy[1] := dy[-1];
      dz[1] := dz[-1];
    end_if;
    
    if iszero(dx[1]) and iszero(dy[1]) and iszero(dz[1]) then
      i := 2;
      while i <= nops(dx) and
            iszero(dx[i]) and iszero(dy[i]) and iszero(dz[i]) do
        i := i+1;
      end_while;
      if i>nops(dx) then
        return(null());
      end_if;
      dx[1] := dx[i];
      dy[1] := dy[i];
      dz[1] := dz[i];
    end_if;
    
    // normalize a vector
    normalize := proc(A)
                   local n;
                 begin
                   n := sqrt(A[1]^2+A[2]^2+A[3]^2);
                   if iszero(n) then A
                   else
                     eval(map(A, `/`, n));
                   end_if;
                 end_proc;
    
    // projection of A to the plane perpendicular to B:
    project := proc(A, B)
                 local scal;
               begin
                 scal := A[1]*B[1] + A[2]*B[2] + A[3]*B[3];
                 A := zip(A, B, (a, b) -> a-scal*b);
               end_proc:
    
    // start "properly"
    findStart := proc(i)
                   local dxyz, i1;
                 begin
                   if i = 1 then i1 := 2 else i1 := i; end;
                   B := [(dy[i1] - dy[i1+1]) * dz[i] - (dz[i1] - dz[i1+1]) * dy[i],
                         (dz[i1] - dz[i1+1]) * dx[i] - (dx[i1] - dx[i1+1]) * dz[i],
                         (dx[i1] - dx[i1+1]) * dy[i] - (dy[i1] - dy[i1+1]) * dx[i]];
                   while iszero(dx[i]) and iszero(dy[i]) and iszero(dz[i]) do
                     i := modp(i, umesh) + 1;
                   end_while;
                   dxyz := normalize([dx[i], dy[i], dz[i]]);
                   B := project(B, dxyz);
                   if iszero(B[1]) and iszero(B[2]) and iszero(B[3]) then
                     B := project([0,0,1], dxyz);
                   end_if;
                   if iszero(B[1]) and iszero(B[2]) and iszero(B[3]) then
                     B := project([0,1,0], dxyz);
                   end_if;
                   P := [B[2] * dz[1] - B[3] * dy[1],
                         B[3] * dx[1] - B[1] * dz[1],
                         B[1] + dy[1] - B[2] * dx[1]];
                 end_proc:
    findStart(1);
    B := normalize(B);
    P := normalize(P);
    
    // we must store the mesh, simply because for Mesh3d,
    // the transposed version of the calculated mesh must be written.
    mesh := array(1..umesh+1, 1..vmesh+1);
    for i from 1 to umesh+1 do
      // normalize (dx, dy, dz)[i]
      tmp := dx[i]^2 + dy[i]^2 + dz[i]^2;
      if tmp < 10^(2-DIGITS) then
        dx[i] := dx[i-1];
        dy[i] := dy[i-1];
        dz[i] := dz[i-1];
      else
        tmp := 1/sqrt(tmp);
        dx[i] := dx[i] * tmp;
        dy[i] := dy[i] * tmp;
        dz[i] := dz[i] * tmp;
      end_if;
      
      // project B to perpendicular to [dx, dy, dz]
      oldB := B;
      B := project(B, [dx[i], dy[i], dz[i]]);
      
      if iszero(B[1]) and iszero(B[2]) and iszero(B[3]) then
        // restart?  for now, just reuse the previous one
        B := oldB;
      end_if;
      B := normalize(B);
      oldP := P;
      P := [B[2] * dz[i] - B[3] * dy[i],
            B[3] * dx[i] - B[1] * dz[i],
            B[1] * dy[i] - B[2] * dx[i]];
      // as the example
      // plot(plot::Tube([0,0,sin(u)], u, u=0..10))
      // shows, we must take care that the circles
      // are oriented consistently:
      if oldP[1]*P[1] + oldP[2]*P[2] + oldP[3]*P[3] < 0 then
        P := map(P, _negate);
      end_if;
      
///      plot::MuPlotML::beginElem("Arrow3d",
///                             "From" = (x[i], y[i], z[i]),
///                             "To"=    (x[i] + r[i]*B[1], y[i] + r[i]*B[2], z[i] + r[i]*B[3]),
///                             "Tubular" = FALSE,
///                             "LineWidth" = "0.002",
///                             "TipLength" = "0.05", Empty);
///      plot::MuPlotML::beginElem("Arrow3d",
///                             "From" = (x[i], y[i], z[i]),
///                             "To"=    (x[i] + r[i]*P[1], y[i] + r[i]*P[2], z[i] + r[i]*P[3]),
///                             "LineColor" = RGB::Green,
///                             "Tubular" = FALSE,
///                             "LineWidth" = "0.002",
///                             "TipLength" = "0.05", Empty);
      for j from 1 to vmesh + 1 do
        tmp :=  [r[i] * P[1] * c(i, j) + r[i] * B[1] * s(i, j),
                 r[i] * P[2] * c(i, j) + r[i] * B[2] * s(i, j),
                 r[i] * P[3] * c(i, j) + r[i] * B[3] * s(i, j)];
        mesh[i, j] := [x[i] + tmp[1],
                       y[i] + tmp[2],
                       z[i] + tmp[3],
                       tmp[1], tmp[2], tmp[3]];
      end_for;
    end_for;

    points := [];
    for j from 1 to vmesh+1 do
      for i from 1 to umesh+1 do
        points := points.[[op(u[i]), 0, op(mesh[i, j])]];
      end_for;
    end_for;
    out::writeRectangularMesh(attributes, table(), points, linecolorfunction, fillcolorfunction,
      max(attributes[UMesh], 3), attributes[VMesh], attributes[USubmesh], attributes[VSubmesh], FALSE );

//    for i from 1 to umesh+1 do
//      if FALSE and frac((ae[i]-ab[i])/2/float(PI)) < 10^(1-DIGITS) then
//      j := 1;
//      else
//      j := vmesh+1;
//      end_if;
//      plot::MuPlotML::prP3(op(mesh[i, j]),
//                         linecolorfunction(u[i], v(i, j),
//                                           op(op(mesh[i, j], 1..3))),
//                         fillcolorfunction(u[i], v(i, j),
//                                           op(op(mesh[i, j], 1..3))));
//    end_for;
    
    mesh := {op(mesh)};
    tmp := x -> if specfunc::abs(x) < 1e-300 then 0.0
                else x end_if;
    [_range(map(op(hull(map(mesh, op, 1))), tmp)),
     _range(map(op(hull(map(mesh, op, 2))), tmp)),
     _range(map(op(hull(map(mesh, op, 3))), tmp))];
  end_proc:
  
