/* -----------------------------------------------------------
    VectorField3d -- The graphical primitive for 3D vectorfields

    Syntax:
    VectorField3d([v1, v2, v3], x = a..b, y = c..d , z = e..f <, op1, op2, ...>)
    VectorField3d(v1, v2, x = a..b, y = c..d <, op1, op2, ...>)
    v1, v2    : x- and y-component of the vectorfield (functional expressions
                in x and y
    x,y       : identifiers
    a, b, c, d, e ,f : arithmetical expressions
    options   : plot options for 3D graphical objects 

   Examples:  
   We demonstrate a plot of the vector field 
   v(x,y,z) = [1, sin(x)+cos(y), z].
   field:= plot::VectorField3d(
                [1,sin(x)+cos(y), z], x=0..3, y=1..2, z=1..2,
                LineColor = RGB::Red):
   plot(field);
----------------------------------------------------------------*/ 
plot::createPlotDomain("VectorField3d",
                       "graphical primitive for 3D vectorfields"):
// methods yet to be implemented:  convert, convert_to, expr
//----------------------------------------------------------------
plot::VectorField3d::styleSheet:=
table(XMesh = 7,
      YMesh = 7,
      ZMesh = 7,
      PointsVisible = TRUE,
      LineWidth = 0.1,
      LineColorDirectionY=0
):
//----------------------------------------------------------------
plot::VectorField3d::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs(["X", "Y", "Z"], args()); 
    // get arguments which are not yet processed
    other := object::other;
    
    if nops(other) > 0 then
      if (other[1])::dom::hasProp(Cat::Matrix) = TRUE then
         other[1]:= [op(other[1])];
      end_if;
  
      if nops(other) = 1 and domtype(other[1]) = DOM_LIST and nops(other[1]) = 3 then
        object::XFunction := other[1][1];
        object::YFunction := other[1][2];
        object::ZFunction := other[1][3];
      
      elif nops(other) = 3 then
        object::XFunction := other[1];
        object::YFunction := other[2];
        object::ZFunction := other[3];
      elif nops(other) > 3 then
        error("unexpected argument: ".expr2text(other[4]));
      end_if;
    end_if;
  
    // semantically check for validity
    dom::checkObject(object);
  end_proc:
//----------------------------------------------------------------
plot::VectorField3d::print :=
  obj -> hold(plot::VectorField3d)([obj::XFunction, obj::YFunction, obj::ZFunction],
                             obj::XName=obj::XRange,
                             obj::YName=obj::YRange,
                             obj::ZName=obj::ZRange):
//----------------------------------------------------------------
plot::VectorField3d::doPlotStatic:=
  proc(out, obj, attributes, inherited)
  local colorfunction, f_x_value, f_y_value, f_z_value,
        x, y, z, f_x, f_y, f_z, indX, indY, indZ,
        stepX, stepY, stepZ, xmesh, xmax, xmin,
        ymesh, ymax, ymin, zmesh, zmax, zmin,
        points;
  begin

    if contains(attributes, LineColorFunction) then
      colorfunction:=  attributes[LineColorFunction] // 1 procedure,
                                                  // 3 expressions (RGB) or 
                                                  // 4 expressions (RGBa)
    else colorfunction:= () -> null();
    end_if;

    f_x := attributes[XFunction];
    f_y := attributes[YFunction];
    f_z := attributes[ZFunction];

    xmin  := attributes[XMin]:
    xmax  := attributes[XMax]:
    xmesh := attributes[XMesh]:
    ymin  := attributes[YMin]:
    ymax  := attributes[YMax]:
    ymesh := attributes[YMesh]:
    zmin  := attributes[ZMin]:
    zmax  := attributes[ZMax]:
    zmesh := attributes[ZMesh]:
    if xmesh = 1 then
    	xmesh :=2 ;
    end_if;
    if ymesh = 1 then
    	ymesh := 2;
    end_if;
    if zmesh = 1 then
    	zmesh := 2;
    end_if;
    stepX := (xmax-xmin)/(xmesh-1);
    stepY := (ymax-ymin)/(ymesh-1);
    stepZ := (zmax-zmin)/(zmesh-1);
    
    points := [];
    for indZ from 0 to zmesh-1 do
      z := (zmesh - 1 - indZ)/(zmesh - 1)*zmin + indZ/(zmesh - 1)*zmax;
      for indY from 0 to ymesh-1 do
        y := (ymesh - 1 - indY)/(ymesh - 1)*ymin + indY/(ymesh - 1)*ymax;
        for indX from 0 to xmesh-1 do
          x := (xmesh - 1 - indX)/(xmesh - 1)*xmin + indX/(xmesh - 1)*xmax;

          // correct Colorfuntion has to be tested!
          // possible: cf(x, y, z); cf(f_x(..),f_y(..),f_z(..));
          //  cf(x, y, z, f_x(..), f_y(..), f_z(..));
          if traperror((f_x_value:= float(f_x(x, y, z));
                        f_y_value:= float(f_y(x, y, z));
                        f_z_value:= float(f_z(x, y, z)))) <> 0 or
            map({f_x_value, f_y_value, f_z_value}, domtype) <> {DOM_FLOAT} then
            f_x_value := 0;
            f_y_value := 0;
            f_z_value := 0;
          end_if;
          points := points.[[x, y, z, f_x_value, f_y_value, f_z_value]];
        end_for;
      end_for;
    end_for;
     
    out::writeField3d(attributes, table(), points,
      float(xmin), float(ymin), float(zmin), float(stepX), float(stepY), float(stepZ), xmesh, ymesh, zmesh,
      colorfunction, 4..6);
    return ([float(xmin)..float(xmax),
             float(ymin)..float(ymax),
             float(zmin)..float(zmax)]);
    
  end_proc:
