//      

/* -----------------------------------------------------------
    Waterman -- Waterman polyhedra

    Syntax:
    plot::Waterman(order)

    order: A positive real number

    Example:
    >> plot::Waterman(4)

    >> plot::Waterman(4, Center=[0.5,0,0])

----------------------------------------------------------------*/ 

plot::createPlotDomain("Waterman",
		       "graphical primitive for Waterman polyhedra",
		       3,
		       [Radius,
			CenterX, CenterY, CenterZ, Center,
			LinesVisible, LineStyle, LineWidth,
			LineColor, LineColorType, LineColor2,
			LineColorFunction,
			Filled, FillColor, FillColorType,
			FillColor2, Color,
			FillColorFunction,
			Shading,
			PointsVisible, PointStyle, PointSize,
			LineColorDirectionX, LineColorDirectionY,
			LineColorDirectionZ, LineColorDirection,
			FillColorDirectionX, FillColorDirectionY,
			FillColorDirectionZ, FillColorDirection]):

//-----------------------------------------------------------------------------
plot::Waterman::styleSheet:= table(CenterX = 0,
				   CenterY = 0,
				   CenterZ = 0,
				   LinesVisible = TRUE,
				   LineWidth = 0.25,
				   Filled = TRUE,
				   LineColor=RGB::Grey40.[0.4],
				   FillColor = RGB::LightOrange,
				   FillColorType = Flat
				  ):
				  
plot::Waterman::hints := table(Scaling = Constrained):

// methods yet to be implemented:  convert, convert_to, expr

// here we have to do something
plot::Waterman::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;
    
    // accept a single expression
    if nops(other) > 1 then
      error("unexpected argument: ".expr2text(op(other, 2)));
    end_if;

    if nops(other) > 0 then
      object::Radius := op(other, 1);
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

plot::Waterman::print :=
  obj -> hold(plot::Waterman)(obj::Radius):

plot::Waterman::doPlotStatic:=
proc(out, obj, attrib, inherited)
  local radius, rad2, center, points, x, y, z,
	xra, xrb, yra, yrb, zra, zrb, R, Ry, s, h, t, r,
	coords, xmin, xmax, ymin, ymax, zmin, zmax,
	c, p, p2, cnt, prod, linecolorfunction,
	fillcolorfunction, i, j, sp, data;
begin
  radius := float(attrib[Radius]);
  center := float([attrib[CenterX], attrib[CenterY], attrib[CenterZ]]);
  
  if contains(attrib, FillColorFunction) then
    fillcolorfunction := float@(attrib[FillColorFunction]);
  else
    fillcolorfunction := () -> null();
  end;
  if contains(attrib, LineColorFunction) then
    linecolorfunction := float@(attrib[LineColorFunction]);
  else
    linecolorfunction := () -> null();
  end;
  
  points := table();
  rad2 := float(attrib[Radius]^2);
  s := radius;
  xra := ceil(center[1]-s); 
  xrb := floor(center[1]+s); 

  for x from xra to xrb do          
    R := rad2 - (x-center[1])^2; 
    if R < 0 then next end_if;
    s := sqrt(R);

    yra := ceil(center[2]-s);
    yrb := floor(center[2]+s);
    for y from yra to yrb do       
      Ry := R - (y-center[2])^2;
                
      if Ry < 0 then next end_if;              // case Ry < 0
      
      if Ry = 0 and iszero(frac(center[3])) then  // case Ry = 0
        if (x+y+round(center[3])) mod 2 <> 0 then
          next
        else
          zra := center[3];
          zrb := center[3];
        end_if
      else                                    // case Ry > 0
      	s := sqrt(Ry);
      	zra := ceil(center[3]-s);
      	zrb := floor(center[3]+s);
      	if (x+y) mod 2 = 0 then              // (x+y) mod 2 = 0
      	  if zra mod 2 <> 0 then
      	    if zra <= center[3] then
      	      zra := zra + 1
      	    else
      	      zra := zra - 1
      	    end_if
      	  end_if
      	else                                // (x+y) mod 2 <> 0
      	  if zra mod 2 = 0 then
      	    if zra <= center[3] then
      	      zra := zra + 1
      	    else
      	      zra := zra - 1               
      	    end_if
      	  end_if
      	end_if
      end_if;
      for z from zra to zrb step 2 do  
        points[nops(points)+1] := [x, y, z] 
      end_for:
    end_for:
  end_for:
  
  points := [points[i]$i=1..nops(points)];
  
  // If PointsVisible:
  out::writePoints3d(attrib, table(), map(points,float), ()->null(), 1..3);
  
  r := 2*(frandom(1876)-0.5);
  
  if nops(points) < 2 then
    return(null());
  end_if;
  while nops(points) < 6 do
    points := points.[zip(points[1], points[-1], (a,b)->(a+b)/2)];
  end_while;
  
  if traperror((h := plot::hull(points, FALSE))) <> 0 then
    // too few points
    return(null());
  end_if;
  
  data := [];
  for i from 1 to nops(h) do // i is needed for color functions
    t := h[i];
    // order polygon
    c := [(/* 0.3*r()+ */1.0) $ i = 1..nops(t)];
    c := map(c, _divide, _plus(op(c)));
    c := [float(_plus(t[j][i]*c[j] $ j=1..nops(t))) $ i = 1..nops(t[1])];

    p := [t[1][1]-c[1], t[1][2]-c[2], t[1][3]-c[3]];
    p := map(p, _divide, sqrt(p[1]^2+p[2]^2+p[3]^2));
    for j from 2 to nops(t) do
      p2 := [t[j][1]-c[1], t[j][2]-c[2], t[j][3]-c[3]];
      p2 := map(p2, _divide, sqrt(p2[1]^2+p2[2]^2+p2[3]^2));
      // if p and p2 are collinear, pick another second point
      sp := specfunc::abs(p[1]*p2[1]+p[2]*p2[2]+p[3]*p2[3]);
      if sp < 0.9 then
	break;
      end_if;
    end;
    cnt := 0;
    repeat
      cnt := cnt+1;
      prod := subs(((p1, p2) -> `#a`*(p1[2]*p2[3]-p1[3]*p2[2]) +
                                `#b`*(p1[3]*p2[1]-p1[1]*p2[3]) +
                                `#c`*(p1[1]*p2[2]-p1[2]*p2[1])),
                    [`#a`=1+r(), `#b`=1+r(), `#c`=1+r()]);
    until specfunc::abs(prod(p, p2)) > 0.1 or cnt > 5
    end_repeat;

    t := prog::sort(t, proc(pt)
                         local delta, acos;
                       begin
                         delta := [pt[1]-c[1], pt[2]-c[2], pt[3]-c[3]];
                         delta := map(delta, _divide, delta[1]^2+delta[2]^2+delta[3]^2);
                         acos := delta[1]*p[1]+delta[2]*p[2]+delta[3]*p[3];
                         if prod(p, delta) < 0 then 2-acos else acos end;
                       end);

    // triangulate it
    for j from 3 to nops(t) do
      data := data.[[[fillcolorfunction(op(t[1]), i, 1)], 0, op(t[1])]];
      data := data.[[[fillcolorfunction(op(t[j-1]), i, j-1)], 0, op(t[j-1])]];
      data := data.[[[fillcolorfunction(op(t[j]), i, j)], 0, op(t[j])]];
    end_for;
    h[i] := t;
  end;
  out::writeTriangles(attrib, table("PointsVisible"=FALSE, "MeshVisible"=FALSE), data,
    ()->null(), (u)->op(u));

  for i from 1 to nops(h) do
    t := h[i];
    // plot the lines
    out::writePoly3d(attrib, table("Closed"=TRUE, "PointsVisible"=FALSE, "Filled"=FALSE),
      zip(t, [([i,j] $ j=1..nops(t))], _concat),linecolorfunction, 1..3);
  end_for;
  
  if nops(points)=0 then
    return(null())
  end_if;
  coords := map(points, op, 1);
  xmin := min(op(coords)):
  xmax := max(op(coords)):
  coords := map(points, op, 2);
  ymin := min(op(coords)):
  ymax := max(op(coords)):
  coords := map(points, op, 3);
  zmin := min(op(coords)):
  zmax := max(op(coords)):
  
  [xmin..xmax, ymin..ymax, zmin..zmax]
end_proc:
