// Adaptive Surface tesselation, based on :
//    Luiz Velho, Luiz Henrique de Figueriredo, and Jonas Gomes
//    A Unified Approach for Hierarchical
//    Adaptive Tesselation of Surfaces
//    ACM Transaction on Graphics, Vol. 18, No. 4, Oct. 1999, pp.329-360
// ToDo: 
//       protection agains 'cannot convert to C double'
//       ignore complex values
//       special case empty plot (purely complex data)
//
// Structure of adaptiveSurfaceEval:
// --------+
// Storage |
// --------+
// The data are stored in 3 tables and two lists of sets:
//   POINTS =    table(i = [u, v, x, y, z])
//   REVPOINTS = table([u, v, x, y, z] = i)
//   EDGES  =    table(i = [[pt1, pt2], [], subdivlevel])
//               with pt1, pt2 = integer indices of POINTS
//               and  tr1, tr2 = integer indices of TRIANGLES
//               The [] is to store intermediate information
//               after splitting this edge
//   REVEDGES =  table([pt1, pt2] = i)
//   TRIANGLES = table(i = [e1, e2, e3])
//               with e1, e2, e3 = integer indices of EDGES
//               a negative e3 indicates that this edge is to be taken
//               in the opposite direction as stored in EDGES[abs(e3)]
//   ULINES =    [{i1, i2, i3, ... }, ...]
//   VLINES =    [{i1, i2, i3, ... }, ...]
//               with i1, i2, ... being indices into EDGES, such that
//               EDGES[i1], EDGES[i2], ... form the U/V-Lines
//-----------+
// Utilities |
//-----------+
// findRealPointBetween(pt1,pt2)
//    called for f(x1,y1) not real, f(x2,y2) real.
//    pt1, pt2 are lists, the first two entries of which are u,v coordinates.
//    tries to find a point as close as possible to (x1,y1)
//    on the straight line between the two points s.t.
//    f is real in that point.  Uses getPoint, i.e., assigns
//    via side-effect.  Returns the index of the newly added point.
// findBorder(a, b) // not implemented, yet
//    a = [ua, va, xa, ya, za] is a point, where x,y,z evaluate
//    to real floats. 
//    b = [ub, vb] is a point, where x,y,z, did not evaluate 
//    to real floats.
//    Returns a point c = [uc,vc,xc,yc,zc] between a and b (as close
//    as possible to b) where x, y, z evaluate to a real float
// subdivide(edge) // only partially implemented
//    Tries the midpoint c = (a+b)/2 of [a,b] = edge[1].
//    1) If c can be evaluated and satisfies the max bend criterion 
//       --> return([[a,b], edge[2], MAXSUBDIVLEVEL)
//    2) If c can be evaluated and does not satisfy the max bend criterion 
//       --> return([[a,c,b], edge[2], edge[3] + 1)
//    3) If c cannot be evaluated then
//          c1 = findBorder(a, c), add_new_point(c1)
//          c2 = findBorder(b, c), add_new_point(c2)
//       --> return([[a,c1,c2,b], edge[2], edge[3] + 1)
// splitTriangle(triangle) // only partially implemented
//    inspects the 3 edges [e1,e2,e3] = triangle, which were marked
//    for subdivision. There is a total of 27 cases dependening on
//    the form of the edges (an edge can have 2 points (-> nothing to do),
//    or 3 points (-> split at the midpoint) or 4 points (-> there is
//    a hole in the middle).
//    For each case do:
//        add_new_point()  // only for some of the 27 cases
//        add_new_edges()  // for the edges that have to be split
//        add_new_triangles() // corresponding to the split edges
//        delete TRIANGLES[triangle]
//--------------+
// Main program |
//--------------+
//    initialize()  // try evaluation on a coarse regular mesh
//    for subdivlevel from 1 to MAXSUBDIVLEVEL do
//        Edges:= map(Edges, subdivide) // mark the edges to be subdivided
//        for triangle in TRIANGLES do
//            splitTriangle(tr1);
//        end_for:
//    end_for:
//    conversion to Triangles[i] = [pt1, pt2, pt3]
//    addnormals();
//    return([ViewingBox, Points]])
//--------------------------------------------------------------------

// There's a *lot* of assertions in this file.
// If not debugging, it saves time to disable them completely:
if FALSE then
//  alias(asserteval(x)=x);
  alias(assert(x)=null());
else
//  alias(asserteval=assert);
end:

plot::adaptiveSurfaceEval:= proc(fX, fY, fZ,
                                 umesh, usubmesh, umin_umax,
                                 vmesh, vsubmesh, vmin_vmax,
                                 xrange, yrange, zrange 
                              // AdaptiveMesh = adaptivemesh
                              // "NoSingularities"
                                 )
local nosingularities,
      MAXBEND, MAXSUBDIVLEVEL, 
      POINTS, REVPOINTS,
      EDGES, REVEDGES, TRIANGLES, REVTRIANGLES,
      ULINES, VLINES,
      NEXTEDGE, NEXTTRIANGLE, NEXTPOINT,
      MINFLOAT, MAXFLOAT, ST,
      WORKTODO, BORDEREDGES,
      addpoint, addedge, addborderedge, treatborder,
      addtriangle, addtriangleFromPoints, removetriangle,
      addULine, addVLine, splitUVedge,
      umin, vmin, umax, vmax, 
      vbox, ubbLO, ubbHI, uwidth, vwidth,
      getPoint, trackRange,
      F, tmp, 
      fmin, fmax, Fmin, Fmax,
      _umin, _umax, _vmin, _vmax,
      i, j, u, du, v, dv, x, triangle, subdivlevel,
      p1, p2, p3, p4, p5, p6,
      um1, vm1, findRealPointBetween,
      revertEdge, subdivide, splitTriangle,
      XYZratio, XYZratio2, oldXYZratio2, minlength2, veryclose,
      negativeCurvature,
      getAutomaticViewingBox, refine, recompute,
      add_normals1, add_normals2,
      checktriangle,    // for debugging
      checkorientation, regroup
    ;
begin
  MAXSUBDIVLEVEL:= op(select([args()], has, AdaptiveMesh), [1,2]):
  nosingularities:= has([args()], "NoSingularities"):
  //------------------------------------------------------------------------
  // The following 'max bend angle' of 10 degrees is mentioned explicitly
  // in the documentation of PLOT/ATTRIBUTES/AdaptiveMesh.tex. Do not change
  // this value without keeping the documentation up to date!
  //------------------------------------------------------------------------
  MAXBEND:= tan::float(10/180*PI/2)^2/2;

  MINFLOAT:= 1e-300: // smallest float that can be converted to a C double
  MAXFLOAT:= 1e300:  // largest float that can be converted to a C double

  //-------------------------------------------------------
  // float input parameters
  //-------------------------------------------------------
  [umin, umax]:= float([op(umin_umax)]);
  [vmin, vmax]:= float([op(vmin_vmax)]);
  uwidth:= umax - umin;
  vwidth:= vmax - vmin;
  if iszero(uwidth) then
     umesh:= 2; // for speed
  end_if;
  if iszero(vwidth) then
     vmesh:= 2; // for speed
  end_if;

  // assumed below
  if umesh < 2 then umesh := 2; end;
  if vmesh < 2 then vmesh := 2; end;
  
  ULINES := [{} $ umesh];
  VLINES := [{} $ vmesh];
  
  umesh:= (umesh - 1)*(1 + usubmesh) + 1:
  du:= 1.435107e-6*uwidth: // 'random' shift used to avoid singularities

  vmesh:= (vmesh - 1)*(1 + vsubmesh) + 1:
  dv:= 1.435107e-6*vwidth: // 'random' shift used to avoid singularities

  //-------------------------------
  // initialize the data containers
  //-------------------------------
  POINTS:= table():    // container
  REVPOINTS := table():

  EDGES:= table():     // container
  REVEDGES := table():
  TRIANGLES:= table(): // container
  REVTRIANGLES := table();

  NEXTEDGE:= 1:     // initialize edge count
  NEXTTRIANGLE:= 1: // initialize triangle count
  NEXTPOINT := 1;
  
  BORDEREDGES := {};
  
  //--------------------------
  // initialize various tables
  //--------------------------
  [fmin, fmax, Fmin, Fmax,
   _umin, _umax, _vmin, _vmax,
   ubbLO, ubbHI, XYZratio
  ] := [table() $ 11]:
  
  // adding points and edges
  addpoint :=
  proc(u,v,x,y,z)
  begin
        assert(map({args()}, domtype@float)={DOM_FLOAT});
    if contains(REVPOINTS, [u,v,x,y,z]) then
      assert(contains(POINTS, REVPOINTS[[u,v,x,y,z]]));
    else
      POINTS[NEXTPOINT] := [u,v,x,y,z];
      REVPOINTS[[u,v,x,y,z]] := NEXTPOINT;
      NEXTPOINT := NEXTPOINT+1;
    end_if;
    REVPOINTS[[u,v,x,y,z]];
  end_proc;
  
  addedge :=
  proc(p1, p2, subdivlevel=0)
  begin
    assert(contains(POINTS, p1));
    assert(not has(POINTS[p1], FAIL));
    assert(contains(POINTS, p2));
    assert(not has(POINTS[p2], FAIL));
    if contains(REVEDGES, [p1, p2]) then
      assert(contains(EDGES, REVEDGES[[p1, p2]]));
      REVEDGES[[p1, p2]];
    elif contains(REVEDGES, [p2, p1]) then
      assert(contains(EDGES, REVEDGES[[p2, p1]]));
      -REVEDGES[[p2, p1]];
    else
      EDGES[NEXTEDGE] := [[p1, p2], [], subdivlevel];
      REVEDGES[[p1, p2]] := NEXTEDGE;
      NEXTEDGE := NEXTEDGE+1;
      NEXTEDGE - 1;
    end_if;
  end_proc;

  addborderedge := 
  proc(p1, p2)
    local i;
  begin
    if args(0) = 1 then
      i := specfunc::abs(p1);
    else
      i := specfunc::abs(addedge(p1, p2));
    end_if;
    assert(_lazy_or(not contains(REVTRIANGLES, i),
                    nops(REVTRIANGLES[i]) < 2));
    // do not use border edges in the usual subdivision
    EDGES[i][3] := MAXSUBDIVLEVEL+1;
    BORDEREDGES := BORDEREDGES union {i};
   end_proc;

   //------------------------------------------
   // For triangle = [edge1, edge2, edge3] with 
   // edge.i = [pt.i.1, pt.i.2], check that the 
   // points do indeed form a closed polygon.
   //------------------------------------------
   checktriangle:= proc(triangle)
   local ne1, ne2, ne3, e1, e2, e3, p1, p2, p3, p4, p5, p6;
   begin
     [ne1, ne2, ne3]:= TRIANGLES[triangle]:
     if ne1 > 0 then
           e1:= EDGES[ne1]; 
     else ne1:= -ne1;
           e1:= revertEdge(EDGES[ne1]);
     end_if;
     if ne2 > 0 then
           e2:= EDGES[ne2]; 
     else ne2:= -ne2;
           e2:= revertEdge(EDGES[ne2]);
     end_if;
     if ne3 > 0 then
           e3:= EDGES[ne3]; 
     else ne3:= -ne3;
           e3:= revertEdge(EDGES[ne3]);
     end_if;
     assert(ne1 <> ne2 and ne2 <> ne3 and ne1 <> ne3);
     p1 := e1[1][1]; p2 := e1[1][-1];
     p3 := e2[1][1]; p4 := e2[1][-1];
     p5 := e3[1][1]; p6 := e3[1][-1];
     assert(p2 = p3 and p4 = p5 and p6 = p1);
   end_proc:
  
  addtriangle :=
  proc(e1, e2, e3)
  begin
    assert(map({e1, e2, e3}, domtype)={DOM_INT});
    if nops(map({e1, e2, e3}, specfunc::abs)) = 3 then
      TRIANGLES[NEXTTRIANGLE] := [e1, e2, e3];
      REVTRIANGLES[specfunc::abs(e1)][NEXTTRIANGLE] := TRUE;
      REVTRIANGLES[specfunc::abs(e2)][NEXTTRIANGLE] := TRUE;
      REVTRIANGLES[specfunc::abs(e3)][NEXTTRIANGLE] := TRUE;
      checktriangle(NEXTTRIANGLE);
      NEXTTRIANGLE := NEXTTRIANGLE + 1;
      NEXTTRIANGLE-1;
    else
      0;
    end_if;
  end_proc;
  
  removetriangle :=
  proc(t)
  begin
    if contains(TRIANGLES, t) then
      delete REVTRIANGLES[specfunc::abs(TRIANGLES[t][1])][t];
      delete REVTRIANGLES[specfunc::abs(TRIANGLES[t][2])][t];
      delete REVTRIANGLES[specfunc::abs(TRIANGLES[t][3])][t];
    end_if;
    delete TRIANGLES[t];
  end_proc;
  
  checkorientation:= proc(tr)
                     local pt1, pt2, pt3, t21, t31;
                     begin
                        pt1:= POINTS[tr[1]];
                        pt2:= POINTS[tr[2]];
                        pt3:= POINTS[tr[3]];
                        t21:= [pt2[i] - pt1[i] $ i= 1..2];
                        t31:= [pt3[i] - pt1[i] $ i= 1..2];
                        bool(t21[2]*t31[1] - t21[1]*t31[2] <= 1e-10);
                     end_proc;

  addtriangleFromPoints :=
  proc(p1, p2, p3, level=0, border=FALSE)
  local e1, e2, e3, t;
  begin
    // ignore degenerate triangles: check for collinearity
    e1 := [POINTS[p3][2]-POINTS[p1][2],  // perpendicular to (p1,p3)
           POINTS[p1][1]-POINTS[p3][1]];
    e2 := [POINTS[p2][1]-POINTS[p1][1],
           POINTS[p2][2]-POINTS[p1][2]];
    if nops({POINTS[p1], POINTS[p2], POINTS[p3]})=3 and
       // normalized scalar product
       (e1[1]*e2[1] + e1[2]*e2[2]) > 
          1e-4*(max(abs(e1[1]), abs(e1[2]))*max(abs(e2[1]), abs(e2[2]))) then
      assert(checkorientation([p1, p2, p3]));
      e1 := addedge(p1, p2, level);
      e2 := addedge(p2, p3, level);
      e3 := addedge(p3, p1, level);
      if ((t := addtriangle(e1, e2, e3))) > 0 and border then
        addborderedge(e1);
      end_if;
      t;
    else
      0;
    end_if;
  end;

  //-----------------------------------------------------------
  // edges with length^2 <= minlength2 will not be subdivided. 
  // Note that relative measures are used (think of the objects
  // as being scaled to a cubic viewing box of length 1 in
  // each direction.)
  //-----------------------------------------------------------
  minlength2:= 1e-4;

  XYZratio2 := [1,1,1];
  veryclose :=
  proc(pt1, pt2)
  local distlr2;
  begin
     distlr2 :=   (pt1[3] - pt2[3])^2/XYZratio2[1]
                  + (pt1[4] - pt2[4])^2/XYZratio2[2]
                  + (pt1[5] - pt2[5])^2/XYZratio2[3];
     bool(distlr2 < minlength2);
  end_proc;

  
  //-------------------
  // utility revertEdge
  //-------------------
  revertEdge:= edge -> [revert(edge[1]), revert(edge[2]), edge[3]]:

  // --------------------------
  // initialize the viewing box
  // --------------------------
  vbox:= [xrange, yrange, zrange];
  for i from 1 to 3 do
    if vbox[i] = Automatic then
       vbox[i]:= Automatic .. Automatic;
    end_if;
  end_for:
  for i from 1 to 3 do
    // initialization for later refinement
    if op(vbox[i], 1) = Automatic then
         ubbLO[i]:= FALSE;    // userboundingbox?
         fmin[i]:=  MAXFLOAT; // The actual viewing box found during the evaluation
         Fmin[i]:= -MAXFLOAT; // The viewing box to be passed to the renderer
    else ubbLO[i]:= TRUE:     // userboundingbox?
         fmin[i]:= op(vbox[i], 1):
         Fmin[i]:= fmin[i];   // The bounding box requested by the user
    end_if;
    if op(vbox[i], 2) = Automatic then
         ubbHI[i]:= FALSE;    // userboundingbox?
         fmax[i]:= -MAXFLOAT; // The actual viewing box found during the evaluation
         Fmax[i]:=  MAXFLOAT; // The viewing box to be passed to the renderer
    else
         ubbHI[i]:= TRUE;     // userboundingbox?
         fmax[i]:= op(vbox[i], 2):
         Fmax[i]:= fmax[i];   // The bounding box requested by the user
    end_if;
  end_for:

  //---------------------------------------------
  // The parametrization functions of the surface
  //---------------------------------------------
  F:= [float@fX, float@fY, float@fZ];

   //-----------------------------------------------------
   // utility getPoint(u,v): try to evaluate X(u,v), Y(u,v), Z(u,v)
   // If the result consist of real numbers, assign them to the
   // variables x = [x[1],x[2],x[3]](of adaptiveSurfaceEval) 
   // (SIDE EFFECT!!) and return TRUE.
   // If an error occurs, FALSE is returned.
   // If a non-real value X(u,v) or Y(u,v) or Z(u,v) is found, it
   // is ignored and FALSE is returned.
   // The call getPoint(u,v) tracks fmin/max[i] ==  xmin, .. , zmax
   // as a SIDE EFFECT.
   //-----------------------------------------------------
   getPoint:= proc(u,v) 
   // assigns x:= [X(u,v),Y(u,v),Z(u,v)] as a side effect
   local i;
   begin
     x:= [0 $ 3]; // this is a variable of plot::adaptiveSurfaceEval!
     if traperror(((x[i]:= F[i](u,v)) $ i=1..3)) <> 0 then
//        singularities:= singularities union {[u,v]};
          return(FALSE)
     elif _lazy_and(
             x[i] <> RD_NAN $ i = 1..3,
             _lazy_or(domtype(x[i]) = DOM_FLOAT,
                      domtype((x[i]:= numeric::complexRound(x[i]))) = DOM_FLOAT)
             $ i = 1..3,
             x[i] >= -MAXFLOAT $ i = 1..3,
             x[i] <=  MAXFLOAT $ i = 1..3,
             _lazy_or((iszero(x[i]),
                       x[i] >= MINFLOAT,
                       x[i] <=-MINFLOAT) $ i = 1..3)
            ) then
//        if args(0) = 2 then
          trackRange(u,v,x);
//        end_if;
          return(TRUE)
     else return(FALSE);
     end_if;
   end_proc:
  //-----------------------------
  // utility findRealPointBetween
  //-----------------------------
  findRealPointBetween :=
  proc(p1, p2)
    option remember; // for consistency
    local x1, x2, y1, y2, tmin, tmax, t, ft, p3, i;
  begin
    x1 := p1[1]; x2 := p2[1];
    y1 := p1[2]; y2 := p2[2];
    assert(getPoint(x2, y2));
// can be caused by round-off effects ...
//    assert(not getPoint(x1, y1));
    ft := t -> getPoint((1-t)*x1+t*x2, (1-t)*y1+t*y2);
    tmin := 0.0;
    tmax := 1.0;
    for i from 1 to 15 do
      t := (tmin+tmax)/2;
      if ft(t) then
        tmax := t;
      else
        tmin := t;
      end_if;
    end_for;
    ft(tmax);
    assert(map(x, domtype)=[DOM_FLOAT$3]);
    p3 := [(1-tmax)*x1+tmax*x2, (1-tmax)*y1+tmax*y2,op(x)];
    if tmax > 0.99 or nops(p2) > 4 and veryclose(p2, p3) then
       // avoid extremely thin triangles at borders
       ft(1.0);
       addpoint(x2, y2, op(x));
    else
       addpoint(op(p3));
    end_if;
  end_proc;
  
   //------------------------------------------------------
   // utility trackRange(u,v,x): check the global extremal
   // values fmin/max[i] and redefine them if necessary.
   //------------------------------------------------------
   trackRange:= proc(u,v,x)
   local i;
   begin
     for i from 1 to 3 do
       if x[i] < fmin[i] then 
          fmin[i]:= x[i]; // the value of the minimum
          _umin[i]:= u;   // the location of the minimum
          _vmin[i]:= v;   // the location of the minimum
       end_if;
       if x[i] > fmax[i] then 
          fmax[i]:= x[i]; // the value of the maximum
          _umax[i]:= u;   // the location of the maximum
          _vmax[i]:= v;   // the location of the maximum
       end_if;
     end_for:
   end_proc:

  //---------------------------------------------
  // Evaluation on the coarse mesh
  //---------------------------------------------

//userinfo(3, "starting evaluation on the initial regular mesh. ");
  ST:= time():
  proc()
  name plot::Surface::evalOnRegularMesh;
  begin
    if umesh = 1 then
      umesh:= 2;
    end_if;
    um1:= umesh - 1:
    if vmesh = 1 then
      vmesh:= 2;
    end_if;
    vm1:= vmesh - 1:
    for i from 1 to vmesh do
      // Beware: do not change this computation of v. 
      // It prevents roundoff effects!
      v:= ((vmesh - i)/vm1)*vmin + ((i - 1)/vm1)*vmax;
      for j from 1 to umesh do
        // Beware: do not change this computation of u. 
        // It prevents roundoff effects!
        u:= ((umesh - j)/um1)*umin + ((j - 1)/um1)*umax;
        // x = [X, Y, Z] is assigned as a side effect
        if   getPoint(u   ,v   ) then POINTS[j+(i-1)*umesh]:= [u   , v   ,op(x)];
        elif getPoint(u+du,v   ) then POINTS[j+(i-1)*umesh]:= [u+du, v   ,op(x)];
        elif getPoint(u-du,v   ) then POINTS[j+(i-1)*umesh]:= [u-du, v   ,op(x)];
        elif getPoint(u   ,v+dv) then POINTS[j+(i-1)*umesh]:= [u   , v+dv,op(x)];
        elif getPoint(u   ,v-dv) then POINTS[j+(i-1)*umesh]:= [u   , v-dv,op(x)];
        elif getPoint(u+du,v+dv) then POINTS[j+(i-1)*umesh]:= [u+du, v+dv,op(x)];
        elif getPoint(u+du,v-dv) then POINTS[j+(i-1)*umesh]:= [u+du, v-dv,op(x)];
        elif getPoint(u-du,v+dv) then POINTS[j+(i-1)*umesh]:= [u-du, v+dv,op(x)];
        elif getPoint(u-du,v-dv) then POINTS[j+(i-1)*umesh]:= [u-du, v-dv,op(x)];
        else
          POINTS[j+(i-1)*umesh] := [u, v, FAIL$3];
        end_if;
      end_for;
    end_for;

    NEXTPOINT:= max(map([op(POINTS)], op, 1))+1;

    addULine :=
    proc(j, p1, p2)
    begin
      if modp(j-1, usubmesh+1) = 0 then
        j := ((j-1) div (usubmesh+1)) + 1;
        ULINES[j] := ULINES[j] union {specfunc::abs(addedge(p1, p2))};
      end_if;
    end_proc;
    addVLine :=
    proc(j, p1, p2)
    begin
      if modp(j-1, vsubmesh+1) = 0 then
        j := ((j-1) div (vsubmesh+1)) + 1;
        VLINES[j] := VLINES[j] union {specfunc::abs(addedge(p1, p2))};
      end_if;
    end_proc;
    
    // The points on the regular mesh are computed. 
    // Collect them to edges and triangles
    for i from 1 to vmesh-1 do
      for j from 1 to umesh-1 do
                                           //  p3---e5---p4
        p1:= j + (i-1)*umesh;              //   |\_      |
        p2:= p1 + 1;                       //  e3  \e2_  e4
        p3:= p1 + umesh; // = j + i*umesh; //   |      \ |
        p4:= p3 + 1;                       //  p1---e1---p2
        case map([p1, p2, p3, p4], p -> has(POINTS[p], FAIL))
          of [FALSE$4] do // f(u,v) defined and real on all corners
            addtriangleFromPoints(p1, p2, p3);
            addtriangleFromPoints(p2, p4, p3);
            addULine(j, p1, p3);
            addULine(j+1, p2, p4);
            addVLine(i, p1, p2);
            addVLine(i+1, p3, p4);
            break;
          of [TRUE, FALSE$3] do
            // f(u,v) not real in p1
            // (ascii drawings are too difficult for this, sorry)
            // p3 := [0,1]: p4 := [1,1]: p2 := [1,0]:
            // p1 := [0.2,0]: p5 := [0,0.3]:
            // e1 := plot::Line2d(p1, p2): e2 := plot::Line2d(p2, p5):
            // e3 := plot::Line2d(p5, p1): e4 := plot::Line2d(p2, p3):
            // e5 := plot::Line2d(p3, p5):
            // e6 := plot::Line2d(p2, p4): e7 := plot::Line2d(p4, p3):
            // plot(eval(e.i$i=1..7), Axes=None)
            p5 := findRealPointBetween(POINTS[p1], POINTS[p2]);
            p1 := findRealPointBetween(POINTS[p1], POINTS[p3]);
            addtriangleFromPoints(p1, p2, p3);
            addtriangleFromPoints(p2, p4, p3);
            addtriangleFromPoints(p1, p5, p2, 0, TRUE);
            addULine(j, p1, p3);
            addULine(j+1, p2, p4);
            addVLine(i, p5, p2);
            addVLine(i+1, p3, p4);
            break;
         of [FALSE, TRUE, FALSE, FALSE] do
           // not real in p2
           p5 := findRealPointBetween(POINTS[p2], POINTS[p4]);
           p2 := findRealPointBetween(POINTS[p2], POINTS[p1]);
           addtriangleFromPoints(p1, p2, p3);
           addtriangleFromPoints(p5, p4, p3);
           addtriangleFromPoints(p2, p5, p3, 0, TRUE);
           addULine(j, p1, p3);
           addULine(j+1, p5, p4);
           addVLine(i, p1, p2);
           addVLine(i+1, p3, p4);
           break;
         of [FALSE, FALSE, TRUE, FALSE] do
           p5 := findRealPointBetween(POINTS[p3], POINTS[p1]);
           p3 := findRealPointBetween(POINTS[p3], POINTS[p4]);
           addtriangleFromPoints(p1, p2, p5);
           addtriangleFromPoints(p2, p4, p3);
           addtriangleFromPoints(p3, p5, p2, 0, TRUE);
           addULine(j, p1, p5);
           addULine(j+1, p2, p4);
           addVLine(i, p1, p2);
           addVLine(i+1, p3, p4);
           break;
         of [FALSE, FALSE, FALSE, TRUE] do
           p5 := findRealPointBetween(POINTS[p4], POINTS[p2]);
           p4 := findRealPointBetween(POINTS[p4], POINTS[p3]);
           addtriangleFromPoints(p1, p2, p3);
           addtriangleFromPoints(p3, p2, p5);
           addtriangleFromPoints(p5, p4, p3, 0, TRUE);
           addULine(j, p1, p3);
           addULine(j+1, p2, p5);
           addVLine(i, p1, p2);
           addVLine(i+1, p3, p4);
           break;
         // two non-real points
         of [TRUE, TRUE, FALSE, FALSE] do
           p1 := findRealPointBetween(POINTS[p1], POINTS[p3]);
           p2 := findRealPointBetween(POINTS[p2], POINTS[p4]);
           addtriangleFromPoints(p2, p4, p3);
           addtriangleFromPoints(p1, p2, p3, 0, TRUE);
           addULine(j, p1, p3);
           addULine(j+1, p2, p4);
           addVLine(i+1, p3, p4);
           break;
         of [FALSE, TRUE, FALSE, TRUE] do
           p2 := findRealPointBetween(POINTS[p2], POINTS[p1]);
           p4 := findRealPointBetween(POINTS[p4], POINTS[p3]);
           addtriangleFromPoints(p1, p2, p3);
           addtriangleFromPoints(p2, p4, p3, 0, TRUE);
           addULine(j, p1, p3);
           addVLine(i, p1, p2);
           addVLine(i+1, p3, p4);
           break;
         of [FALSE, FALSE, TRUE, TRUE] do
           p3 := findRealPointBetween(POINTS[p3], POINTS[p1]);
           p4 := findRealPointBetween(POINTS[p4], POINTS[p2]);
           addtriangleFromPoints(p1, p2, p3);
           addtriangleFromPoints(p4, p3, p2, 0, TRUE);
           addULine(j, p1, p3);
           addULine(j+1, p2, p4);
           addVLine(i, p1, p2);
           break;
         of [TRUE, FALSE, TRUE, FALSE] do
           p3 := findRealPointBetween(POINTS[p3], POINTS[p4]);
           p1 := findRealPointBetween(POINTS[p1], POINTS[p2]);
           addtriangleFromPoints(p3, p2, p4);
           addtriangleFromPoints(p3, p1, p2, 0, TRUE);
           addULine(j+1, p2, p4);
           addVLine(i, p1, p2);
           addVLine(i+1, p3, p4);
           break;
         // two non-neighbouring corners.  Heuristic: If f(u,v) is available
         // in the center of the rectangle, we assume two cut off corners,
         // if it is not, wen assume non-connectedness.
         of [FALSE, TRUE, TRUE, FALSE] do
            p5 := findRealPointBetween(POINTS[p3], POINTS[p4]);
            p6 := findRealPointBetween(POINTS[p2], POINTS[p4]);
            p2 := findRealPointBetween(POINTS[p2], POINTS[p1]);
            p3 := findRealPointBetween(POINTS[p3], POINTS[p1]);
            if getPoint((POINTS[p1][1]+POINTS[p4][1])/2,
                        (POINTS[p1][2]+POINTS[p4][2])/2) then
              addtriangleFromPoints(p1, p2, p3);
              addtriangleFromPoints(p4, p5, p6);
              addtriangleFromPoints(p2, p6, p5, 0, TRUE);
              addtriangleFromPoints(p5, p3, p2, 0, TRUE);
            else
              addtriangleFromPoints(p2, p3, p1, 0, TRUE);
              addtriangleFromPoints(p5, p6, p4, 0, TRUE);
            end_if;
            addULine(j, p1, p3);
            addULine(j+1, p6, p4);
            addVLine(i, p1, p2);
            addVLine(i+1, p5, p4);
            break;
         of [TRUE, FALSE, FALSE, TRUE] do
            p5 := findRealPointBetween(POINTS[p4], POINTS[p3]);
            p6 := findRealPointBetween(POINTS[p1], POINTS[p3]);
            p1 := findRealPointBetween(POINTS[p1], POINTS[p2]);
            p4 := findRealPointBetween(POINTS[p4], POINTS[p2]);
            if getPoint((POINTS[p1][1]+POINTS[p4][1])/2,
                        (POINTS[p1][2]+POINTS[p4][2])/2) then
              addtriangleFromPoints(p2, p3, p6);
              addtriangleFromPoints(p2, p4, p3);
              addtriangleFromPoints(p4, p5, p3, 0, TRUE);
              addtriangleFromPoints(p6, p1, p2, 0, TRUE);
            else
              addtriangleFromPoints(p4, p1, p2, 0, TRUE);
              addtriangleFromPoints(p6, p5, p3, 0, TRUE);
            end_if;
            addULine(j, p6, p3);
            addULine(j+1, p2, p4);
            addVLine(i, p1, p2);
            addVLine(i+1, p3, p5);
            break;
            // three non-real points
         of [TRUE, FALSE, TRUE, TRUE] do
           p1 := findRealPointBetween(POINTS[p1], POINTS[p2]);
           p4 := findRealPointBetween(POINTS[p4], POINTS[p2]);
           addtriangleFromPoints(p4, p1, p2, 0, TRUE);
           addULine(j+1, p2, p4);
           addVLine(i, p1, p2);
           break;
         of [TRUE, TRUE, TRUE, FALSE] do
           p2 := findRealPointBetween(POINTS[p2], POINTS[p4]);
           p3 := findRealPointBetween(POINTS[p3], POINTS[p4]);
           addtriangleFromPoints(p3, p2, p4, 0, TRUE);
           addULine(j+1, p2, p4);
           addVLine(i+1, p3, p4);
           break;
         of [TRUE, TRUE, FALSE, TRUE] do
           p1 := findRealPointBetween(POINTS[p1], POINTS[p3]);
           p4 := findRealPointBetween(POINTS[p4], POINTS[p3]);
           addtriangleFromPoints(p1, p4, p3, 0, TRUE);
           addULine(j, p1, p3);
           addVLine(i+1, p3, p4);
           break;
         of [FALSE, TRUE, TRUE, TRUE] do
           p2 := findRealPointBetween(POINTS[p2], POINTS[p1]);
           p3 := findRealPointBetween(POINTS[p3], POINTS[p1]);
           addtriangleFromPoints(p2, p3, p1, 0, TRUE);
           addULine(j, p1, p3);
           addVLine(i, p1, p2);
           break;
        end_case;
      end_for;
    end_for:
    i:= vmesh;
    assert(modp(i-1, 1+vsubmesh)=0);
/*
    line_index := nops(VLINES);
    for j from 1 to umesh -1 do
        p1:= j + (i-1)*umesh;
        p2:= p1 + 1;
        e1:= addedge(p1, p2);
        VLINES[line_index] := VLINES[line_index] union {e1};
    end_for: */
  end_proc();
  
  POINTS := select(POINTS, _not@has, FAIL);
  if POINTS=table() or TRIANGLES=table() then return([NIL, [], [], []]); end_if;

  assert(_and(op(map([op(map(EDGES,
                             e -> contains(POINTS,e[1][1]) and
                             contains(POINTS,e[1][2])))], op, 2))));
  
  userinfo(3, "evaluation on the initial regular mesh finished. ".
              "Took: ".expr2text(time() - ST)." msec");
   //-------------------------------------------------------
   // Do check the triangles created by evalOnRegularMesh;
   //-------------------------------------------------------
   userinfo(10, "time for checking initial triangles = ".
             expr2text( time((
                               for triangle in TRIANGLES do
                                  checktriangle(op(triangle, 1));
                               end_for:
                               ))). " msec");
   userinfo(3, expr2text(nops(POINTS)). " points, ".
               expr2text(nops(EDGES))." edges, ".
               expr2text(nops(TRIANGLES))." triangles");
   //-------------------------------------------------------

  splitUVedge :=
  proc(old, p1, p2, p3, p4, newlevel)
  begin
    ULINES := subs(ULINES, old=(specfunc::abs(addedge(p1, p2, newlevel)), specfunc::abs(addedge(p3, p4, newlevel))));
    VLINES := subs(VLINES, old=(specfunc::abs(addedge(p1, p2, newlevel)), specfunc::abs(addedge(p3, p4, newlevel))));
    assert(testtype(ULINES, Type::ListOf(Type::SetOf(DOM_INT))));
    assert(testtype(VLINES, Type::ListOf(Type::SetOf(DOM_INT))));
  end_proc:

  //----------------------------------------------
  // utility treatborder:
  // try to bend edges bordering the non-real area
  // outwards if possible, inwards otherwise
  //----------------------------------------------
  treatborder :=
  proc(edge)
  local p1, p2, p3, p4, tr, pt1, pt2, pt3, pt4, pt5, normal, 
        fac_n, bend, distlr2, subdivlevel;
  begin
    if nops(EDGES[edge][1]) <> 2 then return(); end_if;
    [p1, p2] := EDGES[edge][1];
    pt1 := POINTS[p1];
    pt2 := POINTS[p2];
    distlr2 :=   (pt1[3] - pt2[3])^2/XYZratio2[1]
             + (pt1[4] - pt2[4])^2/XYZratio2[2]
             + (pt1[5] - pt2[5])^2/XYZratio2[3];
    if distlr2 < minlength2 then return(); end_if;
    
     // to find out in which direction the border-edge
     // should try to bulge, we must find its (unique) triangle:
    if not contains(REVTRIANGLES, specfunc::abs(edge)) or
       REVTRIANGLES[specfunc::abs(edge)] = table() then
      // no triangle with this edge.
      // this can happen if the triangle would be a degenerate one.
      // Ignore.
      return();
    end_if;
    tr := map([op(REVTRIANGLES[specfunc::abs(edge)])], op, 1);
    assert(nops(tr)=1);
    tr := op(tr, 1);
    if has(TRIANGLES[tr], -edge) then [p1, p2] := [p2, p1]; end;
     // the opposite corner:
    p3 := op({op(map(TRIANGLES[tr],
                     e -> (EDGES[specfunc::abs(e)][1][1],
                           EDGES[specfunc::abs(e)][1][-1])))}
             minus {p1, p2});
    assert(nops({p1, p2, p3}) = 3);
    
    pt3 := POINTS[p3];
    pt5 := [(pt1[1]+pt2[1])/2, (pt1[2]+pt2[2])/2]; // center of side
    if getPoint(op(pt5)) then
       // move out.  First, compute a normal
      normal := [pt2[2]-pt1[2], pt1[1]-pt2[1]];
       // check the direction
      fac_n := 1.0;
      if (pt3[1]-pt5[1])*normal[1]+(pt3[2]-pt5[2])*normal[2] > 0  then
        fac_n := -fac_n;
      end_if;
      repeat
        pt4 := [pt5[1]+fac_n*normal[1], pt5[2]+fac_n*normal[2]];
        fac_n := 2*fac_n;
      until pt4[1] < umin or pt4[1] > umax or
            pt4[2] < vmin or pt4[2] > vmax or
            specfunc::abs(fac_n) > 1 or
            not getPoint(op(pt4)) end_repeat;
      pt4[1] := min(max(pt4[1], umin), umax);
      pt4[2] := min(max(pt4[2], vmin), vmax);
      if getPoint(op(pt4)) then
        p4 := addpoint(op(pt4), op(x));
      else
        p4 := findRealPointBetween(pt4, pt5);
      end_if;
    else
      p4 := findRealPointBetween(pt5, pt3);
    end_if;
    pt4 := POINTS[p4];
     // check if that point has significantly moved
    bend:= (((pt1[3]+pt2[3])/2 - pt4[3])^2/XYZratio2[1]
            + ((pt1[4]+pt2[4])/2 - pt4[4])^2/XYZratio2[2]
            + ((pt1[5]+pt2[5])/2 - pt4[5])^2/XYZratio2[3])/distlr2:
    if bend <= MAXBEND/2 then // do not introduce a mid point
      return();
    end_if;
    
    subdivlevel := min(EDGES[specfunc::abs(TRIANGLES[tr][1])][3],
                       EDGES[specfunc::abs(TRIANGLES[tr][2])][3],
                       EDGES[specfunc::abs(TRIANGLES[tr][3])][3])-1;
    removetriangle(tr);
    if if checkorientation([p3, p1, p4]) then
         addtriangleFromPoints(p3, p1, p4, subdivlevel);
       else
         addtriangleFromPoints(p1, p3, p4, subdivlevel);
       end_if > 0 then
      addborderedge(p1, p4);
    end_if;

    if if checkorientation([p2, p3, p4]) then
         addtriangleFromPoints(p2, p3, p4, subdivlevel);
       else
         addtriangleFromPoints(p3, p2, p4, subdivlevel);
       end_if > 0 then
      addborderedge(p4, p2);
    end_if;
    splitUVedge(edge, p1, p4, p4, p2, MAXSUBDIVLEVEL+1);
  end_proc;

  //---------------------------------------------
  // utility subdivide(edge)
  //---------------------------------------------
  subdivide:= proc(edge)
  local p1, p2, p3, u1, u2, u3, v1, v2, v3,
        x1, x2, x3, y1, y2, y3, z1, z2, z3,
        du, dv, distlr2, pt3, bend;
  name plot::Surface::subdivide;
  begin
      // edge = [[p1, p2], [], subdivlevel] 
      if edge[3] >= MAXSUBDIVLEVEL then
         return(edge);
      end_if;
      if nops(edge[1]) <> 2 then 
         return(subsop(edge, 3 = MAXSUBDIVLEVEL));
      end_if;
      [p1, p2]:= edge[1]:
      [u1, v1, x1, y1, z1]:= POINTS[p1][1..5]:
      [u2, v2, x2, y2, z2]:= POINTS[p2][1..5]:
      distlr2 :=   (x1 - x2)^2/XYZratio2[1]
                 + (y1 - y2)^2/XYZratio2[2]
                 + (z1 - z2)^2/XYZratio2[3];
      if distlr2 <= minlength2 then
         return(subsop(edge, 3 = MAXSUBDIVLEVEL));
      end_if;
      // the interpolated middle:
      [u3, v3, x3, y3, z3]:= [(u1 + u2)/2, (v1 + v2)/2,
                              (x1 + x2)/2, (y1 + y2)/2, (z1 + z2)/2];
      // the exact middle:
      // getPoint assigns x = [X,Y,Z] as a side effect
      du:= 0.849187*uwidth/10^6; // choose a du different to the one in 
      dv:= 0.849187*vwidth/10^6; // evalOnRegularMesh to avoid hitting the singularity 
      if   getPoint(u3     , v3     ) then pt3:= [u3     , v3,      op(x)];
      elif getPoint(u3 + du, v3     ) then pt3:= [u3 + du, v3,      op(x)];
      elif getPoint(u3 - du, v3     ) then pt3:= [u3 - du, v3,      op(x)];
      elif getPoint(u3     , v3 + dv) then pt3:= [u3     , v3 + dv, op(x)];
      elif getPoint(u3     , v3 - dv) then pt3:= [u3     , v3 - dv, op(x)];
      elif getPoint(u3 + du, v3 + dv) then pt3:= [u3 + du, v3 + dv, op(x)];
      elif getPoint(u3 + du, v3 - dv) then pt3:= [u3 + du, v3 + dv, op(x)];
      elif getPoint(u3 - du, v3 + dv) then pt3:= [u3 - du, v3 + dv, op(x)];
      elif getPoint(u3 - du, v3 - dv) then pt3:= [u3 - du, v3 - dv, op(x)];
      else
         return([[p1, [u3, v3, FAIL], p2], [], edge[3]+1]);
      end_if;
      bend:= ((x3 - pt3[3])^2/XYZratio2[1]
            + (y3 - pt3[4])^2/XYZratio2[2]
            + (z3 - pt3[5])^2/XYZratio2[3])/distlr2:
      if bend <= MAXBEND then // do not introduce a mid point
         return(subsop(edge, 3 = MAXSUBDIVLEVEL));
      end_if;
      WORKTODO:= TRUE: // the flag WORKTODO informs splitTriangle
                       // that at least one edge was subdivided
      p3:= addpoint(op(pt3)); // mark for splitting at the mid point
      [[p1, p3, p2], [], edge[3] + 1];
  end_proc:

  //---------------------------------------------
  // utility splitTriangle(triangle)
  //---------------------------------------------
  splitTriangle:= proc(triangle)
  local e1, e2, e3, ne1, ne2, ne3, rev1, rev2, rev3,
        p1, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12,
        center, q1, q2, q3, subdivlevel;
  begin
     [ne1, ne2, ne3]:= TRIANGLES[triangle]:
     if ne1 > 0 then
          e1:= EDGES[ne1]; 
          rev1:= FALSE;
     else ne1:= -ne1;
          e1:= revertEdge(EDGES[ne1]);
          rev1:= TRUE;
     end_if;
     if ne2 > 0 then
          e2:= EDGES[ne2]; 
          rev2:= FALSE;
     else ne2:= -ne2;
          e2:= revertEdge(EDGES[ne2]);
          rev2:= TRUE;
     end_if;
     if ne3 > 0 then
          e3:= EDGES[ne3]; 
          rev3:= FALSE;
     else ne3:= -ne3;
          e3:= revertEdge(EDGES[ne3]);
          rev3:= TRUE;
     end_if;
     assert(ne1 <> ne2 and ne2 <> ne3 and ne1 <> ne3);
    
     //--------------------------------------------------------------
     // combine the cases that are equal up to permutation
     //--------------------------------------------------------------
     case [nops(e1[1]), nops(e2[1]), nops(e3[1])]
     of [2, 2, 2] do 
        return();
     //-----------------------------------
     of [2, 3, 2] do  // reduce to [3, 2, 2]
        [e1,   e2,  e3]:= [ e2,  e3,  e1];
        [ne1, ne2, ne3]:= [ne2, ne3, ne1];
        [rev1, rev2, rev3]:= [rev2, rev3, rev1];
        break;
     of [2, 2, 3] do // reduce to [3, 2, 2]
        [e1,   e2,  e3]:= [ e3,  e1,  e2];
        [ne1, ne2, ne3]:= [ne3, ne1, ne2];
        [rev1, rev2, rev3]:= [rev3, rev1, rev2];
        break;
     of [3, 2, 2] do
        break;
     //-----------------------------------
     of [3, 3, 2] do // reduce to [2, 3, 3]
        [ e1,  e2,  e3]:= [ e3,  e1,  e2];
        [ne1, ne2, ne3]:= [ne3, ne1, ne2];
        [rev1, rev2, rev3]:= [rev3, rev1, rev2];
        break;
     of [3, 2, 3] do // reduce to [2, 3, 3]
        [e1, e2, e3]:=    [ e2,  e3,  e1];
        [ne1, ne2, ne3]:= [ne2, ne3, ne1];
        [rev1, rev2, rev3]:= [rev2, rev3, rev1];
        break;
     of [2, 3, 3] do
        break;
     //-----------------------------------
     of [3, 3, 3] do
     // rotate to move the non-real points to e1, e2, e3, in that order
        case map([e1[1][2], e2[1][2], e3[1][2]], testtype, DOM_LIST)
        of [FALSE, FALSE, FALSE] do
           break;
        of [TRUE, FALSE, FALSE] do
           break;
        of [FALSE, TRUE, FALSE] do
           [e1, e2, e3]:=    [ e2,  e3,  e1];
           [ne1, ne2, ne3]:= [ne2, ne3, ne1];
           [rev1, rev2, rev3]:= [rev2, rev3, rev1];
           break;
        of [FALSE, FALSE, TRUE] do
           [e1, e2, e3]:=    [ e3,  e1,  e2];
           [ne1, ne2, ne3]:= [ne3, ne1, ne2];
           [rev1, rev2, rev3]:= [rev3, rev1, rev2];
           break;
        of [TRUE, TRUE, FALSE] do
           break;
        of [FALSE, TRUE, TRUE] do
           [e1, e2, e3]:=    [ e2,  e3,  e1];
           [ne1, ne2, ne3]:= [ne2, ne3, ne1];
           [rev1, rev2, rev3]:= [rev2, rev3, rev1];
           break;
        of [TRUE, FALSE, TRUE] do
           [e1, e2, e3]:=    [ e3,  e1,  e2];
           [ne1, ne2, ne3]:= [ne3, ne1, ne2];
           [rev1, rev2, rev3]:= [rev3, rev1, rev2];
           break;
        of [TRUE, TRUE, TRUE] do
           break;
        end_case;
        break;
     end_case;

     // the recursive division level of the edges
     subdivlevel:= [e1[3], e2[3], e3[3]];

     //--------------------------------------------------------------
     // If an edge has not been touched before in splitTriangle, its
     // 2nd entry is []. Split such an edge into 2 subedges and write 
     // the indices of the subedges into the edge.
     // If the edge was touched before, its 2nd entry is a list of 
     // 2 subedge indices. Pick these subedges and delete the edge.
     //--------------------------------------------------------------

     case [nops(e1[1]), nops(e2[1]), nops(e3[1])]
     of [2, 2, 2] do
        //------------
        //     p3 
        //    / \
        //  p1---p2  
        //------------
        // nothing to be done
        return();
     //---------------------------------------------
     //---------------------------------------------
//   of [2, 3, 2] do // reduced to [3, 2, 2] above
//   of [2, 2, 3] do // reduced to [3, 2, 2] above
     of [3, 2, 2] do
        //----------------
        //    /\  /\
        //   /  p3  \
        //  e3 /| \ e2
        // /  / e4 \  \
        // \ /  |   \ / 
        //  p1--p4--p2  
        //  |___e1___|
        //----------------
        [p1, p4, p2]:= e1[1]:
        [q2, p3]:= e2[1]:
        [q3, q1]:= e3[1]:
        assert(p1 = q1 and p2 = q2 and p3 = q3);
        // in case of a non-real evaluation at p4, cut around it:
        if testtype(p4, DOM_LIST) then
          assert(p4[3]=FAIL);
          p5 := findRealPointBetween(p4, POINTS[p1]);
          p6 := findRealPointBetween(p4, POINTS[p2]);
          p4 := findRealPointBetween(p4, POINTS[p3]);
          splitUVedge(ne1, p1, p5, p6, p2, subdivlevel[1]+1);
          removetriangle(triangle):
          addtriangleFromPoints(p1, p4, p3, subdivlevel[1]+1);
          addtriangleFromPoints(p4, p2, p3, subdivlevel[1]+1);
          addtriangleFromPoints(p5, p4, p1, subdivlevel[1]+1, TRUE);
          addtriangleFromPoints(p4, p6, p2, subdivlevel[1]+1, TRUE);
        else
          removetriangle(triangle);
          addtriangleFromPoints(p1, p4, p3, subdivlevel[1]+1);
          addtriangleFromPoints(p4, p2, p3, subdivlevel[1]+1);
          splitUVedge(ne1, p1, p4, p4, p2, subdivlevel[1]+1);
        end_if;
        break;
     //---------------------------------------------
     //---------------------------------------------
//   of [3, 3, 2] do // reduced to [2, 3, 3] above
//   of [3, 2, 3] do // reduced to [2, 3, 3] above
     of [2, 3, 3] do
        //----------------------------------
        //     /\  /\             /\  /\
        //    /  p3  \           /  p3  \
        //   e3 /  \ e2         e3 /  \ e2
        //  /  p5--p4  \  bzw. /  p5--p4  \
        //  \ / __/  \ /       \ /  \__ \ /
        //   p1/_____p2         p1_____\p2
        //   |___e1___|         |___e1___|
        //----------------------------------
        [p1, p2]:= e1[1]:
        [q2, p4, p3]:= e2[1]:
        [q3, p5, q1]:= e3[1]:
        assert(p1 = q1 and p2 = q2 and p3 = q3);
        case [testtype(p4, DOM_LIST), testtype(p5, DOM_LIST)]
        of [TRUE, TRUE] do
          // ToDo: the area of non-real points might connect p4 and p5.
          assert(p4[3] = FAIL);
          assert(p5[3] = FAIL);
          p6 := findRealPointBetween(p4, POINTS[p2]);
          p7 := findRealPointBetween(p4, POINTS[p3]);
          p4 := findRealPointBetween(p4, POINTS[p1]);
          
          p8 := findRealPointBetween(p5, POINTS[p3]);
          p9 := findRealPointBetween(p5, POINTS[p1]);
          p5 := findRealPointBetween(p5, POINTS[p4]);
          
          splitUVedge(ne2, p2, p6, p7, p3, subdivlevel[2]+1);
          splitUVedge(ne3, p3, p8, p9, p1, subdivlevel[3]+1);
          removetriangle(triangle);
          addtriangleFromPoints(p4, p3, p5, min(subdivlevel)+1);
          addtriangleFromPoints(p4, p5, p1, min(subdivlevel)+1);
          addtriangleFromPoints(p4, p1, p2, min(subdivlevel)+1);
        
          addtriangleFromPoints(p6, p4, p2, min(subdivlevel)+1, TRUE);
          addtriangleFromPoints(p4, p7, p3, min(subdivlevel)+1, TRUE);
          addtriangleFromPoints(p8, p5, p3, min(subdivlevel)+1, TRUE);
          addtriangleFromPoints(p5, p9, p1, min(subdivlevel)+1, TRUE);
          break;
        of [TRUE, FALSE] do
          assert(p4[3] = FAIL);
          p6 := findRealPointBetween(p4, POINTS[p2]);
          p7 := findRealPointBetween(p4, POINTS[p3]);
          p4 := findRealPointBetween(p4, POINTS[p1]);
          splitUVedge(ne2, p2, p6, p7, p4, subdivlevel[2]+1);
          splitUVedge(ne3, p3, p5, p5, p1, subdivlevel[3]+1);
          removetriangle(triangle);
          addtriangleFromPoints(p4, p7, p3, min(subdivlevel)+1, TRUE);
          addtriangleFromPoints(p4, p3, p5, min(subdivlevel)+1);
          addtriangleFromPoints(p4, p5, p1, min(subdivlevel)+1);
          addtriangleFromPoints(p4, p1, p2, min(subdivlevel)+1);
          addtriangleFromPoints(p6, p4, p2, min(subdivlevel)+1, TRUE);
          break;
        of [FALSE, TRUE] do
          assert(p5[3] = FAIL);
          p6 := findRealPointBetween(p5, POINTS[p1]);
          p7 := findRealPointBetween(p5, POINTS[p3]);
          p5 := findRealPointBetween(p5, POINTS[p2]);
          splitUVedge(ne3, p3, p7, p6, p1, subdivlevel[3]+1);
          splitUVedge(ne2, p2, p4, p4, p3, subdivlevel[2]+1);
          removetriangle(triangle);
          addtriangleFromPoints(p5, p6, p1, min(subdivlevel)+1, TRUE);
          addtriangleFromPoints(p5, p1, p2, min(subdivlevel)+1);
          addtriangleFromPoints(p5, p2, p4, min(subdivlevel)+1);
          addtriangleFromPoints(p5, p4, p3, min(subdivlevel)+1);
          addtriangleFromPoints(p7, p5, p3, min(subdivlevel)+1, TRUE);
          break;
        of [FALSE, FALSE] do
          removetriangle(triangle);
          //--------------------------------------------------------------
          // Introduce the shorter of the edges p1-p4 and p2-p5:
          if   (POINTS[p5][3] - POINTS[p2][3])^2/XYZratio2[1] +
               (POINTS[p5][4] - POINTS[p2][4])^2/XYZratio2[2] +
               (POINTS[p5][5] - POINTS[p2][5])^2/XYZratio2[3] 
            >= (POINTS[p4][3] - POINTS[p1][3])^2/XYZratio2[1] +
               (POINTS[p4][4] - POINTS[p1][4])^2/XYZratio2[2] +
               (POINTS[p4][5] - POINTS[p1][5])^2/XYZratio2[3] then
        
            addtriangleFromPoints(p1, p2, p4, subdivlevel[2]);
            addtriangleFromPoints(p1, p4, p5, min(subdivlevel[2..3]));
            addtriangleFromPoints(p3, p5, p4, min(subdivlevel[2..3]));
          else
            addtriangleFromPoints(p1, p2, p5, subdivlevel[3]);
            addtriangleFromPoints(p2, p4, p5, subdivlevel[2]);
            addtriangleFromPoints(p3, p5, p4, min(subdivlevel[2..3]));
          end_if;
          splitUVedge(ne3, p1, p5, p5, p3, subdivlevel[3]+1);
          splitUVedge(ne2, p2, p4, p4, p3, subdivlevel[2]+1)
        end_case;
        break;
     //---------------------------------------------
     //---------------------------------------------
     of [3, 3, 3] do
        //----------------
        //    /\    /\
        //   /   p3  \
        //  e3  /  \  e2
        // /  p6 -- p5  \
        // \ /  \  /  \ / 
        //  p1---p4---p2  
        //  |____e1____|
        //----------------
        [p1, p4, p2]:= e1[1]:
        [q2, p5, p3]:= e2[1]:
        [q3, p6, q1]:= e3[1]:
        assert(p1 = q1 and p2 = q2 and p3 = q3);
        case map([p4, p5, p6], testtype, DOM_LIST)
        of [TRUE, FALSE, FALSE] do
          p7 := findRealPointBetween(p4, POINTS[p1]);
          p8 := findRealPointBetween(p4, POINTS[p2]);
          p4 := findRealPointBetween(p4, POINTS[p3]);
          splitUVedge(ne1, p1, p7, p8, p2, subdivlevel[1]+1);
          splitUVedge(ne2, p2, p5, p5, p3, subdivlevel[2]+1);
          splitUVedge(ne3, p3, p6, p6, p1, subdivlevel[3]+1);
          removetriangle(triangle);
          addtriangleFromPoints(p1, p7, p6, min(subdivlevel)+1);
          addtriangleFromPoints(p7, p4, p6, min(subdivlevel)+1, TRUE);
          if checkorientation([p4, p5, p6]) then
            addtriangleFromPoints(p4, p5, p6, min(subdivlevel)+1);
            addtriangleFromPoints(p5, p3, p6, min(subdivlevel)+1);
          else
            addtriangleFromPoints(p4, p5, p3, min(subdivlevel)+1);
            addtriangleFromPoints(p4, p3, p6, min(subdivlevel)+1);
          end_if;
          addtriangleFromPoints(p4, p8, p5, min(subdivlevel)+1, TRUE);
          addtriangleFromPoints(p8, p2, p5, min(subdivlevel)+1);
          break;
        of [TRUE, TRUE, FALSE] do
          p7 := findRealPointBetween(p4, POINTS[p2]);
          p8 := findRealPointBetween(p5, POINTS[p2]);
          p9 := findRealPointBetween(p5, POINTS[p3]);
          p10 := findRealPointBetween(p4, POINTS[p1]);
           // check heuristically whether the non-realness at p4 and p5 is connected
          if getPoint((p4[1]+p5[1])/2, (p4[2]+p5[2])/2) and
             ((p4 := findRealPointBetween(p4, POINTS[p3]);
               p5 := findRealPointBetween(p5, POINTS[p1]);
               checkorientation([p6, p4, p5]))) then
            removetriangle(triangle);
            addtriangleFromPoints(p6, p1, p10,min(subdivlevel)+1);
            addtriangleFromPoints(p10,p4, p6, min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p6, p4, p5, min(subdivlevel)+1);
            addtriangleFromPoints(p5, p9, p6, min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p6, p9, p3, min(subdivlevel)+1);
            addtriangleFromPoints(p8, p5, p2, min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p2, p5, p4, min(subdivlevel)+1);
            addtriangleFromPoints(p4, p7, p2, min(subdivlevel)+1, TRUE);
          else
            removetriangle(triangle);
            addtriangleFromPoints(p6, p1, p10,min(subdivlevel)+1);
            addtriangleFromPoints(p10,p9, p6, min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p6, p9, p3, min(subdivlevel)+1);
            addtriangleFromPoints(p8, p7, p2, min(subdivlevel)+1, TRUE);
          end_if;
          splitUVedge(ne1, p1, p10, p7, p2, subdivlevel[1]+1);
          splitUVedge(ne2, p2, p8,  p9, p3, subdivlevel[2]+1);
          splitUVedge(ne3, p3, p6,  p6, p1, subdivlevel[3]+1);
          break;
        of [TRUE, TRUE, TRUE] do
          p7 := findRealPointBetween(p4, POINTS[p2]);
          p8 := findRealPointBetween(p5, POINTS[p2]);
          p9 := findRealPointBetween(p5, POINTS[p3]);
          p10:= findRealPointBetween(p6, POINTS[p3]);
          p11:= findRealPointBetween(p6, POINTS[p1]);
          p12:= findRealPointBetween(p4, POINTS[p1]);
          splitUVedge(ne1, p1, p12, p7,  p2, subdivlevel[1]+1);
          splitUVedge(ne2, p2, p8,  p9,  p3, subdivlevel[2]+1);
          splitUVedge(ne3, p3, p10, p11, p1, subdivlevel[3]+1);
          // heuristic whether the non-real part is connected:
          // if the center of gravity of the triangle evaluates
          // to a real value, yes, else no.
          center := [1/3*POINTS[p1][1]+2/3*p5[1],
                     1/3*POINTS[p1][2]+2/3*p5[2]];
          // center of gravity is chosen because all of these
          // lines go through it:
          p4 := findRealPointBetween(p4, POINTS[p3]);
          p5 := findRealPointBetween(p5, POINTS[p1]);
          p6 := findRealPointBetween(p6, POINTS[p2]);
          if (POINTS[p1][1]-POINTS[p5][1])^2 +
             (POINTS[p1][2]-POINTS[p5][2])^2 >
             (POINTS[p1][1]-center[1])^2 +
             (POINTS[p1][2]-center[2])^2 and
             (POINTS[p2][1]-POINTS[p6][1])^2 +
             (POINTS[p2][2]-POINTS[p6][2])^2 >
             (POINTS[p2][1]-center[1])^2 +
             (POINTS[p2][2]-center[2])^2 and
             (POINTS[p3][1]-POINTS[p4][1])^2 +
             (POINTS[p3][2]-POINTS[p4][2])^2 >
             (POINTS[p3][1]-center[1])^2 +
             (POINTS[p3][2]-center[2])^2 then
            removetriangle(triangle);
            addtriangleFromPoints(p12, p4,  p6,  min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p6,  p11, p12, min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p12, p11, p1,  min(subdivlevel)+1);
            addtriangleFromPoints(p5,  p9,  p10, min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p10, p6,  p5,  min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p5,  p6,  p4,  min(subdivlevel)+1);
            addtriangleFromPoints(p4,  p7,  p5,  min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p8,  p5,  p7,  min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p2,  p8,  p7,  min(subdivlevel)+1);
            addtriangleFromPoints(p3,  p10, p9,  min(subdivlevel)+1);
          else
            removetriangle(triangle);
            addtriangleFromPoints(p12, p11, p1,  min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p8,  p7,  p2,  min(subdivlevel)+1, TRUE);
            addtriangleFromPoints(p10, p9,  p3,  min(subdivlevel)+1, TRUE);
          end_if;
          break;
        of [FALSE, FALSE, FALSE] do
          removetriangle(triangle);
          addtriangleFromPoints(p1, p4, p6, min(subdivlevel[1], subdivlevel[3]));
          addtriangleFromPoints(p2, p5, p4, min(subdivlevel[2], subdivlevel[1]));
          addtriangleFromPoints(p3, p6, p5, min(subdivlevel[2], subdivlevel[3]));
          addtriangleFromPoints(p4, p5, p6, min(subdivlevel));
          splitUVedge(ne1, p1, p4, p4, p2, subdivlevel[1]+1);
          splitUVedge(ne2, p2, p5, p5, p3, subdivlevel[2]+1);
          splitUVedge(ne3, p3, p6, p6, p1, subdivlevel[3]+1);
          break;
        otherwise
          error("should not arrive here");
        end_case;
       break;
     otherwise
       error("should not arrive here");
     end_case;
  end_proc:

  //-----------------------------------------------------
  // This routine does the adaptive work: 
  //-----------------------------------------------------
  refine:= proc() local borders; begin
    for subdivlevel from 1 to MAXSUBDIVLEVEL do
        WORKTODO:= bool(BORDEREDGES = {}); // may be reset by subdivide
        borders := BORDEREDGES;
        BORDEREDGES := {};
        map(borders, treatborder); // expand outwards
        EDGES:= map(EDGES, subdivide); // mark the edges to be subdivided
        context(hold(userinfo)(3, output::ordinal(subdivlevel). " subdivision of the edges. ".
                 "Took: ".expr2text(time() - ST)." msec"));
        // subdivide set the flag WORKTODO to FALSE if no edge was subdivided
        if not WORKTODO then
           break;
        end_if;
        ST:= time():
        for triangle in TRIANGLES do
            splitTriangle(op(triangle, 1));
        end_for:
        context(hold(userinfo)(3, output::ordinal(subdivlevel). 
                               " subdivision of the triangles. ".
                               "Took: ".expr2text((time() - ST))." msec"));
        ST:= time():
        context(hold(userinfo)(3, expr2text(nops(POINTS)). " points, ".
                                  expr2text(nops(EDGES))." edges, ".
                                  expr2text(nops(TRIANGLES))." triangles"));
    end_for:
  end_proc:

  //---------------------------------------------------------
  // utility to decide whether the extrema are singularities.
  // The idea is that the second derivative near a singularity
  // must have the same sign as the function value.
  // We just need the sign of the second derivative along the
  // steepest descent direction (the gradient of fproc).
  // fproc : the function (procedure) to be investigated
  // f1    : fproc(u1, v1)
  // u1, v1: the coordinates of the point to be investigated
  //---------------------------------------------------------
  negativeCurvature:= proc(fproc, f1, u1, v1)
  local eps, u0, v0, fu, fv, tmp, u2, v2, f0, f2;
  save DIGITS;
  begin
     eps:=  0.975*10^(2- DIGITS);
     u0 := u1 - eps*uwidth;
     v0 := v1 - eps*vwidth;
     if traperror((
        fu:= (fproc(u0, v1) - f1)/uwidth; // u-derivative
        fv:= (fproc(u1, v0) - f1)/vwidth; // v-derivative
        )) = 0 then
        tmp:= max(specfunc::abs(fu), specfunc::abs(fv));
        if iszero(tmp) then
           // zero gradient. The second derivative would be
           // dominated by round-off
           return(UNKNOWN);
        end_if;
     else return(UNKNOWN);
     end_if;
     fu:= fu/tmp; // u-component of gradient(f)
     fv:= fv/tmp; // v-component of gradient(f)
     u0 := u1 - eps*uwidth*fu; // perturb
     v0 := v1 - eps*vwidth*fv; // along the
     u2 := u1 + eps*uwidth*fu; // gradient
     v2 := v1 + eps*vwidth*fv; // direction
     DIGITS:= DIGITS + 10;
     if traperror((f0:= fproc(u0,v0); f2:= fproc(u2,v2))) = 0 or // derivative along gradient
        traperror((f0:= fproc(u0,v1); f2:= fproc(u2,v1))) = 0 or // u-derivative at v1
        traperror((f0:= fproc(u1,v0); f2:= fproc(u1,v2))) = 0 or // v-derivative at u1
        traperror((f0:= fproc(u0,v0); f2:= fproc(u2,v0))) = 0 or // u-derivative at v0
        traperror((f0:= fproc(u0,v0); f2:= fproc(u0,v2))) = 0 or // v-derivative at u0
        traperror((f0:= fproc(u0,v2); f2:= fproc(u2,v2))) = 0 or // u-derivative at v2
        traperror((f0:= fproc(u2,v0); f2:= fproc(u2,v2))) = 0    // v-derivative at u2
        then
        tmp:= 2*f1 - (f0 + f2);
        if iszero(tmp) or {domtype(f0), domtype(f2)} <> {Dom::Float} then
             return(UNKNOWN)
        else return(bool(tmp > 0));
        end_if;
     end_if;
     return(UNKNOWN);
  end_proc;
  
  //----------------------------------------------------------------
  // getAutomaticViewingBox tries to clip the ranges in a smart way.
  // It checks heuristically whether very large/small data should be
  // interpreted as singularities.
  // If not, it falls back to the values xmin, ..., zmax found during
  // the numerical evaluation. These are stored as fmin[i], fmax[i] with
  // i = 1,2,3 Finally, it returns the viewing box Fmin .. Fmax in the
  // i-th direction.
  //----------------------------------------------------------------
  getAutomaticViewingBox:= proc(i) // i = 1 = x-direction
                                   // i = 2 = y-direction
                                   // i = 3 = z-direction
  local eq, fdata, tmp, eq1, eq5, eq9;
  name plot::adaptiveSurfaceEval::getAutomaticViewingBox;
  begin
    ST:= time():
    context(hold(userinfo)(3, "trying to compute a decent viewing box in the ".
                              output::ordinal(i)." direction")):
    if fmin[i] = MAXFLOAT or fmax[i] = -MAXFLOAT then
       // no real & finite plot point was found: this plot is empty
       [Fmin[i], Fmax[i]]:= [float(0) $ 2]:
       return():
    end_if;
    // The following context construct serves for activating the
    // userinfo commands inside this sub procedure of adaptiveSurfaceEval
    // by pretending that the userinfo was called in the context
    // of the calling function, i.e., plot::adaptiveSurfaceEval
    fdata:= select(map([op(POINTS)], op, [2, 2 + i]), testtype, DOM_FLOAT);
    if nops(fdata) = 0 then
       return();
    end_if;

    eq:= stats::empiricalQuantile(fdata):
    if not ubbHI[i] then
         eq9:= eq(0.9);
    else eq9:= Fmax[i]; // the user defined viewing box border
    end_if:
    if not ubbLO[i] then
         eq1:= eq(0.1);
    else eq1:= Fmin[i]; // the user defined viewing box border
    end_if:

    if not ubbHI[i] then
      //----------------------------------------------
      // determine Fmax[i]
      //----------------------------------------------
      // Heuristics to decide whether the maximum is a proper maximum or a singularity
      Fmax[i]:= fmax[i];
      assert(fmax[i] = F[i](_umax[i], _vmax[i]));
      if fmax[i] > 0 and negativeCurvature(F[i], fmax[i], _umax[i], _vmax[i]) <> TRUE then
//print("positiveSingularity?");
         // this may be a positive singularity
         if fmax[i] >= eq9 + 100 * (eq9 - eq1) then // this does look like a singularity
//print("positiveSingularity!");
           Fmax[i]:= eq9;
         end_if;
      end_if;
    end_if;
    if not ubbLO[i] then
      //----------------------------------------------
      // determine Fmin[i]
      //----------------------------------------------
      // Heuristics to decide whether the minimum is a proper minimum or a singularity
      Fmin[i]:= fmin[i];
      assert(fmin[i] = F[i](_umin[i], _vmin[i]));
      if fmin[i] < 0 and negativeCurvature(F[i], fmin[i], _umin[i], _vmin[i]) <> FALSE then
//print("negativeSingularity?");
         // this may be a negative singularity
         if fmin[i] <= eq1 - 100 * (eq9 - eq1) then // this does look like a singularity
//print("negativeSingularity!");
           Fmin[i]:= eq1;
         end_if;
      end_if;
    end_if;
    if not (ubbHI[i] and ubbLO[i]) then
      //----------------------------------------------
      // symmetrize Fmin[i], Fmax[i]
      //----------------------------------------------
      // If there is a positive and a negative singularity, symmetrize the
      // viewing box around the 50% quantile eq5 by taking the harmonic
      // mean of its distance to the upper and lower viewingbox bounds
      if Fmax[i] < fmax[i] and Fmin[i] > fmin[i] then
        eq5:= eq(0.5):
        tmp:= sqrt(specfunc::abs(eq5 - Fmin[i])*specfunc::abs(Fmax[i] - eq5));
        if not iszero(tmp)     and
           fmin[i] < eq5 - tmp and
             eq1   > eq5 - tmp and
           fmax[i] > eq5 + tmp and
             eq9   < eq5 + tmp then
             if not ubbLO[i] then
                Fmin[i]:= eq5 - tmp;
             end_if;
             if not ubbHI[i] then
                Fmax[i]:= eq5 + tmp;
             end_if;
        end_if;
      end_if;
    end_if;

    // Heuristics:
    // If 0 < Fmin << Fmax, we probably should set Fmin = 0.0
    // increasing the viewing range up to 10 per cent.
    if (not ubbLO[i]) and Fmin[i] > 0 and Fmin[i] < Fmax[i]/10 then
       Fmin[i]:= float(0);
    end_if;
    // If Fmin << Fmax < 0, we probably should set Fmax = 0.0
    // increasing the viewing range up to 10 per cent.
    if (not ubbHI[i]) and Fmax[i] < 0 and Fmax[i] > Fmin[i]/10 then
       Fmax[i]:= float(0);
    end_if;

    //-----------------------------------------------------
    // automatic viewing box Fmin[i] .. Fmax[i] is computed
    //-----------------------------------------------------

    context(hold(userinfo)(3, "automatic viewing box in the ".
                              output::ordinal(i)." direction: ".
                              expr2text(Fmin[i] .. Fmax[i]).
                              " (original: ".expr2text(fmin[i] .. fmax[i]).
                              "). Took ".expr2text(time()- ST)." msec"
                           )):
  end_proc:

  //--------------------------------------------------------------------
  //-----------------------------------------
  // Estimate the aspect ratios x:y:z needed
  // for the adaptive refinement.
  //-----------------------------------------
  for i from 1 to 3 do
     if ubbLO[i] and ubbHI[i] then
       XYZratio[i]:= Fmax[i] - Fmin[i]; // requested by the user
     elif ubbLO[i] and not ubbHI[i] then
       XYZratio[i]:= fmax[i] - Fmin[i]; 
       if XYZratio[i] < 0 then
          // can only happen if a viewing box was specified and
          // no point in this range was found:
          XYZratio[i]:= 1.0:
       end_if;
     elif (not ubbLO[i]) and ubbHI[i] then
       XYZratio[i]:= Fmax[i] - fmin[i]; 
       if XYZratio[i] < 0 then
          // can only happen if a viewing box was specified and
          // no point in this range was found:
          XYZratio[i]:= 1.0:
       end_if;
     elif (not ubbLO[i]) and (not ubbHI[i]) then
       XYZratio[i]:= fmax[i] - fmin[i]; // from the numerical evaluation
     else
       error("should not arrive here");
     end_if;
     assert(XYZratio[i] >= 0);
     if iszero(XYZratio[i]) then
        XYZratio[i]:= 1.0;
     end_if;
  end_for;
  tmp:= max(XYZratio[1], XYZratio[2], XYZratio[3]):
  XYZratio2:= [(XYZratio[1]/tmp)^2, (XYZratio[2]/tmp)^2, (XYZratio[3]/tmp)^2];

  //--------------------------------------------------------------------
  // Do the adaptive refinement
  //--------------------------------------------------------------------
  refine();
  //--------------------------------------------------------------------
  // call getAutomaticViewingBox
  //--------------------------------------------------------------------
  for i from 1 to 3 do
    if not (ubbLO[i] and ubbHI[i]) then 
       if nosingularities then
            Fmin[i]:= fmin[i];
            Fmax[i]:= fmax[i];
       else getAutomaticViewingBox(i);
       end_if;
    end_if;
    if Fmin[i] > Fmax[i] then 
       // this can only happen if at least one of the
       // viewing box borders is given and no point
       // inside this range was found.
       if ubbLO[i] then // Fmin is binding
          Fmax[i]:= Fmin[i];
       elif ubbHI[i] then // Fmax is binding
          Fmin[i]:= Fmax[i];
       else
          assert(FALSE); // should not arrive here
          // Just to be on the safe side for the release version:
          [Fmin[i], Fmax[i]] := [(Fmin[i] + Fmax[i])/2 $ 2];
       end_if;
    end_if;
  end_for;
  //---------------------------------------------------------------
  // If the aspect ratio changed drastically, we need to recompute.
  // Estimate the new aspect ratios x:y:z after the decision on the
  // viewing box by getAutomaticViewingBox.
  //---------------------------------------------------------------
  recompute:= FALSE;
  oldXYZratio2:= XYZratio2;
  for i from 1 to 3 do
     if not (ubbLO[i] and ubbHI[i]) and
        ( XYZratio[i] < (Fmax[i] - Fmin[i])/10 or
          XYZratio[i] > (Fmax[i] - Fmin[i])*10) then
        XYZratio[i]:= Fmax[i] - Fmin[i]; 
        if iszero(XYZratio[i]) then
           XYZratio[i]:= 1.0;
        end_if;
        recompute:= TRUE:
     end_if;
     assert(XYZratio[i] >= 0);
  end_for:

  //-----------------------------------------
  // do recompute
  //-----------------------------------------
  if recompute then 
     // clear the subdivlevel entry in EDGES
     EDGES:= map(EDGES, subsop, 3 = 0);
     // We need to set the new XYZratio2 need by subdivide (called in refine())
     tmp:= max(XYZratio[1], XYZratio[2], XYZratio[3]):
     XYZratio2:= [(XYZratio[1]/tmp)^2, (XYZratio[2]/tmp)^2, (XYZratio[3]/tmp)^2];

     //--------------------------------------------------------------------
     // Do a second adaptive refinement
     //--------------------------------------------------------------------
     userinfo(3, "the aspect ratios have changed from ".
                  expr2text([oldXYZratio2[i]^(1/2) $ i = 1..3]).
                 " to ".
                  expr2text([XYZratio2[i]^(1/2) $ i = 1..3]));
     userinfo(3, "starting a second adaptive refinement");
     ST:= time();
     refine();
/*
     //--------------------------------------------------------------------
     // 2nd call of getAutomaticViewingBox Problem: the refinement 
     // may have given too much emphasis on the singularities implying
     // that the heuristics in getAutomaticViewingBox breaks down.
     //--------------------------------------------------------------------
     for i from 1 to 3 do
       if not (ubbLO[i] and ubbHI[i]) then 
          if nosingularities then
               Fmin[i]:= fmin[i];
               Fmax[i]:= fmax[i];
          else getAutomaticViewingBox(i);
          end_if;
       end_if;
     end_for;
*/
  end_if;

  //--------------------------------------------------------------------
  //--------------------------------------------------------------------
  //--------------------------------------------------------------------
  // The evaluation of the surface is done
  //--------------------------------------------------------------------
  //--------------------------------------------------------------------
  //--------------------------------------------------------------------

  //----------------------------------------------------------
  // convert TRIANGLE = table() to TRIANGLE = list
  //----------------------------------------------------------
  TRIANGLES:= map([op(TRIANGLES)], op, 2);

  //----------------------------------------------------------
  // convert triangle = [edge1, edge2, edge3] do
  //         triangle = [point1, point2, point3]
  //----------------------------------------------------------
  TRIANGLES:= map(TRIANGLES, proc(edges)
                             local e1, e2, e3;
                             name edges2points;
                             begin
                               [e1, e2, e3]:= edges;
                               if e1 > 0 then
                                    e1:= EDGES[e1][1][1];
                               else e1:= EDGES[-e1][1][-1];
                               end_if;
                               if e2 > 0 then
                                    e2:= EDGES[e2][1][1];
                               else e2:= EDGES[-e2][1][-1];
                               end_if;
                               if e3 > 0 then
                                    e3:= EDGES[e3][1][1];
                               else e3:= EDGES[-e3][1][-1];
                               end_if;
                               assert(e1 <> e2 and e2 <> e3 and e3 <> e1);
                               [e1, e2, e3];
                             end_proc);
  //----------------------------------------------------------
  // Add normals to the point data:
  //     pt = [u,v,x,y,z]   -->   pt = [u,v,x,y,z,nx,ny,nz]
  //----------------------------------------------------------
  // Version 1: compute the normal at a point as the mean
  // of the normals of all triangles sharing this point.
  //----------------------------------------------------------
  add_normals1:= proc()
  local normals, i, j, pt1, pt2, pt3, t21, t31, pt;
  begin
    normals:= [0 $ nops(TRIANGLES)]:
    POINTS := map(POINTS, append, {});
    for i from 1 to nops(TRIANGLES) do
      // add the triangle index as extra info
      // to the points: pt[6] becomes a set of
      // all triangles that share the point pt.
      pt1:= POINTS[TRIANGLES[i][1]]:
      pt2:= POINTS[TRIANGLES[i][2]]:
      pt3:= POINTS[TRIANGLES[i][3]]:
      t21:= [(pt2[j] - pt1[j]) $ j = 3..5]:
      t31:= [(pt3[j] - pt1[j]) $ j = 3..5]:
      // compute the normal of the i-th triangle and
      // store it as the i-th entry of the normals list.
      normals[i]:=[t21[2]*t31[3] - t21[3]*t31[2],
                   t21[3]*t31[1] - t21[1]*t31[3],
                   t21[1]*t31[2] - t21[2]*t31[1]];
      POINTS[TRIANGLES[i][1]][6]:= pt1[6] union {i};
      POINTS[TRIANGLES[i][2]][6]:= pt2[6] union {i};
      POINTS[TRIANGLES[i][3]][6]:= pt3[6] union {i};
    end_for:
    for i in map([op(POINTS)], op, 1) do
      // pt[6] is a set of indices of all triangles that
      // share the point pt. Average the normals of these
      // triangles and add it to the point such that it
      // becomes pt = [u, v, x, z, z, nx, ny, nz].
      // This works properly only if the triangles are
      // oriented in the same fashion!
      pt:= POINTS[i]:
      if pt[6] = {} then
	pt[6] := 0.0,0.0,0.0;
      else
         pt[6]:= _plus(normals[j][i] $ j in pt[6])/nops(pt[6]) $ i = 1..3;
      end_if;
      POINTS[i]:= pt;
    end_for:
  end_proc:
  //-----------------------------------------------------------
  // Version 2: compute some normals via numeric differentiation
  //-----------------------------------------------------------
  add_normals2:= proc()
  local du, dv, umid, vmid, computenormal;
  begin
    du:= 2.516e-6*uwidth: // make sure these perturbations do not
    dv:= 2.516e-6*vwidth: // coincide with the perturbations when
                          // evaluating the function near s singularity
                          // (otherwise we may end up directly on the
                          // singularity)
    if not (iszero(du) or iszero(dv)) then
      umid:= (umin + umax)/2:
      vmid:= (vmin + vmax)/2:
      computenormal:= proc(pt)
      local ndu, ndv, u, v, uu, vv, X, Y, Z, tu, tv;
      name plot::Surface::computenormal;
      begin
        [u, v, X, Y, Z]:= pt[1..5];
        //-----------------------------------------------------
        // Use numeric differentiation only along the boundary
        // of the u-v rectangle. This avoids visible contours
        // in case of closed surfaces.
        // We assume that the points were pre-processed by
        // add_normals1();
        // Remove the following lines to use numeric differentiation
        // for **all** points.
        if u > umin + du and u < umax - du  and
           v > vmin + dv and v < vmax - dv  then
          return(pt);
        end_if:
        //-----------------------------------------------------
        ndu:= du: if u > umid then ndu:= -du; end_if;
        ndv:= dv; if v > vmid then ndv:= -dv; end_if;
        uu:= u + ndu; vv:= v + 0.1*ndv;
        // getPoint(uu, vv) assigns x = [X,Y,Z] as a side effect
        if   getPoint(uu, vv)                             or                          
             getPoint(uu + uwidth/10^9, vv              ) or
             getPoint(uu - uwidth/10^9, vv              ) or
             getPoint(uu,               vv + vwidth/10^9) or
             getPoint(uu,               vv - vwidth/10^9) or
             getPoint(uu + uwidth/10^9, vv + vwidth/10^9) or
             getPoint(uu + uwidth/10^9, vv - vwidth/10^9) or
             getPoint(uu - uwidth/10^9, vv + vwidth/10^9) or
             getPoint(uu - uwidth/10^9, vv - vwidth/10^9) then 
             tu:= [(x[1] - X)/ndu, (x[2] - Y)/ndu, (x[3] - Z)/ndu];
        else // ToDo
          tu := [0.0$3];
//             error("cannot evaluate to a finite real value near the point ".expr2text(uu, vv));
        end_if;
        uu:= u + 0.1*ndu; vv:= v + ndv;
        // getPoint(uu, vv) assigns x = [X,Y,Z] as a side effect
        if   getPoint(uu, vv)                             or  
             getPoint(uu + uwidth/10^9, vv              ) or
             getPoint(uu - uwidth/10^9, vv              ) or
             getPoint(uu,               vv + vwidth/10^9) or
             getPoint(uu,               vv - vwidth/10^9) or
             getPoint(uu + uwidth/10^9, vv + vwidth/10^9) or
             getPoint(uu + uwidth/10^9, vv - vwidth/10^9) or
             getPoint(uu - uwidth/10^9, vv + vwidth/10^9) or
             getPoint(uu - uwidth/10^9, vv - vwidth/10^9) then
             tv:= [(x[1] - X)/ndv, (x[2] - Y)/ndv, (x[3] - Z)/ndv];
        else // ToDo
          tv := [0.0$3];
//             error("cannot evaluate to a finite real value near the point ".expr2text(uu, vv));
        end_if;
        [op(pt, 1..5),-tu[3]*tv[2] + tu[2]*tv[3],
                      -tu[1]*tv[3] + tu[3]*tv[1],
                      -tu[2]*tv[1] + tu[1]*tv[2]];
      end_proc:
      POINTS:= map(POINTS, computenormal);
      assert(nops(op(POINTS, [1,2])) = 8):
    end_if:
  end_proc:

  //------------------------------------------------------------
  // Add the normals to the point data
  //------------------------------------------------------------
  add_normals1(); // average triangle normals
  add_normals2(): // numerical differentiation along the u-v boundaries

  //---------------------------------------------
  // The adaptive refinement is finished. Return
  // the data.
  //---------------------------------------------
  
  // ULINES, VLINES: from edges to points
  assert(_and(op(map(ULINES, s -> _and(op(map(s, i->has(EDGES, specfunc::abs(i)))))))));
  assert(_and(op(map(VLINES, s -> _and(op(map(s, i->has(EDGES, specfunc::abs(i)))))))));
  assert(_and(op(map([op(map(EDGES,
                             e -> contains(POINTS,e[1][1]) and
                             contains(POINTS,e[1][-1])))], op, 2))));
  ULINES := map(ULINES, map, i -> [POINTS[EDGES[specfunc::abs(i)][1][1]],
                                   POINTS[EDGES[specfunc::abs(i)][1][2]]]);
  VLINES := map(VLINES, map, i -> [POINTS[EDGES[specfunc::abs(i)][1][1]],
                                   POINTS[EDGES[specfunc::abs(i)][1][2]]]);
  assert(testtype(ULINES, Type::ListOf(Type::SetOf(Type::ListOf(Type::ListOf(DOM_FLOAT))))));
  assert(testtype(VLINES, Type::ListOf(Type::SetOf(Type::ListOf(Type::ListOf(DOM_FLOAT))))));
  // split disconnected lines
  regroup :=
  proc(lines, index)
    local regroup_line;
  begin
    lines := map(lines, s -> map([op(s)], prog::sort, p->p[index])); // sort edges by v
    lines := map(lines, prog::sort, e -> e[1][index]); // sort by v of first point

    regroup_line :=
    proc(line)
      local split_at, i, j;
    begin
      if line = [] then return(null()); end_if;
      split_at := [1].select([$2..nops(line)], i -> line[i-1][2] <> line[i][1]);
      {line[j][1]$j=split_at[i]..split_at[i+1]-1, line[split_at[i+1]-1][2]}
      $ i = 1..nops(split_at)-1,
      {line[j][1]$j=split_at[-1]..nops(line), line[-1][2]}
    end_proc;
    map(lines, regroup_line);
  end;

  ULINES := regroup(ULINES, 2);
  VLINES := regroup(VLINES, 1);
  
   userinfo(1, "viewing box = ".expr2text([Fmin[i] .. Fmax[i] $ i = 1..3]));
  [
   [Fmin[i] .. Fmax[i] $ i = 1..3],  // viewing box
   [(POINTS[TRIANGLES[i][j]] $ j=1..3) $ i=1..nops(TRIANGLES)],
   ULINES, VLINES
  ];
end_proc:
//----------------------------------------------------------------
