/*---------------------------------------------------------------
functionEval(f, x = xmin..xmax, ymin .. ymax/Automatic ...) 
evaluates a procedure f on a mesh x.i in the interval [xmin, xmax]. 
Errors are trapped and  cause the result to split into several 
branches. Also non-real values of f cause a splitting into several 
branches. The mesh is refined whereever is seems appropriate. 
plot::Function2d calls this utility to compute a function graph.

Calls:  plot::functionEval(f, 
                           x = xmin..xmax, 
                           ymin..ymax,
                           <meshpoints>,
                           <DiscontinuitySearch = TRUE/FALSE>,
                           <AdaptiveMesh = TRUE/FALSE>,
                           <"NoSingularities">)

Parameters:
       x          : identifier (provided by the user, may have properties)
       xmin, xmax,
       ymin, ymax : numerical real values (the viewing box requested
                    by the user) or Automatic (no value set by the user)
       meshpoints : the size of the initial mesh before refinement:
                    an integer >= 2

Return: A list [ViewingBox, [branch1, branch2, ...], SetOfSingularities]
        with ViewingBox = [Xmin .. Xmax, Ymin .. Ymax],
        each branch is a list of points: branch.i= [[x1, y1], [x2, y2], ..],
        SetOfSingularities = {x1, x2, ...}

Example:
>> plot::functionEval(x -> 1/sin(1/x), x = -PI .. PI, Automatic, 1000);
---------------------------------------------------------------*/
plot::functionEval:= proc(f, xrange, yrange = Automatic, meshpoints = 101
      /*,DiscontinuitySearch = TRUE/FALSE, 
         AdaptiveMesh = TRUE/FALSE,
         "NoSingularities"
      */) 
local 
      _xmax, _xmin, adaptive, adaptivemesh, bisect, branches, branchindex, branch,
      closebranch, currentmeshpoints, cut, dx, eps, findReal, 
      functionEval_rec, getAutomaticViewingBox, getf, getXYData, getYRange, 
      i, lastx, lasty, lastymax, lastymin, lastyOK, lock, lowslopes,
      maxrecursionlevel, maxtan, poleOrder, rawbranches, refine,
      s, singularities, _singularities, slopeTable, symbolicBranch, symbolicBranches,
      trackYRange, ubbLO, ubbHI, usediscont, width,
      x, xmax, xmin, xdata, Xmax, Xmin,
      y, ymax, ymin, ydata, Ymin, Ymax, YXratio, oldYXratio, 
      tmp, MINFLOAT, MAXFLOAT, ST, st,
      nosingularities, x1, y1, x2, y2, y3
      ;
begin
   MINFLOAT:= 1e-300: // smallest float that can be converted to a C double
   MAXFLOAT:= 1e300:  // largest float that can be converted to a C double
   ST:= time(); // Start time for timinings in userinfos
   //-------------------------------------------------------------------------
   // Set the tuning parameters for the adaptive mode:
   // The following 'max bend angle' of 10 degrees is mentioned explicitly 
   // in the documentation of PLOT/ATTRIBUTES/AdaptiveMesh.tex. Do not change
   // this value without keeping the documentation up to date!
   // Further, keep this synchronized with curveEval!
   //-------------------------------------------------------------------------
   adaptivemesh:= op(select([args()], has, AdaptiveMesh), [1, 2]);
   if adaptivemesh = FAIL then
      adaptivemesh:= 0;
   end_if;
   nosingularities:= has([args()], "NoSingularities"):
   maxrecursionlevel:= adaptivemesh:
     // refine the initial mesh by up to 2^maxrecursionlevel
     // points between each pair of points on the initial mesh
   maxtan:= tan::float(10/180*PI);
   // stop the refinement when consecutive
   // line segments are (almost) parallel
   // within 10 degrees
   // set the options
   adaptive:= bool(adaptivemesh > 0);
   if has([args()], DiscontinuitySearch = FALSE) then
        usediscont:= FALSE;
   else usediscont:= TRUE;
   end_if:
   xrange:= op(xrange, 2);
   [xmin, xmax, width]:= float([op(xrange,1), op(xrange, 2), 
                                op(xrange,2) -op(xrange,1)]);
   Xmin:= xmin;         // initialization for later refinement
   Xmax:= xmax;         // initialization for later refinement
   if op(args(3), 1) = Automatic and op(args(3), 2) = Automatic then
        // neither ViewingboxYMin nor ViewingboxYMax were given by the user
        ubbLO:= FALSE;       // userboundingbox ?
        ubbHI:= FALSE;       // userboundingbox ?
        Ymin:=-MAXFLOAT;     // initialization for later refinement
        Ymax:= MAXFLOAT;     // initialization for later refinement
   elif op(args(3), 1) <> Automatic and op(args(3), 2) = Automatic then
        // ViewingboxYMin was given by the user, ViewingBoxYMax wasn't
        ubbLO:= TRUE;        // userboundingbox ?
        ubbHI:= FALSE;       // userboundingbox ?
        Ymin:= op(yrange, 1);// The bounding box requested by the user
        Ymax:=-MAXFLOAT;     // initialization for later refinement
   elif op(args(3), 1) = Automatic and op(args(3), 2) <> Automatic then
        // ViewingboxYMax was given by the user, ViewingBoxYMin wasn't
        ubbLO:= FALSE;       // userboundingbox ?
        ubbHI:= TRUE;       // userboundingbox ?
        Ymin:=-MAXFLOAT;     // initialization for later refinement
        Ymax:= op(yrange, 2);// The bounding box requested by the user
   else //  both ViewingboxYMin and ViewingBoxYMax were given by the user
        ubbLO:= TRUE;       // userboundingbox ?
        ubbHI:= TRUE;       // userboundingbox ?
        Ymin:= op(yrange, 1);// The bounding box requested by the user
        Ymax:= op(yrange, 2);// The bounding box requested by the user
   end_if;
   f:= float@f:
   if meshpoints <= 1 then 
      warning("expecting a number of mesh points >= 2, got: ".
              expr2text(meshpoints).
              " Setting the number of mesh points to 2.");
      meshpoints:= 2:
   end_if:
   dx:= width/(meshpoints - 1):
   //---------------------------------------------------------
   // A heuristic numerical utility to check whether potential
   // singularities found by discont are indeed singularities.
   // Assuming that the singularity is a pole, the order 
   //   s = (g'(x))^2/((g'(x))^2 - g(x)*g''(x)), g(x)=1/f(x)
   // of the pole is estimated. Return values:
   //  0 (no singularity detected) or an approximation of s.
   //---------------------------------------------------------
   poleOrder:= proc(x)
   local eps, x1, x2, x3, y1, y2, y3, f0, f1, f2, s;
   save DIGITS;
   begin
       if not usediscont then 
          return(0); // don't say anything about the pole order
       end_if;
       eps:= width*10.0^(-10-DIGITS);
       x:= x + width/10^10;  // do disturb the approximation
                             // of the potential singularity
       DIGITS:= 2*(10 + DIGITS);
       [x1, x2, x3]:= [x - eps/2, x + eps/2, x + 3/2*eps]:
       if traperror((
           y1:= 1/f(x1);
           y2:= 1/f(x2);
           y3:= 1/f(x3);
         )) = 0 then
            f0:= y2: 
            f1:= (y3 - y1)/2/eps;
            f2:= ((y1 - y2) + (y3 - y2))/eps^2:
       else return(0);
       end_if;
       // The following s should be the order of the pole
       if traperror((
           s:= float(f1^2/(f0*f2 - f1^2));
           )) <> 0 then
            return(0);  // cannot say anything about the pole order
       elif not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(s)) then
            return(0);  // cannot say anything about the pole order
       else return(Re(s));
       end_if;
   end_proc;
   //-----------------------------------------------------
   // utility getf(x): try to evaluate f(x). 
   // If the result is a real number, assign it to the 
   // variable y (of functionEval_rec) (SIDE EFFECT!!)
   // and returns TRUE.
   // If an error occurs, x is appended to the
   // set of (potential) singularities and FALSE is retuned.
   // If a non-real value f(x) is found, it is ignored and
   // FALSE is returned.
   // The call getf(x) tracks ymin, ymax as a SIDE EFFECT.
   // The call getf(x, any_second_argument) does not track 
   // ymin, ymax.
   //-----------------------------------------------------
   getf:= proc(x) // assigns y:= f(x) as a side effect
   begin
     if traperror((y:= f(x))) <> 0 then
//        singularities:= singularities union {x};
          return(FALSE)
     elif y <> RD_NAN
          and _lazy_or(domtype(y) = DOM_FLOAT,
                      domtype((y:= numeric::complexRound(y))) = DOM_FLOAT)
          and y >=-MAXFLOAT 
          and y <= MAXFLOAT 
        //and (iszero(y) or
        //     y >= MINFLOAT or
        //     y <=-MINFLOAT)
          then
               if specfunc::abs(y) < MINFLOAT then
                  y:= float(0):
               end_if;
               if args(0) = 1 then
                  trackYRange(x, y);
               end_if;
               return(TRUE)
     else return(FALSE);
     end_if;
   end_proc:
   //-----------------------------------------------------
   // utility trackYRange(y): check the global extrmal 
   // values ymin, ymax and redefine them if necessary.
   //-----------------------------------------------
   trackYRange:= proc(x, y) begin
     if y < ymin then ymin:= y; _xmin:= x; end_if;
     if y > ymax then ymax:= y; _xmax:= x; end_if;
   end_proc:
   //--------------------------------------------------------
   // Initialize the variables for the actual bounding box
   //--------------------------------------------------------
   ymin:=  MAXFLOAT;  // the actual bounding box found during evaluation
   ymax:= -MAXFLOAT;  // the actual bounding box found during evaluation
   //--------------------------------------------------------
   // exception: the x range has length 0
   //--------------------------------------------------------
   singularities:= {}:
   if iszero(width) then
      if getf(xmin, 1) then
           return([[xmin .. xmin, y .. y], [[[xmin, y]]], {}]);
      else return([NIL, [[]], singularities]);
      end_if: 
   end_if:
   //--------------------------------------------------------
   // exception: xmin > xmax 
   //--------------------------------------------------------
   if xmin > xmax then
      return([NIL, [[]], {}]);
   end_if:
   //----------------------------------------------------------------
   // pre-processing: search for singularities via numeric::discont:
   //----------------------------------------------------------------
   symbolicBranches:= [[xmin, xmax]];
   if usediscont then
     userinfo(1, "Starting search for discontinuities.");
     st:= time():
     eps:= 1e-4*width;
     if traperror((
        singularities:= float(numeric::discont(
                     f(`#x`), `#x` = xmin .. xmax, eps/100, eps/5)):
        )) = 0 then
       if type(singularities) = DOM_SET then
         singularities:= select(singularities, x -> (domtype(x) = DOM_FLOAT));
         _singularities:= sort([op(singularities)]);
         s:= nops(_singularities);
         _xmin:= xmin:
         _xmax:= xmax:
         if s > 0 then
           if specfunc::abs(_singularities[1] - xmin) <= eps then 
             _xmin:= xmin + eps; 
           end_if;
           if specfunc::abs(_singularities[s] - xmax) <= eps then 
             _xmax:= xmax - eps; 
           end_if;
         end_if;
         _singularities:= select(_singularities, s -> ((_xmin + 2*eps < s) and 
                                                       (s < _xmax - 2*eps)));
         s:= nops(_singularities);
         if s > 0 then symbolicBranches:= [
                [_xmin, max(_xmin, _singularities[1] - eps)], 
                [_singularities[i] + eps, _singularities[i+1] - eps] $ i=1..s-1,
                [min(_singularities[s] + eps, _xmax), _xmax] ];
                // Beware: we do need the left end of the branch
                // to be smaller than the right end:
                symbolicBranches:= select(symbolicBranches,
                                          branch ->  (branch[1] < branch[2]));
         end_if;
       else // we do not want to return a singularity set
            // consisting of symbolical stuff. Pretend
            // that we did not find any singularities
            singularities:= {}:
       end_if;
     end_if;
     userinfo(2, "Number of branches: ".expr2text(nops(symbolicBranches))):
     userinfo(3, "Time for discontinuity search: ".
                 expr2text(time() - st)." msec"):
   end_if;
   //-----------------------------------------------
   // findReal(a, b, fb) - find a value x close to 'a'
   // that produces a real & finite value f(x).
   // This utility is used to find points x with
   // real & finite function value f(x) close to a
   // problematic point 'a' (either a singularity
   // or f(a) = non-real.
   // On input: f(a) = non-real or +/-infinity, 
   //           f(b) = finite & real,
   // findReal(a, b, fb) is *guaranteed* to return an
   // x with finite real f(x).  However, x may coincide 
   // with b.
   //-----------------------------------
   findReal:= proc(a, b, fb)
   local lastx, lasty, x, maxcount, count;
   begin
      maxcount:= 20: // 2^maxcount intermediate points
      lastymin:= ymin:
      lastymax:= ymax:
      if has(singularities, a) and
         getf(a + (b - a)/10^DIGITS) then
            b:= a + (b - a)/10^DIGITS;
            fb:= y;
            if fb < lastymin or fb > lastymax then
               branch[b]:= fb;
            end_if;
            return([b, fb]);
      end_if;
      lastx:= b:  // Remember the last OK-pair x, y
      lasty:= fb: // with y = f(x) = real & finite
      x:= (a + b)/2;  // bisect
      if iszero(x - lastx) then
         return([lastx, lasty]); //cancellation
      end_if;
      for count from 1 to maxcount do
        lastymin:= ymin:
        lastymax:= ymax:
        if getf(x) then  // getf(x) assigns y as a side effect
             lastx:= x;  // This pair (x, y) is OK.
             lasty:= y;  // Remember it.
             if y < lastymin or y > lastymax then
                branch[x]:= y;
             end_if;
             b:= x: // Shorten the search interval (a, b] to (a, x]
        else a:= x; // Shorten the search interval (a, b] to (x, b]
        end_if;
        x:= (a + b)/2; // bisect
        if iszero(x - lastx) then
           break;
        end_if;
      end_for:
      return([lastx, lasty]); // return the last real pair
   end_proc:
   //-----------------------------------------------------
   // The utility closebranch(branch) closes the current branch
   // by inserting it into the table 'branches' and increments 
   // the 'global' counter 'branchindex' for the next branch.
   //-----------------------------------------------------
   closebranch:= proc(branch) begin 
     if nops(branch) <> 0 then
       branches[branchindex]:= branch: // write branch to branches
       branchindex:= branchindex + 1; // increase branch index for next branch
    end_if;
   end_proc:

   //--------------------------------------------------------
   // initialize the branches table and open the first branch
   //--------------------------------------------------------
   st:= time():
   branches:= table();// table to contain all branches
   branchindex:= 1:   // branch index of the current branch
   //--------------------------------------------------------
   // Start the work: non-adaptive evaluation of the function
   // on the (default or user specified) coarse mesh.
   //--------------------------------------------------------
   for symbolicBranch in symbolicBranches do
     userinfo(5, "investigating branch ".expr2text(symbolicBranch));
     branch:= table():  // open a new branch
     [xmin, xmax]:= symbolicBranch;
     if xmax - xmin <= width/10^4 then
        next; // ignore very small branches. This is important for
              // avoiding the branches [x1 - eps, x1 + eps] found
              // in numeric::piecewise([x < x1, ..], [x >= x1, ..])
     end_if;

     // reset the meshpoints to make sure that the meshpoints are
     // distributed uniformly and symmetrically in the current branch
     currentmeshpoints:= max(2, 1 + ceil((xmax - xmin)/width * (meshpoints - 1))):
     dx:= (xmax - xmin)/(currentmeshpoints-1);
     // reset singularities for those singularities that 
     // will be detected during the numerical evaluation.
     // This set will be filled via getf.
//   singularities:= {};
     //-----------------------------------------------------------------
     // Search for a starting point to be inserted into the first branch
     //-----------------------------------------------------------------
     x:= xmin;
     lastx:= x:
     while TRUE do
       if getf(x, 1) then // do not track ymin, ymax
          break;
       end_if;
       lastx:= x;
       x:= x + dx;
       if x > xmax then break; end_if;
     end_while;
     if x > xmax then
        next; // no plot point in the current branch
     end_if;
     if x = xmax and not getf(x) then
        next; // no plot point in the current branch
              // There is no need to close this branch,
              // just use it for the next symbolicBranch
     end_if;
     // A starting point (x, y) was found, but not
     // written into the branch table, yet. First,
     // improve this starting point:
     if x <> xmin then
        userinfo(10, "start searching for a left border"):
        [x, y]:= findReal(lastx, x, y);
     end_if;
     userinfo(10, "left border found. starting with ".expr2text(x));
     trackYRange(x, y); // just in case findReal returned its
                        // input values x, y (y was found via
                        // getf(x, 1), i.e., without tracking)
     branch[x]:= y;
     lastx:= x;
     lasty:= y;
     lastyOK:= TRUE;
     x:= x + dx;
     if x > xmax - dx/100 then
        x:= xmax;
     end_if;
     //------------------------------------------------
     // A starting point was found. Start the iteration
     //------------------------------------------------
     while x <= xmax do
        lastymin:= ymin;
        lastymax:= ymax;
        if getf(x) then
            if y < lastymin or y > lastymax then
               branch[x]:= y;
            end_if;
            // getf(x) produced a real value y = f(x)
            if not lastyOK then
               // improve the beginning of the new branch
               [x, y]:= findReal(lastx, x, y);
            end_if;
            branch[x]:= y;
            lastyOK:= TRUE;
        else
           if lastyOK then
              [x, y]:= findReal(x, lastx, lasty);
              branch[x]:= y;
              closebranch(branch);
              branch:= table(): // open a new branch
           end_if;
           lastyOK:= FALSE;
        end_if:
        lastx:=x;
        lasty:=y;
        if x = xmax then 
           break; 
        end_if;
        x:= x + dx;
        if x > xmax - dx/100 then
           x:= xmax;
        end_if;
     end_while;
     closebranch(branch);
     userinfo(10, "this branch is done"):
  end_for: // for symbolicBranch in symbolicBranches
  userinfo(1, "Evaluation on the initial mesh finished."):
  userinfo(2, "Number of branches: ".expr2text(nops(branches))):
  userinfo(2, "Total number of mesh points: ".
              expr2text(_plus(nops(op(branch, 2)) $ branch in branches))):
  userinfo(3, "Time to compute the initial mesh: ".
               expr2text(time() - st)." msec");
  //------------------------------------------------------------
  // non-adaptive numerical evaluation on the initial mesh
  // finished. The data are stored in the table 'branches',
  // and a copy 'rawbranches'.
  //------------------------------------------------------------

  if _plus(nops(branches[i]) $ i = 1..nops(branches)) = 0 then
      return([NIL, [[]], {}]);
  end_if;

  if not (ubbLO and ubbHI) then
     rawbranches:= branches:
  end_if;

  //------------------------------------------------------------
  // Determine the scaling ratio between the x and the y values.
  // This aspect ratio is required by the adaptive refinement
  // of the raw mesh further down below. 
  //------------------------------------------------------------
  //----------------------------------------------------------------
  // utility
  //----------------------------------------------------------------
  getYRange:= proc()
  local branch, _ymin, _ymax;
  begin 
      _ymin:= RD_INF;
      _ymax:= RD_NINF;
      for branch in branches do
          branch:= map([op(op(branch, 2))], op, 2);
          _ymin:= min(_ymin, op(branch));
          _ymax:= max(_ymax, op(branch));
      end_for:
      return([_ymin, _ymax]):
  end_proc:

  //----------------------------------------------------------------
  // utility: let 'branch' be a table containing the points [x, y] via
  // branch[x] = y. We need to extract the x values and sort them in
  // ascending order, sorting the y values correspondingly. We sort the
  // indices 'x' and use the table to obtain a list of correspondingly
  // ordered y values.
  // This way, we can use the kernel sort without library call backs.
  // Instead of passing/returning the data table/data lists, we use
  // the local variables branch, xdata, ydata of functionEval for speed.
  //----------------------------------------------------------------
  getXYData:= proc(/*branch*/)
  local x;
  begin
    xdata:= sort(map([op(branch)], op, 1)):
    ydata:= [branch[x] $ x in xdata];
  end_proc:
  //----------------------------------------------------------------
  // getAutomaticViewingBox tries to clip the y range in a smart way.
  // It checks heuristically whether very large/small data should be 
  // interpreted as singularities in the function.
  // If not, it falls back to the values ymin, ymax found during 
  // the numerical evaluation.
  // Finally, it sets the viewing box Xmin, Xmax, Ymin, Ymax.
  //----------------------------------------------------------------
  getAutomaticViewingBox:= proc()
  local n, maxslope, slopes, try, newYXratio,
        minydata, maxydata, tmp, mindone, maxdone,
        allslopes, alldata, eq, eq5;
  name plot::functionEval::getAutomaticViewingBox;
  begin
    if ymin = MAXFLOAT or ymax =-MAXFLOAT then
       // no real & finite plot point was found: this plot is empty
       return([NIL, NIL]):
    end_if;
    //-----------------------------------------------------
    // The following context construc serves for activating the
    // userinfo commands inside this sub procedure of functionEval
    // by pretending that the userinfo was called in the context
    // of the calling function, i.e., plot::functionEval
    //-----------------------------------------------------
    context(hold(userinfo)(10, "Testing minimum (x, y) = ".expr2text([_xmin, ymin]).
                 ". Pole order = ".expr2text(poleOrder(_xmin))));
    mindone:= ubbLO:
    if (not mindone) and poleOrder(_xmin) > 0 then // no negative pole
       Ymin:= ymin;
       mindone:= TRUE;
    end_if;
    //-----------------------------------------------------
    context(hold(userinfo)(10, "Testing maximum (x, y) = ".expr2text([_xmax, ymax]).
                 ". Pole order = ".expr2text(poleOrder(_xmax))));

    maxdone:= ubbHI:
    if (not maxdone) and poleOrder(_xmax) > 0 then // no positive pole
       Ymax:= ymax:
       maxdone:= TRUE;
    end_if;
    //-----------------------------------------------------
    if mindone and maxdone then
       return([Ymin, Ymax]);
    end_if;
    //-----------------------------------------------------
    // Now,  mindone = FALSE or maxdone = FALSE
    //-----------------------------------------------------

    allslopes:= []:
    slopeTable:= table():
    for branch in branches do
       branch:= op(branch, 2):
       n:= nops(branch);
       getXYData(); // uses branch and sets xdata, ydata
       if n = 0 then next end_if;
       // the right slopes:
       if n = 1 then
          // A single point forming a branch. Make sure that
          // it will be displayed by assigning a 0 slope to it.
          slopes:= [float(0)]
       else
          slopes:= [(ydata[i+1] - ydata[i])/(xdata[i+1] - xdata[i]) $ 
                    i = 1..n - 1];
          // Beware: the heuristics for eliminating singular points
          // will fail miserably, wenn singular points have small
          // slopes. This may happen, when points close to an even 
          // pole are spaced symmetrically, leading to a 0 slope. 
          // The cure: average between neighbouring slopes.
          // However, if only one point very close to an even 
          // pole is found, the averaged slope may become small, 
          // because the left slope is >> 0 and the right slope is <<0.
          // The cure: use averaging of *absolute* slope values:
          slopes:= map(slopes, specfunc::abs);
          // the averaged slopes:
          slopes:= [slopes[1], (slopes[i-1]+slopes[i])/2 $ i=2..n-1, slopes[n-1]];
       end_if;
       for i from 1 to n do
           if type(slopeTable[slopes[i]]) = "_index" then
                slopeTable[slopes[i]]:= ydata[i];
           else slopeTable[slopes[i]]:= slopeTable[slopes[i]], ydata[i];
           end_if;
       end_for:
       if allslopes = [] then
            allslopes:= slopes;
       else allslopes:= allslopes . slopes;
       end_if;
    end_for;
    if nops(allslopes) = 0 then
       return([NIL, NIL]);
    end_if;
    assert(getYRange() = [ymin, ymax]);
    if not mindone then
       Ymin:= ymin:
    end_if;
    if not maxdone then
       Ymax:= ymax:
    end_if;
    YXratio:= (Ymax - Ymin)/width;
    //-------------------------------------------------------
    // The crucial loop of the heuristics starts: for try ...
    //-------------------------------------------------------
    lowslopes:= allslopes;
    for try from 1 to 6 do
      cut:= min(10.0^(try/3), 100)*YXratio;
      // select from the old lowslopes instead of all data,
      // because we are decreasing YXratio in each 'try' step
      lowslopes:= select(lowslopes, _less, cut);
      if nops(lowslopes) = 0 then 
         // try again with larger YXratio (less points will
         // be removed in the next version of 'lowslopes'):
         YXratio:= 10^3*YXratio;
         // reset lowslopes to *all* the original data,
         // since we are increasing the YXratio
         lowslopes:= allslopes;
         next;
      end_if;
      // use 'sl' in the **set** of lowslopes to avoid
      // redundant multiple entries in the ydata sequence:
      ydata:= op(map({op(lowslopes)}, sl -> slopeTable[sl]));
      [minydata, maxydata]:= [min(ydata), max(ydata)];
      if try < 2 then
         maxslope:= max(op(lowslopes));
         if maxslope < (maxydata - minydata)/width*10^2 or
            nops(lowslopes) < 10 then
            YXratio:= min((ymax - ymin)/width, 5*YXratio);
            // reset lowslopes to *all* the original data, since
            // we are increasing the YXratio
            lowslopes:= allslopes;
            context(hold(userinfo)(10, "Step ".expr2text(try).
                     ". Experimentally increasing the aspect ratio Y:X to ".
                     expr2text(YXratio))):
            next;
         end_if;
      end_if; // if try < 2
      if not mindone then
         Ymin:= minydata;
      end_if;
      if not maxdone then
         Ymax:= maxydata;
      end_if;
      newYXratio:= (Ymax - Ymin)/width;
      if try > 4 and YXratio = newYXratio then
           break;
      else YXratio:= newYXratio;
      end_if;
      context(hold(userinfo)(10, "Step ".expr2text(try).". Modifying the y range to ".
                    expr2text(Ymin .. Ymax).". Trying the new aspect ratio Y:X = ".
                    expr2text(YXratio))):
    end_for: // for try from 1 to 6 do

    //----------------------------------------------------------------------
    // A heuristic clipping range Ymin .. Ymax was found.
    //----------------------------------------------------------------------
    // Check, whether ymax is an approximation of infinity
    // or whether ymax is a regular maximum
    if not maxdone then
       if ymax <> Ymax and 
         (ymax <= 0 or 
          Ymax >= ymax/10 or 
          Ymax <= Ymin) then 
          Ymax:= ymax; 
          userinfo(10, "Resetting maximum of y. Choosing y range = ".  expr2text(Ymin .. Ymax)):
       end_if;
    end_if;
    // Check, whether ymin is an approximation of -infinity
    // or whether ymin is a regular minimum
    if not mindone then
       if ymin <> Ymin and 
         (ymin >= 0 or 
          Ymin <= ymin/10 or 
          Ymin >= Ymax) then 
          Ymin:= ymin; 
          userinfo(10, "Resetting minimum of y. Choosing y range = ".  expr2text(Ymin .. Ymax)):
       end_if;
    end_if;
    //----------------------------------------------
    // symmetrize Ymin, Ymax
    //----------------------------------------------
    if not (mindone and maxdone) then
      if Ymax < ymax and Ymin > ymin then
        // there is a positive and a negative singularity
        alldata:= []:
        for branch in branches do
            branch:= op(branch, 2):
            getXYData(); // uses branch and sets xdata, ydata
            alldata:= alldata . ydata;
        end_for:
        eq:= stats::empiricalQuantile(alldata):
        eq5:= eq(0.5):
        tmp:= sqrt(specfunc::abs(eq5 - Ymin)*specfunc::abs(Ymax - eq5));
        if  (not iszero(tmp))  and
             ymin  < eq5 - tmp and
           eq(0.2) > eq5 - tmp and
             ymax  > eq5 + tmp and
           eq(0.8) < eq5 + tmp then
             if not ubbLO then
                Ymin:= eq5 - tmp;
             end_if;
             if not ubbHI then
                Ymax:= eq5 + tmp;
             end_if;
        end_if;
      end_if;
    end_if;

    if Ymin > Ymax then 
       Ymin:= (Ymin + Ymax)/2:
       Ymax:= Ymin:
    end_if;

    //-----------------------------------------------------
    // Heuristics:
    // If 0 < Ymin << Ymax, we probably should set Ymin = 0.0
    // increasing the viewing range up to 10 per cent.
    //-----------------------------------------------------
    if (not ubbLO) and Ymin > 0 and Ymin < Ymax/10 then
       Ymin:= float(0);
    end_if;
    //-----------------------------------------------------
    // If Ymin << Ymax < 0, we probably should set Ymax = 0.0
    // increasing the viewing range up to 10 per cent.
    //-----------------------------------------------------
    if (not ubbHI) and Ymax < 0 and Ymax > Ymin/10 then
       Ymax:= float(0);
    end_if;

    //-----------------------------------------------------
    // automatic viewing box Ymin .. Ymax is computed
    //-----------------------------------------------------
    return([Ymin, Ymax]);
  end_proc:

  //-----------------------------
  // Call getAutomaticViewingBox
  //-----------------------------
  if not (ubbLO and ubbHI) then
       if nosingularities then
          tmp:= [ymin, ymax]
       else
          st:= time():
          userinfo(1, "Determining the automatic viewing box."):
          tmp:= getAutomaticViewingBox():
       end_if;
       if not has(tmp, NIL) then
          if not ubbLO then
             Ymin:= tmp[1];
          end_if;
          if not ubbHI then
             Ymax:= tmp[2];
          end_if;
       end_if;
       userinfo(2, "Automatic viewing box determined: ".
                expr2text([Xmin..Xmax, Ymin..Ymax]));
       userinfo(3, "Time for determining the automatic viewing box: ".
                expr2text(time() - st)." msec"):
  end_if;
  //--------------------------------------------------------
  // Determine the aspect ratio y:x  = YXratio:1 needed
  // for the following refinement.
  //--------------------------------------------------------
  YXratio:= (Ymax - Ymin)/width;
  //-------------------------------------------------------
  // Next, we go for an adaptive refinement of the branches
  //-------------------------------------------------------
  //------ the bisect utility -----------------------------
  // It just checks whether 2 slopes fp1 and fp2 represent
  // lines with directions that are no more than 10 degrees
  // apart (maxtan = tan(10 degrees) is set above. 
  // It requires the YXratio computed above.
  //------------------------------------------------------
  if iszero(YXratio) then
       bisect:= () -> FALSE
  else bisect:= proc(x1, f1, xm, fm, x2, f2)
       local fp1, fp2, tmp;
       begin
         fp1:= (fm - f1)/(xm - x1);
         fp2:= (f2 - fm)/(x2 - xm);
         // with c = cos(epsilon), t = tan(epsilon), the criterion for
         // bisect is (r^2+fp1*fp2) <= c*sqrt((r^2+fp1^2)*(r^2+fp2^2))
         // or, equivalently, (r^2+fp1*fp2)* t  <= r*|fp1 - fp2|. The
         // constants r = YXratio and t = maxtan are set above.
         tmp:= fp1*fp2:
         if tmp < 0 then return(TRUE) end_if;
         tmp:= YXratio^2 + fp1*fp2 ;
         return(bool(tmp * maxtan < YXratio * specfunc::abs(fp1 - fp2)));
       end_proc:
  end_if;
  //----- recursive utility -------------------------
  // Test, whether the mid point c = (a + b)/2 of an
  // interval [a, b] is reasonably represented by the
  // straight line through a and b. If not, split the
  // interval into 2 halves [a, c] and [c, b].
  //-------------------------------------------------
  functionEval_rec:= proc(a, fa, c, fc, b, fb, reclevel)
  local c1, fc1, c2, fc2;
  begin
     // different calls to functionEval_rec communicate
     // with one another via the 'lock' table
     if lock[a, b] = TRUE then return(): end_if:
     if reclevel >= maxrecursionlevel then return(); end_if;
     if a = c or c = b or a = b then return() end_if;
     if bisect(a, fa, c, fc, b, fb) then
        c1:= (a + c)/2: 
        if getf(c1) then
            fc1:= y:
            branch[c1]:= fc1;
            functionEval_rec(a, fa, c1, fc1, c, fc, reclevel + 1);
            lock[a, c]:= TRUE;
        end_if;
        c2:= (c + b)/2: 
        if getf(c2) then
            fc2:= y:
            branch[c2]:= fc2;
            functionEval_rec(c, fc, c2, fc2, b, fb, reclevel + 1);
            lock[c, b]:= TRUE;
        end_if;
        if lock[a, c] = TRUE and lock[c, b] = TRUE then
            // these conditions garantee that fc1, fc2 were ok.
            functionEval_rec(c1, fc1, c, fc, c2, fc2, reclevel + 1);
      //    lock[c1, c2]:= TRUE;  // no need to lock this interval
        end_if;
     end_if;
     return();
  end_proc:
  //-------------------------------------------------
  // adaptive refinement of the branches. Implement 
  // 'refine' as a procedure (it may be called twice)
  //-------------------------------------------------
  refine:= proc() 
  local tmp, i;
  begin
    // do not declare 'branch', 'branchindex', 'lock' as local 
    // to 'refine', because 'functionEval_rec' refers to these
    // values!
    for branch in branches do
        // lock is a table that is used by functionEval_rec.
        // When the entry lock[x2, x3] = TRUE is set by
        // functionEval_rec(x1, x2, x3), then the next call
        // functionEval_rec(x2, x3, x4) will not recurse into
        // the locked interval [x2, x3] (it was handled by the
        // first call functionEval_rec(x1, x2, x3)).
        lock:= table(): // initialize the locks 
        // turn the table 'branch' (with branch[x] = y) into a
        // sorted list of the form branch = [.., [x, y] ,..]
        branchindex:= op(branch, 1); // global used by getf
        branch:= op(branch, 2): // branch is a table to be enlarged
                                // by the following call to
                                // functionEval_rec
        getXYData(); // uses branch and sets xdata, ydata
        // The following call fills in the 'global' branch
        // as a side effect. The calls for consecutive values
        // of i communicate via the 'lock' table:
        for i from 2 to (nops(xdata) - 1) do
          functionEval_rec(xdata[i-1], ydata[i-1],
                           xdata[ i ], ydata[ i ],
                           xdata[i+1], ydata[i+1], 0);
          // Keep the lock table as small as possible. The entry
          // for the first half of the interval above is not 
          // needed anymore, because the next call cannot descend
          // into this interval, anyway.
          delete lock[xdata[i-1], xdata[i]];
        end_for:
        // reinsert the enlarged branch into the branches table:
        branches[branchindex]:= branch;
    end_for:
  end_proc:
  //----------------------------------------------------
  // The function was evaluated on the initial mesh.
  // If appropriate, the data were split into several
  // branches (stored in the 'branches' table).
  // Next, if AdpativeMesh = TRUE, refine the branches.
  //----------------------------------------------------
  oldYXratio:= YXratio; // remember YXratio for double checking
  if adaptive then
     userinfo(1, "Starting adaptive refinement of the initial mesh."):
     userinfo(3, "Assuming aspect ratio Y:X = ".expr2text(YXratio)):
     st:= time():
     refine(); 
     userinfo(2, "New total number of mesh points: ".
                 expr2text(_plus(nops(op(branch,2)) $ branch in branches))):
     userinfo(3, "Time for the adaptive refinement: " .
                 expr2text(time() - st)." msec"):
     //-------------------------------------------------
     // Adaptive refinement of the branches is finished.
     //-------------------------------------------------
     // Double check the automatic ViewingBox
     //-------------------------------------------------
     if not (ubbLO and ubbHI) then
       oldYXratio:= YXratio; // remember YXratio for double checking
       if nosingularities then
          tmp:= [ymin, ymax]
       else
          st:= time():
          userinfo(1, "Determining the automatic viewing box."):
          tmp:= getAutomaticViewingBox():
       end_if;
       if not has(tmp, NIL) then
          if not ubbLO then
             Ymin:= tmp[1];
          end_if;
          if not ubbHI then
             Ymax:= tmp[2];
          end_if;
       end_if;
       userinfo(2, "New automatic viewing box determined: ".
                expr2text([Xmin..Xmax, Ymin..Ymax]));
       userinfo(3, "Time for determining the new automatic viewing box: ".
                    expr2text(time() - st)." msec"):
       //--------------------------------------------------------
       // Determine the aspect ratios x:t, y:t, y:t needed
       // for the following refinement.
       //--------------------------------------------------------
       YXratio:= (Ymax - Ymin)/width;
       if YXratio <> oldYXratio then
            userinfo(3, "The aspect ratio Y:X has changed from ".expr2text(oldYXratio).
                        " to ".expr2text(YXratio)):
       else userinfo(3, "Using the aspect ratio Y:X = ".expr2text(oldYXratio));
       end_if;
       //----------------------------------------------------------------
       // re-refine: the refinement depends crucially on the
       // scaling ratio estimated before the refinement. We
       // double check the scaling ratio. If it has dropped to 
       // less than a quarter of its original value, the maximal
       // bend angle of 6 degrees may have amplified to 24 degrees
       // in the actual plot. In this case, we need to recompute 
       // the refinement.
       //----------------------------------------------------------------
       if YXratio < oldYXratio/4 then
          userinfo(1, "The aspect ratio Y:X changed drastically by ".
                      "the 1st adaptive refinement. Need to recompute.");
          userinfo(1, "Starting 2nd adaptive refinement of the mesh."):
          st:= time():
          // do not use the refined mesh in 'branches', because a second
          // refinement would lead to meshs that are much too fine.
          // Use the initial raw mesh instead. The improved second 
          // refinement relies on the better value of YXratio.
          branches:= rawbranches: 
          // Do the work:
          refine();
          userinfo(2, "New total number of meshpoints:".
                      expr2text(_plus(nops(op(branch, 2)) $ branch in branches)));
          userinfo(3, "Time for 2nd adaptive refinement: " .
                      expr2text(time() - st)." msec"):
        end_if; // if YXratio < oldYXratio/4
     end_if: // if ubb <> TRUE
  end_if; // if adaptive
  //----------------------------------------------------------
  // pass the values of the automatic viewing box to the renderer.
  // Enlarge the viewing box by a tiny amount.
  //----------------------------------------------------------
  eps:= (Xmax - Xmin)*10^(-DIGITS):
  [Xmin, Xmax]:= [Xmin - eps, Xmax + eps];
  eps:= (Ymax - Ymin)*10^(-DIGITS):
  [Ymin, Ymax]:= [Ymin - eps, Ymax + eps];
  //----------------------------------------------------------
  // Each branch in the 'branches' table is a table with
  // branch[x] = [f(x), f'(x)]. Turn it into a sorted
  // list branch = [.., [x,f(x)],..]:
  //----------------------------------------------------------
  userinfo(1, "Computation finished. Preparing the return data."):
  st:= time():
  for branch in branches do
      branchindex:= op(branch, 1); // global used by getf
      branch:= op(branch, 2):
      getXYData(); // uses branch and sets xdata, ydata
      branches[branchindex]:= [[xdata[i], ydata[i]] $ i=1..nops(branch)];
  end_for;
  userinfo(3, "Time for preparing the return data: ".
              expr2text(time() - st)." msec"):

  //----------------------------------------------------------
  // Clean the singularities. Step 1.
  // Heuristic check whether the 'singularity' found by
  // numeric::discont is regular 0/0 or a jump discontinuity
  //----------------------------------------------------------
  if nops(singularities) > 0 then
     nosingularities:= {}; 
     for s in singularities do
         [y1, y2, y3]:= [NIL, NIL, NIL]:
         if getf(s - 2.456*width/10^5, 1) then
            y1:= y;
         end_if;
         if getf(s + 1.987*width/10^5, 1) then
            y2:= y;
            y3:= y;
         end_if;
         if  y1 = NIL and y2 = NIL then
            // it seems that the expression cannot be
            // evaluated to reals close to f. Do not
            // put an asymptote here.
            nosingularities := nosingularities union {s};
         end_if;
         if (y1 <> NIL and y2 <> NIL and
            specfunc::abs(y2 - y1) <= 0.9*specfunc::abs(Ymax - Ymin)
            ) then
            //----------------------
            // is this just a jump?? 
            // Check the steepness when approaching from the left:
            if getf(s - 2.1234*width/10^5, 1) then
               y2:= y;
            end_if;
            if y2 <> NIL and
               specfunc::abs(y2 - y1) <= specfunc::abs(Ymax - Ymin)/10^4 then
               nosingularities := nosingularities union {s};
            end_if;

            // Check the steepness when approaching from the left:
            y2:= y3:
            if getf(s + 1.9123*width/10^5, 1) then
               y3:= y;
            end_if;
            if y3 <> NIL and
               specfunc::abs(y3 - y2) <= specfunc::abs(Ymax - Ymin)/10^4 then
               nosingularities := nosingularities union {s};
            end_if;
         end_if;
     end_for:
     if nops(nosingularities) > 0 then
        singularities := singularities minus nosingularities;
     end_if;
  end_if;

  //----------------------------------------------------------
  // convert the table 'branches' to a list
  // branches:=  [[[x1,y1],[x2,y2],...], [[x.i,y.i],  ...  ], ...]
  //              |--- branches[1] ---|  |-- branches[2] --|, ...]
  // Note that the Hatch object needs the branches ordered from
  // small x to large x. We also need the ordering to identify the
  // gaps between the branches (for eliminating VerticalAsymptotes)
  //----------------------------------------------------------
  branches:= [branches[i] $ i=1..nops(branches)];
  // eliminate empty branches
  branches:= select(branches, b -> (nops(b) <> 0));

  //----------------------------------------------------------
  // Clean the singularities. Step 2. Do not indicate the 
  // singularities in the gaps between the branches by 
  // vertical asymptotes! 
  //----------------------------------------------------------
  if nops(branches) = 0 then
     singularities:= {};
  end_if;
  if nops(singularities) > 0 then
     eps:= 1e-4*width; // this eps should be much larger than the 
                       // precision with which the singularities
                       // were isolated by numeric::discont
     x1:= branches[1][1][1]: // the smallest x value in the i-th branch
     singularities := select(singularities, s -> (x1 - eps <= s));
     for i from 2 to nops(branches) do
         x1:= branches[i-1][nops(branches[i-1])][1];
         x2:= branches[i][1][1]: // the smallest x value in the i-th branch
         if specfunc::abs(x2 - x1) > eps then
            // A gap of size eps/100 around the singularities found by
            // numeric::discont was introduced above. The gap [x1, x2] 
            // seemes to be a much larger gap not stemming from the splitting 
            // into branches because of poles (i.e., this gap stems from
            // non-real values). Eliminate the singularities in this gap
            // [x1, x2]. However, eliminate only the singularities in the
            // somewhat smaller interval [x1 + eps, x2 - eps] to make sure
            // that singularities at the end points x1 and x2 of the gap
            // are indeed found and highlighted via vertical asymptotes:
            singularities := select(singularities, 
                                    s -> _lazy_or(s <= x1+eps, x2-eps <= s));
         end_if;
     end_for:
     x2:= branches[nops(branches)][nops(branches[nops(branches)])][1];
     singularities := select(singularities, s -> (s <= x2 + eps));
  end_if;
                          
  //----------------------------------------------------------
  userinfo(3, "Total time needed: ".expr2text(time() - ST)." msec"):
  userinfo(3, "--------------------------------------------------"):
  return([[Xmin..Xmax, Ymin..Ymax],  // the viewing box
     //  map([op(branches)], op, 2), // the plot data as
                                     // unordered branches
         branches,      // the plot data as ordered branches
         singularities  // the set of singularities
        ]);
end_proc:
//--------------------------------------------------------------
