/*---------------------------------
surfaceEval(X(u,v), Y(u,v), Z(u,v), 
            nu, smoothnessu, umin..umax, umin .. umax,
            nv, smoothnessv, vmin..vmax, vmin .. vmax,
            xmin..xmax/Automatic,
            ymin..ymax/Automatic,
            zmin..zmax/Automatic,
            AdpativeMesh = TRUE/FALSE)
evaluates the procedures X, Y, Z on an equidistant regular
mesh u.i, v.i in the rectangle umin .. umax, vmin .. vmax.
If umin > umax or vmin > vmax or nu < 1 or nv < 1, an
empty object of the form [NIL, []] is returned.
nu = 1 and nv = 1 are reset to nu =2 and nv = 2, automatically. 

plot::Surface calls this utility for AdaptiveMesh = 0 to compute 
a regular numerical mesh for a surface.

Singularties are avoided by trying evaluation in the neighbourhood.
A heuristics is used to determine an automatic viewing box if
singularities are found.

Calls: plot::surfaceEval(X, Y, Z,
                         nu, nsu, umin..umax,
                         nv, nsv, vmin..vmax,
                         xrange, yrange, zrange,
                      <, AdaptiveMesh = TRUE/FALSE>
                      <, "NoSingularities">)

Parameters:
       X, Y, Z    : procedures. X(u,v) is the parametrization of the x-coordinate etc.
       nu, nv     : number of mesh points of the 'coarse' mesh: integers >= 2
       nsu, nsv   : number of additional points between 2 points of the 'coarse' mesh:
                    integers >= 0.
       umin, umax : the range of the u parameter
       vmin, vmax : the range of the v parameter
       xrange     : Viewingbox x range: xmin .. xmax
       yrange     : Viewingbox y range: xmin .. xmax
       zrange     : Viewingbox z range: xmin .. xmax
       The string "NoSingularities" prevents that the extremely expensive
       subroutine 'getAutomaticViewingBox' is called. The viewing box is
       computed by the min/max values of the parametrization

ReturnValue: the list [ViewingBox, data]
             ViewingBox = [Xmin .. Xmax, Ymin .. Ymax, Zmin .. Zmax],
             Presently:
             data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
             Planned:
             data = [[u1, v1, x1, y1, z1, nx1, ny1, nz1], 
                     [u2, v2, x2, y2, z2, nx2, ny2, nz2], ...]
             where (nx.i, ny.i, nz.i) is the normal to the surface
             at the point (x.i, y.i, z.i)
Example:
>> plot::surfaceEval((u, v) -> u*cos(v), 
                     (u, v) -> u*sin(v),
                     (u, v) -> u*sin(2*v),
                      3, 0, 0 .. 2,
                     20, 3, 0 .. 2*PI,
                     Automatic, Automatic, Automatic)
---------------------------------*/

plot::surfaceEval:= proc(X, Y, Z,
                         umesh, usubmesh, urange,
                         vmesh, vsubmesh, vrange,
                         xrange, yrange, zrange
                     //< AdaptiveMesh = adaptivemesh >
                         )
local MINFLOAT, MAXFLOAT,
      u, du, umin, umax, uwidth, um1,
      v, dv, vmin, vmax, vwidth, vm1,
      x, // xmin, xmax, ymin, ymax, zmin, zmax,
      /*normalX, normalY, normalZ,*/
      F, i, j, data, empty,
      fmin, fmax, Fmin, Fmax,
      _umin, _umax, _vmin, _vmax,
      ubbLO, ubbHI, vbox, ST,
      getPoint, trackRange, negativeCurvature,
      getAutomaticViewingBox, nosingularities
     ; 
begin

  MINFLOAT:= 1e-300: // smallest float that can be converted to a C double
  MAXFLOAT:= 1e300:  // largest float that can be converted to a C double

  empty:= [NIL, []]:
  [umin, umax]:= float([op(urange)]): 
  [vmin, vmax]:= float([op(vrange)]): 
  uwidth:= umax - umin;
  vwidth:= vmax - vmin;

  if umin > umax or
     vmin > vmax then
        return(empty): 
  end_if:

  umesh:= (umesh - 1)*(1 + usubmesh) + 1: 
  du:= 1.234567*uwidth/10^6:

  vmesh:= (vmesh - 1)*(1 + vsubmesh) + 1: 
  dv:= 1.234567*vwidth/10^6:

  //--------------------------
  // initialize various tables
  //--------------------------
  [fmin, fmax, Fmin, Fmax,
   _umin, _umax, _vmin, _vmax,
   ubbLO, ubbHI
  ] := [table() $ 10]:

  // --------------------------
  // initialize the viewing box
  // --------------------------
  nosingularities:= has([args()], "NoSingularities"):
  vbox:= [xrange, yrange, zrange];
  for i from 1 to 3 do
    if vbox[i] = Automatic then
       vbox[i]:= Automatic .. Automatic;
    end_if;
  end_for:
  for i from 1 to 3 do
    // initialization for later refinement
    if op(vbox[i], 1) = Automatic then
         ubbLO[i]:= FALSE;    // userboundingbox ?
         fmin[i]:=  MAXFLOAT; // The actual viewing box found during the evaluation
         Fmin[i]:= -MAXFLOAT; // The viewing box to be passed to the renderer
    else ubbLO[i]:= TRUE:     // userboundingbox ?
         fmin[i]:= op(vbox[i], 1):
         Fmin[i]:= fmin[i];   // The bounding box requested by the user
    end_if;
    if op(vbox[i], 2) = Automatic then
         ubbHI[i]:= FALSE;    // userboundingbox ?
         fmax[i]:= -MAXFLOAT; // The actual viewing box found during the evaluation
         Fmax[i]:=  MAXFLOAT; // The viewing box to be passed to the renderer
    else
         ubbHI[i]:= TRUE;     // userboundingbox ?
         fmax[i]:= op(vbox[i], 2):
         Fmax[i]:= fmax[i];   // The bounding box requested by the user
    end_if;
  end_for:

  //-----------------------------------------------------
  // utility getPoint(u,v): try to evaluate X(u,v), Y(u,v), Z(u,v)
  // If the result consist of real numbers, assign them to the
  // variables x = [x[1],x[2],x[3]](of surfaceEval)
  // (SIDE EFFECT!!) and return TRUE.
  // If an error occurs, FALSE is retuned.
  // If a non-real value X(u,v) or Y(u,v) or Z(u,v) is found, 
  // FALSE is returned.
  //-----------------------------------------------------
  getPoint:= proc(u,v)
  // assigns x:= [X(u,v),Y(u,v),Z(u,v)] as a side effect
  local i;
  begin
    x:= [0 $ 3]; // this is a variable of plot::surfaceEval!
    if traperror(((x[i]:= F[i](u,v)) $ i=1..3)) <> 0 then
         return(FALSE)
    elif _lazy_and(
            x[i] <> RD_NAN $ i = 1..3,
            _lazy_or(domtype(x[i]) = DOM_FLOAT,
                     domtype((x[i]:= numeric::complexRound(x[i]))) = DOM_FLOAT)
            $ i = 1..3,
            x[i] >= -MAXFLOAT $ i = 1..3,
            x[i] <=  MAXFLOAT $ i = 1..3,
            _lazy_or((iszero(x[i]),
                      x[i] >= MINFLOAT,
                      x[i] <=-MINFLOAT) $ i = 1..3)
           ) then
         trackRange(u,v,x);
         return(TRUE)
    else return(FALSE);
    end_if;
  end_proc:

   //------------------------------------------------------
   // utility trackRange(u,v,x): check the global extremal
   // values fmin/max[i] (== xmin, ... , zmax) and redefine
   // them if necessary.
   //------------------------------------------------------
   trackRange:= proc(u,v,x)
   local i;
   begin
     for i from 1 to 3 do
       if x[i] < fmin[i] then
          fmin[i]:= x[i]; _umin[i]:= u; _vmin[i]:= v;
       end_if;
       if x[i] > fmax[i] then
          fmax[i]:= x[i]; _umax[i]:= u; _vmax[i]:= v;
       end_if;
     end_for:
   end_proc:

  //---------------------------------------------------------
  // utility to decide whether the extrema are singularities.
  // The idea is that the second derivative near a singularity
  // must have the same sign as the function value.
  // We just need the sign of the second derivative along the
  // steepest descent direction (the gradient of fproc).
  // fproc : the function to be investigated: a procedure
  // f1    : the value fproc(u1, v1)
  // u1, v1: the coordinates of the point to be investigated
  //---------------------------------------------------------
  negativeCurvature:= proc(fproc, f1, u1, v1)
  local eps, u0, v0, fu, fv, tmp, u2, v2, f0, f2;
  save DIGITS; 
  begin
     eps:=  0.975*10^(2- DIGITS);
     u0 := u1 - eps*uwidth;
     v0 := v1 - eps*vwidth;
     if traperror((
        fu:= (fproc(u0, v1) - f1)/uwidth; // u-derivative
        fv:= (fproc(u1, v0) - f1)/vwidth; // v-derivative
        )) = 0 then
        tmp:= max(specfunc::abs(fu), specfunc::abs(fv));
        if iszero(tmp) then
           // zero gradient. The second derivative would be
           // dominated by round-off
           return(UNKNOWN);
        end_if;
     else return(UNKNOWN);
     end_if;
     fu:= fu/tmp; // u-component of gradient(f)
     fv:= fv/tmp; // v-component of gradient(f)
     u0 := u1 - eps*uwidth*fu; // perturb
     v0 := v1 - eps*vwidth*fv; // along the
     u2 := u1 + eps*uwidth*fu; // gradient
     v2 := v1 + eps*vwidth*fv; // direction
     DIGITS:= DIGITS + 10;
     if traperror((f0:= fproc(u0,v0); f2:= fproc(u2,v2))) = 0 or // derivative along gradient
        traperror((f0:= fproc(u0,v1); f2:= fproc(u2,v1))) = 0 or // u-derivative at v1
        traperror((f0:= fproc(u1,v0); f2:= fproc(u1,v2))) = 0 or // v-derivative at u1
        traperror((f0:= fproc(u0,v0); f2:= fproc(u2,v0))) = 0 or // u-derivative at v0
        traperror((f0:= fproc(u0,v0); f2:= fproc(u0,v2))) = 0 or // v-derivative at u0
        traperror((f0:= fproc(u0,v2); f2:= fproc(u2,v2))) = 0 or // u-derivative at v2
        traperror((f0:= fproc(u2,v0); f2:= fproc(u2,v2))) = 0    // v-derivative at u2
        then
        tmp:= 2*f1 - (f0 + f2);
        if iszero(tmp) or {domtype(f0), domtype(f2)} <> {Dom::Float} then
             return(UNKNOWN)
        else return(bool(tmp > 0));
        end_if;
     end_if;
     return(UNKNOWN);
  end_proc;

  //----------------------------------------------------------------
  // getAutomaticViewingBox tries to clip the ranges in a smart way.
  // It checks heuristically whether very large/small data should be
  // interpreted as singularities.
  // If not, it falls back to the values xmin, .. , zmax found during
  // the numerical evaluation. These are stored as fmin[i], fmax[i] with
  // i = 1,2,3 Finally, it returns the viewing box Fmin .. Fmax in the
  // i-th direction.
  //----------------------------------------------------------------
  getAutomaticViewingBox:= proc(i) // i = 1 = x-direction
                                   // i = 2 = y-direction
                                   // i = 3 = z-direction
  local eq, fdata, tmp, eq1, eq5, eq9;
  name plot::adaptiveSurfaceEval::getAutomaticViewingBox;
  begin
    ST:= time():
    context(hold(userinfo)(3, "trying to compute a decent viewing box in the ".
                              output::ordinal(i)." direction")):
    if fmin[i] = MAXFLOAT or fmax[i] = -MAXFLOAT then
       // no real & finite plot point was found: this plot is empty
       [Fmin[i], Fmax[i]]:= [float(0) $ 2]:
       return():
    end_if;
    // The following context construct serves for activating the
    // userinfo commands inside this sub procedure of adaptiveSurfaceEval
    // by pretending that the userinfo was called in the context
    // of the calling function, i.e., plot::adaptiveSurfaceEval
    fdata:= map(data, op, 2 + i);
    if nops(fdata) = 0 then
       return();
    end_if;

    eq:= stats::empiricalQuantile(fdata):
    if not ubbHI[i] then
         eq9:= eq(0.9);
    else eq9:= Fmax[i]; // the user defined viewing box border
    end_if:
    if not ubbLO[i] then
         eq1:= eq(0.1);
    else eq1:= Fmin[i]; // the user defined viewing box border
    end_if:

    if not ubbHI[i] then
      //----------------------------------------------
      // determine Fmax[i]
      //----------------------------------------------
      // Heuristics to decide whether the maximum is a proper maximum or a singularity
      Fmax[i]:= fmax[i];
      assert(fmax[i] = F[i](_umax[i], _vmax[i]));
      if fmax[i] > 0 and negativeCurvature(F[i], fmax[i], _umax[i], _vmax[i]) <> TRUE then
//print("positiveSingularity?");
         // this may be a positive singularity
         if fmax[i] >= eq9 + 100 * (eq9 - eq1) then // this does look like a singularity
//print("positiveSingularity!");
           Fmax[i]:= eq9;
         end_if;
      end_if;
    end_if;
    if not ubbLO[i] then
      //----------------------------------------------
      // determine Fmin[i]
      //----------------------------------------------
      // Heuristics to decide whether the minimum is a proper minimum or a singularity
      Fmin[i]:= fmin[i];
      assert(fmin[i] = F[i](_umin[i], _vmin[i]));
      if fmin[i] < 0 and negativeCurvature(F[i], fmin[i], _umin[i], _vmin[i]) <> FALSE then
//print("negativeSingularity?");
         // this may be a negative singularity
         if fmin[i] <= eq1 - 100 * (eq9 - eq1) then // this does look like a singularity
//print("negativeSingularity!");
           Fmin[i]:= eq1;
         end_if;
      end_if;
    end_if;
    if not (ubbHI[i] and ubbLO[i]) then
      //----------------------------------------------
      // symmetrize Fmin[i], Fmax[i]
      //----------------------------------------------
      // If there is a positive and a negative singularity, symmetrize the
      // viewing box around the 50% quantile eq5 by taking the harmonic
      // mean of its distance to the upper and lower viewingbox bounds
      if Fmax[i] < fmax[i] and Fmin[i] > fmin[i] then
        eq5:= eq(0.5):
        tmp:= sqrt(specfunc::abs(eq5 - Fmin[i])*specfunc::abs(Fmax[i] - eq5));
        if not iszero(tmp)     and
           fmin[i] < eq5 - tmp and
             eq1   > eq5 - tmp and
           fmax[i] > eq5 + tmp and
             eq9   < eq5 + tmp then
             if not ubbLO[i] then
                Fmin[i]:= eq5 - tmp;
             end_if;
             if not ubbHI[i] then
                Fmax[i]:= eq5 + tmp;
             end_if;
        end_if;
      end_if;
    end_if;

    // Heuristics:
    // If 0 < Fmin << Fmax, we probably should set Fmin = 0.0
    // increasing the viewing range up to 10 per cent.
    if (not ubbLO[i]) and Fmin[i] > 0 and Fmin[i] < Fmax[i]/10 then
       Fmin[i]:= float(0);
    end_if;
    // If Fmin << Fmax < 0, we probably should set Fmax = 0.0
    // increasing the viewing range up to 10 per cent.
    if (not ubbHI[i]) and Fmax[i] < 0 and Fmax[i] > Fmin[i]/10 then
       Fmax[i]:= float(0);
    end_if;
    
    //-----------------------------------------------------
    // automatic viewing box Fmin[i] .. Fmax[i] is computed
    //-----------------------------------------------------

    context(hold(userinfo)(3, "automatic viewing box in the ".
                              output::ordinal(i)." direction: ".
                              expr2text(Fmin[i] .. Fmax[i]).
                              " (original: ".expr2text(fmin[i] .. fmax[i]).
                              "). Took ".expr2text(time()- ST)." msec"
                           )):
  end_proc:

  //----------------------------------------------------------
  // evaluation on a regular mesh
  //----------------------------------------------------------
  F:= [float@X, float@Y, float@Z]; // called in getPoint
  data:= table():
  um1:= max(1, umesh - 1):
  vm1:= max(1, vmesh - 1):
  for i from 1 to vmesh do
    // Beware: do not change this computation of v.
    // It prevents roundoff effects!
    v:= ((vmesh - i)/vm1)*vmin + ((i - 1)/vm1)*vmax;
    for j from 1 to umesh do
      // Beware: do not change this computation of u.
      // It prevents roundoff effects!
      u:= ((umesh - j)/um1)*umin + ((j - 1)/um1)*umax;
      // x = [X, Y, Z] is assigned as a side effect
      if   getPoint(u,      v     ) then data[i,j]:= [u   , v   , op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u + du, v     ) then data[i,j]:= [u+du, v   , op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u - du, v     ) then data[i,j]:= [u-du, v   , op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u,      v + dv) then data[i,j]:= [u   , v+dv, op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u + du, v - dv) then data[i,j]:= [u+du, v-dv, op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u,      v - dv) then data[i,j]:= [u   , v-dv, op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u + du, v + dv) then data[i,j]:= [u+du, v+dv, op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u - du, v + dv) then data[i,j]:= [u-du, v+dv, op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      elif getPoint(u - du, v - dv) then data[i,j]:= [u-du, v-dv, op(x)]; //,normalX(u,v),normalY(u,v),normalZ(u,v)];
      else
        // ToDO
        error("cannot evaluate to a real numerical value ".
              "near the point (".expr2text(u, v).")");
      end_if;
    end_for;
  end_for;
  //----------------------------------------------------------
  // evaluation on the regular mesh is finished. Convert the
  // data array to a linear list
  //--------------------------------------------------------------------
  data:= [data[i,j] $ j=1.. umesh $ i=1..vmesh]:

  //--------------------------------------------------------------------
  // call getAutomaticViewingBox
  //--------------------------------------------------------------------
  for i from 1 to 3 do
    if not (ubbLO[i] and ubbHI[i]) then 
       if nosingularities then
            Fmin[i]:= fmin[i];
            Fmax[i]:= fmax[i];
       else getAutomaticViewingBox(i);
       end_if;
    end_if; 
    if Fmin[i] > Fmax[i] then  
       // this can only happen if at least one of the
       // viewing box borders is given and no point
       // inside this range was found.
       if ubbLO[i] then // Fmin is binding
          Fmax[i]:= Fmin[i];
       elif ubbHI[i] then // Fmax is binding
          Fmin[i]:= Fmax[i];
       else
          assert(FALSE); // should not arrive here
          // Just to be on the safe side for the release version:
          [Fmin[i], Fmax[i]] := [(Fmin[i] + Fmax[i])/2 $ 2];
       end_if;
    end_if;
  end_for;
  //--------------------------------------------------------------------

  userinfo(1, "viewing box = ".expr2text([Fmin[i] .. Fmax[i] $ i = 1..3]));

  if nops(data) > 0 then
     return([[Fmin[i] .. Fmax[i] $ i=1..3], data]):
  else
     return([empty]):
  end_if:
end_proc:
