/*
  An entry in plot::attributes is a list with 5 entries.
  The index is a attribute name as specified in the plot
  data model.

  An entry list contains:

  1)  Type of the attibute:
      ["Mandatory" |       is to be set explicitly
       "Optional"  |       is optional
       "Inherited" |       is inherited from above
       "Library"]          is only a library shortcut

  2)  Default value:  NIL (no default) or a value
                      which is accepted by the check procedure in 3)

  3)    Check-procedure
          Input:         Type, OptionName, OptionValue
          Return value:  (sequence of) option-equation(s)
                      or List mit ??? fuer Dinge, die
                         als Elemente rausgeschrieben werden
                      or DOM_STRING with an error message
      OR  a set of valid values for this attribute

  4)  a set of object types, where this attribute is used

  5)  FAIL or a procedure for combining attribute values of children
              to an own value, e.g. a procedure which propagates the
              setting  AxesInFront = TRUE from a polygon plot to the
              CoordinateSystem2d containing the plot
     Input:         Option name + hint attribute values of child-objects
     Return value:  [(sequence of) option-equation(s)]  computed
                    from the corresponding child entries
*/


/*
 *  The set plot::extraValues is uses to identify
 *  extra optional values from explicit checking procedures.
 */
plot::extraValues := {Visible}:


/*
 *  Typechecking Procedures used in the table plot::attributes
 */


// returns o=x, iff x is a valid color value
plot::checkColor :=
(typ, o, x) ->
  if testtype(x, DOM_IDENT) and
    strmatch("".x, "^#[A-Fa-f0-9]{6}([A-Fa-f0-9]{2})?$") then
    x := "".x;
    x := [text2int(x[2..3], 16)/255.0,
          text2int(x[4..5], 16)/255.0,
          text2int(x[6..7], 16)/255.0,
          if length(x)=9 then
            text2int(x[8..9], 16)/255.0
          end];
    o = x;
  elif testtype(x, DOM_LIST) and
    (nops(x) in {3, 4}) and
    map({op(x)}, type@float) = {DOM_FLOAT} then
    o=map(x, float)
  elif contains(plot::attributes, o."Function") and
    contains(plot::attributes[o."Function"][4], typ) then
    "expecting an RGB or an RGBa color; maybe you want to use the attribute ".expr2text(o."Function") 
  else
    "expecting an RGB or an RGBa color"
  end_if:

// returns o=x, iff x is a valid RGB color value
plot::checkRGBColor :=
(typ, o, x) ->
  if testtype(x, DOM_IDENT) and
    strmatch("".x, "^#[A-Fa-f0-9]{6}$") then
    x := "".x;
    x := [text2int(x[2..3], 16)/255.0,
          text2int(x[4..5], 16)/255.0,
          text2int(x[6..7], 16)/255.0];
    o = x;
  elif testtype(x, DOM_LIST) and
    nops(x) = 3 and
    map({op(x)}, type@float) = {DOM_FLOAT} then
    o=map(x, float)
  else
    "expecting an RGB color"
  end_if:

/** Utility function for checking output size parameters:
 *  
 *
 */
plot::checkOutputSize :=
  proc(typ, o, x, mupType, str : DOM_STRING)
  begin
    if domtype(float(x)) <> DOM_FLOAT then
      // x has a unit
      if traperror((x := unit::convert(x, unit::mm)/unit::mm)) = 0 then
        if testtype(x, mupType) then
          o=x
        else
          str
        end_if;
      else
        str
      end_if;
    else
      // x has no unit
      if testtype(x, mupType) then
        // convert to MM
        o=unit::convert(x*plot::attributes[OutputUnits][2], unit::mm)/unit::mm
      else
        str
      end_if:
    end_if;
  end_proc:

// returns o=x, iff x is a valid output size value
plot::checkPosOutputSize :=
(typ, o, x) -> 
  plot::checkOutputSize(args(), Type::Positive,
            "expecting a positive output size such as 1.2*unit::mm"):

// returns o=x, iff x is a valid output size value
plot::checkNonNegOutputSize :=
(typ, o, x) ->
  plot::checkOutputSize(args(), Type::NonNegative,
            "expecting a nonnegative output size such as 1.2*unit::mm"):

// returns o=x, iff x is a valid output position value
plot::checkOutputPos := 
(typ, o, x) ->
  plot::checkOutputSize(args(), Type::NonNegative,
            "expecting a nonnegative output size such as 1.2*unit::mm"):

// returns o=x, iff x is a DOM_STRING
plot::checkText :=
(typ, o, x) ->
  if testtype(x, DOM_STRING) then
    o=x
  else
    "expecting a character string"
  end_if:

// returns o=x, iff x is a nonnegative integer
plot::checkNonNegInteger :=
(typ, o, x) ->
  if testtype(x, Type::NonNegInt) then
    o=x
  else
    "expecting a nonnegative integer number"
  end_if:

// returns o=x, iff x is a positive integer
plot::checkPosInteger :=
(typ, o, x) ->
  if testtype(x, Type::PosInt) then
    o=x
  else
    "expecting a positive integer number"
  end_if:

// returns o=x, iff x is a positive integer
plot::checkInteger :=
(typ, o, x) ->
  if testtype(x, Type::Integer) then
    o=x
  else
    "expecting an integer number"
  end_if:

// returns o=x, iff x is a positive real
plot::checkPosReal :=
(typ, o, x) ->
  if testtype(float(x), Type::Positive) then
    o=float(x)
  else
    "expecting a positive real number"
  end_if:

// returns o=x, iff x is a nonnegative real
plot::checkNonNegReal :=
(typ, o, x) ->
  if testtype(float(x), Type::NonNegative) then
    o=float(x)
  else
    "expecting a nonnegative real number"
  end_if:

// returns o=x, iff x is real
plot::checkReal :=
(typ, o, x) ->
  if testtype(float(x), Type::Real) then
    o=float(x)
  else
    "expecting a real number"
  end_if:

// returns o=x, iff x is real; used for CameraDirection
plot::checkReal_ :=
(typ, o, x) ->
  if typ = "Scene3d" then
    if testtype(float(x), Type::Real) then
      o=NIL, o."_"=float(x)
    else
      "expecting a real number"
    end_if:
  else
    if testtype(float(x), Type::Real) then
      o=float(x)
    else
      "expecting a real number"
    end_if:
  end_if:

plot::readReal :=
  (obj, o, t) ->
  if contains(t, o)  then
    t[o]
  else
    FAIL
  end_if:


// returns o=x, iff x is a valid font
/*  font encoding in MuPAD is as follows:
    a) a string in the encoding of the DTD
    b) a list (all entries are optional):

       [FontFamily : DOM_STRING,
        FontSize   : PosInt,
        Bold,
        Italic,
        RGB(A) color value,
        alignment (one of Left, Center, Right)]
 
*/
plot::checkFont :=
proc(typ, o, x)
  local font, ind, tmp, l, ret, list;
begin
  if testtype(x, DOM_STRING) then
    // string in hopefully correct encoding
    l := select(text2list(x, ["'"]), x->x<>"");
    if l[1] = "'" and l[3] = "'" then
      list := [l[2]];
      l := select(text2list(l[4], [" "]), x->x<>"")
    else
      return("font name missing")
    end:
    l := select(l, x->x<>" ");
    if traperror((ret := text2expr(l[1]))) = 0 and
      testtype(ret, Type::PosInt) then
      list := list.[ret];
      delete l[1];
    else
      return("font size missing")
    end_if;
    ind := contains(l, "Bold");
    if ind <> 0 then
      list := list.[l[ind]];
      delete l[ind];
    end:
    ind := contains(l, "Italic");
    if ind <> 0 then
      list := list.[l[ind]];
      delete l[ind];
    end:
    if nops(l) > 0 and contains({"Left", "Center", "Right"}, l[1]) then
      list := list.[text2expr(l[1])];
      delete l[1];
    end:
    if nops(l) > 0 and length(l[1]) = 7 and l[1][1] = "#" then
      list := list.[l[1]];
      delete l[1];
    end:
    if nops(l) > 0 then
      return("unexpected value in font specification: ".l[1])
    end:
    return(o=x)
  end_if;
  if testtype(x, DOM_LIST) then
    list := x;
    tmp := select(x, testtype, DOM_STRING);
    if nops(tmp) = 0 then
      // no font family, use
      font := "'sans-serif' "
    elif nops(tmp) = 1 then
      // string is font family
      font := "'", tmp[1], "' ";
      ind := contains(x, tmp[1]);
      delete x[ind];
    else
      return("expecting only one font family")
    end_if;
    tmp := select(x, testtype, Type::PosInt);
    if nops(tmp) = 0 then
      // no font size, use 10
      font := font, "10" 
    elif nops(tmp) = 1 then
      // number is font size
      font := font, expr2text(tmp[1]);
      ind := contains(x, tmp[1]);
      delete x[ind];      
    else
      return("expecting only one font size")
    end_if;
    ind := contains(x, Italic);
    if ind <> 0 then
      font := font, " Italic";
      delete x[ind];
    end_if;
    ind := contains(x, Bold);
    if ind <> 0 then
      font := font, " Bold";
      delete x[ind];
    end_if;
    ind := contains(x, Left);
    if ind <> 0 then
      font := font, " Left";
      delete x[ind];
    end_if;
    ind := contains(x, Center);
    if ind <> 0 then
      font := font, " Center";
      delete x[ind];
    end_if;
    ind := contains(x, Right);
    if ind <> 0 then
      font := font, " Right";
      delete x[ind];
    end_if;
    if nops(x) > 0 and // valid color?
      testtype(x[1], Type::ListOf(Type::NonNegative, 3, 4)) then
      font := font, " ", plot::MuPlotML::colStr(x[1]);
      delete x[1];
    end_if;
    if nops(x) = 0 then // no unprocessed options left over
      return(o=_concat(font));
    end_if;
  end_if;
  "expecting a font"
end_proc:

// for slot-methods:  convert string representation into list
plot::font2list :=
proc(x : DOM_STRING)
  local l, ret, list;
begin
  l := select(text2list(x, ["'", "'", " "]), x->x<>"");
  if l[1] = "'" and l[3] = "'" then
    list := [l[2]];
    l := l[4..-1]
  else
    return("font name missing")
  end:
  l := select(l, x->x<>" ");
  if traperror((ret := text2expr(l[1]))) = 0 and
    testtype(ret, Type::PosInt) then
    list := list.[ret];
    delete l[1];
  else
    return("font size missing")
  end_if;
  if nops(l) > 0 and l[1] = "Bold" then
    list := list.[Bold];
    delete l[1];
  end:
  if nops(l) > 0 and l[1] = "Italic" then
    list := list.[Italic];
    delete l[1];
  end:
  if nops(l) > 0 and contains({"Left", "Center", "Right"}, l[1]) then
    list := list.[text2expr(l[1])];
    delete l[1];
  end:
  if nops(l) > 0 and length(l[1]) = 7 and l[1][1] = "#" then
    list := list.[l[1]];
    delete l[1];
  end:
  if nops(l) > 0 then
    return("unexpected value in font specification: ".l[1])
  end:
  list
end_proc:



plot::extraValues :=
  plot::extraValues union {Bold, Italic, Left, Center, Right}:


/*  Attributes (w.r.t. the plot data model document) which take expression
 *  values are NOT written as XML attributes, but as elements.  We use
 *  the domain plot::ExprAttributes to store those values.  
 */

alias(EA = plot::ExprAttribute):

/*  activate when renderer can do 2d formulas
// 
plot::elementText :=
(typ, o, x) ->
  if testtype(x, output::mathText) or testtype(args(3..args(0)), DOM_STRING) then
    o = EA::namedText("".o, x)
  else
    o = EA::namedText("".o, output::mathText(args(3..args(0))))  
  end_if:
*/
plot::elementText :=
(typ, o, x) ->
  if testtype(x, DOM_STRING) then
    o = EA::namedText("".o, x)
  else
    "expecting a string"
  end_if:


// 
plot::elementExpr :=
(typ, o, x) ->
  if testtype(plot::ExprAttribute::getExpr(x), Type::Arithmetical) then
    o = EA::namedExpr("".o, plot::ExprAttribute::getExpr(x))
  else
    "expecting an arithmetical expression"
  end_if:

// Expressions which are printed as Expr with Opt attribute
// for graphical primitives.  E.g.  XMin used in a Box or a
// Function2d/3d is written as <Expr Opt="XMin">...</Expr>
plot::elementOptExpr :=
(typ, o, x) ->
  if testtype(plot::ExprAttribute::getExpr(x), Type::Arithmetical) then
    if typ <> "Hatch" and contains({-infinity, infinity}, x) then
      "infinity not allowed"
    elif contains({-infinity, infinity}, x) then 
      o = EA::optExpr("".o, x)
    else
      o = EA::optExpr("".o, plot::ExprAttribute::getExpr(x))
    end_if
  else
    "expecting an arithmetical expression"
  end_if:

plot::elementOptFunctionExpr :=
(typ, o, x) ->
  if testtype(plot::ExprAttribute::getExpr(x), Type::Arithmetical) or
    testtype(plot::ExprAttribute::getExpr(x), Type::Function) then
    o = EA::optExpr("".o, plot::ExprAttribute::getExpr(x))
  else
    "expecting an arithmetical expression or a function, got a '".
    expr2text(domtype(plot::ExprAttribute::getExpr(x)))."'"
  end_if:

plot::elementOptTextExpr :=
(typ, o, x) ->
  if testtype(x, Type::Function) or // testtype(x, output::mathText) or
    testtype(args(3..args(0)), DOM_STRING) then
    o = EA::optExpr("".o, x)
  else
//    o = EA::optExpr("".o, output::mathText(args(3..args(0))))
    "expecting a function or a string"
  end_if:


plot::elementOptProp :=
proc(t)
  option escape;
begin
  (typ, o, x) ->
  if testtype(x, t) then
    o = EA::optProp("".o, x)
  else
    "type ".expr2text(t)." expected"
  end_if:
end_proc:

// returns o=x, iff x is a valid list of color values
plot::checkColorList := 
(typ, o, x) ->
  if testtype(x, DOM_LIST) and map({op(x)}, testtype, Type::ListOf(Type::Real, 3, 4)) = {TRUE} then
    o=EA::optColors(op(map(x, float)))
  else
    "expecting a list of RGB or RGBa color values"
  end_if:

plot::elementOptPropSet :=
proc(s : Type::SetOf(DOM_IDENT))
  option escape;
begin
  (typ, o, x) ->
  if contains(s, x) then
    o = EA::optProp("".o, x)
  else
    "expecting an element of ".expr2text(s)
  end_if:
end_proc:

// expressions with float values
plot::elementVExpr :=
(typ, o, x) ->
  if testtype(float(x), DOM_FLOAT) then
    o = EA::namedVExpr("".o, x)
  else
    "expecting an expression evaluating to a float"
  end_if:

// expressions with float values
plot::elementViewingBoxVExpr :=
(typ, o, x) ->
  if x = Automatic or testtype(float(x), DOM_FLOAT) then
    o = EA::namedVExpr("".o, x)
  else
    "expecting an expression evaluating to a float"
  end_if:

// returns o=x, iff x is a valid value for TicksAt
// a valid value is a list of mixed expressions and
// equations of the form expressions = string where string
// is the optional tick label
//
//  [expr1, expr2 = label, ...  ]
//
// all expressions must evaluate to real values
plot::checkTicksAt := 
proc(typ, o, x)
  local tickPos;
begin
  // builds the representation of one tick position
  tickPos := y -> if testtype(y, "_equal") then
                    // explicit label given
                    EA::namedCompound("TickPos", {}, [],
                                      EA::VExpr(op(y, 1)),
                                      EA::namedText("Title", op(y,2)))
                  else
                    // no label
                    EA::namedCompound("TickPos", {}, [],
                                      EA::VExpr(y))
                  end_if:

  if testtype(x, DOM_LIST) and
    map({op(x)},
        y -> testtype(float(y), DOM_FLOAT) or
             testtype(y, "_equal") and
               testtype(float(op(y, 1)), DOM_FLOAT) and
               testtype(op(y, 2), DOM_STRING)) = {TRUE} then
    o = EA::namedCompound("".o, {}, [x], op(map(x, tickPos)))
  else
    "expecting a list of tick positions or equations of tick positions and label strings"
  end_if:
end_proc:

// a valid color function is a list of 3 or 4 expressions
// encoding a RGB(A) color or a procedure returning such a list.
// a list may contain the free parameters and animation parameter
// of the corresponding object.  When numerical values are inserted,
// the list entries must evaluate to real values between 0.0 and 1.0.
plot::checkColorFunction :=
proc(typ, o, x)
  local i;
begin
  if testtype(x, DOM_LIST) and (nops(x) = 3 or nops(x) =4) then
    o = EA::namedCompound("ColorFunc", {"Opt"="".o}, [x],
                          EA::IExpr(i) $ i in x),
    if contains(plot::legalAttributes[typ][3], LineColorType) and
       o = LineColorFunction then
      LineColorType = Functional else null() end_if,
    if contains(plot::legalAttributes[typ][3], FillColorType) and
       o = FillColorFunction  then
      FillColorType = Functional else null() end_if
  elif testtype(x, Type::Function) then
    o = EA::namedCompound("ColorFunc", {"Opt"="".o}, [x],
                      EA::IExpr(x)),
    if contains(plot::legalAttributes[typ][3], LineColorType) and
       o = LineColorFunction then
      LineColorType = Functional else null() end_if,
    if contains(plot::legalAttributes[typ][3], FillColorType) and
       o = FillColorFunction  then
      FillColorType = Functional else null() end_if
  else
    "expecting a list of 3 or 4 expressions or a procedure"
  end_if;
end_proc:


// checks text orientation vectors for TextOrientation
// orientation vectors have to be encoded as a list of 6 expressions.
plot::checkTextOrientation :=
proc(typ, o, x)
  local i;
begin
  if testtype(x, DOM_LIST) and nops(x) = 6 then
    o = EA::namedCompound("".o, {}, [x],
                          EA::IExpr(i) $ i in x)
  else
    "expecting a list of 6 expressions"
  end_if;
end_proc:

// checks for a 2x2 matrix encoded as a list with 4 entries
plot::checkMatrix2d :=
proc(typ, o, x)
  local i;
begin
  if x::dom::hasProp(Cat::Matrix) = TRUE then
    if linalg::ncols(x) = 2 and linalg::nrows(x) = 2 then
      x := [op(x)]
    else
      return("expecting a 2x2 matrix")
    end_if
  end_if;
  if testtype(x, DOM_LIST) and nops(x) = 4 then
    o = EA::namedCompound("".o, {}, [x],
                          EA::IExpr(i) $ i in x)
  else
    "expecting a list of 4 expressions"
  end_if;
end_proc:


// checks for a 3x3 matrix encoded as a list with 9 entries
plot::checkMatrix3d :=
proc(typ, o, x)
  local i;
begin
  if x::dom::hasProp(Cat::Matrix) = TRUE then
    if linalg::ncols(x) = 3 and linalg::nrows(x) = 3 then
      x := [op(x)]
    else
      return("expecting a 3x3 matrix")
    end_if
  end_if;
  if testtype(x, DOM_LIST) and nops(x) = 9 then
    o = EA::namedCompound("".o, {}, [x],
                          EA::IExpr(i) $ i in x)
  else
    "expecting a list of 9 expressions"
  end_if;
end_proc:

// check for a list of expressions
plot::checkList :=
  proc(typ, o, x)
    local i;
  begin
    if testtype(x, DOM_LIST) then
      if nops(x) > plot::pointsInObjectInspector() or
         nops(x) = 1 and testtype(x[1], DOM_HFARRAY) and
         op(x[1], [0,1])>1 and
         (op(x[1], [0,2,2])-op(x[1], [0,2,1]))*(op(x[1], [0,3,2])-op(x[1], [0,3,1])) > plot::pointsInObjectInspector()
        then
        o = NIL, o."_" = x
      else
        o = EA::namedCompound("ExprSeq", {"Opt"=o}, [x],
                              EA::IExpr(i) $ i in x),
        o."_" = x
      end_if;
    elif testtype(x, Type::Arithmetical) then
        o = EA::namedCompound("ExprSeq", {"Opt"=o}, [[x]],
                              EA::IExpr(x)),
        o."_" = [x]
    else
      "expecting a list of expressions"
    end_if;
  end_proc:


// checks Points vectors for PointList2d/3d and Polygon2d/3d
// orientation vectors have to be encoded as a list of lists of
// 2,5,6 expressions (2D) or 3,6,7 expression (3D).
plot::checkPoints :=
proc(typ, o, x)
  local i, j;
begin
  // convert matrices into lists of lists
  if x::dom::hasProp(Cat::Matrix)=TRUE then
    x := [[op(linalg::col(x,i))] $ i=1..linalg::ncols(x)]
  end_if;
  if testtype(x, DOM_LIST) and
     map({op(x)}, x -> (x::dom::hasProp(Cat::Matrix))) = {TRUE} then
         x := map(x, x -> [op(x)]);
  end_if;

  if contains(plot::allGraphPrim2d, typ) then
    if testtype(x, DOM_LIST) and
      map({op(x)}, domtype) = {DOM_LIST} and
      map({op(x)}, nops) minus {2, 3} = {} and
      map(map({op(x)}, op, 1..2), testtype, Type::Arithmetical) = {TRUE} and
      (map({op(x)}, nops) = {2} or
       map(map({op(x)}, op, 3), testtype,
           Type::ListOf(Type::Arithmetical, 3, 4)) = {TRUE}) then
      if nops(x) <= plot::pointsInObjectInspector() then
        o = EA::namedCompound("Points2d", {}, [x],
                              EA::namedCompound("Point2d", {}, [],
                                                (EA::IExpr(j) $ j in i[1..2]),
                                                if nops(i) > 2 then
                                                  (EA::IExpr(j) $ j in op(i[3]))
                                                else
                                                  null()
                                                end_if)
                              $ i in x), o."_" = x
      else
        o=NIL, o."_" = x
      end_if;
    else
      "expecting a list of lists of 2 expressions and an optional color value"
    end_if;      
  else  // not 2D, hopefully 3D
    if testtype(x, DOM_LIST) and
      map({op(x)}, domtype) = {DOM_LIST} and
      map({op(x)}, nops) minus {3, 4} = {} and
      map(map({op(x)}, op, 1..3), testtype, Type::Arithmetical) = {TRUE} and
      (map({op(x)}, nops) = {3} or
       map(map({op(x)}, op, 4), testtype,
           Type::ListOf(Type::Arithmetical, 3, 4)) = {TRUE}) then
      if nops(x) <= plot::pointsInObjectInspector() then
        o = EA::namedCompound("Points3d", {}, [x],
                              EA::namedCompound("Point3d", {}, [],
                                                (EA::IExpr(j) $ j in i[1..3]),
                                                if nops(i) > 3 then
                                                  (EA::IExpr(j) $ j in op(i[4]))
                                                else
                                                  null()
                                                end_if)
                              $ i in x), o."_" = x
      else
        o=NIL, o."_" = x
      end_if;
    else
      "expecting a list of lists of 3 expressions and an optional color value"
    end_if;      
  end_if;
end_proc:


//----------------------------------------------------------------------------
// Check Functions for library attributes
//----------------------------------------------------------------------------

// library interface for FillColorDirection/LineColorDirection
plot::libListOfReal := 
proc(typ, o, x)
begin
  x := map(x, float);
  if contains(plot::all2dObjects, typ) then
    // 2D
    if testtype(x, Type::ListOf(Type::Real, 2, 2)) then
      o."X" = x[1], o."Y" = x[2]
    else
    "expecting a list of 2 floats"      
    end_if
  elif contains(plot::all3dObjects, typ) then
    // 3D
    if testtype(x, Type::ListOf(Type::Real, 3, 3)) then
      o."X" = x[1], o."Y" = x[2], o."Z" = x[3]      
    else
    "expecting a list of 3 floats"      
    end_if
  else
    // neither 2D nor 3D: Canvas - accept both
    if testtype(x, Type::ListOf(Type::Real, 2, 3)) then
      o."X" = x[1], o."Y" = x[2],
      if nops(x) = 3 then o."Z" = x[3] else null() end
    else
    "expecting a list of 2 or 3 floats"      
    end_if
  end_if;
end_proc:
plot::readListOfReal := 
  (obj, o, t) ->
  if contains(t, o."X") and contains(t,  o."Y") then
    [t[o."X"], t[o."Y"],
          if contains(t, o."Z") then
            t[o."Z"]
          else
            null()
          end_if]
  else
    FAIL
  end_if:



// library interface TicksAt
plot::libTicksAt := 
proc(typ, o, x)
begin
  if testtype(x, Type::ListOf(DOM_LIST, 2, 3)) then
    if nops(x) = 2 then
      plot::checkTicksAt(typ, XTicksAt, x[1]),
      plot::checkTicksAt(typ, YTicksAt, x[2])
    else
      plot::checkTicksAt(typ, XTicksAt, x[1]),
      plot::checkTicksAt(typ, YTicksAt, x[2]),
      plot::checkTicksAt(typ, ZTicksAt, x[3])
    end_if;
  else
    "expecting a list of 2 or 3 lists"
  end_if;
end_proc:
plot::readTicksAt := 
  (obj, o, t) ->
  if contains(t, XTicksAt) and contains(t, YTicksAt) then
    ([t[XTicksAt], t[YTicksAt],
      if contains(t, ZTicksAt) then
        t[ZTicksAt]
      else
        null()
      end_if]);
    map(%, plot::ExprAttribute::getExpr)
  else
    FAIL
  end_if:

// library interface for  attribute Color
plot::libColor :=
  proc(typ, o, x)
    local colors, colorval;
  begin
    if testtype(x, DOM_IDENT) and
      strmatch("".x, "^#[A-Fa-f0-9]{6}([A-Fa-f0-9]{2})?$") then
      x := "".x;
      x := [text2int(x[2..3], 16)/255.0,
            text2int(x[4..5], 16)/255.0,
            text2int(x[6..7], 16)/255.0,
            if length(x)=9 then
              text2int(x[8..9], 16)/255.0
            end];
    end_if;
    if testtype(x, DOM_LIST) and
      (nops(x) in {3, 4}) and
      map({op(x)}, type@float) = {DOM_FLOAT} then
      colorval := map(x, float);
    else
      return("expecting an RGB or an RGBa color");
    end_if:
    colors := (plot::legalAttributes[typ][2] union
               plot::legalAttributes[typ][3]) intersect
              {FillColor, LineColor, PointColor, LightColor, Colors};
    if (slot(plot, typ))::primaryColor <> FAIL then
      if (slot(plot, typ))::primaryColor = Colors then
        (slot(plot, typ))::primaryColor =
        op(plot::checkColorList(typ, Colors, [colorval]), 2)
      elif not contains({FillColor, LineColor, PointColor, LightColor, Colors},
                      (slot(plot, typ))::primaryColor) then
        (slot(plot, typ))::primaryColor =
              EA::optProp("".(slot(plot, typ))::primaryColor, colorval)
      else
        (slot(plot, typ))::primaryColor = colorval
      end_if;
    elif contains(colors, Colors) then
      Colors = op(plot::checkColorList(typ, Colors, [colorval]), 2)
    elif contains(colors, FillColor) then
      FillColor = colorval
    elif contains(colors, LineColor) then
      LineColor = colorval
    elif contains(colors, PointColor) then
      PointColor = colorval
    elif contains(colors, LightColor) then
      LightColor = colorval
    else
      context(hold(error)("attribute 'Color' not applicable for plot::".typ))
    end_if
  end_proc:
plot::readColor :=
  proc(obj, o, x)
    local colors, tmp, res;
  begin
    colors := (plot::legalAttributes[obj::dom::objType][2] union
               plot::legalAttributes[obj::dom::objType][3]) intersect
              {FillColor, LineColor, PointColor, LightColor, Colors};
    
    if contains(colors, obj::dom::primaryColor) then
      tmp := "".obj::dom::primaryColor
    elif contains(colors, Colors) then
      tmp := "Colors"
    elif contains(colors, FillColor) then
      tmp := "FillColor"
    elif contains(colors, LineColor) then
      tmp := "LineColor"
    elif contains(colors, PointColor) then
      tmp := "PointColor"
    elif contains(colors, LightColor) then
      tmp := "LightColor"
    else
      return(FAIL)
    end_if;
    res := slot(obj, tmp);
    if tmp = "Colors" then
      res := plot::ExprAttribute::getExpr(res);
      if nops(res) = 1 then
        op(res)
      else
        FAIL
      end_if
    else
      res
    end_if;
  end_proc:

// library interface for *Mesh attributes
plot::libMesh :=
  proc(typ, o, x)
    local meshvars;
  begin
    meshvars := plot::legalAttributes[typ][3] intersect
                {UMesh, VMesh, XMesh, YMesh, ZMesh};
    if meshvars = {UMesh} then
      if testtype(x, Type::PosInt) then
        UMesh = x
      else
        "expecting a positive integer number"
      end_if;
    elif meshvars = {UMesh, VMesh} then
      if testtype(x, Type::ListOf(Type::PosInt, 2, 2)) then
        UMesh = x[1], VMesh = x[2]
      else
        "expecting a list of 2 positive integer numbers"
      end_if;
    elif meshvars = {XMesh} then
      if testtype(x, Type::PosInt) then
        XMesh = x
      else
        "expecting a positive integer number"
      end_if;
    elif meshvars = {XMesh, YMesh} then
      if testtype(x, Type::ListOf(Type::PosInt, 2, 2)) then
        XMesh = x[1], YMesh = x[2]
      else
        "expecting a list of 2 positive integer numbers"
      end_if;
    elif meshvars = {XMesh, YMesh, ZMesh} then
      if testtype(x, Type::ListOf(Type::PosInt, 3, 3)) then
        XMesh = x[1], YMesh = x[2], ZMesh = x[3]
      else
        "expecting a list of 3 positive integer numbers"
      end_if;
    else
      if nops(meshvars) > 0 then
        context(hold(error)("'Mesh' must be set directly in object"))
      else
        context(hold(error)("attribute 'Mesh' not applicable for plot::".typ))
      end_if
    end_if;
  end_proc:
plot::readMesh :=
  proc(obj, o, x)
    local meshvars;
  begin
    meshvars := plot::legalAttributes[obj::dom::objType][3] intersect
                {UMesh, VMesh, XMesh, YMesh, ZMesh};
    meshvars := sort([op(map(meshvars, expr2text))]);
    meshvars := map(meshvars, x->slot(obj, x));
    if meshvars = [] or has(meshvars, FAIL) then
      meshvars := FAIL;
    end_if;
    if nops(meshvars) = 1 then
      meshvars := op(meshvars);
    else
      meshvars
    end_if;
  end_proc:
// library interface for *Submesh attributes
plot::libSubmesh :=
  proc(typ, o, x)
    local submeshvars;
  begin
    submeshvars := plot::legalAttributes[typ][3] intersect
                {USubmesh, VSubmesh, XSubmesh, YSubmesh};
    if submeshvars = {USubmesh} then
      if testtype(x, Type::NonNegInt) then
        USubmesh = x
      else
        "expecting a nonnegative integer number"
      end_if;
    elif submeshvars = {USubmesh, VSubmesh} then
      if testtype(x, Type::ListOf(Type::NonNegInt, 2, 2)) then
        USubmesh = x[1], VSubmesh = x[2]
      else
        "expecting a list of 2 nonnegative integer numbers"
      end_if;
    elif submeshvars = {XSubmesh} then
      if testtype(x, Type::NonNegInt) then
        XSubmesh = x
      else
        "expecting a nonnegative integer number"
      end_if;
    elif submeshvars = {XSubmesh, YSubmesh} then
      if testtype(x, Type::ListOf(Type::NonNegInt, 2, 2)) then
        XSubmesh = x[1], YSubmesh = x[2]
      else
        "expecting a list of 2 nonnegative integer numbers"
      end_if;
    else
      if nops(submeshvars) > 0 then
        context(hold(error)("'Submesh' must be set directly in object"))
      else
        context(hold(error)("attribute 'Submesh' not applicable for plot::".typ))
      end_if
    end_if;
  end_proc:
plot::readSubmesh :=
  proc(obj, o, x)
    local submeshvars;
  begin
    submeshvars := plot::legalAttributes[obj::dom::objType][3] intersect
                {USubmesh, VSubmesh, XSubmesh, YSubmesh};
    submeshvars := sort([op(map(submeshvars, expr2text))]);
    submeshvars := map(submeshvars, x->slot(obj, x));
    if submeshvars = [] or has(submeshvars, FAIL) then
      submeshvars := FAIL;
    end_if;
    if nops(submeshvars) = 1 then
      submeshvars := op(submeshvars);
    else
      submeshvars
    end_if;
  end_proc:

// special case, libVisibleBefore sets serveral different attributes
plot::libVisibleBefore :=
(typ, o, x) ->
  if testtype(float(x), Type::Real) then
    TimeBegin=float(x), TimeEnd=float(x),
    VisibleBeforeBegin=TRUE, VisibleAfterEnd=FALSE
  else
    "expecting a real number"
  end_if:
plot::readVisibleBefore :=
  (obj, o, t) ->
  if contains(t, TimeBegin) and contains(t, TimeEnd) and
    t[TimeBegin]=t[TimeEnd] and
    t[VisibleBeforeBegin]=TRUE and t[VisibleAfterEnd]=FALSE then
    t[TimeBegin]:
  else
    FAIL
  end_if:

// special case, libVisibleAfter sets serveral different attributes
plot::libVisibleAfter :=
(typ, o, x) ->
  if testtype(float(x), Type::Real) then
    TimeBegin=float(x), TimeEnd=float(x),
    VisibleBeforeBegin=FALSE, VisibleAfterEnd=TRUE
  else
    "expecting a real number"
  end_if:
plot::readVisibleAfter := 
  (obj, o, t) ->
  if contains(t, TimeBegin) and contains(t, TimeEnd) and
    t[TimeBegin]=t[TimeEnd] and
    t[VisibleBeforeBegin]=FALSE and t[VisibleAfterEnd]=TRUE then
    t[TimeBegin]:
  else
    FAIL
  end_if:

// special case, libVisibleFromTo sets serveral different attributes
plot::libVisibleFromTo :=
(typ, o, x) ->
  if testtype(float(x), "_range") and
    map({op(float(x))}, testtype, Type::Real) = {TRUE} then
    TimeBegin=float(op(x, 1)), TimeEnd=float(op(x, 2)),
    VisibleBeforeBegin=FALSE, VisibleAfterEnd=FALSE
  else
    "expecting a range of real numbers"
  end_if:
plot::readVisibleFromTo := 
  (obj, o, t) ->
  if contains(t, TimeBegin) and contains(t, TimeEnd) and
    t[VisibleBeforeBegin]=FALSE and t[VisibleAfterEnd]=FALSE then
    t[TimeBegin]..t[TimeEnd]:
  else
    FAIL
  end_if:



// Legend:  set LegendText, LegendEntry, LegendVisible simultanously
plot::libLegend :=
(typ, o, x) ->
  if testtype(x, DOM_STRING) then
    LegendText = EA::namedText("LegendText", x),
    LegendEntry = TRUE, LegendVisible = TRUE
  else
    "expecting a string"
  end_if:
plot::readLegend :=
  (obj, o, t) ->
  if contains(t, LegendText) and
    t[LegendEntry] = TRUE and t[LegendVisible] = TRUE then
    plot::ExprAttribute::getExpr(t[LegendText]):
  else
    FAIL
  end_if:


/* generator for check functions of the alias type:
 *  attributename 'aliasName' may be used instead of 'existingName'
 * [aliasName, existingName]
 */
plot::libAlias :=
proc(l : DOM_LIST)
  option escape;
begin
  (typ, o, x) -> plot::attributes[l[2]][3](typ, l[2], x)
end_proc:
// the read function corresponding to plot::libAlias
plot::readAlias :=
proc(l : DOM_LIST)
  option escape;
begin
  (typ, o, x) -> if contains(x, l[2]) then
                   plot::ExprAttribute::getExpr(x[l[2]])
                 else
                   FAIL
                 end
end_proc:
   

// generator for check functions of the type :
// one library interface sets serveral attributes with the same value
// [AxesVisible, {admissible values} or checkFunction,
//               [XAxisVisible, YAxisVisible, ZAxisVisible]]
plot::libOneToMore :=
  proc(l : DOM_LIST)
    option escape;
  begin
    funcenv(
    proc(typ, o, x)
      local res;
    begin
      if domtype(l[2]) = DOM_SET then
        if contains(l[2], x) then
          if contains(plot::all2dObjects, typ) then
            map(op(l[3], 1..2), _equal, x)
          else
            map(op(l[3]), _equal, x)
          end_if
        else
          "value from ".expr2text(l[2])." expected"
        end_if;
      else
        res := l[2](typ, l[3][1], x);
        if domtype(res) <> DOM_STRING then
          if contains(plot::all2dObjects, typ) then
            map(op(l[3], 1..2), _equal, op(res,2))
          else
            map(op(l[3]), _equal, op(res, 2))
          end_if;
        else
          //res was an error string
          res
        end_if;
      end_if;
    end_proc,
            NIL, table("accepted" = l[2]));
  end_proc:
// read function corresponding to plot::libOneToMore
plot::readOneToMore :=
  proc(l : DOM_LIST)
    option escape;
  begin
    (obj, o, t) ->
    if obj::dom::dimension = 2 then
      if nops({t[l[2][1]], t[l[2][2]]}) = 1 then
        (plot::ExprAttribute::getExpr@op)({t[l[2][1]], t[l[2][2]]})
      else
        FAIL
      end_if;
    else
      if nops({t[l[2][1]], t[l[2][2]], t[l[2][3]]}) = 1 then
        (plot::ExprAttribute::getExpr@op)({t[l[2][1]], t[l[2][2]], t[l[2][3]]})
      else
        FAIL
      end_if;
    end_if:
  end_proc:
plot::libAxesVisible :=
  plot::libOneToMore([AxesVisible, {TRUE, FALSE},
                      [XAxisVisible, YAxisVisible, ZAxisVisible]]):
plot::readAxesVisible :=
  plot::readOneToMore([AxesVisible,
                       [XAxisVisible, YAxisVisible, ZAxisVisible]]):

plot::libTicksVisible :=
  plot::libOneToMore([TicksVisible, {TRUE, FALSE},
                      [XTicksVisible, YTicksVisible, ZTicksVisible]]):
plot::readTicksVisible :=
  plot::readOneToMore([TicksVisible,
                       [XTicksVisible, YTicksVisible, ZTicksVisible]]):

plot::libAxesTitleAlignment :=
  plot::libOneToMore([AxesTitleAlignment, {Begin, Center, End},
                      [XAxisTitleAlignment, YAxisTitleAlignment,
                       ZAxisTitleAlignment]]):
plot::readAxesTitleAlignment :=
  plot::readOneToMore([AxesTitleAlignment,
                       [XAxisTitleAlignment, YAxisTitleAlignment,
                        ZAxisTitleAlignment]]):

plot::libGridVisible :=
  plot::libOneToMore([GridVisible, {TRUE, FALSE},
                      [XGridVisible, YGridVisible, ZGridVisible]]):
plot::readGridVisible :=
  plot::readOneToMore([GridVisible,
                       [XGridVisible, YGridVisible, ZGridVisible]]):

plot::libSubgridVisible :=
  plot::libOneToMore([SubgridVisible, {TRUE, FALSE},
                      [XSubgridVisible, YSubgridVisible, ZSubgridVisible]]):
plot::readSubgridVisible :=
  plot::readOneToMore([SubgridVisible,
                       [XSubgridVisible, YSubgridVisible, ZSubgridVisible]]):

plot::libTicksLabelStyle :=
  plot::libOneToMore([TicksLabelStyle,
                      {Horizontal, Vertical, Diagonal, Shifted},
                      [XTicksLabelStyle, YTicksLabelStyle, ZTicksLabelStyle]]):
plot::readTicksLabelStyle :=
  plot::readOneToMore([TicksLabelStyle,
                       [XTicksLabelStyle, YTicksLabelStyle, ZTicksLabelStyle]]):

plot::libTicksLabelsVisible :=
  plot::libOneToMore([TicksLabelsVisible,
                      {TRUE, FALSE},
                      [XTicksLabelsVisible, YTicksLabelsVisible, ZTicksLabelsVisible]]):
plot::readTicksLabelsVisible :=
  plot::readOneToMore([TicksLabelsVisible,
                       [XTicksLabelsVisible, YTicksLabelsVisible, ZTicksLabelsVisible]]):

plot::libTicksNumber :=
  plot::libOneToMore([TicksNumber, {None, Low, Normal, High},
                      [XTicksNumber, YTicksNumber, ZTicksNumber]]):
plot::readTicksNumber :=
  plot::readOneToMore([TicksNumber,
                       [XTicksNumber, YTicksNumber, ZTicksNumber]]):

plot::libTicksBetween :=
  plot::libOneToMore([TicksBetween, plot::checkNonNegInteger,
                      [XTicksBetween, YTicksBetween, ZTicksBetween]]):
plot::readTicksBetween :=
  plot::readOneToMore([TicksBetween,
                       [XTicksBetween, YTicksBetween, ZTicksBetween]]):

plot::libTicksAnchor :=
  plot::libOneToMore([TicksAnchor, plot::elementVExpr,
                      [XTicksAnchor, YTicksAnchor, ZTicksAnchor]]):
plot::readTicksAnchor :=
  plot::readOneToMore([TicksAnchor,
                       [XTicksAnchor, YTicksAnchor, ZTicksAnchor]]):

plot::libTicksDistance :=
  plot::libOneToMore([TicksDistance, plot::elementVExpr,
                      [XTicksDistance, YTicksDistance, ZTicksDistance]]):
plot::readTicksDistance :=
  plot::readOneToMore([TicksDistance,
                       [XTicksDistance, YTicksDistance, ZTicksDistance]]):

plot::libMargins :=
  proc(l : DOM_LIST)
    option escape;
  begin
    funcenv(
    proc(typ, o, x)
      local res;
    begin
      res := l[2](typ, l[3][1], x);
      if domtype(res) <> DOM_STRING then
        map(op(l[3]), _equal, op(res, 2))
      else
        //res was an error string
        res
      end_if;
    end_proc,
            NIL, table("accepted"=l[2]));
  end_proc:
// read function corresponding to plot::libOneToMore
plot::readMargins :=
proc(l : DOM_LIST)
    option escape;
  begin
    (obj, o, t) ->
    if nops({t[l[2][1]], t[l[2][2]], t[l[2][3]]}) = 1 then
      t[l[2][1]];
    else
      FAIL
    end_if;
  end_proc:
plot::libMargin :=
  plot::libMargins([Margin, plot::checkNonNegOutputSize,
                      [TopMargin, BottomMargin, LeftMargin, RightMargin]]):
plot::readMargin :=
  plot::readMargins([Margin,
                       [TopMargin, BottomMargin, LeftMargin, RightMargin]]):

// used for AxesOrigin, AxesTitles, TitlePosition
plot::libListOfExpr :=
  proc(l : DOM_LIST)
    option escape;
  begin
    funcenv(
    proc(typ, o, x)
      local res1, res2, res3;
            begin
      if nops(x) < 2 then
        return("at least 2 list entries expected")
      end_if;
      res1 := l[2](typ, l[3][1], x[1]);
      res2 := l[2](typ, l[3][2], x[2]);
      if domtype(res1) = DOM_STRING then
        //res1 was an error string
        return(res1);
      elif domtype(res2) = DOM_STRING then
        //res2 was an error string
        return(res2);
      end_if;
      if contains(plot::all2dObjects, typ) or nops(l[3]) = 2 or
            typ = "Canvas" and nops(x) = 2 then
          res1, res2
      else
        // 3D
        if nops(x) < 3 then
          return("3 list entries expected")
        end_if;
        res3 := l[2](typ, l[3][3], x[3]);
        if domtype(res3) = DOM_STRING then
        //res3 was an error string
          return(res3);
        end_if;
          res1, res2, res3
      end_if;
    end_proc,
            NIL, table("acceptingListOf" = l[2]));
  end_proc:
plot::readListOfExpr :=
  proc(l : DOM_LIST)
    option escape; //  KAPUTT
  begin
    (obj, o, t) ->
    if obj::dom::dimension = 2 or nops(l[2]) = 2 or
       obj = plot::Canvas and t[l[2][3]] = FAIL then
      if contains(t, l[2][1]) and contains(t, l[2][2]) then
        [t[l[2][1]], t[l[2][2]]]
      else
        FAIL
      end_if;
    else
      if contains(t, l[2][1]) and contains(t, l[2][2]) and
	 contains(t, l[2][3]) then
        [t[l[2][1]], t[l[2][2]], t[l[2][3]]]
      else
        FAIL
      end_if;
    end_if:
  end_proc:
plot::libTitlePosition :=
  plot::libListOfExpr([TitlePosition, plot::elementExpr,
                      [TitlePositionX, TitlePositionY, TitlePositionZ]]):
plot::readTitlePosition :=
  plot::readListOfExpr([TitlePosition,
                      [TitlePositionX, TitlePositionY, TitlePositionZ]]):
plot::libAxesTitles :=
  plot::libListOfExpr([AxesTitles, plot::elementText,
                      [XAxisTitle, YAxisTitle, ZAxisTitle]]):
plot::readAxesTitles :=
  plot::readListOfExpr([AxesTitles,
                      [XAxisTitle, YAxisTitle, ZAxisTitle]]):
plot::libAxesOrigin :=
  plot::libListOfExpr([AxesOrigin, plot::elementVExpr,
                      [AxesOriginX, AxesOriginY, AxesOriginZ]]):
plot::readAxesOrigin :=
  plot::readListOfExpr([AxesOrigin,
                      [AxesOriginX, AxesOriginY, AxesOriginZ]]):

// vector-values attributes like Center, Position, ..
plot::libListOfOptExprXYZ :=
(typ, o, x) ->
  if testtype(x, Type::ListOf(Type::Arithmetical)) or
    x::dom::hasProp(Cat::Matrix)=TRUE then
    if contains(plot::all2dObjects, typ) then
      if nops(x) = 2 then
        o."X"=EA::optExpr("".o."X", [op(x)][1]),
        o."Y"=EA::optExpr("".o."Y", [op(x)][2])
      else
        "expecting a list with 2 entries"
      end_if;
    elif nops(x) = 3 then
      // 3D, we need 3 list entries
        o."X"=EA::optExpr("".o."X", [op(x)][1]),
        o."Y"=EA::optExpr("".o."Y", [op(x)][2]),
        o."Z"=EA::optExpr("".o."Z", [op(x)][3])
    else
    "expecting a list with 3 entries"      
    end_if
  else
    "expecting a list of arithmetical expressions"
  end_if:
plot::readListOfOptExprXYZ :=
(obj, o, t) -> if obj::dom::dimension = 2 then
                 if contains(t, o."X") and contains(t, o."Y") then
                   [t[o."X"], t[o."Y"]]
                 else
                   FAIL
                 end
               else
                 if contains(t, o."X") and contains(t, o."Y") and
                   contains(t, o."Z")  then
                   [t[o."X"], t[o."Y"], t[o."Z"]]
                 else
                   FAIL
                 end
               end_if:

// Special case for SemiAxes <-> [SemiAxisX, ...]
plot::libSemiAxes :=
(typ, o, x) ->
  if contains(plot::all2dObjects, typ) then
    if (testtype(x, Type::ListOf(Type::Arithmetical)) or
        x::dom::hasProp(Cat::Matrix)=TRUE ) and nops(x) = 2 then
      SemiAxisX=EA::optExpr("SemiAxisX", [op(x)][1]),
      SemiAxisY=EA::optExpr("SemiAxisY", [op(x)][2])
    else
      "expecting a list of 2 arithmetical expressions"
    end_if;
  else
    if testtype([op(x)], Type::ListOf(Type::Arithmetical)) and nops(x) = 3 then
      SemiAxisX=EA::optExpr("SemiAxisX", [op(x)][1]),
      SemiAxisY=EA::optExpr("SemiAxisY", [op(x)][2]),
      SemiAxisZ=EA::optExpr("SemiAxisZ", [op(x)][3])
    else
      "expecting a list of 3 arithmetical expressions"
    end_if:
  end_if:
plot::readSemiAxes :=
(obj, o, t) -> if contains(t, SemiAxisX) and contains(t, SemiAxisY) then
                 if contains(t, SemiAxisZ) then
                   [t[SemiAxisX], t[SemiAxisY], t[SemiAxisZ]]
                 else
                   [t[SemiAxisX], t[SemiAxisY]]
                 end_if
               else
                 FAIL
               end_if:

// used for XRange, YRange, ZRange, URange, VRange
plot::libRangeOfXYZOptExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (typ, o, x) ->
  if type(x) = "_range" and
    map({op(x)}, testtype, Type::Arithmetical) = {TRUE} then
    if typ <> "Hatch" and hastype(x, stdlib::Infinity) then
      "infinity not allowed in range bounds"
    elif numeric::isless(op(x, 2),op(x, 1))  <> TRUE then
        hold(``).str."Min"=EA::optExpr(str."Min", op(x, 1)),
        hold(``).str."Max"=EA::optExpr(str."Max", op(x, 2))
    else
      "invalid range bounds"
    end_if
  else
    "expecting a range expression"
  end_if;
end_proc:
plot::readRangeOfXYZOptExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (obj, o, t) -> 
  if contains(t, hold(``).str."Min") and contains(t, hold(``).str."Max") then
    (t[hold(``).str."Min"]) .. expr(t[hold(``).str."Max"]):
    map(%, plot::ExprAttribute::getExpr)
  else
    FAIL
  end_if
end_proc:

// used for AngleRange <==> AngleBegin, AngleEnd
plot::libRangeOfOptExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (typ, o, x) ->
  if type(x) = "_range" and
    map({op(x)}, testtype, Type::Arithmetical) = {TRUE} then
    if numeric::isless(op(x, 2),op(x, 1))  <> TRUE then
        hold(``).str."Begin"=EA::optExpr(str."Begin", op(x, 1)),
        hold(``).str."End"=EA::optExpr(str."End", op(x, 2))
    else
      "invalid range bounds"
    end_if
  else
    "expecting a range expression"
  end_if;
end_proc:
plot::readRangeOfOptExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (obj, o, t) -> 
  if contains(t, hold(``).str."Begin") and contains(t, hold(``).str."End") then
    (t[hold(``).str."Begin"]) .. expr(t[hold(``).str."End"]):
    map(%, plot::ExprAttribute::getExpr)
  else
    FAIL
  end_if
end_proc:

// used for ParameterRange
plot::libRangeOfXYZExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (typ, o, x) ->
  if type(x) = "_range" and
    map({op(x)}, z -> testtype(float(z), Type::Real)) = {TRUE} then
    hold(``).str."Begin"=EA::namedVExpr(str."Begin", op(x, 1)),
    hold(``).str."End"=EA::namedVExpr(str."End", op(x, 2))
  else
    "expecting a range of numerical values"
  end_if;
end_proc:
plot::readRangeOfXYZExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (obj, o, t) ->
  if contains(t, hold(``).str."Begin") and contains(t, hold(``).str."End") then
    (t[hold(``).str."Begin"]) .. expr(t[hold(``).str."End"]):
    map(%, plot::ExprAttribute::getExpr)
  else
    FAIL
  end_if
end_proc:

// used for ViewingBox?Range
plot::libRangeOfVBExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (typ, o, x) ->
  if type(x) = "_range" and
    map({op(x)}, testtype, Type::Arithmetical) = {TRUE} then
    hold(``).str."Min"=EA::namedVExpr(str."Min", op(x, 1)),
    hold(``).str."Max"=EA::namedVExpr(str."Max", op(x, 2))
  else
    "expecting a range expression"
  end_if;
end_proc:
plot::readRangeOfVBExpr :=
proc(str : DOM_STRING)
  option escape;
begin
  (obj, o, t) ->
  if contains(t, hold(``).str."Min") and contains(t, hold(``).str."Max") then
    (t[hold(``).str."Min"]) .. expr(t[hold(``).str."Max"]):
    map(%, plot::ExprAttribute::getExpr)
  else
    FAIL
  end_if
end_proc:

// used for TimeRange
plot::libRangeOfXYZ :=
proc(str : DOM_STRING)
  option escape;
begin
  (typ, o, x) ->
  if type(x) = "_range" and
    map({op(x)}, testtype, Type::Arithmetical) = {TRUE} then
    if numeric::isless(op(x, 2),op(x, 1))  <> TRUE then
        hold(``).str."Begin"=op(x, 1), hold(``).str."End"=op(x, 2)
    else
      "invalid range bounds"
    end_if
  else
    "expecting a range expression"
  end_if;
end_proc:
plot::readRangeOfXYZ :=
proc(str : DOM_STRING)
  option escape;
begin
  (obj, o, t) ->
  if contains(t, hold(``).str."Begin") and contains(t, hold(``).str."End") then
    t[hold(``).str."Begin"] .. t[hold(``).str."End"]:
  else
    FAIL
  end_if
end_proc:

// used for ViewingBox
plot::libViewingBox :=
  proc(typ, o, x)
    local i;
  begin
    if domtype(x) = DOM_LIST and contains({2, 3}, nops(x)) then
    // we allow the identifier Automatic and ranges (possibly with  Automatic)
      for i from 1 to nops(x) do
        if type(x[i]) <> "_range" and x[i] <> Automatic then
          return("expecting a list of range expression")
        end_if;
        if x[i] = Automatic then
          x[i] := Automatic..Automatic
        end_if;
      end_for;
      if nops(x) = 2 then
        if not contains(plot::all3dObjects, typ) then
          return(ViewingBoxXMin =
                 EA::namedVExpr("ViewingBoxXMin", op(x, [1, 1])),
                 ViewingBoxXMax =
                 EA::namedVExpr("ViewingBoxXMax", op(x, [1, 2])),
                 ViewingBoxYMin =
                 EA::namedVExpr("ViewingBoxYMin", op(x, [2, 1])),
                 ViewingBoxYMax =
                 EA::namedVExpr("ViewingBoxYMax", op(x, [2, 2])))
        else
          return("expecting a list of 3 range expressions")
        end_if;
      else // nops(x) = 3
        if not contains(plot::all2dObjects, typ) then
          return(ViewingBoxXMin =
                 EA::namedVExpr("ViewingBoxXMin", op(x, [1, 1])),
                 ViewingBoxXMax =
                 EA::namedVExpr("ViewingBoxXMax", op(x, [1, 2])),
                 ViewingBoxYMin =
                 EA::namedVExpr("ViewingBoxYMin", op(x, [2, 1])),
                 ViewingBoxYMax =
                 EA::namedVExpr("ViewingBoxYMax", op(x, [2, 2])),
                 ViewingBoxZMin =
                 EA::namedVExpr("ViewingBoxZMin", op(x, [3, 1])),
                 ViewingBoxZMax =
                 EA::namedVExpr("ViewingBoxZMax", op(x, [3, 2])))
        else
          return("expecting a list of 2 range expressions")
        end_if;
      end_if;
    else
      "expecting a list of range expression"
    end_if
  end_proc:
plot::readViewingBox :=
  (obj, o, t) ->
  if contains(t, ViewingBoxXMin) and contains(t, ViewingBoxXMax) and 
    contains(t, ViewingBoxYMin) and contains(t, ViewingBoxYMax) then
    ([map(t[ViewingBoxXMin] .. t[ViewingBoxXMax], plot::ExprAttribute::getExpr),
     map(t[ViewingBoxYMin] .. t[ViewingBoxYMax], plot::ExprAttribute::getExpr),
     if contains(t, ViewingBoxZMin) and contains(t, ViewingBoxZMax) then
       map(t[ViewingBoxYMin] .. t[ViewingBoxYMax], plot::ExprAttribute::getExpr)
     else
       null()
     end_if]):
  else
    FAIL
  end_if:

//----------------------------------------------------------------------------
// Functions for hint combining
// Arguments:   attribute    -- the attribute name, like AxesInFront
//              [hint1, ...] -- list of hints given by children;
//                              has at least one entry
//----------------------------------------------------------------------------

// combine boolean values with _or
plot::combineBooleanOr := (attr, list) -> attr=_or(op(list)):

// combine boolean values with _or
plot::combineBooleanAnd := (attr, list) -> attr=_and(op(list)):

// combine hints by selecting the first one
plot::combineTakeFirst := (attr, list) -> attr=list[1]:

// combine hints by taking minimum value
plot::combineMin := (attr, list) -> attr=min(op(list)):

// combine hints by taking maximum value
plot::combineMax := (attr, list) -> attr=max(op(list)):

// combine hints by taking minimum value
plot::combineMinOrAutomatic :=
(attr, list) -> if contains(list, Automatic) <> 0 then
                  attr=Automatic
                else
                  attr=min(op(list))
                end:

// combine hints by taking maximum value
plot::combineMaxOrAutomatic := 
(attr, list) -> if contains(list, Automatic) <> 0 then
                  attr=Automatic
                else
                  attr=max(op(list))
                end:



plot::checkOutputOptions :=
  proc(typ, o, x)
    local tab, dummy, equations, tmp;
  begin
    if domtype(x) = DOM_LIST then
      [equations, tmp, dummy] := split(x, testtype, "_equal");
      tab := table(equations, map(tmp, _equal, TRUE));

      x := [];
      if contains(tab, ReduceTo256Colors) then
        if contains({TRUE, FALSE}, tab[ReduceTo256Colors]) then
          x := x.[ReduceTo256Colors = tab[ReduceTo256Colors]];
          delete tab[ReduceTo256Colors];
        else
          return("entry ReduceTo256Colors: TRUE or FALSE expected")
        end_if;
      end_if;
      if contains(tab, DotsPerInch) then
        if testtype(tab[DotsPerInch], Type::PosInt) then
          x := x.[DotsPerInch = tab[DotsPerInch]];
          delete tab[DotsPerInch];
        else
          return("entry DotsPerInch: positive integer expected")
        end_if;
      end_if;
      if contains(tab, FramesPerSecond) then
        if testtype(tab[FramesPerSecond], Type::PosInt) then
          x := x.[FramesPerSecond = tab[FramesPerSecond]];
          delete tab[FramesPerSecond];
        else
          return("entry FramesPerSecond: positive integer expected")
        end_if;
      end_if;
      if contains(tab, Quality) then
        if testtype(tab[Quality], Type::PosInt) and
          tab[Quality] > 0 and tab[Quality] <= 100 then
          x := x.[Quality = tab[Quality]];
          delete tab[Quality];
        else
          return("entry Quality: positive integer between 1 and 100 expected")
        end_if;
      end_if;
      if contains(tab, JPEGMode) then
        if contains({0, 1, 2}, tab[JPEGMode]) then
          x := x.[JPEGMode = tab[JPEGMode]];
          delete tab[JPEGMode];
        else
          return("entry JPEGMode: 0, 1, or 2 expected")
        end_if;
      end_if;
      if contains(tab, EPSMode) then
        if contains({0, 1}, tab[EPSMode]) then
          x := x.[EPSMode = tab[EPSMode]];
          delete tab[EPSMode];
        else
          return("entry EPSMode: 0 or 1 expected")
        end_if;
      end_if;
      if contains(tab, AVIMode) then
        if contains({0, 1, 2}, tab[AVIMode]) then
          x := x.[AVIMode = tab[AVIMode]];
          delete tab[AVIMode];
        else
          return("entry AVIMode: 0, 1, or 2 expected")
        end_if;
      end_if;
      if contains(tab, WMFMode) then
        if contains({0, 1, 2}, tab[WMFMode]) then
          x := x.[WMFMode = tab[WMFMode]];
          delete tab[WMFMode];
        else
          return("entry WMFMode: 0, 1, or 2 expected")
        end_if;
      end_if;
      if contains(tab, PlotAt) then
        if testtype(tab[PlotAt], Type::ListOf(Type::Real)) then
          x := x.[PlotAt = tab[PlotAt]];
          delete tab[PlotAt];
        else
          return("entry PlotAt: list of reals expected")
        end_if;
      end_if;
      if tab <> table() then
        "unknown output options: ".expr2text(op(tab))
      else
        o=x:
      end_if;
    else
      "list expected"
    end_if;
  end:

//============================================================================
//           THE BIG TABLE CONTAINING ALL PLOT ATTRIBUTES
//============================================================================
plot::attributes:=
table
 (
//----------------------------------------------------------------------------
// Only used in plot::new 
//----------------------------------------------------------------------------
  OutputFile =
      ["Optional", "",       plot::elementOptTextExpr,
       {},   FAIL,
       FAIL],
  OutputOptions =
      ["Optional", [],   plot::checkOutputOptions,
       {},   FAIL,
       FAIL],
//----------------------------------------------------------------------------
// Canvas options
//----------------------------------------------------------------------------
  Name =
      ["Optional", NIL,        plot::checkText,
       plot::allObjectTypes minus {"View2d", "View3d"},   FAIL,
       ["Definition", "XMLAttribute", "String",
        "The name of the object.", FALSE]],
  Layout =
      ["Optional", Tabular,   {Tabular, Horizontal, Vertical, Absolute, Relative},
       {"Canvas"},             plot::combineTakeFirst,
       ["Layout", "XMLAttribute", "Layout",
        "Layout of the scenes in the canvas.", FALSE]],
  Rows =
      ["Optional",  0,  plot::checkNonNegInteger,
       {"Canvas"},      FAIL,
       ["Animation", "XMLAttribute", "PosInt",
        "Number of rows for tabular scene positioning.", TRUE]],
  Columns =
      ["Optional",  0,  plot::checkNonNegInteger,
       {"Canvas"},      FAIL,
       ["Animation", "XMLAttribute", "PosInt",
        "Number of columns for tabular scene positioning.", TRUE]],
  Spacing =
      ["Optional", 1.0, plot::checkNonNegOutputSize,
       {"Canvas"},      FAIL,
       ["Layout", "XMLAttribute", "NonNegSize",
        "Space between scenes in automatic layout.", FALSE]],
  OutputUnits =
      ["Optional", unit::mm,   {unit::mm, unit::cm, unit::dm, unit::m, unit::km, unit::inch, unit::pt},
       {"Canvas"},             FAIL,
       ["Layout", "XMLAttribute", "OutputUnits",
        "Units in which output sizes are displayed.", FALSE]],
  Width =
      ["Inherited", 120,        plot::checkPosOutputSize,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineMax,
       ["Layout", "XMLAttribute", "PosSize",
        "Width.", FALSE]],
  Height =
      ["Inherited", 80,         plot::checkPosOutputSize,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineMax,
       ["Layout", "XMLAttribute", "PosSize",
        "Height.", FALSE]],
  TopMargin =
      ["Inherited", 1,        plot::checkNonNegOutputSize,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineMax,
       ["Layout", "XMLAttribute", "NonNegSize",
        "Top margin height.", FALSE]],
  BottomMargin =
      ["Inherited", 1,        plot::checkNonNegOutputSize,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineMax,
       ["Layout", "XMLAttribute", "NonNegSize",
        "Bottom margin height.", FALSE]],
  LeftMargin =
      ["Inherited", 1,        plot::checkNonNegOutputSize,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineMax,
       ["Layout", "XMLAttribute", "NonNegSize",
        "Left margin width.", FALSE]],
  RightMargin =
      ["Inherited", 1,        plot::checkNonNegOutputSize,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineMax,
       ["Layout", "XMLAttribute", "NonNegSize",
        "Right margin width.", FALSE]],
  Margin =
      ["Library", NIL,         plot::libMargin,
       {"Canvas", "Scene2d", "Scene3d"}, plot::readMargin,
       [[{TopMargin, BottomMargin, LeftMargin, RightMargin}]]],
  BorderWidth =
      ["Inherited", 0,          plot::checkNonNegOutputSize,
       {"Canvas", "Scene2d", "Scene3d"}, plot::combineMax,
       ["Layout", "XMLAttribute", "NonNegSize",
        "Border width.", FALSE]],
  BorderColor =
      ["Inherited", RGB::Grey50, plot::checkColor,
       {"Canvas", "Scene2d", "Scene3d"}, plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Color",
        "Border color.", FALSE]],
  BackgroundColor =
      ["Inherited", RGB::White, plot::checkColor,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Color",
        "Background color.", FALSE]],
  Header =
      ["Optional", NIL,        plot::elementText,
       {"Canvas", "Scene2d", "Scene3d"}, plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "Title","", FALSE]],
  Footer =
      ["Optional", NIL,        plot::elementText,
       {"Canvas", "Scene2d", "Scene3d"}, plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "Title", "", FALSE]],
  HeaderAlignment =
      ["Inherited", Center,     {Left, Center, Right}, 
       {"Canvas", "Scene2d", "Scene3d"}, plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "HorAlignment", "", FALSE]],
  FooterAlignment =
      ["Inherited", Center,     {Left, Center, Right},
       {"Canvas", "Scene2d", "Scene3d"}, plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "HorAlignment", "", FALSE]],
  HeaderFont =
      ["Inherited", "'sans-serif' 12", plot::checkFont,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Font", "Canvas header font.", FALSE]],
  FooterFont =
      ["Inherited", "'sans-serif' 12", plot::checkFont,
       {"Canvas", "Scene2d", "Scene3d"},             plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Font", "Canvas footer font.", FALSE]],
  AnimationStyle =
      ["Inherited", RunOnce,     {Loop, RunOnce, BackAndForth},
       {"Canvas"}, plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "AnimationStyle", "", FALSE]],
  InitialTime =
      ["Optional",  NIL,  plot::checkReal, {"Canvas"},
       FAIL,
       ["Animation", "XMLAttribute", "Float",
        "Initial animation time.", TRUE]],
  CurrentTime =
      ["Optional",  NIL,  plot::checkReal, {"Canvas"},
       FAIL, FAIL],
  AutoPlay =
      ["Optional", TRUE,          {TRUE, FALSE},
       {"Canvas"},
       FAIL, FAIL],

  
//----------------------------------------------------------------------------
// Scene{2,3}d options
//----------------------------------------------------------------------------
  Bottom =
      ["Optional",  0,            plot::checkOutputPos,
       {"Scene2d", "Scene3d"},    plot::combineMax,
       ["Layout", "XMLAttribute", "ScenePos", "", FALSE]],
  Left =
      ["Optional",  0,            plot::checkOutputPos,
       {"Scene2d", "Scene3d"},    plot::combineMax,
       ["Layout", "XMLAttribute", "ScenePos", "", FALSE]],
  BackgroundTransparent =
      ["Inherited", FALSE,        {TRUE, FALSE},
       {"Scene2d", "Scene3d"},    plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Bool", "", FALSE]],
  LegendVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"Scene2d", "Scene3d"},    plot::combineBooleanOr,
       ["Annotation", "XMLAttribute", "Bool", "", FALSE]],
  LegendPlacement =
      ["Inherited", Bottom,       {Top, Bottom},
       {"Scene2d", "Scene3d"},    plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "LegendPlacement", "", FALSE]],
  LegendAlignment =
      ["Inherited", Center,       {Left, Center, Right},
       {"Scene2d", "Scene3d"},    plot::combineTakeFirst,
       ["Annotation", "XMLAttribute", "HorAlignment", "", FALSE]],
  LegendFont =
      ["Inherited", "'sans-serif' 8", plot::checkFont,
       {"Scene2d", "Scene3d"},    plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Font", "Legend font.", FALSE]],
  YXRatio =
      ["Inherited", 1,        plot::checkNonNegReal,
       {"Scene3d"},    plot::combineTakeFirst,
       ["Style", "XMLAttribute", "PosFloat", "", FALSE]],

//----------------------------------------------------------------------------
// Scene3d-only options
//----------------------------------------------------------------------------
 
  ZXRatio =
      ["Inherited", 2/3,        plot::checkNonNegReal,
       {"Scene3d"},    plot::combineTakeFirst,
       ["Style", "XMLAttribute", "PosFloat", "", FALSE]],
  BackgroundStyle =
      ["Inherited", Flat,  {Flat, LeftRight, TopBottom, Pyramid},
       {"Scene3d"},             plot::combineTakeFirst,
       ["Style", "XMLAttribute", "BackgroundStyle", "", FALSE]],
  BackgroundColor2 =
      ["Inherited",  RGB::Grey75, plot::checkColor,
       {"Scene3d"},             plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Color", "", FALSE]],
  Lighting =
      ["Inherited",  Automatic, {None, Automatic, Explicit},
       {"Scene3d"},             plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Lighting", "", FALSE]],
       
//----------------------------------------------------------------------------
// CoordinateSystem{2,3}d options
//----------------------------------------------------------------------------

  Scaling =
      ["Inherited", Unconstrained,   {Constrained, Unconstrained, Automatic},
       {"CoordinateSystem2d", "CoordinateSystem3d"}, plot::combineTakeFirst,
       ["Style", "XMLAttribute", "Scaling", "", FALSE]],
  CoordinateType =
      ["Inherited", LinLin,
       {LogLog, LinLog, LogLin, LinLin},
       {"CoordinateSystem2d"},
       plot::combineTakeFirst,
// problem here:  different types in different contexts
       ["Definition", "XMLAttribute", "CoordinateType", "", TRUE]
       ],
  Axes =
      ["Inherited", Automatic,   {Boxed, Frame, Origin, Automatic, None},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "Axes", "", FALSE]],
  ViewingBoxXMin =
      ["Optional",  Automatic,           plot::elementViewingBoxVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineMinOrAutomatic,
       ["Definition", "XMLAttribute", "Expression", "", TRUE]],
  ViewingBoxXMax =
      ["Optional",  Automatic,           plot::elementViewingBoxVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineMaxOrAutomatic,
       ["Definition", "XMLAttribute", "Expression", "", TRUE]],
  ViewingBoxXRange =
      ["Library", NIL,         plot::libRangeOfVBExpr("ViewingBoxX"),
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readRangeOfVBExpr("ViewingBoxX"),
       [[ViewingBoxXMin..ViewingBoxXMax]]],
  ViewingBoxYMin =
      ["Optional",  Automatic,           plot::elementViewingBoxVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineMinOrAutomatic,
       ["Definition", "XMLAttribute", "Expression", "", TRUE]],
  ViewingBoxYMax =
      ["Optional",  Automatic,           plot::elementViewingBoxVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineMaxOrAutomatic,
       ["Definition", "XMLAttribute", "Expression", "", TRUE]],
  ViewingBoxYRange =
      ["Library", NIL,         plot::libRangeOfVBExpr("ViewingBoxY"),
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readRangeOfVBExpr("ViewingBoxY"),
       [[ViewingBoxYMin..ViewingBoxYMax]]],
  ViewingBoxZMin =
      ["Optional",  Automatic,           plot::elementViewingBoxVExpr,
       {"CoordinateSystem3d"},
       plot::combineMinOrAutomatic,
       ["Definition", "XMLAttribute", "Expression", "", TRUE]],
  ViewingBoxZMax =
      ["Optional",  Automatic,           plot::elementViewingBoxVExpr,
       {"CoordinateSystem3d"}, 
       plot::combineMaxOrAutomatic,
       ["Definition", "XMLAttribute", "Expression", "", TRUE]],
  ViewingBoxZRange =
      ["Library", NIL,         plot::libRangeOfVBExpr("ViewingBoxZ"),
       {"CoordinateSystem3d"},
       plot::readRangeOfVBExpr("ViewingBoxZ"),
       [[ViewingBoxZMin..ViewingBoxZMax]]],
  ViewingBox =
      ["Library", NIL,             plot::libViewingBox,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readViewingBox,
       [[[ViewingBoxXMin..ViewingBoxXMax, ViewingBoxYMin..ViewingBoxYMax]],
        [[ViewingBoxXMin..ViewingBoxXMax, ViewingBoxYMin..ViewingBoxYMax,
          ViewingBoxZMin..ViewingBoxZMax]]]],
  AxesOriginX =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "ValExpression", "", FALSE]],
  AxesOriginY =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "ValExpression", "", FALSE]],
  AxesOriginZ =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "ValExpression", "", FALSE]],
  AxesOrigin =
      ["Library", NIL,             plot::libAxesOrigin,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readAxesOrigin,
       [[[AxesOriginX, AxesOriginY]],
        [[AxesOriginX, AxesOriginY, AxesOriginZ]]]],
  XAxisVisible =
      ["Inherited", TRUE,          {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Axes", "XMLAttribute", "Bool", "", FALSE]],
  YAxisVisible =
      ["Inherited", TRUE,          {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Axes", "XMLAttribute", "Bool", "", FALSE]],
  ZAxisVisible =
      ["Inherited", TRUE,       {TRUE, FALSE},
       {"CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Axes", "XMLAttribute", "Bool", "", FALSE]],
  AxesVisible =
      ["Library",  NIL,            plot::libAxesVisible,  
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readAxesVisible,
       [[{XAxisVisible, YAxisVisible}],
        [{XAxisVisible, YAxisVisible, ZAxisVisible}]]],
  AxesInFront =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem2d"},  // 2D-only
       plot::combineBooleanOr,
       ["Axes", "XMLAttribute", "Bool", "", FALSE]],
  AxesLineWidth = // Do not change: DIN
      ["Inherited", 0.18,         plot::checkNonNegOutputSize,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "PosSize", "", FALSE]],
  AxesLineColor =
      ["Inherited", RGB::Black,    plot::checkColor,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "Color", "", FALSE]],
  AxesTips =
      ["Inherited", TRUE,          {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Axes", "XMLAttribute", "Bool", "", FALSE]],
  XTicksAnchor =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "ValExpression", "", FALSE]],
  YTicksAnchor =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "ValExpression", "", FALSE]],
  ZTicksAnchor =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "ValExpression", "", FALSE]],
  TicksAnchor =
      ["Library", NIL,           plot::libTicksAnchor,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksAnchor,
       [[{XTicksAnchor, YTicksAnchor}],
        [{XTicksAnchor, YTicksAnchor, ZTicksAnchor}]]],
  XTicksDistance =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "ValExpression", "", FALSE]],
  YTicksDistance =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "ValExpression", "", FALSE]],
  ZTicksDistance =
      ["Optional",  0,           plot::elementVExpr,
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "ValExpression", "", FALSE]],
  TicksDistance =
      ["Library", NIL,           plot::libTicksDistance,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksDistance,
       [[{XTicksDistance, YTicksDistance}],
        [{XTicksDistance, YTicksDistance, ZTicksDistance}]]],
  XTicksAt =
      ["Optional",  NIL,           plot::checkTicksAt,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Ticks", "", FALSE]],
  YTicksAt =
      ["Optional",  NIL,           plot::checkTicksAt,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Ticks", "", FALSE]],
  ZTicksAt =
      ["Optional",  NIL,           plot::checkTicksAt,
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Ticks", "", FALSE]],
  TicksAt =
      ["Library", NIL,            plot::libTicksAt,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksAt,
       [[[XTicksAt, YTicksAt]],
        [[XTicksAt, YTicksAt, ZTicksAt]]]],
  XTicksVisible =
      ["Inherited", TRUE,          {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Tick Marks", "XMLAttribute", "Bool", "", FALSE]],
  YTicksVisible =
      ["Inherited", TRUE,          {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Tick Marks", "XMLAttribute", "Bool", "", FALSE]],
  ZTicksVisible =
      ["Inherited", TRUE,          {TRUE, FALSE},
       {"CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Tick Marks", "XMLAttribute", "Bool", "", FALSE]],
  TicksVisible =
      ["Library", NIL,         plot::libTicksVisible,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksVisible,
       [[{XTicksVisible, YTicksVisible}],
        [{XTicksVisible, YTicksVisible, ZTicksVisible}]]],
  TicksLength =
      ["Inherited", 2,         plot::checkPosOutputSize,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "PosSize", "", FALSE]],
  XTicksLabelStyle =
      ["Inherited", Horizontal,
       {Horizontal, Vertical, Diagonal, Shifted},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "TicksLabelStyle", "", FALSE]],
  XTicksLabelsVisible =
      ["Inherited", TRUE,      {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Bool", "", FALSE]],
  YTicksLabelsVisible =
      ["Inherited", TRUE,      {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Bool", "", FALSE]],
  ZTicksLabelsVisible =
      ["Inherited", TRUE,      {TRUE, FALSE},
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Bool", "", FALSE]],
  TicksLabelsVisible =
      ["Library", NIL,         plot::libTicksLabelsVisible,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksLabelsVisible,
       [[{XTicksLabelsVisible, YTicksLabelsVisible}],
        [{XTicksLabelsVisible, YTicksLabelsVisible, ZTicksLabelsVisible}]]],
  YTicksLabelStyle =
      ["Inherited", Horizontal,
       {Horizontal, Vertical, Diagonal, Shifted},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "TicksLabelStyle", "", FALSE]],
  ZTicksLabelStyle =
      ["Inherited", Horizontal,
       {Horizontal, Vertical, Diagonal, Shifted},
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "TicksLabelStyle", "", FALSE]],
  TicksLabelStyle =
      ["Library", NIL,         plot::libTicksLabelStyle,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksLabelStyle,
       [[{XTicksLabelStyle, YTicksLabelStyle}],
        [{XTicksLabelStyle, YTicksLabelStyle, ZTicksLabelStyle}]]],
  XTicksNumber =
      ["Inherited", Normal,         {None, Low, Normal, High},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "TicksNumber", "", FALSE]],
  YTicksNumber =
      ["Inherited", Normal,         {None, Low, Normal, High},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "TicksNumber", "", FALSE]],
  ZTicksNumber =
      ["Inherited", Normal,         {None, Low, Normal, High},
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "TicksNumber", "", FALSE]],
  TicksNumber =
      ["Library", NIL,         plot::libTicksNumber,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksNumber,
       [[{XTicksNumber, YTicksNumber}],
        [{XTicksNumber, YTicksNumber, ZTicksNumber}]]],
  XTicksBetween =
      ["Inherited", 1,             plot::checkNonNegInteger,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "NonNegInt", "", FALSE]],
  YTicksBetween =
      ["Inherited", 1,             plot::checkNonNegInteger,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "NonNegInt", "", FALSE]],
  ZTicksBetween =
      ["Inherited", 1,             plot::checkNonNegInteger,
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "NonNegInt", "", FALSE]],
  TicksBetween =
      ["Library", NIL,         plot::libTicksBetween,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readTicksBetween,
       [[{XTicksBetween, YTicksBetween}],
        [{XTicksBetween, YTicksBetween, ZTicksBetween}]]],
  TicksLabelFont =
      ["Inherited", "'sans-serif' 8", plot::checkFont,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Tick Marks", "XMLAttribute", "Font",
        "Tick mark labels font.", FALSE]],
  XAxisTitle =
      ["Optional",  "x",           plot::elementText,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "Title", "", FALSE]],
  YAxisTitle =
      ["Optional",  "y",           plot::elementText,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "Title", "", FALSE]],
  ZAxisTitle =
      ["Optional",  "z",           plot::elementText,
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "Title", "", FALSE]],
  AxesTitles =
      ["Library", NIL,             plot::libAxesTitles,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readAxesTitles,
       [[[XAxisTitle, YAxisTitle]],
        [[XAxisTitle, YAxisTitle, ZAxisTitle]]]],
  TubeDiameter =
      ["Inherited",  1.0,         plot::checkPosOutputSize,
       {"Arrow3d", "Line3d"},
       plot::combineTakeFirst,
       [["Style", "Lines"], "XMLAttribute", "PosSize",
        "Diameter of tubular lines shafts.", FALSE]],
  XAxisTitleAlignment =
      ["Inherited", End,           {Begin, Center, End},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "AxisTitleAlignment", "", FALSE]],
  YAxisTitleAlignment =
      ["Inherited", End,           {Begin, Center, End},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "AxisTitleAlignment", "", FALSE]],
  ZAxisTitleAlignment =
      ["Inherited", End,           {Begin, Center, End},
       {"CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "AxisTitleAlignment", "", FALSE]],
  AxesTitleAlignment =
      ["Library", NIL,         plot::libAxesTitleAlignment,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readAxesTitleAlignment,
       [[{XAxisTitleAlignment, YAxisTitleAlignment}],
        [{XAxisTitleAlignment, YAxisTitleAlignment, ZAxisTitleAlignment}]]],
  YAxisTitleOrientation =
      ["Inherited", Horizontal,      {Horizontal, Vertical},
       {"CoordinateSystem2d"},  // 2d-only
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "TitleOrientation", "", FALSE]],
  AxesTitleFont =
      ["Inherited", "'sans-serif' 10", plot::checkFont,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Axes", "XMLAttribute", "Font",
        "Axes title font.", FALSE]],
  XGridVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  YGridVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  ZGridVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  GridVisible =
      ["Library", NIL,         plot::libGridVisible,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readGridVisible,
       [[{XGridVisible, YGridVisible}],
        [{XGridVisible, YGridVisible, ZGridVisible}]]],
  GridInFront =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem2d"},  // 2d-only
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  GridLineWidth = // Do not change: DIN
      ["Inherited", 0.10,       plot::checkPosOutputSize,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Grid Lines", "XMLAttribute", "PosSize", "", FALSE]],
  GridLineStyle =
      ["Inherited", Solid,        {Solid, Dashed, Dotted},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Grid Lines", "XMLAttribute", "GridLineStyle", "", FALSE]],
  GridLineColor =
      ["Inherited", RGB::Grey60,     plot::checkColor,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Grid Lines", "XMLAttribute", "Color", "", FALSE]],
  XSubgridVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  YSubgridVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  ZSubgridVisible =
      ["Inherited", FALSE,         {TRUE, FALSE},
       {"CoordinateSystem3d"},
       plot::combineBooleanOr,
       ["Grid Lines", "XMLAttribute", "Bool", "", FALSE]],
  SubgridVisible =
      ["Library", NIL,         plot::libSubgridVisible,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::readSubgridVisible,
       [[{XSubgridVisible, YSubgridVisible}],
        [{XSubgridVisible, YSubgridVisible, ZSubgridVisible}]]],
  SubgridLineWidth = // Do not change: DIN
      ["Inherited", 0.10,  plot::checkPosOutputSize,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Grid Lines", "XMLAttribute", "PosSize", "", FALSE]],
  SubgridLineStyle =
      ["Inherited", Solid,        {Solid, Dashed, Dotted},
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Grid Lines", "XMLAttribute", "GridLineStyle", "", FALSE]],
  SubgridLineColor =
      ["Inherited", RGB::Grey80, plot::checkColor,
       {"CoordinateSystem2d", "CoordinateSystem3d"},
       plot::combineTakeFirst,
       ["Grid Lines", "XMLAttribute", "Color", "", FALSE]],

//----------------------------------------------------------------------------
//  View2/3d
//----------------------------------------------------------------------------

  ViewXMin =
      ["Optional",  NIL,   plot::checkReal,
       {"View2d"},   FAIL,  FAIL],
  ViewXMax =
      ["Optional",  NIL,   plot::checkReal,
       {"View2d"},   FAIL,  FAIL],
  ViewYMin =
      ["Optional",  NIL,   plot::checkReal,
       {"View2d"},   FAIL,  FAIL],
  ViewYMax =
      ["Optional",  NIL,   plot::checkReal,
       {"View2d"},   FAIL,  FAIL],
  ViewPositionX =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewPositionY =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewPositionZ =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewFocalPointX =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewFocalPointY =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewFocalPointZ =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewTranslationX =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewTranslationY =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewTranslationZ =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewUpVectorX =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewUpVectorY =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewUpVectorZ =
      ["Optional",  NIL,   plot::checkReal,
       {"View3d"},   FAIL,  FAIL],
  ViewKeepUpVector =
      ["Optional",  NIL,   {TRUE, FALSE},
       {"View3d"},   FAIL,  FAIL],  
  ViewOrthogonalProjection =
      ["Optional",  NIL,   {TRUE, FALSE},
       {"View3d"},   FAIL,  FAIL],  
  ViewDirectional =
      ["Optional",  TRUE,   {TRUE, FALSE},
       {"View3d"},   FAIL,  FAIL],  
  ViewViewingAngle =
      ["Optional",  NIL,   plot::checkPosReal,
       {"View3d"},   FAIL,  FAIL],

//----------------------------------------------------------------------------
//  options common  for all animated object
//----------------------------------------------------------------------------

  ParameterName =
      ["Optional",  NIL,  plot::elementExpr, plot::animatedObjects,
       FAIL,
       ["Definition", "XMLAttribute", "Expression",
        "Animation parameter.", FALSE]],
  ParameterBegin =
      ["Optional",  NIL,  plot::elementVExpr, plot::animatedObjects,
       FAIL,
       ["Definition", "XMLAttribute", "Expression",
        "Lower bound of animation parameter.", FALSE]],
  ParameterEnd =
      ["Optional",  NIL,  plot::elementVExpr, plot::animatedObjects,
       FAIL,
       ["Definition", "XMLAttribute", "Expression",
        "Upper bound of animation parameter.", FALSE]],
  ParameterRange =
      ["Library", NIL,    plot::libRangeOfXYZExpr("Parameter"),
       plot::animatedObjects, plot::readRangeOfXYZExpr("Parameter"),
       [[ParameterBegin..ParameterEnd]]],
  TimeBegin =
      ["Inherited",  0.0,  plot::checkReal, plot::animatedObjects,
       FAIL,
       ["Animation", "XMLAttribute", "Float",
        "Begin of animation time.", TRUE]],
  TimeEnd =
      ["Inherited", 10.0,  plot::checkReal, plot::animatedObjects,
       FAIL,
       ["Animation", "XMLAttribute", "Float",
        "End of animation time.", TRUE]],
  TimeRange =
      ["Library", NIL,    plot::libRangeOfXYZ("Time"),
       plot::animatedObjects, plot::readRangeOfXYZ("Time"),
       [[TimeBegin..TimeEnd]]],
  Frames =
      ["Inherited",  50,  plot::checkPosInteger, plot::animatedObjects,
       FAIL,
       ["Animation", "XMLAttribute", "PosInt",
        "Number of animation time frames.", TRUE]],
  VisibleBeforeBegin =
      ["Inherited", TRUE, {TRUE, FALSE},    plot::allGraphPrim,
       FAIL,
       ["Animation", "XMLAttribute", "Bool",
        "Is the object visible before TimeBegin?", FALSE]],
  VisibleAfterEnd =
      ["Inherited", TRUE, {TRUE, FALSE},    plot::allGraphPrim,
       FAIL,
       ["Animation", "XMLAttribute", "Bool",
        "Is the object visible after TimeEnd?", FALSE]],
  VisibleBefore =
      ["Library", NIL,    plot::libVisibleBefore, plot::allGraphPrim,
       plot::readVisibleBefore,
       {TimeBegin, TimeEnd, VisibleBeforeBegin, VisibleAfterEnd}],
  VisibleAfter =
      ["Library", NIL,    plot::libVisibleAfter, plot::allGraphPrim,
       plot::readVisibleAfter,
       {TimeBegin, TimeEnd, VisibleBeforeBegin, VisibleAfterEnd}],
  VisibleFromTo =
      ["Library", NIL,    plot::libVisibleFromTo, plot::allGraphPrim,
       plot::readVisibleFromTo,
       {TimeBegin, TimeEnd, VisibleBeforeBegin, VisibleAfterEnd}],


//----------------------------------------------------------------------------
// Camera options
//----------------------------------------------------------------------------

  PositionX =
      ["Mandatory", NIL,   plot::elementOptExpr,
       {"Camera", "PointLight", "DistantLight", "SpotLight", "Point2d",
        "Point3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The x coordinate of the object position.", FALSE]],
  PositionY =
      ["Mandatory", NIL,   plot::elementOptExpr,
       {"Camera", "PointLight", "DistantLight", "SpotLight", "Point2d",
        "Point3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y coordinate of the object position.", FALSE]],
  PositionZ =
      ["Mandatory", NIL,   plot::elementOptExpr,
       {"Camera", "PointLight", "DistantLight", "SpotLight", "Point3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The z coordinate of the object position.", FALSE]],
  Position =
      ["Library",  NIL,    plot::libListOfOptExprXYZ,
       {"Camera", "PointLight", "DistantLight", "SpotLight", "Point2d",
        "Point3d"},
       plot::readListOfOptExprXYZ,
       [[[PositionX, PositionY]],
        [[PositionX, PositionY, PositionZ]]]],
  // a position for the implicit Camera in Scene3d
  CameraDirectionX =
      ["Library", NIL,   plot::checkReal_,
       {"Scene3d"},
       plot::readReal,
       [{CameraDirectionX_}]],
  CameraDirectionY =
      ["Library", NIL,   plot::checkReal_,
       {"Scene3d"},
       plot::readReal,
       [{CameraDirectionY_}]],
  CameraDirectionZ =
      ["Library", NIL,   plot::checkReal_,
       {"Scene3d"},
       plot::readReal,
       [{CameraDirectionZ_}]],
  CameraDirection =
      ["Library",  NIL,    plot::libListOfReal,
       {"Scene3d"},
       plot::readListOfReal,
       [{},
        [[CameraDirectionX, CameraDirectionY, CameraDirectionZ]]]],
  FocalPointX =
      ["Optional", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The x coordinate of the point where the camera looks to.", FALSE]],
  FocalPointY =
      ["Optional", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The y coordinate of the point where the camera looks to.", FALSE]],
  FocalPointZ =
      ["Optional", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The z coordinate of the point where the camera looks to.", FALSE]],
  FocalPoint =
      ["Library", NIL, plot::libListOfOptExprXYZ, {"Camera"},
       plot::readListOfOptExprXYZ,
       [{},
        [[FocalPointX, FocalPointY, FocalPointZ]]]],
  UpVectorX =
      ["Optional", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The x coordinate of the cameras vertical orientation.", FALSE]],
  UpVectorY =
      ["Optional", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The y coordinate of the cameras vertical orientation.", FALSE]],
  UpVectorZ =
      ["Optional", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The z coordinate of the cameras vertical orientation.", FALSE]],
  UpVector =
      ["Library", NIL, plot::libListOfOptExprXYZ, {"Camera"},
       plot::readListOfOptExprXYZ,
       [{},
        [[UpVectorX, UpVectorY, UpVectorZ]]]],
  KeepUpVector =
      ["Inherited", TRUE, {TRUE, FALSE},        {"Camera"}, FAIL,
       ["Definition", "XMLAttribute", "Bool",
        "", FALSE]],
  OrthogonalProjection =
      ["Inherited", FALSE, {TRUE, FALSE},        {"Camera"}, FAIL,
       ["Definition", "XMLAttribute", "Bool",
        "", FALSE]],
  ViewingAngle =
      ["Mandatory", NIL,   plot::elementOptExpr, {"Camera"}, FAIL,
       ["Definition", "Expr", FAIL,
        "Angle representing the focal distance of the camera.", FALSE]],


//----------------------------------------------------------------------------
// Light options
//----------------------------------------------------------------------------

  TargetX =
      ["Mandatory", NIL, plot::elementOptExpr,
       {"DistantLight", "SpotLight"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The x coordinate of the point where the light shines to.", FALSE]],
  TargetY =
      ["Mandatory", NIL, plot::elementOptExpr,
       {"DistantLight", "SpotLight"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The y coordinate of the point where the light shines to.", FALSE]],
  TargetZ =
      ["Mandatory", NIL, plot::elementOptExpr,
       {"DistantLight", "SpotLight"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The z coordinate of the point where the light shines to.", FALSE]],
  Target =
      ["Library",  NIL,  plot::libListOfOptExprXYZ,
       {"DistantLight", "SpotLight"},
       plot::readListOfOptExprXYZ,
       [{},
        [[TargetX, TargetY, TargetZ]]]],
  LightColor =
      ["Inherited", RGB::White, plot::checkRGBColor, plot::allLights, FAIL,
       ["Definition", "XMLAttribute", "Color",
        "The color of the light.", FALSE]],
  LightIntensity =
      ["Optional",  1.0,    plot::elementOptExpr,   plot::allLights, FAIL,
       ["Definition", "Expr", "Intensity",
        "The intensity of the light.", FALSE]],
  CameraCoordinates =
      ["Inherited", FALSE, {TRUE, FALSE},
       plot::allLights minus {"AmbientLight"}, FAIL,
       ["Definition", "XMLAttribute", "Bool",
        "Keep the light position relative to the camera?", FALSE]],
  SpotAngle =
      ["Mandatory",  NIL,        plot::elementOptExpr, {"SpotLight"}, FAIL,
       ["Definition", "Expr", "SpotAngle",
        "The open angle of a spotlight.", FALSE]],


//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
// Options of graphical primitives
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

  Title =
      ["Optional",  NIL,  plot::elementText, plot::allGraphPrim,   FAIL,
       ["Annotation", "XMLAttribute", "Title",
        "The title of the object.", FALSE]],
  TitleAlignment =
      ["Inherited", Center,       {Left, Center, Right},
       plot::allGraphPrim,    plot::combineTakeFirst,
       ["Style", "XMLAttribute", "HorAlignment", "", FALSE]],
  TitlePositionX =
      ["Optional",  NIL,  plot::elementExpr, plot::allGraphPrim,   FAIL,
       ["Annotation", "XMLAttribute", "Expression",
        "The x position of the title.", FALSE]],
  TitlePositionY =
      ["Optional",  NIL,  plot::elementExpr, plot::allGraphPrim,   FAIL,
       ["Annotation", "XMLAttribute", "Expression",
        "The y position of the title.", FALSE]],
  TitlePositionZ =
      ["Optional",  NIL,  plot::elementExpr, plot::allGraphPrim3d, FAIL,
       ["Annotation", "XMLAttribute", "Expression",
        "The z position of the title.", FALSE]],
  TitlePosition =
      ["Library", NIL,    plot::libTitlePosition, plot::allGraphPrim,
       plot::readTitlePosition,
       [[[TitlePositionX, TitlePositionY]],
        [[TitlePositionX, TitlePositionY, TitlePositionZ]]]],
  TitleFont =
      ["Inherited", "'sans-serif' 11",
                          plot::checkFont,   plot::allGraphPrim,   FAIL,
       ["Style", "XMLAttribute", "Font",
        "Text font.", FALSE]],
  LegendEntry =
      ["Inherited", FALSE, {TRUE, FALSE},     plot::allGraphPrim,   FAIL,
       ["Annotation", "XMLAttribute", "Bool",
        "Is the object to be described in the legend?", FALSE]],
  LegendText =
      ["Optional",  NIL,  plot::elementText, plot::allGraphPrim,   FAIL,
       ["Annotation", "XMLAttribute", "Title",
        "Legend text for the object.", FALSE]],
  Legend =
      ["Library", NIL,    plot::libLegend, plot::allGraphPrim,
       plot::readLegend,
       {LegendText, LegendEntry, LegendVisible}],
  Visible =
      ["Optional", TRUE, {TRUE, FALSE},
       plot::allGraphPrim union
       {"Group2d", "Group3d", "ClippingBox", "Camera"} union
       plot::allLights,   FAIL,
       ["Definition", "XMLAttribute", "Bool",
        "Is the object visible?", FALSE]],
  AffectViewingBox =
      ["Inherited", TRUE, {TRUE, FALSE},
       plot::allGraphPrim union plot::allTransformations,   FAIL,
       ["Definition", "XMLAttribute", "Bool",
        "Is the object considered in ViewingBox calculation?", TRUE]],
    
//----------------------------------------------------------------------------
// Function2d
//----------------------------------------------------------------------------

  XMin =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"ClippingBox", "Function2d", "Function3d", "Box",
        "Rectangle", "Implicit2d", "Implicit3d", "Hatch",
        "VectorField2d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Lower bound of objects x variable.", FALSE]],
  XMax =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"ClippingBox", "Function2d", "Function3d", "Box",
        "Rectangle", "Implicit2d", "Implicit3d", "Hatch",
        "VectorField2d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Upper bound of objects x variable.", FALSE]],
  XRange =
      ["Library", NIL,         plot::libRangeOfXYZOptExpr("X"),
       {"ClippingBox", "Function2d", "Function3d", "Box", "Hatch",
        "Rectangle", "Implicit2d", "Implicit3d", "VectorField2d",
        "VectorField3d"},
       plot::readRangeOfXYZOptExpr("X"),
       [[XMin..XMax]]],
  YMin =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"ClippingBox", "Function3d", "Box", "Rectangle",
        "Implicit2d", "Implicit3d", "VectorField2d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Lower bound of objects y variable.", FALSE]],
  YMax =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       { "ClippingBox", "Function3d", "Box", "Rectangle",
        "Implicit2d", "Implicit3d", "VectorField2d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Upper bound of objects y variable.", FALSE]],
  YRange =
      ["Library", NIL,         plot::libRangeOfXYZOptExpr("Y"),
       {"ClippingBox", "Function3d", "Box", "Rectangle",
        "Implicit2d", "Implicit3d", "VectorField2d", "VectorField3d"},
       plot::readRangeOfXYZOptExpr("Y"),
       [[YMin..YMax]]],
  ZMin =
      ["Mandatory",  NIL,        plot::elementOptExpr,
       {"ClippingBox", "Box", "Implicit3d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Lower bound of objects z variable.", FALSE]],
  ZMax =
      ["Mandatory",  NIL,        plot::elementOptExpr,
       {"ClippingBox", "Box", "Implicit3d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Upper bound of objects z variable.", FALSE]],
  ZRange =
      ["Library", NIL,         plot::libRangeOfXYZOptExpr("Z"),
       {"ClippingBox", "Box", "Implicit3d", "VectorField3d"},
       plot::readRangeOfXYZOptExpr("Z"),
       [[ZMin..ZMax]]],
  Function =
      ["Mandatory", NIL,          plot::elementOptFunctionExpr,
       {"Function2d", "Function3d", "Implicit2d", "Implicit3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The function definition.", FALSE]],
  XName =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Function2d", "Function3d", "Implicit2d", "Implicit3d",
        "VectorField2d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "First (x) variable used to define the object", FALSE]],
  UMesh =
      ["Inherited", 25,          plot::checkPosInteger,
       {"Curve2d", "Curve3d", "Surface"},
       FAIL,
       ["Calculation", "XMLAttribute", "PosInt",
        "Number of mesh points to calculate.", TRUE]],
  Mesh =
      ["Library", NIL,          plot::libMesh,
       {"Curve2d", "Curve3d", "Surface", "Function2d", "Function3d",
        "Implicit2d", "Implicit3d", "VectorField2d", "VectorField3d"},
       plot::readMesh,
       {UMesh, VMesh, XMesh, YMesh, ZMesh}],
  Submesh =
      ["Library", NIL,          plot::libSubmesh,
       {"Curve2d", "Curve3d", "Surface", "Function2d", "Function3d"},
       plot::readSubmesh,
       {USubmesh, VSubmesh, XSubmesh, YSubmesh}],
  XSubmesh =
      ["Inherited", 0,            plot::checkNonNegInteger,
       {"Function2d", "Function3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "NonNegInt",
        "Number of in-between mesh points to calculate.", TRUE]],
  YSubmesh =
      ["Inherited", 0,            plot::checkNonNegInteger,
       {"Function3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "NonNegInt",
        "Number of in-between mesh points to calculate.", TRUE]],
  AdaptiveMesh =
      ["Inherited", 0,       plot::checkNonNegInteger,
       {"Function2d", "Curve2d", "Curve3d", "Surface", "Function3d",
        "Implicit3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "NonNegInt",
        "Maximal adaptive refinement level.", TRUE]],
  DiscontinuitySearch =
      ["Inherited", TRUE,        {TRUE, FALSE},
       {"Function2d", "Curve2d", "Curve3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "Bool",
        "Search for objects discontinuities?", TRUE]],
  LineStyle =
      ["Inherited", Solid,        {Solid, Dashed, Dotted},
       {"Function2d", "Curve2d", "Curve3d", "Surface", "Function3d",
        "Arrow2d", "Arrow3d", "Box", "Circle2d", "Circle3d", "Cone",
        "Ellipse2d", "Line2d", "Line3d", "Polygon2d", "Polygon3d",
        "Rectangle", "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "LineStyle",
        "Line style.", FALSE]],
  LinesVisible =
      ["Inherited", TRUE,        {TRUE, FALSE},
       {"Box", "Circle2d", "Circle3d", "Cone", "Curve2d", "Curve3d",
        "Ellipse2d", "Function2d", "Function3d", /* "Surface",*/
        "Polygon2d", "Polygon3d", "Rectangle",
        "Implicit2d"/*, "Implicit3d"*/},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Are lines visible?", FALSE]],
  LineWidth = // Do not change: DIN
      ["Inherited", 0.35,         plot::checkPosOutputSize,
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Arrow2d", "Arrow3d", "Box",
        "Circle2d", "Circle3d", "Cone", "Ellipse2d", "Line2d",
        "Line3d", "Polygon2d", "Polygon3d", "Rectangle",
        "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "PosSize",
        "Line width.", FALSE]],
  LineColor = // nach Diskussion um Defaults
      ["Inherited", RGB::Blue,   plot::checkColor,
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Arrow2d", "Arrow3d", "Box",
        "Circle2d", "Circle3d", "Cone", "Ellipse2d", "Line2d",
        "Line3d", "Polygon2d", "Polygon3d", "Rectangle",
        "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Color",
        "Main line color.", FALSE]],
  LineColorDirectionX = 
      ["Inherited", 0,   plot::checkReal,
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Arrow2d", "Arrow3d", "Box",
        "Circle2d", "Circle3d", "Cone", "Ellipse2d", "Polygon2d",
        "Polygon3d", "Rectangle", "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Float",
        "x-direction for line coloring.", FALSE]],
  LineColorDirectionY = 
      ["Inherited", 1,   plot::checkReal,
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Arrow2d", "Arrow3d", "Box",
        "Circle2d", "Circle3d", "Cone", "Ellipse2d", "Polygon2d",
        "Polygon3d", "Rectangle", "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Float",
        "y-direction for line coloring.", FALSE]],
  LineColorDirectionZ = 
      ["Inherited", 1,   plot::checkReal,
       {"Curve3d", "VectorField3d",
        "Surface", "Function3d", 
        "Circle3d",  "Polygon3d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Float",
        "z-direction for line coloring.", FALSE]],
  LineColorDirection = 
      ["Library", NIL,   plot::libListOfReal,
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Arrow2d", "Arrow3d", "Box",
        "Circle2d", "Circle3d", "Cone", "Ellipse2d", "Polygon2d",
        "Polygon3d", "Rectangle", "Implicit2d", "Implicit3d"},
       plot::readListOfReal,
       [[[LineColorDirectionX, LineColorDirectionY]],
        [[LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ]]]],
  FillColorDirectionX = 
      ["Inherited", 0,   plot::checkReal,
       {"Surface", "Function3d", "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Float",
        "x-direction for surface coloring.", FALSE]],
  FillColorDirectionY = 
      ["Inherited", 0,   plot::checkReal,
       {"Surface", "Function3d", "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Float",
        "y-direction for surface coloring.", FALSE]],
  FillColorDirectionZ = 
      ["Inherited", 1,   plot::checkReal,
       {"Surface", "Function3d", "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Float",
        "z-direction for surface coloring.", FALSE]],
  FillColorDirection = 
      ["Library", NIL,   plot::libListOfReal,
       {"Surface", "Function3d", "Implicit3d"},
       plot::readListOfReal,
       [[[FillColorDirectionX, FillColorDirectionY]],
        [[FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ]]]],
  VerticalAsymptotesStyle =
      ["Inherited", Dashed,        {Solid, Dashed, Dotted},
       {"Function2d"},
       FAIL,
       [["Style", "Asymptotes"], "XMLAttribute", "LineStyle",
        "Style of vertical asymptotes.", FALSE]],
  VerticalAsymptotesVisible =
      ["Inherited", TRUE,        {TRUE, FALSE},
       {"Function2d"},
       FAIL,
       [["Style", "Asymptotes"], "XMLAttribute", "Bool",
        "Are vertical asymptotes (pole lines) visible?", FALSE]],
  VerticalAsymptotesWidth =
      ["Inherited", 0.2,         plot::checkPosOutputSize,
       {"Function2d"},
       FAIL,
       [["Style", "Asymptotes"], "XMLAttribute", "PosSize",
        "Width of vertical asymptotes.", FALSE]],
  VerticalAsymptotesColor =
      ["Inherited", RGB::Grey50,   plot::checkColor,
       {"Function2d"},
       FAIL,
       [["Style", "Asymptotes"], "XMLAttribute", "Color",
        "Color of vertical asymptotes.", FALSE]],
  PointStyle =
      ["Inherited", FilledCircles,
       [// 2D
        {Squares, FilledSquares, Circles, FilledCircles, Crosses, XCrosses,
        Diamonds, FilledDiamonds, Stars},
        // 3D
        {FilledSquares, FilledCircles}],
// problem here:  different types in different contexts
       {"Curve2d", "Curve3d", "Function2d", 
        "Surface", "Function3d", "Point2d",
        "Point3d", "Polygon2d", "Polygon3d",
        "Implicit3d", "VectorField3d"},
       FAIL,
       [["Style", "Points"], "XMLAttribute", ["PointStyle", "PointStyle3d"],
        "Point style.", FALSE]],
  PointsVisible =
      ["Inherited", FALSE,        {TRUE, FALSE},
       {"Curve2d", "Curve3d", "Function2d", "Function3d", 
        "Polygon2d", "Polygon3d", "Surface",
        "Implicit3d", "VectorField3d"},
       FAIL,
       [["Style", "Points"], "XMLAttribute", "Bool",
        "Are points visible?", FALSE]],
  PointSize =
      ["Inherited", 1.5,          plot::checkPosOutputSize,
       {"Function2d", "Curve2d", "Curve3d", 
        "Surface", "Function3d", "Point2d",
        "Point3d", "Polygon2d", "Polygon3d", "VectorField3d"},
       FAIL,
       [["Style", "Points"], "XMLAttribute", "PosSize",
        "Point size.", FALSE]],
  LineColorType =
      ["Inherited", Flat,
       {Flat, Monochrome, Dichromatic, Rainbow, Functional},
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Polygon2d", "Polygon3d",
        "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "ColorType",
        "Type of line coloring.", FALSE]],
  LineColor2 =
      ["Inherited", RGB::DeepPink,     plot::checkColor,
       {"Function2d", "Curve2d", "Curve3d", "VectorField2d", "VectorField3d",
        "Surface", "Function3d", "Polygon2d", "Polygon3d",
        "Implicit2d", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Color",
        "Secondary line color.", FALSE]],

  
//----------------------------------------------------------------------------
// Function3d
//----------------------------------------------------------------------------

  YName =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Function3d", "Implicit2d", "Implicit3d", "VectorField2d",
        "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Second (y) variable used to define the object", FALSE]],

//----------------------------------------------------------------------------
// Curve2/3d
//----------------------------------------------------------------------------

  UName =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Curve2d", "Curve3d", "Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "First (u) variable used to define the object", FALSE]],
  UMin =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Curve2d", "Curve3d", "Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Lower bound of objects u variable.", FALSE]],
  UMax =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Curve2d", "Curve3d", "Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Upper bound of objects u variable.", FALSE]],
  URange =
      ["Library", NIL,         plot::libRangeOfXYZOptExpr("U"),
       {"Curve2d", "Curve3d", "Surface"},
       plot::readRangeOfXYZOptExpr("U"),
       [[UMin..UMax]]],
  XFunction =
      ["Mandatory", NIL, plot::elementOptFunctionExpr,
       {"Curve2d", "Curve3d", "VectorField2d", "VectorField3d", "Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The x components function of the object.", FALSE]],
  YFunction =
      ["Mandatory", NIL, plot::elementOptFunctionExpr,
       {"Curve2d", "Curve3d", "VectorField2d", "VectorField3d", "Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y components function of the object.", FALSE]],
  ZFunction =
      ["Mandatory", NIL, plot::elementOptFunctionExpr,
       {"Curve3d", "Surface", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The z components function of the object.", FALSE]],


//----------------------------------------------------------------------------
// VectorField2d
//----------------------------------------------------------------------------

  VName =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Second (v) variable used to define the object", FALSE]],
  VMin =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Lower bound of objects v variable.", FALSE]],
  VMax =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Surface"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Upper bound of objects v variable.", FALSE]],
  VRange =
      ["Library", NIL,         plot::libRangeOfXYZOptExpr("V"),
       {"Surface"},
       plot::readRangeOfXYZOptExpr("V"),
       [[VMin..VMax]]],
  ArrowLength =
      ["Inherited", Proportional, {Fixed, Proportional, Logarithmic},
       {"VectorField2d", "VectorField3d"},
       FAIL,
       [["Style", "Arrows"], "XMLAttribute", "ArrowLength",
        "Length of vector field arrows.", FALSE]],


//----------------------------------------------------------------------------
// Surface
//----------------------------------------------------------------------------

  VMesh =
      ["Inherited", 25,           plot::checkPosInteger,
       {"Surface"},
       FAIL,
       ["Calculation", "XMLAttribute", "PosInt",
        "Number of mesh points to calculate in v direction.", TRUE]],
  USubmesh =
      ["Inherited", 0,      plot::checkNonNegInteger,
       {"Curve2d", "Curve3d", "Surface"},
       FAIL,
       ["Calculation", "XMLAttribute", "NonNegInt",
        "Number of in-between mesh points to calculate in u direction.", TRUE]],
  VSubmesh =
      ["Inherited", 0,      plot::checkNonNegInteger,
       {"Surface"},
       FAIL,
       ["Calculation", "XMLAttribute", "NonNegInt",
        "Number of in-between mesh points to calculate in v direction.", TRUE]],
  ULinesVisible =
      ["Inherited",  TRUE, {TRUE, FALSE},
       {"Surface"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Are lines in u direction visible?", FALSE]],
  VLinesVisible =
      ["Inherited",  TRUE, {TRUE, FALSE},
       {"Surface"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Are lines in v direction visible?", FALSE]],
  XLinesVisible =
      ["Inherited",  TRUE, {TRUE, FALSE},
       {"Function3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Are lines in u direction visible?", FALSE]],
  YLinesVisible =
      ["Inherited",  TRUE, {TRUE, FALSE},
       {"Function3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Are lines in v direction visible?", FALSE]],
  MeshVisible =
      ["Inherited",  FALSE, {TRUE, FALSE},
       {"Function3d", "Surface", "Implicit3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Render the adaptive mesh visible?", FALSE]],
  Filled =
      ["Inherited",  TRUE,  {TRUE, FALSE},
       {"Surface", "Box", "Circle2d", "Circle3d", "Cone", "Ellipse2d",
        "Polygon2d", "Polygon3d", "Rectangle", "Function3d", 
        "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Bool",
        "Are patches visible?", FALSE]],
  Shading =
      ["Inherited", Smooth, {Flat, Smooth},
       {"Surface", "Cone", "Sphere", "Function3d", "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Shading",
        "Type of surface shading.", FALSE]],
  LineColorFunction =
      ["Optional",  NIL,    plot::checkColorFunction,
       {"Function3d", "Surface", "Implicit3d", "Function2d", "Curve2d",
        "Curve3d", "VectorField2d", "VectorField3d", "Polygon2d",
        "Polygon3d", "Implicit2d"},
       FAIL,
       ["Definition", "ColorFunc", FAIL,
        "Function defining surface line color.", FALSE]],
  FillColorType =
      ["Inherited",  Dichromatic,   
       {Flat, Monochrome, Dichromatic, Rainbow, Functional},
       {"Function3d", "Surface", "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "ColorType",
        "Type of fill coloring.", FALSE]],
  FillColorFunction =
      ["Optional",  NIL,    plot::checkColorFunction,
       {"Function3d", "Surface", "Implicit3d"},
       FAIL,
       ["Definition", "ColorFunc", FAIL,
        "Function defining surface fill color.", FALSE]],
  FillColor =
      ["Inherited", RGB::Red,   plot::checkColor,
       {"Function3d", "Surface", "Box", "Circle2d", "Circle3d", "Cone",
        "Ellipse2d", "Polygon2d", "Polygon3d", "Rectangle", "Sphere",
        "Implicit3d", "Hatch"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Color",
        "Main fill color.", FALSE]],
  FillColor2 =
      ["Inherited", RGB::CornflowerBlue,   plot::checkColor,
       {"Function3d", "Surface", "Implicit3d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "Color",
        "Secondary fill color.", FALSE]],

//----------------------------------------------------------------------------
// Arrow2d/3d
//----------------------------------------------------------------------------
  FromX =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Arrow2d", "Arrow3d", "Line2d", "Line3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The x component of the objects begin.", FALSE]],
  FromY =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Arrow2d", "Arrow3d", "Line2d", "Line3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the objects begin.", FALSE]],
  FromZ =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Arrow3d", "Line3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the objects begin.", FALSE]],
  From =
      ["Library",  NIL,           plot::libListOfOptExprXYZ,
       {"Arrow2d", "Arrow3d", "Line2d", "Line3d"},
       plot::readListOfOptExprXYZ,
       [[[FromX, FromY]],
        [[FromX, FromY, FromZ]]]],
  ToX =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Arrow2d", "Arrow3d", "Line2d", "Line3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The x component of the objects end.", FALSE]],
  ToY =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Arrow2d", "Arrow3d", "Line2d", "Line3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the objects end.", FALSE]],
  ToZ =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Arrow3d", "Line3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The z component of the objects end.", FALSE]],
  To =
      ["Library",  NIL,           plot::libListOfOptExprXYZ,
       {"Arrow2d", "Arrow3d", "Line2d", "Line3d"},
       plot::readListOfOptExprXYZ,
       [[[ToX, ToY]],
        [[ToX, ToY, ToZ]]]],
  TipAngle = // 15 Grad = PI/12 nach DIN
             // DIN sieht bescheuert aus. --> 24 Grad = 2*PI/15
      ["Inherited", 2*PI/15,    plot::checkNonNegReal,
       {"Arrow2d", "Arrow3d", "VectorField2d"},
       FAIL,
       [["Style", "Tip"], "XMLAttribute", "TipAngle",
        "Arrow tips angle.", FALSE]],
  TipLength = // Approximation einer DIN Norm ??
      ["Inherited", 4,     plot::checkNonNegOutputSize,
       {"Arrow2d", "Arrow3d", "VectorField2d"},
       FAIL,
       [["Style", "Tip"], "XMLAttribute", "NonNegSize",
        "Arrow tips length.", FALSE]],
  TipStyle =
      ["Inherited", Filled,           {Filled, Open, Closed},
       {"Arrow2d", "Arrow3d", "VectorField2d"},
       FAIL,
       [["Style", "Tip"], "XMLAttribute", "TipStyle",
        "Arrow tips style.", FALSE]],
  Tubular =
      ["Inherited",  FALSE,          {TRUE, FALSE},
       {"Arrow3d", "Line3d"},
       FAIL,
       [["Style", "Lines"], "XMLAttribute", "Bool",
        "Draw tubular line shaft?", FALSE]],
  Extension =
      ["Inherited", Finite,         {Finite, SemiInfinite, Infinite},
       {"Line2d", "Line3d"},
       FAIL,
       [["Definition"], "XMLAttribute", "Extension",
        "Extend lienes?", FALSE]],

//----------------------------------------------------------------------------
// Circle2d
//----------------------------------------------------------------------------
  CenterX =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Circle2d", "Circle3d", "Ellipse2d", "Sphere", "Rotate2d", "Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The x component of the objects center.", FALSE]],
  CenterY =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Circle2d", "Circle3d", "Ellipse2d", "Sphere", "Rotate2d", "Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the objects center.", FALSE]],
  Center =
      ["Library", NIL,             plot::libListOfOptExprXYZ,
       {"Circle2d", "Circle3d", "Ellipse2d", "Sphere", "Rotate2d", "Rotate3d"},
       plot::readListOfOptExprXYZ,
       [[[CenterX, CenterY]],
        [[CenterX, CenterY, CenterZ]]]],
  Radius =
      ["Mandatory",  NIL,          plot::elementOptExpr,
       {"Circle2d", "Circle3d", "Sphere"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The radius.", FALSE]],
  FillPattern =
      ["Inherited", DiagonalLines, {Solid, HorizontalLines, VerticalLines,
                                    DiagonalLines, FDiagonalLines,
                                    CrossedLines, XCrossedLines},
       {"Circle2d", "Ellipse2d", "Polygon2d", "Rectangle", "Hatch"}, FAIL,
       [["Style", "Surface"], "XMLAttribute", "FillPattern",
        "Pattern used to fill.", FALSE]],

//----------------------------------------------------------------------------
// Circle3d
//----------------------------------------------------------------------------
  CenterZ =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Circle3d", "Sphere", "Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The z component of the objects center.", FALSE]],
  NormalX =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Circle3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The x component of the objects normal vector.", FALSE]],
  NormalY =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Circle3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the objects normal vector.", FALSE]],
  NormalZ =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Circle3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The z component of the objects normal vector.", FALSE]],
  Normal =
      ["Library", NIL,              plot::libListOfOptExprXYZ,
       {"Circle3d"},
       plot::readListOfOptExprXYZ,
       [[[NormalX, NormalY]],
        [[NormalX, NormalY, NormalZ]]]],

//----------------------------------------------------------------------------
// Cone
//----------------------------------------------------------------------------
  BaseX =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The x component of the center of the cone's base.", FALSE]],
  BaseY =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the center of the cone's base.", FALSE]],
  BaseZ =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The z component of the center of the cone's base.", FALSE]],
  Base =
      ["Library",    NIL, plot::libListOfOptExprXYZ,
       {"Cone"}, plot::readListOfOptExprXYZ,
       [[[BaseX, BaseY]],
        [[BaseX, BaseY, BaseZ]]]],
  TopX =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The x component of the center of the cone's top.", FALSE]],
  TopY =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The y component of the center of the cone's top.", FALSE]],
  TopZ =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The z component of the center of the cone's top.", FALSE]],
  Top =
      ["Library",    NIL, plot::libListOfOptExprXYZ,
       {"Cone"}, plot::readListOfOptExprXYZ,
       [[[TopX, TopY]],
        [[TopX, TopY, TopZ]]]],
  BaseRadius =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The radius of the cone's base.", FALSE]],
  TopRadius =
      ["Mandatory",  NIL, plot::elementOptExpr, {"Cone"}, FAIL,
       ["Definition", "Expr", FAIL,
        "The radius of the cone's top.", FALSE]],

//----------------------------------------------------------------------------
// Ellipse2d
//----------------------------------------------------------------------------
  SemiAxisX =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Ellipse2d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The length of the semi axis in x direction.", FALSE]],
  SemiAxisY =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {"Ellipse2d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The length of the semi axis in y direction.", FALSE]],
  SemiAxisZ =
      ["Mandatory",  NIL,           plot::elementOptExpr,
       {},
       FAIL,
       ["Definition", "Expr", FAIL,
        "The length of the semi axis in z direction.", FALSE]],
  SemiAxes =
      ["Library",    NIL,           plot::libSemiAxes,
       {"Ellipse2d"}, plot::readSemiAxes,
       [[[SemiAxisX, SemiAxisY]],
        [[SemiAxisX, SemiAxisY, SemiAxisZ]]]],

//----------------------------------------------------------------------------
// Point2d/3d
//----------------------------------------------------------------------------
  PointColor =
      ["Inherited",  RGB::MidnightBlue,           plot::checkColor,
       {"Point2d", "Point3d"},
       FAIL,
       [["Style", "Points"], "XMLAttribute", "Color",
        "Point color.", FALSE]],

//----------------------------------------------------------------------------
// Polygon2d/3d
//----------------------------------------------------------------------------
  Points2d =
      ["Mandatory",  NIL,           plot::checkPoints,
       {"Polygon2d"},
       FAIL,
       ["Definition", "XMLAttribute", "Points2d",
        "The points of the polygon.", TRUE]],
  Points3d =
      ["Mandatory",  NIL,           plot::checkPoints,
       {"Polygon3d"},
       FAIL,
       ["Definition", "XMLAttribute", "Points3d",
        "The points of the polygon.", TRUE]],
  Closed =
      ["Inherited",  FALSE,           {TRUE, FALSE},
       {"Polygon2d", "Polygon3d"},
       FAIL,
       ["Definition", "XMLAttribute", "Bool",
        "Is the polygon closed?", FALSE]],
  FillStyle =
      ["Inherited",  EvenOdd,           {Winding, EvenOdd},
       {"Polygon2d"},
       FAIL,
       [["Style", "Surface"], "XMLAttribute", "FillStyle",
        "Style used to fill self-overlapping polygon.", FALSE]],

//----------------------------------------------------------------------------
// Text2d/3d
//----------------------------------------------------------------------------
  TextRotation =
      ["Optional",  0,          plot::elementOptExpr,
       {},
       FAIL,
       ["Definition", "Expr", "Angle",
        "The rotation angle of the texts baseline.", FALSE]],
  HorizontalAlignment =
      ["Inherited",  Left,          {Left, Center, Right},
       {},
       FAIL,
       ["Style", "XMLAttribute", "HorAlignment",
        "Horizontal alignment of text.", FALSE]],
  VerticalAlignment =
      ["Inherited",  BaseLine,          {Top, BaseLine, Center, Bottom},
       {},
       FAIL,
       ["Style", "XMLAttribute", "VertAlignment",
        "Vertical alignment of text.", FALSE]],
  TextOrientation =
      ["Optional",  [1, 0, 0, 0, 0, 1], plot::checkTextOrientation,
       {},
       FAIL,
       ["Definition", "XMLAttribute", "TextOrientation",
        "", FALSE]],
  TextFont =
      ["Inherited", "'sans-serif' 11",
                          plot::checkFont,   {},   FAIL,
       ["Style", "XMLAttribute", "Font",
        "Text font.", FALSE]],
  Billboarding =
      ["Inherited",  TRUE,           {TRUE, FALSE},
       {},
       FAIL,
       ["Style", "XMLAttribute", "Bool",
        "Is the text always directed to the user?", FALSE]],

//----------------------------------------------------------------------------
// Translate2d/3d
//----------------------------------------------------------------------------
  ShiftX =
      ["Optional",  0,          plot::elementOptExpr,
       plot::allTransformations,
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  ShiftY =
      ["Optional",  0,          plot::elementOptExpr,
       plot::allTransformations,
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  ShiftZ =
      ["Optional",  0,          plot::elementOptExpr,
       plot::allTransformations3d,
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  Shift =
      ["Library", NIL,          plot::libListOfOptExprXYZ,
       plot::allTransformations, plot::readListOfOptExprXYZ,
       [[[ShiftX, ShiftY]],
        [[ShiftX, ShiftY, ShiftZ]]]],

//----------------------------------------------------------------------------
// Rotate2d/3d
//----------------------------------------------------------------------------
  Angle =
      ["Optional",  0,            plot::elementOptExpr,
       {"Rotate2d", "Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  AxisX =
      ["Optional",  NIL,          plot::elementOptExpr,
       {"Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  AxisY =
      ["Optional",  NIL,          plot::elementOptExpr,
       {"Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  AxisZ =
      ["Optional",  NIL,          plot::elementOptExpr,
       {"Rotate3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  Axis =
      ["Library", NIL,         plot::libListOfOptExprXYZ,
       {"Rotate3d"}, plot::readListOfOptExprXYZ,
       [{}, [[AxisX, AxisY, AxisZ]]]],

//----------------------------------------------------------------------------
// Scale2d/3d
//----------------------------------------------------------------------------
  ScaleX =
      ["Optional",  1,          plot::elementOptExpr,
       {"Scale2d", "Scale3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  ScaleY =
      ["Optional",  1,          plot::elementOptExpr,
       {"Scale2d", "Scale3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  ScaleZ =
      ["Optional",  1,          plot::elementOptExpr,
       {"Scale3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "", FALSE]],
  Scale =
      ["Library", NIL,          plot::libListOfOptExprXYZ,
       {"Scale2d", "Scale3d"}, plot::readListOfOptExprXYZ,
       [[[ScaleX, ScaleY]],
        [[ScaleX, ScaleY, ScaleZ]]]],

//----------------------------------------------------------------------------
// Transform2d/3d
//----------------------------------------------------------------------------
  Matrix2d =
      ["Optional",  [1,0,0,1],          plot::checkMatrix2d,
       plot::allTransformations2d,
       FAIL,
       ["Definition", "XMLAttribute", "Matrix2d", "", FALSE]],
  Matrix3d =
      ["Optional",  [1,0,0, 0,1,0, 0,0,1], plot::checkMatrix3d,
       plot::allTransformations3d,
       FAIL,
       ["Definition", "XMLAttribute", "Matrix3d", "", FALSE]],

//----------------------------------------------------------------------------
// Implicit2d/3d
//----------------------------------------------------------------------------
  XMesh =
      ["Inherited", 11,           plot::checkPosInteger,
       {"Function2d", "Function3d", "Implicit2d", "Implicit3d",
        "VectorField2d", "VectorField3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "PosInt",
        "Number of mesh points to calculate in x direction.", TRUE]],
  YMesh =
      ["Inherited", 11,           plot::checkPosInteger,
       {"Implicit2d", "Implicit3d", "Function3d", "VectorField2d",
        "VectorField3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "PosInt",
        "Number of mesh points to calculate in y direction.", TRUE]],
  ZMesh =
      ["Inherited", 11,             plot::checkPosInteger,
       {"Implicit3d", "VectorField3d"},
       FAIL,
       ["Calculation", "XMLAttribute", "PosInt",
        "Number of grid points to calculate in z direction.", TRUE]],
  Contours =
      ["Mandatory", NIL,         plot::checkList,
       {"Implicit2d", "Implicit3d"},
       FAIL,
       ["Definition", "ExprSeq", FAIL,
        "The contour levels of the function.", FALSE]],
  XContours =
      ["Optional", [],         plot::checkList,
       {"Implicit3d", "Surface"},
       FAIL,
       ["Definition", "ExprSeq", FAIL,
        "Contour levels at constant x values.", FALSE]],
  YContours =
      ["Optional", [],         plot::checkList,
       {"Implicit3d", "Surface"},
       FAIL,
       ["Definition", "ExprSeq", FAIL,
        "Contour levels at constant y values.", FALSE]],
  ZContours =
      ["Optional", [],         plot::checkList,
       {"Implicit3d", "Surface", "Function3d"},
       FAIL,
       ["Definition", "ExprSeq", FAIL,
        "Contour levels at constant z values.", FALSE]],
  ZName =
      ["Mandatory", NIL,          plot::elementOptExpr,
       {"Implicit3d", "VectorField3d"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Third (z) variable used to define the object", FALSE]],
//----------------------------------------------------------------------------
// Hatch
//----------------------------------------------------------------------------
  Function1 =
      ["Mandatory", NIL,          plot::elementOptProp(DOM_STRING),
       {"Hatch"},
       FAIL,
       ["Definition", "Prop", "String",
        "Name of the upper bound function of the hatch.", FALSE]],
  Function2 =
      ["Optional", NIL,          plot::elementOptProp(DOM_STRING),
       {"Hatch"},
       FAIL,
       ["Definition", "Prop", "String",
        "Name of the lower bound function of the hatch.", FALSE]],
  Baseline =
      ["Optional", NIL,          plot::elementOptExpr,
       {"Hatch"},
       FAIL,
       ["Definition", "Expr", FAIL,
        "Lower bound of the hatch.", FALSE]],
  Color =
      ["Library", NIL, plot::libColor,
       {"Arrow2d", "Arrow3d", "Box", "Circle2d", "Circle3d", "Cone",
        "Curve2d", "Curve3d", "Ellipse2d", "Function2d",
        "Function3d", "Hatch", "Implicit2d", "Implicit3d",
        "Line2d", "Line3d", "Point2d", "Point3d", "Polygon2d",
        "Polygon3d", "Rectangle", "Sphere", "Surface", "VectorField2d",
        "VectorField3d"},
       plot::readColor,
       {LightColor, Colors, FillColor, LineColor, PointColor}],
  Colors =
      ["Optional", NIL, plot::checkColorList,
       {},
       FAIL,
       ["Style", "Prop", "ColorList",
        "List of Colors", TRUE]],
  AntiAliased =
      ["Inherited", TRUE,            {TRUE, FALSE},
       {"Arrow2d", "Circle2d", "Curve2d", "Ellipse2d", "Function2d", "Hatch",
        "Implicit2d", "Line2d", "Point2d", "Polygon2d", "Rectangle",
        "VectorField2d"},
       FAIL,
       ["Style", "XMLAttribute", "Bool",
        "Are lines antialiased?", FALSE]]
):



// attributes which can contain variables 
plot::varNames := {UName, VName, XName, YName, ZName}:
// names of attribute which represent functions
plot::functions := {Function, XFunction, YFunction, ZFunction}:
// names of attribute which represent color functions
plot::colorFunctions :=
  {ColorFunction, LineColorFunction, FillColorFunction}:


// used in plot::getInherited
plot::allInheritedAttributes :=
    map({op(select(plot::attributes,
                   x -> op(x, [2, 1]) = "Inherited"))}, op, 1):

// used in plot::getOptionals
plot::allOptionalAttributes :=
    map({op(select(plot::attributes,
                   x -> op(x, [2, 1]) = "Optional"))}, op, 1):

// used in plot::createPlotDomain 
plot::allMandatoryAttributes :=
    map({op(select(plot::attributes,
                   x -> op(x, [2, 1]) = "Mandatory"))}, op, 1):

// used in plot::createPlotDomain for extensions
plot::allLibraryAttributes :=
    map({op(select(plot::attributes,
                   x -> op(x, [2, 1]) = "Library"))}, op, 1):

// all Fonts
plot::allFontAttributes :=
    map({op(select(plot::attributes,
                   x -> op(x, [2, 3]) = plot::checkFont))}, op, 1):

// used in plot::createPlotDomain for extensions
plot::updateLibraryAttributes :=
proc(domType : DOM_STRING, attrib : DOM_SET)
  local i, entry, set, entries;
begin
  for i in plot::allLibraryAttributes do
    entry := plot::attributes[i];
    set   := entry[4];
    if domtype(entry[6]) = DOM_SET then
      entries := indets(entry[6])
    else
      // use only 2D-attributes
      entries := indets(entry[6][1])
    end_if;
    if entries <> {} and entries minus attrib = {} then
      // add domType to set of objects where i applies
      sysassign(plot::attributes[i], subsop(entry, 4=set union {domType}));
    end_if;
  end:
end_proc:

// map check function on defaults of optional attributes and
// replace the given defaults by internal representation
proc()
  local i;
begin
  plot::setDefaultInternal(if plot::attributes[i][2] <> NIL then
                             i = plot::attributes[i][2]
                           else
                             null()
                           end_if $
                           i in plot::allOptionalAttributes):
end_proc():

// used in slot methods of plot domaians
plot::outputSizeAttributes :=
    {Spacing, Width, Height, 
     TopMargin, BottomMargin, LeftMargin, RightMargin,
     AxesLineWidth, TicksLength, TubeDiameter, GridLineWidth, SubgridLineWidth,
     LineWidth, VerticalAsymptotesWidth, PointSize, TipLength,
     Bottom, Left}:

// all attributes which may be set as hints for parents
plot::hintAllowed :=
    map({op(select(plot::attributes, x -> op(x, [2, 5]) <> FAIL or
                   op(x, [2, 1]) = "Library"))}, op, 1):

// a selection of hints, which will be in TAB-Completions
plot::standardHints :=
   {AxesInFront, GridInFront, GridVisible, Axes,
    SubgridVisible, ViewingBox}:


  
/*  Computing plot::legalAttributes.
 *
 *  Beginning from Canvas scan the object hierachy and compute a list of
 *  3 sets of valid attributes for a given object type containing the
 *  mandatory, optional and inherited attribute names which are legal.
 *
 *  This table takes about 1 MB memory and its computation takes about
 *  370 ms.  The table is necessary in order to decide whether an attribute
 *  is a hint or not.
 */
plot::computeLegalAttributes :=
proc()
  local i, j, attrTyp, idx, attrib, p;
begin
  // initialize empty table
  sysassign(plot::legalAttributes,
            table(i = [{}, {}, {}, {}, {}] $ i in plot::allObjectTypes)):

  for i in plot::attributes do
    attrib := {op(i, 1)};
    attrTyp := op(i, [2, 1]);
    idx := contains(["Mandatory", "Optional", "Inherited", "Library"], attrTyp);
    for j in op(i, [2, 4]) do
      sysassign(plot::legalAttributes[j][idx], plot::legalAttributes[j][idx] union attrib)
    end_for;
  end_for;

  // propagate inherited options up
  p := proc(typ)
         local children;
       begin
         children := select(plot::objectHierarchy[typ], testtype, DOM_STRING)
                     minus {typ} minus plot::allTransformations;
         children := op(map(children, p));
         sysassign(plot::legalAttributes[typ][5],
                   plot::legalAttributes[typ][3]);
         sysassign(plot::legalAttributes[typ],
                   subsop(plot::legalAttributes[typ], 3=
                          op(plot::legalAttributes[typ], 3) union children));
         plot::legalAttributes[typ][3];
       end_proc;
  p("Canvas");
  null()
end_proc:
plot::computeLegalAttributes():


// check that all entries are valid:
plot::checkAttributesTable :=
proc()
  local attrib, printWarning, inspector, i, res, tmp, remaining;
begin
  printWarning :=
  proc(str : DOM_STRING, s : DOM_SET)
    name plot::checkAttributesTable;
  begin
    warning(str.if nops(s) > 1 then "s:" else ":" end."\n\t".
            expr2text(op(s)))
  end:
  
  // check for lists
  attrib := split(plot::attributes, x -> testtype(op(x, 2), DOM_LIST));
  if attrib[2] <> table() then
    printWarning("non-list entry defined for attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;

  // check number of list entries
  attrib := split(attrib[1], x -> contains({6}, nops(op(x, 2))));
  if attrib[2] <> table() then
    printWarning("6 list entries expected for attribute",
                 map({op(attrib[2])}, op, 1)
)
  end_if;
  
  // check 1. list entries
  attrib := split(attrib[1],
                  x -> contains({"Inherited", "Optional",
                                 "Mandatory", "Library"},
                                op(x, [2, 1])));
  if attrib[2] <> table() then
    printWarning("wrong type of attribute", map({op(attrib[2])}, op, 1))
  end_if;
  
  // check check-functions
  attrib := split(attrib[1], x -> testtype(op(x, [2, 3]), Type::Function) or
                                  testtype(op(x, [2, 3]), DOM_SET) or
                                  testtype(op(x, [2, 3]), Type::ListOf(DOM_SET, 2,2)));
  if attrib[2] <> table() then
    printWarning("check-procedure or set or list of 2 sets expected for attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;
  
  // check for objects where attribute applies
  attrib := split(attrib[1],
                  x -> _lazy_and(testtype(op(x, [2, 4]), DOM_SET),
                                 op(x, [2, 4]) minus
                                 plot::allObjectTypes = {}));
  if attrib[2] <> table() then
    warning("attribute(s) apply to nonexistent plot object:\n\t".
            expr2text(op(map({op(attrib[2])}, op, 1))))
  end_if;
  
  // check defaults: compliant with check-function or NIL ?
  attrib := split(attrib[1],
                  proc(x)
                    local checker, value;
                  begin
                    value := op(x, [2, 2]);
                    if testtype(value, DOM_NIL) then
                      return(TRUE)
                    else
                      if testtype(value, EA) then
                        value := plot::ExprAttribute::getExpr(value);
                      end_if;
                      checker := op(x, [2, 3]);
                      if testtype(checker, DOM_SET) then
                        return(contains(checker, value))
                      else  // apply checker procedure on value
                        return(not testtype(checker(op(x, [2, 4, 1]),
                                                    op(x, 1),
                                                    value),
                                            DOM_STRING));
                      end_if;
                    end_if
                  end_proc);
  if attrib[2] <> table() then
    printWarning("invalid default value for attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;
  
  // check defaults: have all inherited options a default value?
  attrib := split(attrib[1],
                  x -> _lazy_or(not op(x, [2, 1]) = "Inherited",
                                not op(x, [2, 2]) = NIL));
  if attrib[2] <> table() then
    printWarning("missing default value for inherited attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;
  
  // check defaults: have all mandatory options no default value?
  // There MUST be no defaults!  Defaults are only allowed in style sheets!
  attrib := split(attrib[1],
                  x -> _lazy_or(not op(x, [2, 1]) = "Mandatory",
                                op(x, [2, 2]) = NIL));
  if attrib[2] <> table() then
    printWarning("default value for mandatory attribute <> NIL",
                 map({op(attrib[2])}, op, 1))
  end_if;
  
  // check defaults: have all library options a read function?
  attrib := split(attrib[1],
                  x -> _lazy_or(not op(x, [2, 1]) = "Library",
                                domtype(op(x, [2, 5])) = DOM_PROC));
  if attrib[2] <> table() then
    printWarning("missing read function for library interface attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;

  // Library attributes must have 6 list entries; last one a set
  attrib := split(attrib[1],
                  x -> _lazy_or(not op(x, [2, 1]) = "Library",
                                not domtype(op(x, [2, 6])) = DOM_SET,
                                not domtype(op(x, [2, 6])) = DOM_LIST));
  if attrib[2] <> table() then
    printWarning("wrong type of interface wrapper for library interface attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;

  // all other attributes must have an inspector definition (6th list entry)
  attrib := split(attrib[1],
                  x -> _lazy_or(op(x, [2, 1]) = "Library",
                                nops(op(x, 2)) = 6));
  if attrib[2] <> table() then
    printWarning("inspector entry missing for attribute",
                 map({op(attrib[2])}, op, 1))
  end_if;


  // check inspector entry for consistency
  inspector := split(attrib[1], x-> not op(x, [2, 1]) = "Library");
  // inspector entry may be FAIL ->  no inspector entry
  inspector := split(inspector[1], x -> testtype(op(x, [2, 6]), DOM_LIST));
  inspector := split(inspector[1],
                     x -> testtype(op(x, [2, 6, 1]),
                                   Type::Union(DOM_STRING, 
                                               Type::ListOf(DOM_STRING))));
  if inspector[2] <> table() then
    printWarning("string or list expected for first entry of inspector entry of attribute",
                 map({op(inspector[2])}, op, 1))
  end_if;
  // inspector DOM type OK?
  inspector := split(inspector[1],
                     x ->
                     contains({"XMLAttribute", "Expr", "ColorFunc",
                               "Prop", "ExprSeq"}, op(x, [2, 6, 2])));
  if inspector[2] <> table() then
    printWarning("invalid DOM type in inspector entry of attribute",
                 map({op(inspector[2])}, op, 1))
  end_if;
  // inspector attribute type OK?
  inspector := split(inspector[1],
                     x -> op(x, [2, 6, 2]) in {"XMLAttribute", "Expr"} and
                          (domtype(op(x, [2, 6, 3])) = DOM_STRING and
                           contains(plot::inspector::PropertyType,
                                    op(x, [2, 6, 3])) or
                           contains(plot::inspector::PropertyType,
                                    op(x, [2, 6, 3, 1])) and
                           contains(plot::inspector::PropertyType,
                                    op(x, [2, 6, 3, 2]))) or
                     op(x, [2, 6, 2]) in {"Prop", "ExprSeq", "ColorFunc", "Expr"} and
                     op(x, [2, 6, 3]) <> FAIL or
                     op(x, [2, 6, 3]) = FAIL or
                     domtype(op(x, [2, 6, 3])) = DOM_SET);
  if inspector[2] <> table() then
    printWarning("invalid inspector attribute type definition of attribute",
                 map({op(inspector[2])}, op, 1))
  end_if;
  // inspector attribute type OK?
  inspector := split(inspector[1],
                     x -> op(x, [2, 6, 2]) = "XMLAttribute" and
                     (domtype(op(x, [2, 6, 3])) = DOM_STRING and
                      contains(plot::inspector::PropertyType,
                               op(x, [2, 6, 3])) or
                      contains(plot::inspector::PropertyType,
                               op(x, [2, 6, 3, 1])) and
                      contains(plot::inspector::PropertyType,
                               op(x, [2, 6, 3, 2]))) or
                     op(x, [2, 6, 2]) = "Prop" and
                     op(x, [2, 6, 3]) <> FAIL or
                     contains({"Expr", "ColorFunc", "ExprSeq"},
                              op(x, [2, 6, 2])));
  if inspector[2] <> table() then
    printWarning("invalid inspector attribute type definition of attribute",
                 map({op(inspector[2])}, op, 1))
  end_if;
  // inspector info string
  inspector := split(inspector[1],
                     x -> domtype(op(x, [2, 6, 4])) = DOM_STRING);
  if inspector[2] <> table() then
    printWarning("invalid inspector info string of attribute",
                 map({op(inspector[2])}, op, 1))
  end_if;
  // inspector recalc flag
  inspector := split(inspector[1],
                     x -> contains({TRUE, FALSE}, op(x, [2, 6, 5])));
  if inspector[2] <> table() then
    printWarning("invalid inspector recalc flag of attribute",
                 map({op(inspector[2])}, op, 1))
  end_if;

  // check for combine-function or FAIL ?
  attrib := split(attrib[1],
                  x -> contains({DOM_FAIL, DOM_PROC}, domtype(op(x, 5))));
  if attrib[2] <> table() then
    printWarning("combine-function or FAIL expected for attribute", 
                 map({op(attrib[2])}, op, 1))
  end_if;

  res := {};
  for i in plot::attributes do
    tmp := op(i, 2)[4];
    remaining := tmp minus plot::animatedObjects minus {"Ode2d", "Ode3d"}:
    if op(i, 2)[1] <> "Inherited" or remaining = tmp or
      remaining = {} then
     // only graphical primitives or only something else
    else
      res := res union {op(i, 1)};
    end_if;
  end_for;
  if res <> {} then
    printWarning("conflicting usage of attribute", res);
  end_if;
end:
