/*  Collection for functions for attribute/option checking */

/** plot::checkOptions:
 * 
 *  Utility function for plot::setDefault and other methods.  
 *   
 *
 *   Arguments:
 *       objType  -- DOM_STRING identifying the type
 *       l        -- DOM_LIST of equations with OptionName = Value
 *   Return value: a DOM_TABLE
 */
plot::checkOptions :=
proc(objType : DOM_STRING, l : DOM_LIST)
  local i, j, t, res, checker, op1, mapFunc;
  name plot;
begin
  t := table():
  for i in l do
    if not testtype(i, "_equal") then
      context(hold(error)("equation expected"))
    end_if;
    op1 := op(i, 1);
    if domtype(op1) = plot::StyleSheetEntry then
      objType := plot::StyleSheetEntry::objType(op1);
      op1 := op1::dom::attribute(op1);
      mapFunc := proc(x)
                 begin
                   if domtype(x) = plot::StyleSheetEntry then
                     x
                   else
                     plot::StyleSheetEntry(objType, x)
                   end_if;
                 end:
    else
      mapFunc := id:
    end_if;
    if not contains(plot::attributes, op1) then
      error("invalid attribute name: ".expr2text(op1))
    end_if;
    checker := plot::attributes[op1][3];
    if domtype(checker) = DOM_SET or
      domtype(checker) = DOM_LIST then
      if domtype(checker) = DOM_LIST then
        //  different values for 2D and 3D allowed, e.g. PointStyle
        if contains(plot::all2dObjects, objType) then
          checker := op(checker, 1);
        elif contains(plot::all3dObjects, objType) then
          checker := op(checker, 2);
        else
          // neither 2D nor 3D:  Canvas;  allow the union of values
          checker := _union(op(checker))
        end_if
      end_if;
      // check if value is in the given set
      if contains(checker, op(i, 2)) then
        res := op1 = op(i, 2);
      else
        checker := DOM_SET::sort(checker);
        checker := map(checker,
            x -> if domtype(x) = DOM_STRING then x else expr2text(x) end);
        if nops(checker) = 2 then
          res := "expecting '".checker[1]."' or '".checker[2]."'"
        else
          res := _concat("expecting ",
                         (("'", checker[j],"', ")  $
                          j = 1 .. nops(checker)-1),
                         "or '", checker[-1], "'"
                         )
        end_if;
      end_if;
    else
      // checker is an explicit check procedure, call it
      res := checker(objType, op1, op(i, 2));
    end_if;
    if domtype(res) = DOM_STRING then
      error(res." for attribute '".expr2text(op(i, 1))."' in ".objType." object");
    end_if;
    if type(res) = "_equal" then
      t[mapFunc(op(res, 1))] := op(res, 2);
    else
      assert(testtype(res, Type::SequenceOf("_equal")));
      for i in res do
        t[mapFunc(op(i, 1))] := op(i, 2);
      end_for;
    end_if
  end_for;
  return(t);
end_proc:


/** plot::setDefaultInternal:
 *
 *   Checks if ALL option names and all values are valid and then
 *   changes the table plot::attributes accordingly.
 *
 *   Arguments:    one or more equation(s)
 *   Return value: TRUE, iff options set, otherwise an error is raised
 */
plot::setDefaultInternal :=
proc()
  local values, i, op1, attrName;
begin
  values := plot::checkOptions("Default", [args()]);
  stdlib::syseval((
    for i in values do
      op1 := op(i,1);
      attrName := expr2text(op1);            
      if domtype(op1) = DOM_IDENT and attrName[-1] <> "_" then
        plot::attributes[op1] := subsop(plot::attributes[op1],
                                        2 = op(i, 2))
      end_if;
    end_for;
  ));
  TRUE
end_proc:



/** plot::setDefault:
 *
 *   Checks if ALL option names and all values are valid and then
 *   changes the table plot::attributes accordingly.
 *
 *   Arguments:    one or more equation(s)
 *   Return value: old defaults, iff options set, otherwise an error is raised
 */
plot::setDefault :=
proc()
  local values, i, op1, res;
begin
  res := null();
  values := plot::checkOptions("Default", [args()]);
  stdlib::syseval((
    for i in values do
      op1 := op(i,1);
      if expr2text(op1)[-1] = "_" then
        // internal value overwrites converted value
        next;
      end_if;
      if domtype(op1) = DOM_IDENT then
        if contains({OutputFile, OutputOptions}, op1) then
          plot::attributes[op1] := subsop(plot::attributes[op1],
                                          2 = op(i, 2))
        else
          error("only  qualified access such as 'plot::".
                op(plot::attributes[op1][4], 1)."::".op1.
                " = ".expr2text(op(i, 2))."' allowed");
        end_if;
      else
        // make object specific entry
        assert(domtype(op1) = plot::StyleSheetEntry);
        res := res, plot::getDefault(op1);           
        (slot(plot, op1::dom::objType(op1)))::styleSheet[op1::dom::attribute(op1)] := plot::ExprAttribute::getExpr(op(i,2));
      end_if;
    end_for;
  ));
  res
end_proc:



/** plot::getDefault:
 *
 *   Checks if ALL option names and all values are valid and then
 *   changes the table plot::attributes accordingly.
 *
 *   Arguments:    one or more equation(s)
 *   Return value: TRUE, iff options set, otherwise an error is raised
 */
plot::getDefault :=
proc(attrName)
  local myAttrName, myObjType, myEntry, myIndets, myDefaults, res,
        extra, attrNameStr;
begin
  if attrName = FAIL then
    return(FAIL)
  end:
  if not testtype(attrName, Type::Union(DOM_IDENT, plot::StyleSheetEntry)) then
    error("invalid argument");
  end_if;
  if domtype(attrName) = plot::StyleSheetEntry then
    if contains((slot(plot, attrName::dom::objType(attrName)))::styleSheet,
                attrName::dom::attribute(attrName)."_") then
      return((slot(plot, attrName::dom::objType(attrName)))::styleSheet[
                                attrName::dom::attribute(attrName)."_"]);
    elif contains((slot(plot, attrName::dom::objType(attrName)))::styleSheet,
                attrName::dom::attribute(attrName)) then
      res := (slot(plot, attrName::dom::objType(attrName)))::styleSheet[
                                        attrName::dom::attribute(attrName)];
      if contains(plot::allFontAttributes,
                  attrName::dom::attribute(attrName)) then
        return(plot::font2list(res))
      else
        return(res)
      end_if;
    else
      myAttrName := attrName::dom::attribute(attrName);
      myObjType  := attrName::dom::objType(attrName);
      if contains(plot::attributes, myAttrName) and
        plot::attributes[myAttrName][1] = "Library"  then
        myEntry := plot::attributes[myAttrName];
        if domtype(myEntry[6]) = DOM_SET then
          // look in plotDom::libInterface
          extra := (slot(plot, myObjType))::libInterface;
          if contains(extra, myAttrName) then
            myEntry := extra[myAttrName];
          else
            return(NIL)
          end_if
        elif myObjType = "Canvas" then
          if nops(myEntry[6]) = 1 then
            // no dimensional difference
            myEntry := op(myEntry[6][1]);
          else
            // Canvas has no dimension; dont know what to do
            return(NIL)
          end_if
        elif contains(plot::all2dObjects, myObjType) then
          myEntry := op(myEntry[6][1]);
        else // 3D object
          if nops(myEntry[6]) = 1 then
            // no dimensional difference
            myEntry := op(myEntry[6][1]);
          else
            // use 3D-entry
            myEntry := op(myEntry[6][2]);
          end_if
        end_if;
        
        myIndets := [op(indets(myEntry))];
        myDefaults := map(map(myIndets,
                              x -> plot::StyleSheetEntry(myObjType, x)),
                          plot::getDefault);
        if has(myDefaults, NIL) then
          return(NIL)
        end_if;

        // do return NIL when Colors is accesses via Color and there
        // is more than one color in the list
        if myAttrName = Color and myIndets = [Colors] and
          nops(myDefaults[1]) > 1 then
          return(NIL);
        end_if;

        
        res := subs(myEntry, op(zip(myIndets, myDefaults, _equal)));
        if domtype(res) = DOM_SET then
          // shortcut for serveral attributes with equal values
          if nops(res) = 1 then
            // values are equal
            res := op(res)
          else
            return(NIL)
          end_if
        end_if;
        if contains(plot::allFontAttributes, myAttrName) then
           res := plot::font2list(res)
        end_if;
        return(res)
      else
        attrName := attrName::dom::attribute(attrName);
        // fall through
      end_if
    end_if;
  end_if;
  attrNameStr := "".attrName;
  if contains(plot::attributes, attrName) then
    if plot::attributes[attrName][1] = "Library" then
      error("'".attrName."' is a library shortcut for ".
            expr2text(indets(plot::attributes[attrName][6])).
            ";\nlook for these attributes directly")
    else
      res := plot::ExprAttribute::getExpr(plot::attributes[attrName][2]);
    end_if;
  elif contains(plot::attributes, hold(``).attrNameStr[1..-2]) and
    attrNameStr[-1] = "_" then
    res := plot::attributes[hold(``).attrNameStr[1..-2]][2]
  else
    error("unkown attribute ".attrName)
  end_if:
  if contains(plot::allFontAttributes, attrName) then
    res := plot::font2list(res)
  end_if;
  return(res)
end_proc:


/** plot::getInherited:
 *
 *   Get default values for all inherited options from plot::attributes.
 *   
 *   Arguments:     plot Object (OPTIONAL)
 *   Return value:  A table
 */
plot::getInherited :=
proc(obj = FAIL)
  local inherited, inheritedIdx, res;
begin
  if obj = FAIL then
    res := table(map(plot::allInheritedAttributes,
                        x -> x = plot::attributes[x][2]));
  else
    if obj::dom::objType = "Canvas" then
      res := table(map(plot::legalAttributes["Canvas"][5],
                        x -> x = plot::attributes[x][2]),
                   op(plot::Canvas::styleSheet));
    else
      res := table();
    end_if;
    
    inherited := {op((extop(obj, 1))::inherited)};
    inheritedIdx := map(inherited, op, 1);
    inherited := table(inherited);
    res := table(res, map(inheritedIdx, x -> x = inherited[x]));
  end_if;
  res;
end_proc:

/** plot::getAttributeNames:
 *
 *   The resulting set will contain all attribute names or all
 *   adissible attribute names for the given object type typ.
 *
 *   Arguments:     (opt) typ 
 *   Return value:  a DOM_SET of valid index entries in plot::attributes
 */
plot::getAttributeNames :=
proc(typ = "all" : DOM_STRING)
begin
  if typ = "all" then
    return(map({op(plot::attributes)}, op, 1))
  else
    // return only those attributes which are valid for object type
    // typ.
    _union(op(plot::legalAttributes[typ]));
 end_if;
end_proc:

/** plot::getOwnAttributeNames:
 *
 *   The resulting set will contain all attribute names or all
 *   adissible attribute names for the given object type typ.
 *
 *   Arguments:     typ 
 *   Return value:  a DOM_SET of index entries in plot::attributes for typ
 */
plot::getOwnAttributeNames :=
proc(typ : DOM_STRING)
begin
  map({op(select(plot::attributes, x -> contains(op(x, [2, 4]), typ)))}, op, 1);end_proc:


/** plot::getOptionAndValuesNames:
 *
 *   Get all Option names from plot::attributes.
 *   The resulting set will contain all attribute names and
 *   admissible values for the attributes.
 *
 *   Arguments:     --
 *   Return value:  a DOM_SET of valid index entries in plot::attributes
 */
plot::getOptionAndValuesNames :=
proc()
begin
  select(// all attribute names
         map({op(plot::attributes)}, op, 1) union
         // some values checked explicitly in check procedures
         plot::extraValues union
         // values from sets
         select(map(op(plot::attributes), op, [2, 3]), testtype, DOM_SET),
         testtype, DOM_IDENT)
end_proc:

/** plot::getOptionNames:
 *
 *   Get all Option names from plot::attributes.
 *   The resulting set will contain all attribute names but not
 *   the admissible values for the attributes (see above).
 *
 *   Arguments:     --
 *   Return value:  a DOM_SET of valid index entries in plot::attributes
 */
plot::getOptionNames := () -> map({op(plot::attributes)}, op, 1):


/**  plot::getOptionals:
 *
 *   Get all optional attribute values from plot::attributes
 *   for the given object type.  These attributes are split into
 *   expression values and non-expression values.  Then merge in
 *   the optional attributes given in the given object obj.
 * 
 *   Arguments:      -- a plot object
 *                  
 *   Return value:  a list of 3 DOM_TABLEs:
 *       [non-expr attributes, expr attributes, extra attributes]
 */
plot::getOptionals :=
proc(obj)
  local typ, possibleOpt, optTable, i, rep, default, optExprTable,
        styleSheetDefaults, mapfunc, extra, tmp, dummy;
begin
  typ := obj::dom::objType;

  // optional and mandatory attributes which have NO expressions values
  optTable := table():
  // optional and mandatory attributes which have expressions values
  optExprTable := table():
  extra := table():

  // attributes to be checked in plot::attributes
  possibleOpt := plot::legalAttributes[typ][1] union
                 plot::legalAttributes[typ][2];

  // get global defaults for optional attributes
  for i in possibleOpt do
    default := op(plot::attributes[i], 2);
    if default <> NIL then
      if testtype(default, plot::ExprAttribute) then
        optExprTable[i] := default;
      else
        optTable[i] := default;
      end_if;
    end_if;
  end_for;
  
  // merge in settings from style sheets
  // start with optional style sheet entries
  styleSheetDefaults := select(obj::dom::styleSheet,
                         y -> plot::attributes[op(y, 1)][1] = "Optional"):
  // apply check functions
  styleSheetDefaults := plot::checkOptions(typ, [op(styleSheetDefaults)]);
  mapfunc := x -> domtype(op(x, 2)) = plot::ExprAttribute or
                   type(op(x, 2)) = "_exprseq" and
                   map({op(x, 2)}, domtype) = {plot::ExprAttribute};
  
  optExprTable := table(optExprTable,
                        select(styleSheetDefaults,
                               mapfunc)):
  extra := [op(select(styleSheetDefaults, _not@mapfunc))];
  [extra, tmp, dummy] := split(extra, x -> expr2text(op(x,1))[-1] = "_");
  optTable := table(optTable, tmp);


  // merge in settings with expression type from obj
  rep := (extop(obj, 1))::legalExpr;
  optExprTable := table(optExprTable, rep);
  // merge in settings with non-expression type from obj
  rep := (extop(obj, 1))::legal;
  optTable := table(optTable, rep);

  if contains(optTable, OutputUnits) then
    optTable[OutputUnits] := case optTable[OutputUnits]
                               of unit::mm   do "MM"; break;
                               of unit::inch do "In"; break;
                               of unit::pt   do "Pt"; break;
                             end_case;
  end_if;

  [optTable, optExprTable, table(extra, op((extop(obj, 1))::extra))]
end_proc:


/**  plot::combineHints:
 *   gets the hint attribute lists from all children in list and calls
 *   the combining functions for all those attributes.
 * 
 *   Arguments:     the child objects
 *   Return value:  list of equations
 */
plot::combineHints :=
proc()
  local childrenHints, hints, i, idx, j, k, values;
begin
  childrenHints := map(args(), obj -> table(op((extop(obj, 1))::hints)));
  // childrenHints is now a list of tables
  
  hints := [];
  for i from 1 to nops(childrenHints) do
    for j in childrenHints[i] do
      idx := op(j, 1);
      values := [op(j, 2)];
      for k from i+1 to nops(childrenHints) do
        // scan following tables for idx
        if contains(childrenHints[k], idx) then
          values := values.[childrenHints[k][idx]];
          delete childrenHints[k][idx];
        end_if;
      end_for;
      if nops({op(values)}) = 1 then
        // if there are serveral identical hints, use the first, no conflict
        values := [values[1]];
      end_if;
      if nops(values) > 1 then
        // more than one hint for an attribute, try to combine
        if plot::attributes[idx][5] <> FAIL then
          hints := hints.[plot::attributes[idx][5](idx, map(values, plot::ExprAttribute::getExpr))];
        else
          error("conflicting hints for option '".idx."' can not be combined");
        end_if;
      else
        // only one hint, add it
        hints := hints.[idx=plot::ExprAttribute::getExpr(values[1])];
      end_if;
    end_for;
  end_for;
  hints;
end_proc:
