//      

/*++
    Date:    2008-09-30
    Version: 26

    plot::easy -- easy plotting

    Syntax:

    plot::easy(<, po> <, pa> <, go> <, ga> ...>)

    po, ... : plot (graphical) objects
    pa, ... : plot graphical) attributes see plot::new for details
    go, ... : guess objects
    ga, ... : guess attributes see plot::easy for details

    Returns its input parameter after trying to convert all non-plot objects
    into regular plot (graphical) objects. The regular plot objects and plot
    attributes given to plot::easy are not changed by it. Guess attributes
    are filtered out after interpreting them. The currently accepted guess
    attributes are:

     `#3` = `#3D`,           // = create 3d scene instead of 2d scene, if possible
     `#A` = `#Arrows`,       // = display arrows instead of points
     `#C` = `#Constrained`,  // = same scaling for all axes (Scaling=Constrained)
     `#D` = `#Debug`,        // = display debug messages (not in final version)
     `#G` = `#Grid`,         // = display grid lines (GridVisible)
     `#L` = `#Legend`,       // = create and display legends (LegendVisible)
     `#O` = `#Origin`,       // = display axes origin in any case
     `#P` = `#Points`,       // = display arrows/curves as points and display mesh points (ggf. PointsVisible)
     `#X` = `#XRange`,       // = set XRange of the ViewingBox
     `#Y` = `#YRange`,       // = set YRange of the ViewingBox
     `#Z` = `#ZRange`,       // = set ZRange of the ViewingBox
     `#<name>`               // = short-cut for Color=RGB::<name>..., if RGB::<name> exists
                             // = <name> is in html color style (e.g. #FF00FF or #FF00FF33)
     `#<name>..#<name2>`     // = as above, plus {Line|Fill}Color=RGB::<namew>, if RGB::<name2> exists

    Currently accepted objects and their regular plot domains are:

     - number, identifier, polynomials           => plot::Function2d
     - [arithex,arithex <,arithex>]              => plot::Point{2|3}d, plot::Arrow{2|3}d or plot::Curve{2|3}d
     - matrix([arithex,arithex <,arithex>])      => plot::Arrow{2|3}d
     - [[arithex,arithex <,arithex>]...]         => plot::Polygon{2|3}d or plot::Arrow{2|3}d
     - matrix([[arithex,arithex <,arithex>]...]) => plot::Polygon{2|3}d or plot::Arrow{2|3}d
     - [[x,y<,z>,RGBa],...]                      => plot::PointdList{2|3} or plot::Arrow{2|3}d
     - #x=n,#y=n,#z=n                            => plot::Line2d or plot::Plane
     - [arithex,arithex <,arithex>]=text|proc    => plot::Text2d or plot::Text3d
     - arithex=arithex                           => plot::Implicit{2|3}d
     - arithex<arithex, arithex<=arithex         => plot::Inequality
     - [arithex<arithex,arithex=arithex,...]     => plot::Inequality
     - other arithmetical expression             => plot::Function{2|3}d
     - [cond,arithex], [[cond,arithex],...]      => plot::Function{2|3}d (piecewise)

    plot::easy uses curly bracktes '{....}' to define a group of objects.
    This objects share local regular plot attributes and guess attributes
    and have the same color (if this is not explicitely changed).
       
    plot::easy also accepts Colors=[], Mesh=n|[n,m], Submesh=n|[n,m].

    plot::easy is called by plot::new. The behavior of plot::new should
    not be changed but only be extended.

    Example:

    >> plot::easy(1, x, sin(t), t = -PI..PI)
    >> plot::easy([1,2], matrix([1,2]))
    >> plot::easy([1,2], {matrix([1,2]), `#Legend`})
    >> plot::easy([1,2], `#Origin`)
    >> plot::easy([1,2], `#Arrows`)
    >> plot(sin(x), `#x`=1, `#y`=PI)


    Technical Information:

    1. The input parameters are split into regular plot objects, regular
       plot attributes, guess objects, guess attributes and guess ranges.
    2. First, the dimension of the scene might be 2, 3 or undefined. The
       dimension can either be determined during the conversion process
       or is finally set to a default in the end. 'piecewise' is used to
       delay the creation of the graphical objects and attributes until
       the scene dimension has been determined.
    3. plot::easy trys to convert guess objects to regular plot objects.
       The guess ranges that corresponds to the variables of a guess object
       are used when creating the regular plot object. If not enough ranges
       are available, the default -5..5 is taken.
    4. If a guess object cannot be converted then it is returned unchanged.
       If plot::easy is called by plot::new, then plot::new will display
       an error message.

    Wish List:

    - #U = Unconstrained  (siehe #C), Jrgen 2007-07
      => nicht so wichtig, lieber Schnitstelle klein halten
    - #? Koordinatensystem anzeigen/nicht anzeigen
      => nicht so wichtig, lieber Schnitstelle klein halten
    - [x,y<,z>] = "text" mit mehr Gestaltungsmglichkeiten,
      insbesondere zentrierter Text, ggf. [x, y] = ["text"]?
      => schn, aber die Syntax ["text", Center] wird m.E.
      so komplex, wie die Original-Syntax. Das macht dann
      wenig Sinn.
    - Kreise/Kugeln? [[x,y], r] bzw. [[x,y,z], r]
      => Ist da Bedarf? Muss sich erst noch zeigen.

++*/

// ----------------------------------------------------------------------
// Konverter fr #Red, #Red.[0.3] sowie #FFFF00, #FFFF0033 Farbattribute
// liefert FAIL, falls kein entsprechendes Farbattribut bergeben wurde.
// ----------------------------------------------------------------------
plot::easy_helper_attrHashColor := proc(hashColor)
    local isHtmlColor, c1, c2, color, trans;
begin
    if type(hashColor) = "_range" then  
       if (c1:= plot::easy_helper_attrHashColor(op(hashColor,1))) <> FAIL and
          (c2:= plot::easy_helper_attrHashColor(op(hashColor,2))) <> FAIL then
          return(c1..c2);
       end_if;
       return(FAIL);
    end_if;
    if domtype(hashColor) <> DOM_IDENT then
       return(FAIL);
    end_if;
    hashColor:= "".hashColor;
    if hashColor[1] <> "#" then
       return(FAIL);
    end_if;
    isHtmlColor:= x -> (
    if strmatch(x, "^#[A-Fa-f0-9]{6}([A-Fa-f0-9]{2})?$") then
       [float(text2int(x[2..3], 16))/255.0,
        float(text2int(x[4..5], 16))/255.0,
        float(text2int(x[6..7], 16))/255.0,
        if length(x)=9 then float(text2int(x[8..9], 16))/255.0 else null(); end];
    else
       FAIL;
    end_if
    );
    if (color:= isHtmlColor(hashColor)) <> FAIL then
       return(color);
    end_if;
    color:= (hashColor)[2..-1];
    trans:= text2list(color, ["[","]"]);
    if nops(trans) = 4 and
       trans[1] > "" and trans[2] = "[" and trans[4] = "]" and
       traperror((trans[3]:= float(text2expr(trans[3])))) = 0 and
       testtype(trans[3], Type::Numeric) then
       color:=  trans[1];
       trans:= [trans[3]];
    else
       trans:= [];
    end_if;
    if contains(RGB::ColorNames(color), hold(``).color) > 0 then
       return(slot(RGB,color).trans);
    end_if;
    return(FAIL);
end_proc;


plot::easy:= proc()
   // ----------------------------------------------------------------------
   // 'dim' muss fr piecewise-Konstrukte als Bezeichner definiert sein !!!
   // ----------------------------------------------------------------------
   save dim;  // wird im Folgenden gelscht!

   // -------------------------------------------------------------------------
   // Lokale Parameter aller Szenen
   // -------------------------------------------------------------------------
   local guessAttributes, guessAttributesSet,
         has3D, hasArrows, hasDebug, hasOrigin, hasPoints,
         viewXmin, viewXmax,
         inScene, moreThanOneScene, setSceneDimension,
         userColorList, userDefaultAttributes,
         doScene;
begin
   // -------------------------------------------------------------------------
   // Attribute (Optionen), die plot::easy fr seine Objekte akzeptiert
   // -------------------------------------------------------------------------
   guessAttributes:= {
     `#3` = `#3D`,           // = create a 3d scene instead of 2d, if possible
     `#A` = `#Arrows`,       // = Punkte als Pfeile zeichnen
     `#C` = `#Constrained`,  // = Punkte als Pfeile zeichnen (Scaling=Constrained)
     `#D` = `#Debug`,        // = Debug-Meldung ausgeben
     `#G` = `#Grid`,         // = Koodinatengitter einzeichnen (GridVisible)
     `#L` = `#Legend`,       // = Legenden anzeigen (LegendVisible)
     `#O` = `#Origin`,       // = Den Koordinatenursprung mit aufnehmen
     `#P` = `#Points`,       // = Punkte anzeigen (ggf. PointsVisible)
     `#X` = `#XRange`,       // = set XRange of the ViewingBox
     `#Y` = `#YRange`,       // = set YRange of the ViewingBox
     `#Z` = `#ZRange`,       // = set ZRange of the ViewingBox
     null()
   };
   guessAttributesSet:= map(guessAttributes, e->(e[1],e[2]));

   // -------------------------------------------------------------------------
   // Global aktive Attribute zurcksetzen
   // -------------------------------------------------------------------------
   [has3D, hasArrows, hasDebug, hasOrigin, hasPoints]:= [FALSE $ 5];

   // -------------------------------------------------------------------------
   // userColorList: vom Anwender spezifizierte Farbliste oder Standardliste
   // -------------------------------------------------------------------------
   userColorList := RGB::ColorList;

   // -------------------------------------------------------------------------
   // userDefaultAttributes: vom Anwender spezifizierte Default-Attribute
   // -------------------------------------------------------------------------
   userDefaultAttributes:= null();
   
   // -------------------------------------------------------------------------
   // inScene-Flag: zeigt an, ob wir uns in einer {{..}}-Szene befinden
   // -------------------------------------------------------------------------
   moreThanOneScene:= FALSE;
   inScene:= FALSE;

   // -------------------------------------------------------------------------
   // Dimension der Szene als undefiniert kennzeichnen
   // -------------------------------------------------------------------------
   delete dim;

   // -------------------------------------------------------------------------
   // Dimension der Szene nur setzen, falls diese noch undefiniert ist
   // -------------------------------------------------------------------------
   setSceneDimension:= proc(d)
      name plot::easy[setSceneDimension];
   begin
      if dim = hold(dim) then
         dim:= d;
      elif dim <> d then
         error("dimension mismatch");
      end_if;
   end_proc;

   // -------------------------------------------------------------------------
   // Maximale Ausdehnung der Szene in x-Richtung (soweit mglich) berwachen
   // Experimentell fr mgliche Erweiterung der aktuellen Funktionalitt
   // -------------------------------------------------------------------------
   [viewXmin, viewXmax]:= [infinity,-infinity];

   // -------------------------------------------------------------------------
   // Konvertieren der einzelnen Szenen
   // -------------------------------------------------------------------------
   doScene:= proc()
      name plot::easy[doScene];
      local nextColor2, nextColor3, currColor2, currColor3, currColorUsed, 
            i, t, dummy,  /* __c, needed when used with MuPAD Pro 4.0.6 */
            // ----------------------------------------------------------------
            // Lokale Parameter eine Szene
            // ----------------------------------------------------------------
            attrStack, attrStackDepth,
            attrStackAll, attrStackPop, attrStackPush,
            userSceneAttributes,
            doGuess,
            hasUndefined,
            result;
      save x, y, z; // werden gelscht und als default ranges verwendet !!!    
   begin
      // ----------------------------------------------------------------------
      // x, y, z werden in getRanges() als default ranges verwendet !!!
      // ----------------------------------------------------------------------
      sysdelete(x, y, z);
      
      // ----------------------------------------------------------------------
      // Argumente werden im Folgenden zum Teil noch konvertiert / gefiltert.
      // ----------------------------------------------------------------------
      result:= [args()];
      
      // -------------------------------------------------------------------------
      // userDefaultAttributes: vom Anwender spezifizierte Default-Attribute
      // -------------------------------------------------------------------------
      userSceneAttributes:= userDefaultAttributes;
      
      // ----------------------------------------------------------------------
      // Farbgenerator fr die von plot::easy generierten plot-Objekte
      // ----------------------------------------------------------------------
      %if 10*version()[1]+version()[2] < 44
         then nextColor2:= (__c:= -1; ()-> (__c:= __c +1; userColorList[(__c mod nops(userColorList))+1]));
         else nextColor2:= RGB::generator(userColorList);
      end_if;
      %if 10*version()[1]+version()[2] < 44
         then nextColor3:= (__c:= -1; ()-> (__c:= __c +1; userColorList[(__c mod nops(userColorList))+1]));
         else nextColor3:= RGB::generator(userColorList);
      end_if;
      currColorUsed:= TRUE;

      // ----------------------------------------------------------------------
      // Attribut-Stack fr geschachtelte Gruppen
      // ----------------------------------------------------------------------
      attrStack     := table();
      attrStackDepth:= 0;
      attrStackAll  := proc(e,revertStack=FALSE)
         name plot::easy[attrStackAll];
      begin	
         if revertStack then
            map(map(revert([attrStack[i] $ i=1..attrStackDepth]),_index,e),op);
         else
            map(map([attrStack[i] $ i=1..attrStackDepth],_index,e),op);
         end_if;
      end_proc;
      attrStackPop  := () -> (delete attrStack[attrStackDepth]; attrStackDepth:= attrStackDepth-1);
      attrStackPush := proc(o)
         name plot::easy[attrStackPush];
         local i, dummy;
      begin
         attrStackDepth:= attrStackDepth + 1;

         // -------------------------------------------------------------------
         // Parameter-Sequence 'o' zerlegen und auf den Attribut-Stack legen
         // -------------------------------------------------------------------
         [attrStack[attrStackDepth][`#ploto`], o, dummy]:= split(o,
                   e -> bool(type(e)="plotObject"));

         [attrStack[attrStackDepth][`#plota`], o, dummy]:= split(o,
                   e -> _lazy_or(
                        _lazy_and(type(e)="_equal",  type(lhs(e))=plot::StyleSheetEntry),
                        _lazy_and(type(e)="_equal",  contains(plot::attributes, lhs(e))),
                        _lazy_and(type(e)=DOM_IDENT, contains(plot::attributes,     e)))
                        );

         [attrStack[attrStackDepth][`#guessa`], o, dummy]:= split(o,
                   e->contains(map(guessAttributes,e->(e[1],e[2])), e));

         [attrStack[attrStackDepth][`#guessr`], o, dummy]:= split(o,
                   e->_lazy_and(type(e)="_equal", type(lhs(e))=DOM_IDENT or type(lhs(e))="_index", type(rhs(e))="_range"));

         attrStack[attrStackDepth][`#guesso`]:= o;

         // -------------------------------------------------------------------
         // guess-Objekte der Form '#Red' als plot-Attribute 'Color=RGB::Red'
         // in Liste der aktuellen plot-Attribute aufnehmen. Spter, bei der
         // Konvertierung der an plot::easy bergebenen Objekte  (siehe auch
         // 'FARBWERTE'), muss explizit 'Color=RGB::Red' generiert werden!!!
         // -------------------------------------------------------------------
         for i from 1 to nops(attrStack[attrStackDepth][`#guesso`]) do
            dummy:= plot::easy_helper_attrHashColor(attrStack[attrStackDepth][`#guesso`][i]);
            if dummy <> FAIL then
               if type(dummy) = "_range" then
                  attrStack[attrStackDepth][`#plota`]:=
                  attrStack[attrStackDepth][`#plota`].[Color = op(dummy,1)];
               else
                  attrStack[attrStackDepth][`#plota`]:=
                  attrStack[attrStackDepth][`#plota`].[Color = dummy];
              end_if;
            end_if;
         end_for;

         // -------------------------------------------------------------------
         // Falls mglich, dann die Dimension der aktuellen Szene bestimmen
         // -------------------------------------------------------------------
         if nops(select(attrStack[attrStackDepth][`#ploto`],
                        e->bool(slot(slot(e,"dom"),"dimension")=2))) > 0 then
            setSceneDimension(2);
         end_if;
         if nops(select(attrStack[attrStackDepth][`#ploto`],
                        e->bool(slot(slot(e,"dom"),"dimension")=3))) > 0 then
            setSceneDimension(3);
         end_if;

         // ------------------------------------------------------------------
         // 3d-Hint `#3D` in globale guess-Attribute bernehmen. Die
         // Auswertung erfolgt nach der Konvertierung aller guess-Objekte.
         // -------------------------------------------------------------------
         if contains(attrStack[attrStackDepth][`#guessa`], `#3D`) > 0 then
            has3D:= TRUE;
         end_if;

         // -------------------------------------------------------------------
         // Punkte als Pfeile zeichnen aktivieren
         // -------------------------------------------------------------------
         if contains(attrStack[attrStackDepth][`#guessa`], `#Arrows`) > 0 then
            if attrStackDepth = 1 then
               hasArrows:= TRUE;
            end_if;
         end_if;

         // -------------------------------------------------------------------
         // Debug-Messages aktivieren
         // -------------------------------------------------------------------
         if contains(attrStack[attrStackDepth][`#guessa`], `#Debug`) > 0 then
            if attrStackDepth = 1 then
               hasDebug:= TRUE;
            end_if;
         end_if;

         // -------------------------------------------------------------------
         // Koordinatenursprung mit einzeichnen
         // -------------------------------------------------------------------
         if contains(attrStack[attrStackDepth][`#guessa`], `#Origin`) > 0 then
            hasOrigin:= TRUE;
         end_if;

         // -------------------------------------------------------------------
         // Punkte einzeichnen (ggf. PointsVisible)
         // -------------------------------------------------------------------
         if contains(attrStack[attrStackDepth][`#guessa`], `#Points`) > 0 then
            if attrStackDepth = 1 then
               hasPoints:= TRUE;
            end_if;
         end_if;
      end_proc;

      // ----------------------------------------------------------------------
      // Konvertierungsfunktion fr Gruppen und Objekte
      // ----------------------------------------------------------------------
      doGuess:= proc()
         name  plot::easy[doGuess];
         local o, doConvert, hasValues;
      begin
         o:= args();

         // -------------------------------------------------------------------
         // Die zentrale Konvertierungsfunktion fr guess-Objekte
         // -------------------------------------------------------------------
         doConvert:= proc(/* objects */)
            name  plot::easy[doConvert];
            local o, includeAttributes, getRanges, i, j, vars, t, d, dummy, 
                  isArrow, isPointList2dWithColor, isPointList3dWithColor,
                  isListOfPiecewise, var1;
         begin
            o:= args();
            isArrow:= FALSE;

            // ----------------------------------------------------------------
            // plot-Attribute fr guess-Objekte aus den bergebenen Attributen
            // und dem Attribut-Stack zusammenmischen.
            // ----------------------------------------------------------------
            includeAttributes:= proc(ot=FAIL, pa=[], opa=[])
               name plot::easy[includeAttributes];
               local ago, fgo, as, fgs, oa, text, opaHasCo, opaHasLS;
            begin
               text:= contains({plot::Text2d,plot::Text3d},ot);

               // Menge gltiger plot-Attribute fr 'ot'
               if ot = FAIL then
                  ago:= map({op(plot::attributes)}, e->hold(``).op(e,1));
               else
                  ago:= map({op(ot)}, e->hold(``).op(e,1));
               end_if;

               // Menge gltiger Farbattribute fr 'ot'. Es werden nur die
               // "Haupt-Farbattribute" bercksichtigt, nicht LineColor2
               // und FillColor2!!
               fgo:= ago intersect {PointColor, LineColor, FillColor, Color};

               // Menge aller aktuellen plot-Attribute vom Attribut-Stack
               as:= {op(attrStackAll(`#plota`))};

               // Menge der fr 'ot' gltigen Farbattribute vom Attribut-Stack
               fgs:= select(as, e->contains(fgo,op(e,1)));

               // Color/LineStyle in opa (=optional plot attributes) enthalten?
               opaHasCo:= bool(contains(map(opa,op,1), Color) > 0);
               opaHasLS:= bool(contains(map(opa,op,1), LineStyle) > 0);

               // Wenn kein fr 'ot' gltiges Farbattribut auf dem Attribut-Stack
               // liegt, genau dann die "Hauptfarbe" automatisch setzen und die
               // aktuelle Farbe "entwerten"
               if fgs = {} and not opaHasCo and not text then
                  if ot <> FAIL and slot(ot,"dimension") = 3 then
                     // Im 3D-Fall, bei Verwendung der Default-Farbtabelle,
                     // die Farbe Blau berspringen. Dies simuliert das Ver-
                     // halten von plotfunc3d.
                     if userColorList = RGB::ColorList and currColor3 = RGB::Blue then                
                        currColor3:= nextColor3();
                     end_if;
                     oa:= Color = currColor3;
                  else
                     oa:= Color = currColor2;
                  end_if;
                  currColorUsed:= TRUE;
               else
                  oa:= null();
               end_if;

               // Color ggf. aus opa entfernen
               if opaHasCo and contains(map(fgs,op,1), Color) then;
                  opa:= select(opa, e->bool(op(e,1)<>Color));
               end_if;

               // LineStyle ggf. aus opa entfernen
               if opaHasLS and contains(map(as,op,1), LineStyle) then;
                  opa:= select(opa, e->bool(op(e,1)<>LineStyle));
               end_if;

               // Einen Legenden-Text explizit setzen!
               if text then
                  oa:= oa, LegendEntry = FALSE;
               else
                  oa:= oa, LegendText = expr2text(o); // Lnge auf 20(?) Zeichen begrenzen!!!
               end_if;
                              
               // Die bergebenen, und fr das gegebene plot-Objekt gltigen,
               // plot-Attribute einfgen: optionale vorn, erforderliche hinten.
               oa:= userSceneAttributes, // derzeit Mesh und Submesh
                    op(select(opa, e->contains(ago,op(e,1)))), 
                    oa, 
                    op(select(pa, e->contains(ago,op(e,1))));

               return(oa);
            end_proc:

            // ----------------------------------------------------------------
            // Bereiche fr die Variablen in 'o' ermitteln. Dazu werden die Be-
            // reiche aus dem Attribut-Stack geholt und nach Bedarf aufgestockt
            // oder reduziert. Sofern es "gefahrlos" mglich ist, wird die Wahl
            // zustzlicher Bereiche optimiert, ebenso von Wahl der verwendeten
            // Variablennamen in Bereichen.
            // ----------------------------------------------------------------
            getRanges:= proc(o,n=0)
               name plot::easy[getRanges];
               local theVars, theRanges, further, dummy, var, vars, found, missed,
                     newRanges, ranges;
            begin
               // print(o, n);
               
               // -------------------------------------------------------------
               // Objektvariablen ermitteln
               // -------------------------------------------------------------
               if domtype(o) = DOM_PROC then
                  //theVars:= numeric::indets([op(o,1), op(o,4)]); // parameters and vars in body
                  theVars:= numeric::indets([op(o,1)]); // parameters
               elif domtype(o) = DOM_FUNC_ENV then
                  theVars:= numeric::indets([op(o,[1,1])]); // local of its DOM_PROC
               else
                  theVars:= numeric::indets(o);
               end_if;
               
               // theVars:= theVars 
                         // Folgende Bezeichner sollen ggf. nicht als freie Variablen
                         // interpretiert werden. Siehe lib\LIBFILES\stdlib.mu:
                         // minus stdlib::ENVIRONMENT_VARIABLES
                         // minus stdlib::SYSTEM_CONSTANTS
                         // minus stdlib::LIBRARY_CONSTANTS
                         // minus stdlib::OPTIONS;

               // -------------------------------------------------------------
               // Im folgenden Beispiel sollen die gegebenen Bereichsgrenzen
               // auch von Objekten ohne freie Variablen eingehalten werden.
               // plot(#3D, sqrt(1-x^2-y^2), 0, x=-1..1, y=-1..1)
               // -------------------------------------------------------------
               // Bereiche mssen im Folgenden auch fr den Fall theVars = {} 
               // ermittelt werden!
               
               // -------------------------------------------------------------
               // Die Bereiche entsprechend Ihrer Gruppen-Hierarchie vom Stack
               // auslesen und dabei die #-ViewingBox-Ranges ausfiltern.
               // -------------------------------------------------------------
               [further, theRanges, dummy]:= split(
                   attrStackAll(`#guessr`,TRUE),
                   e->(e:=lhs(e); bool(e=`#XRange` or e=`#YRange` or e=`#ZRange`))
               );

               // -------------------------------------------------------------
               // Anzahl der fehlenden bzw. berzhligen Bereiche ermitteln und 
               // die Bereiche mit #i aufstocken bzw. deren Anzahl reduzieren.
               // Die Reihenfolge der Bereiche nach Konvention der plot-Library 
               // erhalten.
               // -------------------------------------------------------------
               missed:= max(nops(theVars),n) - nops(theRanges);
               if missed = 0 then
                  newRanges:= [];
                  ranges:= theRanges;
               elif missed < 0 then
                  newRanges:= [];
                  ranges:= theRanges[1..max(nops(theVars),n)];
               else
                  // ----------------------------------------------------------
                  // Ranges aufstocken
                  // ----------------------------------------------------------
                  // Sonderfall: Genau ein Bereich gegeben? Dann diesen vor dem
                  // Default -5..5 bevorzugen. Wnschenswert fr Beispiele wie:
                  // plot(sin(x), x=-2*PI..2*PI, {[a,sin(a)],`#P`}, sin'(a)*(x-a)+sin(a))
                  // ----------------------------------------------------------
                  if nops(theRanges) = 1 then
                     newRanges:= [`#`.i = rhs(op(theRanges)) $ i=1..missed];
                  // ggf. auch #X/Y/Z-Ranges sowie die Ranges der "echten"
                  // plot-Objekte bercksichtigen.
                  else
                     newRanges:= [`#`.i = -5..5 $ i=1..missed];
                  end_if;
                  ranges:= theRanges.newRanges;
               end_if;

               //print(o, n = nops(ranges), [missed]);
               
               // -------------------------------------------------------------
               // Objektvariablen ermitteln, die keinen Range haben
               // -------------------------------------------------------------
               missed:= theVars minus {op(map(ranges,lhs))};
               
               // -------------------------------------------------------------
               // Gesamtheit der Rangevariablen, Ordnung erhalten!
               // -------------------------------------------------------------
               vars:= map(theRanges.newRanges,lhs);
               
               // -------------------------------------------------------------
               // Falls diese nicht aufgrund Ihrer hinteren Position in der 
               // Gesamtheit aller Ranges ausgeschlossen wurden, so knnen
               // sie gefahrlos  in die zu verwendenden Ranges substituiert
               // werden, da eine Vertauschung von Bereichen und damit eine
               // Tuschung des Anwenders ausgeschlossen ist.
               // -------------------------------------------------------------
               // Strategie 1: Nicht-Objektvariablen/-bereiche austauschen
               // sofern entsprechender Bereich in Gesamtheit der Bereiche
               // verfgbar.
               // print(theVars, missed, theRanges.newRanges);
               if nops(missed) > 0 then
                  map(ranges, e-> (
                     if nops(missed) > 0 and not contains(theVars, lhs(e)) then
                        found:= contains(vars,missed[1]);
                        if found > 0 then
                           ranges:= subs(ranges, e=(theRanges.newRanges)[found]);
                           // print(e ==> (theRanges.newRanges)[found]);
                           missed:= missed minus {missed[1]};
                        end_if;
                     end_if;
                  ));
               end_if;
               // Strategie 2: Nicht-Objektvariablen mit Objektvariablen
               // ersetzen.
               if nops(missed) > 0 and not missed subset {op(vars)} then
                  map(ranges, e-> (
                     e:= lhs(e);
                     if nops(missed) > 0 and not contains(theVars, e) then
                        // print(e ==> missed[1]);
                        ranges:= subs(ranges, e=missed[1]);
                        missed:= missed minus {missed[1]};
                     end_if;
                  ));
               end_if;
               // Strategie 3: Die erste drei #-Parameter durch x, y bzw. 
               // z ersetzen, falls mglich. Andernfalls wre die Achsen-
               // beschriftung falsch bzw. unschn: 
               // plot(1); plot(1,#3); plot(x,#3);
               vars:= map(ranges,lhs);
               if nops(ranges) >= 1 and ("".lhs(ranges[1]))[1] = "#" and 
                  contains(vars, hold(x)) = 0 then
                     ranges[1]:= hold(x) = rhs(ranges[1]);
               end_if;
               if nops(ranges) >= 2 and ("".lhs(ranges[2]))[1] = "#" and 
                  contains(vars, hold(y)) = 0 then
                     ranges[2]:= hold(y) = rhs(ranges[2]);
               end_if;
               if nops(ranges) >= 3 and ("".lhs(ranges[3]))[1] = "#" and 
                  contains(vars, hold(z)) = 0 then
                     // print(lhs(ranges[3]) ==> hold(z));
                     ranges[3]:= hold(z) = rhs(ranges[3]);
               end_if;

               // -------------------------------------------------------------
               // Sicherstellen, dass keine Bereichsvariable mehrfach auftritt.
               // -------------------------------------------------------------
               vars:= map(ranges,lhs);
               if nops(vars) <> nops({op(vars)}) then
                  // Liste - Menge = Dublikate
                  for var in {op(vars)} do 
                     found:= contains(vars,var);
                     if found > 0 then delete vars[found] end_if;
                  end_for;
                  error("more than one range defined for '".expr2text(op(vars))."'.");
               end_if;

               // print(o = ranges);
               return(op(ranges));
               
               // -------------------------------------------------------------
               // -------------------------------------------------------------

               /* NO LONGER USED
               print("Using old range strategy!!!");
               
               // -------------------------------------------------------------
               // Bereiche nach Gruppen-Hierarchie sortiert nehmen, ViewingBox-
               // Ranges nach hinten sortieren und bei Bedarf noch mit #i=-5..5
               // auffllen. Hier sicherstellen, dass die Ranges ausreichen!!!!
               // -------------------------------------------------------------
               [further, theRanges, dummy]:= split(
                   attrStackAll(`#guessr`,TRUE),
                   e->(e:=lhs(e); bool(e=`#XRange` or e=`#YRange` or e=`#ZRange`))
               );

               // Sicherstellen, dass fr keine Bereichsvariable mehr als eine
               // Bereichsdefinition auftritt.
               dummy:= map(theRanges,lhs);
               if nops(dummy) <> nops({op(dummy)}) then
                  for missed in {op(dummy)} do
                    found:= contains(dummy,missed);
                    if found > 0 then delete dummy[found] end_if;
                  end_for;
                  error("more than one range defined for '".expr2text(op(dummy))."'.");
               end_if;

               // Sonderfall: Falls genau ein Bereich gegeben ist, dann besser 
               // mit dem gegebenen Bereich auffllen, statt mit -5..5. Insbe-
               // sondere im Kontext von Animationen ist das gewnscht:
               // plot(sin(x), x=-2*PI..2*PI, {[a,sin(a)],`#P`}, sin'(a)*(x-a)+sin(a))
               if nops(theRanges) = 1 then 
                  missed:= rhs(op(theRanges));
               else 
                  missed:= -5..5;
               end_if;

               // Bereiche sollen "x", "y" und "z" heien. Andernfalls wre die 
               // Achsenbeschriftung falsch bzw. unschn: 
               // plot(1); plot(1,#3); plot(x,#3);
               if nops(theRanges) = 0 then
                  theRanges:= theRanges.[hold(x)=missed, hold(y)=missed, hold(z)=missed];
               elif nops(theRanges) = 1 then
                  if not has(theRanges,hold(y)) then theRanges:= theRanges.[hold(y)=missed]; end_if;
                  if not has(theRanges,hold(z)) then theRanges:= theRanges.[hold(z)=missed]; end_if;
               elif nops(theRanges) = 2 then
                  if not has(theRanges,hold(z)) then theRanges:= theRanges.[hold(z)=missed]; end_if;
               end_if;

               theRanges:= theRanges.sort(further)
               .[ `#`.i=missed $ i=1..max(max(nops(theVars),n)-nops(theRanges)-nops(further),0) ];

               // -------------------------------------------------------------
               // Bereiche fr die im Objekt enthaltenen Variablen bernehmen
               // -------------------------------------------------------------
               [ranges, theRanges, dummy]:= split(theRanges, e->contains(theVars,lhs(e)));
               found := map(ranges, lhs);
               missed:= listlib::setDifference(theVars, found);

               // -------------------------------------------------------------
               // Fehlen noch Ranges fr *Variablen im Ausdruck*? Dann bedienen
               // wir uns zunchst bei den rechten Seiten der restlichen Ranges.
               // -------------------------------------------------------------
               if nops(missed) > 0 then
                  // Wir bedienen uns genau beim ersten der *restlichen* Ranges. 
                  // Da die Benennung von Variablen, und somit Ihre Reihenfolge, 
                  // beliebig ist, kann keine Unterscheidung von Definitions-,
                  // Werte- und Animationsbereichen erfolgen. Daher wird allen
                  // Variablen der gleiche Bereich zugeordnet. Die Hoffnung ist, 
                  // dass Animationen so sinnvoll synchronisiert bleiben.
                  // plot(sin(x), x=-2*PI..2*PI, {[a,sin(a)],`#P`}, sin'(a)*(x-a)+sin(a))
                  further:= zip(sort(missed), [theRanges[1] $ nops(missed)], (m,f)->m=rhs(f));
                  ranges := ranges.further;
               end_if;

               // -------------------------------------------------------------
               // Ist n grer als die Anzahl der zugeordneten Ranges? Dann mit
               // den *restlichen* Ranges weiter auffllen.
               // -------------------------------------------------------------
               if n-nops(ranges) > 0 then
                  if nops(theVars) = 0 then
                     // Im folgenden Beispiel sollen die gegebenen Bereichsgrenzen
                     // auch von Objekten ohne freie Variablen eingehalten werden.
                     // In dem Fall also die ersten n-nops Bereiche verwenden!
                     // plot(#3D, sqrt(1-x^2-y^2), 0, x=-1..1, y=-1..1)
                     further:= theRanges[1..n-nops(ranges)];
                  else
                     // Hier nur den ersten Range verwenden, damit es nicht so
                     // leicht berraschungen bei Animationen bzgl. der Reihen
                     // folge von Ranges gibt.
                     further:= [theRanges[1] $ n-nops(ranges)];
                     further:= theRanges[1..n-nops(ranges)];
                  end_if;
                  ranges := ranges.further;
                  // Nur zur doppelten Kontrolle. Sollte nie erreicht werden!
                  if n > nops(ranges) then
                     error("missing ".expr2text(n-nops(ranges))." parameter range(s) for '".expr2text(o)."'");
                  end_if;
               end_if;

               return(op(ranges));
               */
            end_proc;

            // ----------------------------------------------------------------
            // Regulre plot-Objekte modifizieren oder direkt zurck geben.
            // ----------------------------------------------------------------
            if contains(attrStack[attrStackDepth][`#ploto`],o) > 0 then
               if userSceneAttributes <> null() then
                  // Eine Modifikation ist derzeit nicht erwnscht!
                  // return(plot::modify(o,userSceneAttributes));
               end_if;
               return(o);
            end_if;

            // ----------------------------------------------------------------
            // Regulre plot-Attribute ggf. ausfiltern, auswerten oder direkt 
            // zurck geben.
            // ----------------------------------------------------------------
            if contains(attrStack[attrStackDepth][`#plota`],o) > 0 then
               if type(o) = "_equal" then
                  // hier ggf. einen Filter einbauen
               end_if;
               return(o);
            end_if;

            // ----------------------------------------------------------------
            // Einige guess-Attribute in regulre plot-Atribute konvertieren.
            // Den Rest ausfiltern. Sie wurden an anderer Stelle ausgewertet.
            // ----------------------------------------------------------------
            if contains(guessAttributesSet,o) then
               case(o)
                 of hold(`#3D`)          do return(null());
                 of hold(`#Arrows`)      do return(null());
                 of hold(`#Constrained`) do return(Scaling=Constrained);
                 of hold(`#Debug`)       do return(null());
                 of hold(`#Grid`)        do return(GridVisible=TRUE);
                 of hold(`#Legend`)      do return(LegendEntry=TRUE,
                                                   LegendVisible=TRUE
                                            );
                 of hold(`#Origin`)      do return(null());
                 of hold(`#Points`)      do return(PointsVisible=TRUE, hold(op@piecewise)(
                                               [dim=2, [LinesVisible=FALSE]],
                                               [dim=3, [LinesVisible=FALSE,
                                                        MeshVisible=FALSE,
                                                        XLinesVisible=FALSE,
                                                        YLinesVisible=FALSE,
                                                        ULinesVisible=FALSE,
                                                        VLinesVisible=FALSE]])
                                            );
                 otherwise                  return(null());
               end_case;
            end_if;

            // ----------------------------------------------------------------
            // Spezielle guess-Ranges als ViewingBoxRange interpretieren. Alle
            // restlichen ausfiltern. Sie wurden an anderer Stelle ausgewertet.
            // ----------------------------------------------------------------
            if type(o) = "_equal" and type(op(o,2)) = "_range" then
               case op(o,1)
                  of hold(`#XRange`) do: return(ViewingBoxXRange=op(o,2));
                  of hold(`#YRange`) do: return(ViewingBoxYRange=op(o,2));
                  of hold(`#ZRange`) do: return(ViewingBoxZRange=op(o,2));
                  otherwise:
                     if contains(attrStack[attrStackDepth][`#guessr`],o) > 0  then
                        return(null());
                  end_if;
               end_case;
            end_if;

            // ----------------------------------------------------------------
            // Farb-Attribute der Form '#Red' in regulre plot-Attribute der
            // Form 'Color=RGB::Red' konvertieren und zurckgeben. Siehe auch
            // bei 'FARBWERTE'.
            // ----------------------------------------------------------------
            if (dummy:= plot::easy_helper_attrHashColor(o)) <> FAIL then
               if type(dummy) = "_range" then
                  return(Color = op(dummy,1), hold(op@piecewise)(
                     [dim=2, [LineColorType=Dichromatic, LineColor2 = op(dummy,2)]],
                     [dim=3, [FillColorType=Dichromatic, FillColor2 = op(dummy,2)]]
                  ));
               else
                  return(Color = dummy, hold(op@piecewise)(
                     [dim=2, [LineColorType = Flat]],
                     [dim=3, [LineColorType = Flat, FillColorType = Flat]]
                  ));
              end_if;
            end_if;

            // ----------------------------------------------------------------
            // Ab hier gilt: Es liegt ein guess-Objekt vor!!!
            // ----------------------------------------------------------------

            // ----------------------------------------------------------------
            // Element eines Domains mit berladener plot Methode?
            // ----------------------------------------------------------------
            if traperror((dummy:= o::dom::plot)) = 0 and dummy <> FAIL then
               dummy:= dummy(o,
                  [getRanges(o)],             // Ranges fr enhaltene Variablen
                  [includeAttributes()],      // alle Attribute vom Stack
                  dim,                        // Dimension der Szene
                  [hasArrows or bool(contains(attrStackAll(`#guessa`), `#Arrows`) > 0),
                   hasPoints or bool(contains(attrStackAll(`#guessa`), `#Points`) > 0)],
                  attrStack
               );
               if dummy <> FAIL then
                  return(dummy);
               end_if;
            end_if;

            // ----------------------------------------------------------------
            // DOM_ARRAY und DOM_HFARRAY in Dom::Matrix() transformieren
            // ----------------------------------------------------------------
            if domtype(o) = DOM_ARRAY or
               domtype(o) = DOM_HFARRAY then
               o:= matrix(o);
            end_if;

            // ----------------------------------------------------------------
            // Dom::Matrix() in DOM_LIST transformieren
            // ----------------------------------------------------------------
            if domtype(o) = Dom::Matrix() then
               if linalg::ncols(o) = 1 then
                  o:= [ o[i] $ i=1..linalg::nrows(o) ];
                  // 2x1 und 3x1 Vektoren als Pfeile darstellen!
                  isArrow:= not (hasPoints or contains(attrStackAll(`#guessa`), `#Points`) > 0);
                else
                  o:= [ [o[i,j] $ j=1..linalg::ncols(o)] $ i=1..linalg::nrows(o) ];
               end_if;
            end_if;

            // ----------------------------------------------------------------
            // DOM_TABLE in DOM_LIST konvertieren
            // ----------------------------------------------------------------
            if domtype(o) = DOM_TABLE then
               if nops({op(map(rhs(o), e->if domtype(e) = DOM_LIST then nops(e) else 1 end))}) <> 1 then
                  error("dimension mismatch in '".expr2text(o)."'");
               end_if;
               // table( x=y, y=[y,z] or x=[x,RGBa], x=[x,y,RGBa] )
               o:= map([op(o)], e->if type(op(e,2)) = DOM_LIST then [op(e,1)].op(e,2) else [op(e)] end);
            end_if;

            // ----------------------------------------------------------------
            // DOM_POLY in DOM_EXPR transformieren
            // ----------------------------------------------------------------
            if domtype(o) = DOM_POLY then
               o:= expr(o);
            end_if;

            // ----------------------------------------------------------------
            // DOM_LIST [condition, term] ggf. in piecewise transformieren
            // ----------------------------------------------------------------
            if domtype(o) = DOM_LIST then
               if nops(o) = 2 and contains({"_leequal","_less","_equal"}, type(o[1])) and
                              not contains({"_leequal","_less","_equal"}, type(o[2])) then
                              // !!! otherwise it is a list of (in)equalities !!!
                  o:= hold(piecewise)(o);
               else
                  isListOfPiecewise:= L -> {op(map(L, e->_lazy_and(
                     type(e) = DOM_LIST,
                     nops(e) = 2,
                     contains({"_leequal","_less","_equal"}, type(e[1]))
                     )))};
                  if isListOfPiecewise(o) = {TRUE} then
                     o:= hold(piecewise)(op(o));
                  end_if;
               end_if;
            end_if;

            // ----------------------------------------------------------------
            // Anzahl der freien Bezeichner ermittlen
            // ----------------------------------------------------------------
            vars:= numeric::indets(o);

            // ----------------------------------------------------------------
            // Konstanten, Bezeichner und stckweise definierte Funktionen
            // ----------------------------------------------------------------
            if testtype(o, Type::Numeric) or
               domtype(o) = DOM_IDENT     or
               domtype(o) = piecewise     or
               domtype(o) = Series::Puiseux or
               domtype(o) = Series::gseries then
               // Die folgende Eingabe muss als 3D-Funktion interpretiert werden.
               // Daher muss die Szene hier bei mehr als 2 Variablen explizit auf
               // 3D umgeschaltet werden:
               // plot::easy(piecewise([a+b+c=1, a+b+c], [Otherwise, a+b-c]))
               if nops(vars) > 2 then
                 setSceneDimension(3);
               end_if;
               if nops(vars) > 3 then
                 error("too many indeterminates in '".expr2text(o)."', ");
               end_if;
               // Im folgenden Beispiel sollen die gegebenen Bereichsgrenzen
               // auch von Objekten ohne freie Variablen eingehalten werden.
               // Daher beim Aufruf von getRanges 1 bzw. 2 Bereiche fordern!
               // plot(#3D, sqrt(1-x^2-y^2), 0, x=-1..1, y=-1..1)
               return(hold(piecewise)(
                  [dim=2, hold(plot::Function2d)(o, getRanges(o,1), includeAttributes(plot::Function2d))],
                  [dim=3, hold(plot::Function3d)(o, getRanges(o,2), includeAttributes(plot::Function3d))]
               ));

            // ----------------------------------------------------------------
            // Funktionen
            // ----------------------------------------------------------------
            elif domtype(o) = DOM_PROC     or
                 domtype(o) = DOM_FUNC_ENV then
               // -------------------------------------------------------------
               // Funktionen:
               // Die Anzahl der freien Variablen ist hier 0, 1 oder unbekannt.
               // ==> In Abhngigkeit von der Dimension der Szene wird entweder
               //     eine 2d- oder 3d-Funktion generiert. Diese Entscheidung
               //     wird erst nach Konvertierung aller Objekte gefllt.
               // -------------------------------------------------------------
               if domtype(o) = DOM_FUNC_ENV and nops(op(op(o,1),1)) = 1 then
                  // ----------------------------------------------------------
                  // Eine Funktionsumgebung mit Prozedur mit nur einem Argument
                  // ----------------------------------------------------------
                  %if 10*version()[1]+version()[2] < 44 then
                     var1:= `_x` --> o(`_x`);
                  else
                     var1:= `#x` --> o(`#x`);
                  end_if;
                  return(piecewise(
                     [dim=2, hold(plot::Function2d)(o, getRanges(o,1), includeAttributes(plot::Function2d))],
                     [dim=3, hold(plot::Function3d)(var1, getRanges(o,2), includeAttributes(plot::Function3d))]
                  ));
               else
                  if nops([getRanges(o)]) = 1 then
                     setSceneDimension(2);
                     return(plot::Function2d(o, getRanges(o,1), includeAttributes(plot::Function2d)))
                  else 
                     setSceneDimension(3);
                     return(plot::Function3d(o, getRanges(o,2), includeAttributes(plot::Function3d)));
                  end_if;
               end_if;

            // ----------------------------------------------------------------
            // Listen-Objekte
            // ----------------------------------------------------------------
            elif domtype(o) = DOM_LIST then
               t:= {op(map(o, type))};
               // -------------------------------------------------------------
               // Ungleichungen
               // -------------------------------------------------------------
               if nops(t) > 0 and t subset {"_leequal","_less","_equal"} then
                  setSceneDimension(2);
                  return(plot::Inequality(o, getRanges(o,2), includeAttributes(plot::Inequality)));
               end_if;

               // -------------------------------------------------------------
               // Listendarstellung standardisieren
               // -------------------------------------------------------------
               if t subset {DOM_ARRAY, DOM_HFARRAY, Dom::Matrix(), DOM_TABLE} then
                  o:= map(o, o->if domtype(o) = DOM_TABLE then [misc::maprec([op(op(o))], {DOM_LIST}=(o->op(o)))] else [op(o)] end);
               end_if;
                
               t:= {op(map(o, testtype, Type::Arithmetical))};
               // -------------------------------------------------------------
               // 2d Kurve, Punkte oder Pfeile
               // -------------------------------------------------------------
               if t = {TRUE} and nops(o) = 2 then
                  setSceneDimension(2);
                  if isArrow or hasArrows or contains(attrStackAll(`#guessa`), `#Arrows`) > 0 then
                     return(plot::Arrow2d(o, getRanges(o), includeAttributes(plot::Arrow2d)));
                  end_if;
                  if hasPoints or contains(attrStackAll(`#guessa`), `#Points`) > 0 then
                     return(plot::Point2d(o, getRanges(o), includeAttributes(plot::Point2d)));
                  end_if;
                  if {op(map(float(o), testtype, Type::Numeric))} = {TRUE} then
                     return(plot::Point2d(o, getRanges(o), includeAttributes(plot::Point2d)));
                  end_if;
                  %if 10*version()[1]+version()[2] < 44 then
                     return(plot::Curve2d(o, getRanges(o,1), includeAttributes(plot::Curve2d)));
                  else
                     return(plot::Curve2d(o, getRanges(o), includeAttributes(plot::Curve2d)));
                  end_if;
               end_if;

               // -------------------------------------------------------------
               // 3d Kurve, Punkte oder Pfeile
               // -------------------------------------------------------------
               if t = {TRUE} and nops(o) = 3 then
                  setSceneDimension(3);
                  if isArrow or hasArrows or contains(attrStackAll(`#guessa`), `#Arrows`) > 0 then
                     return(plot::Arrow3d(o, getRanges(o), includeAttributes(plot::Arrow3d)));
                  end_if;
                  if hasPoints or contains(attrStackAll(`#guessa`), `#Points`) > 0 then
                     return(plot::Point3d(o, getRanges(o), includeAttributes(plot::Point3d)));
                  end_if;
                  if  {op(map(float(o), testtype, Type::Numeric))} = {TRUE} then
                     return(plot::Point3d(o, getRanges(o), includeAttributes(plot::Point3d)));
                  end_if;
                  %if 10*version()[1]+version()[2] < 44 then
                     return(plot::Curve3d(o, getRanges(o,1), includeAttributes(plot::Curve3d)));
                  else
                     return(plot::Curve3d(o, getRanges(o), includeAttributes(plot::Curve3d)));
                  end_if;
               end_if;

               // -------------------------------------------------------------
               // 2d Polygone oder Pfeile
               // -------------------------------------------------------------
               if testtype(o, Type::ListOf(Type::ListOf(Type::Arithmetical,2,2),1)) then
                  setSceneDimension(2);
                  if (isArrow or hasArrows or contains(attrStackAll(`#guessa`), `#Arrows`) > 0) then
                     if nops(o) <= 2 then  // one arrow with arbitrary starting point
                        return(plot::Arrow2d(op(o), getRanges(o), includeAttributes(plot::Arrow2d)));
                     else                  // multiple arrows starting at point (0|0)
                        return(op(map(o,dummy->plot::Arrow2d(dummy, getRanges(o), includeAttributes(plot::Arrow2d)))));
                     end_if;
                  end_if;
                  return(plot::Polygon2d(o, getRanges(o), includeAttributes(plot::Polygon2d)));
               end_if;

               // -------------------------------------------------------------
               // 3d Polygone oder Pfeile
               // -------------------------------------------------------------
               if testtype(o, Type::ListOf(Type::ListOf(Type::Arithmetical,3,3),1)) then
                  setSceneDimension(3);
                  if (isArrow or hasArrows or contains(attrStackAll(`#guessa`), `#Arrows`) > 0) then
                     if nops(o) <= 2 then  // one arrow with arbitrary starting point
                        return(plot::Arrow3d(op(o), getRanges(o), includeAttributes(plot::Arrow3d)));
                     else                  // multiple arrows starting at point (0|0|0)
                        return(op(map(o,dummy->plot::Arrow3d(dummy, getRanges(o), includeAttributes(plot::Arrow3d)))));
                     end_if;
                  end_if;
                  return(plot::Polygon3d(o, getRanges(o), includeAttributes(plot::Polygon3d)));
               end_if;

               // -------------------------------------------------------------
               // 2d Punktelisten mit Farbangaben
               // -------------------------------------------------------------
               isPointList2dWithColor:= L -> {op(map(L, e->_lazy_and(
                  type(e) = DOM_LIST,
                  nops(e) = 3,
                  testtype(e[1], Type::Arithmetical),
                  testtype(e[2], Type::Arithmetical),
                  testtype(e[3], Type::ListOf(Type::Arithmetical, 3,4))
                  )))};
               if isPointList2dWithColor(o) = {TRUE} then
                  setSceneDimension(2);
                  if isArrow or hasArrows or contains(attrStackAll(`#guessa`), `#Arrows`) > 0 then
                     t:= getRanges(o);
                     d:= includeAttributes(plot::Arrow2d);
                     return(op(map(o,e->plot::Arrow2d([e[1],e[2]], t, d, LineColor=e[3]))));
                  end_if;
                  return(plot::PointList2d(o, getRanges(o), includeAttributes(plot::PointList2d)));
               end_if;

               // -------------------------------------------------------------
               // 3d Punktelisten mit Farbangaben
               // -------------------------------------------------------------
               isPointList3dWithColor:= L -> {op(map(L, e->_lazy_and(
                  type(e) = DOM_LIST,
                  nops(e) = 4,
                  testtype(e[1], Type::Arithmetical),
                  testtype(e[2], Type::Arithmetical),
                  testtype(e[3], Type::Arithmetical),
                  testtype(e[4], Type::ListOf(Type::Arithmetical, 3,4))
                  )))};
               if isPointList3dWithColor(o) = {TRUE} then
                  setSceneDimension(3);
                  if isArrow or hasArrows or contains(attrStackAll(`#guessa`), `#Arrows`) > 0 then
                     t:= getRanges(o);
                     d:= includeAttributes(plot::Arrow2d);
                     return(op(map(o,e->plot::Arrow3d([e[1],e[2],e[3]], t, d, LineColor=e[4]))));
                  end_if;
                  return(plot::PointList3d(o, getRanges(o), includeAttributes(plot::PointList3d)));
               end_if;

               // OTHERWISE: ILLEGAL => IGNORE => DO NOT CONVERT

            // ----------------------------------------------------------------
            // Ausdrcke
            // ----------------------------------------------------------------
            elif domtype(o) = DOM_EXPR then
               // -------------------------------------------------------------
               // Gleichungen
               // -------------------------------------------------------------
               if type(o) = "_equal" then
                  // ----------------------------------------------------------
                  // Geraden und Ebenen
                  // ----------------------------------------------------------
                  if testtype(float(op(o,2)), Type::Arithmetical) then
                     d:= float(10^-100);  // min. Ausdehnung fr Geraden und Ebene

                     // WILLKOMMEN IN DER #X/#Y/#Z-WELT :-)
                     if hold(`#x`) = op(o,1) then // #X=n ==> x-Asymptote oder x-Ebene
                        return(hold(piecewise)(
                           [dim=2, hold(plot::Line2d)([op(o,2),0], [op(o,2),d], getRanges(op(o,2)), includeAttributes(plot::Line2d, [Extension=Infinite, AntiAliased=FALSE], [Color=RGB::DarkGray, LineStyle=Dashed]))],
                           [dim=3, hold(plot::Plane)( [op(o,2),0,0], [d,0,0], getRanges(op(o,2)), includeAttributes(plot::Plane, [], [Color=RGB::Black.[0.2]]))]
                        ));
                     elif hold(`#y`) = op(o,1) then // #Y=n ==> x-Asymptote oder x-Ebene
                        return(hold(piecewise)(
                           [dim=2, hold(plot::Line2d)([0,op(o,2)], [d,op(o,2)], getRanges(op(o,2)), includeAttributes(plot::Line2d, [Extension=Infinite, AntiAliased=FALSE], [Color=RGB::DarkGray, LineStyle=Dashed]))],
                           [dim=3, hold(plot::Plane)([0,op(o,2),0], [0,d,0], getRanges(op(o,2)), includeAttributes(plot::Plane, [], [Color=RGB::Black.[0.2]]))]
                        ));
                     elif hold(`#z`) = op(o,1) then // #Z=n ==> z-Ebene
                        if dim = 2 then
                           error("dimension mismatch, '".expr2text(o)."' not allowed in 2d scene");
                        end_if;
                        setSceneDimension(3);
                        return(hold(piecewise)(
                           [dim=2, o], // = ILLEGAL, UNEXPECTED 3D OBJECT!
                           [dim=3, hold(plot::Plane)([0,0,op(o,2)], [0,0,d], getRanges(op(o,2)), includeAttributes(plot::Plane, [], [Color=RGB::Black.[0.2]]))]
                        ));
                     end_if;
                  end_if;

                  // ----------------------------------------------------------
                  // 2d Texte und 3d Texte
                  // ----------------------------------------------------------
                  if domtype(op(o,2)) = DOM_STRING or domtype(op(o,2)) = DOM_PROC then
                     if testtype(op(o,1), Type::ListOf(Type::Arithmetical,2,3)) then
                        if nops(op(o,1)) = 2 then
                           setSceneDimension(2);
                           return(plot::Text2d(op(o,2), op(o,1), getRanges(o), includeAttributes(plot::Text2d)));
                        else
                           setSceneDimension(3);
                           return(plot::Text3d(op(o,2), op(o,1), getRanges(o), includeAttributes(plot::Text3d)));
                        end_if;
                     end_if;
                     if domtype(op(o,1)) = Dom::Matrix() and linalg::ncols(op(o,1)) = 1 then
                        if linalg::nrows(op(o,1)) = 2 then
                           setSceneDimension(2);
                           return(plot::Text2d(op(o,2), op(o,1), getRanges(o), includeAttributes(plot::Text2d)));
                        elif linalg::nrows(op(o,1)) = 3 then
                           setSceneDimension(3);
                           return(plot::Text3d(op(o,2), op(o,1), getRanges(o), includeAttributes(plot::Text3d)));
                        end_if;
                     end_if;
                  end_if;

                  if not testtype(op(o,1), Type::Arithmetical) or not testtype(op(o,2), Type::Arithmetical) then
                     return(o);
                  end_if;

                  // ----------------------------------------------------------
                  // Implizite 2d oder 3d Funktion
                  // ----------------------------------------------------------
                  if nops(vars) > 3 then
                     // Animierte implizite 3d Funktion
                     setSceneDimension(3);
                     return(plot::Implicit3d(o, getRanges(o,3), includeAttributes(plot::Implicit3d)));
                  end_if;

                  return(hold(piecewise)(
                     [dim=2, hold(plot::Implicit2d)(o, getRanges(o,2), includeAttributes(plot::Implicit2d))],
                     [dim=3, hold(plot::Implicit3d)(o, getRanges(o,3), includeAttributes(plot::Implicit3d))]
                  ));

               // -------------------------------------------------------------
               // Ungleichungen
               // -------------------------------------------------------------
               elif type(o) = "_less" or type(o) = "_leequal" then
                  setSceneDimension(2);
                  return(plot::Inequality(o, getRanges(o,2), includeAttributes(plot::Inequality)));

               // -------------------------------------------------------------
               // Ausdruck als Funktion interpretieren [ggf. weitere Zerlegung
               // fr bessere Fehlerbehandlung ntig?]
               // -------------------------------------------------------------
               else
                  if contains({"_seqgen", "_seqstep", "_range"}, type(o)) then
                     return(o);
                  end_if;

                  if nops(vars) = 3 then
                     // -------------------------------------------------------
                     // Die Anzahl der freien Variablen ist 3.
                     // => Es muss sich um eine animierte 3d-Funktion handeln.
                     // -------------------------------------------------------
                     setSceneDimension(3);
                     return(plot::Function3d(o, getRanges(o,2), includeAttributes(plot::Function3d)));

                  elif nops(vars) < 3 then
                     // -------------------------------------------------------
                     // Die Anzahl der freien Variablen ist kleiner als 3.
                     // => In Abhngigkeit von der Dimension der Szene eine 2d-
                     //    oder 3d-Funktion generieren. Diese Entscheidung wird
                     //    nach Konvertierung aller Szene-Objekte gefllt.
                     // -------------------------------------------------------
                     return(hold(piecewise)(
                        [dim=2, hold(plot::Function2d)(o, getRanges(o,1), includeAttributes(plot::Function2d))],
                        [dim=3, hold(plot::Function3d)(o, getRanges(o,2), includeAttributes(plot::Function3d))]
                     ));

                  else // ILLEGAL
                     error("too many indeterminates in '".expr2text(o)."', ");
                  end_if;
               end_if;
            end_if;

            // ----------------------------------------------------------------
            // guess-Objekt konnte nicht auf ein plot-Objekt abgebildet werden.
            // ----------------------------------------------------------------
            return(o);
         end_proc;

         // -------------------------------------------------------------------
         // Neue Szene {{...}}? Objekte als neue Szene generieren.
         // -------------------------------------------------------------------
         if type(o) = DOM_SET and nops(o) = 1 and type(op(o)) = DOM_SET then
            if inScene then
               error("Nested scenes are not allowed");
            end_if;
            // ----------------------------------------------------------------
            // Global aktive Attribute sichern und spter zurcksetzen
            // ----------------------------------------------------------------
            moreThanOneScene:= TRUE;
            inScene:= TRUE;
            hasValues:= [has3D, hasArrows, hasDebug, hasOrigin, hasPoints];
            o:= op(o);
            o:= o[i] $ i=1..nops(o);  // interaktiv sichtbare Reihenfolge!
            o:= doScene(o);
            // ----------------------------------------------------------------
            // Den Koordinaten-Ursprung in die Szene mit einzeichnen
            // ----------------------------------------------------------------
            if hasOrigin then
               o:= o, hold(piecewise)(
                      [dim=2, hold(plot::Point2d)(0,0,   Visible=FALSE, Name="#Origin")],
                      [dim=3, hold(plot::Point3d)(0,0,0, Visible=FALSE, Name="#Origin")]
                   );
            end_if;
            [has3D, hasArrows, hasDebug, hasOrigin, hasPoints]:= hasValues;
            inScene:= FALSE;

            return(hold(piecewise)([dim=2, hold(plot::Scene2d)(o)], [dim=3, hold(plot::Scene3d)(o)]));
         end_if;

         // -------------------------------------------------------------------
         // Neue Farbe fr top-level Objekt bzw. Gruppe von Objekten whlen.
         // -------------------------------------------------------------------
         if attrStackDepth = 1 and currColorUsed then
            currColor2:= nextColor2();
            currColor3:= nextColor3();
            currColorUsed:= FALSE;
         end_if;

         // -------------------------------------------------------------------
         // Gruppe {...}? Gruppenattribute auf den Stack, Elemente konvertieren
         // -------------------------------------------------------------------
         if domtype(o) = DOM_SET then
            // ----------------------------------------------------------------
            // Daten der Gruppe "zerlegen" und auf den Attribut-Stack legen
            // ----------------------------------------------------------------
            attrStackPush([op(o)]);

            // ----------------------------------------------------------------
            // Debug-Meldung: vor Beginn der Konvertierung
            // ----------------------------------------------------------------
            if hasDebug or contains(attrStackAll(`#guessa`), `#Debug`) > 0  then
               print(Typeset, `#Dimension` = dim, `#Push:`);
               print(Typeset, attrStack);
            end_if;

            // ----------------------------------------------------------------
            // Objekte der Gruppe elementweise konvertieren
            // ----------------------------------------------------------------
            o:= [o[i] $ i=1..nops(o)];  // interaktiv sichtbare Reihenfolge!
            o:= op(map(o, doGuess));
            o:= hold(piecewise)(
                   [dim=2, hold(plot::Group2d)(o)],
                   [dim=3, hold(plot::Group3d)(o)]
                );

            // ----------------------------------------------------------------
            // Daten des aktuellen Gruppe vom Attribut-Stack abrumen
            // ----------------------------------------------------------------
            attrStackPop();

            // ----------------------------------------------------------------
            // Debug-Meldung: nach Ende der Konvertierung
            // ----------------------------------------------------------------
            if hasDebug or contains(attrStackAll(`#guessa`), `#Debug`) > 0  then
               print(Typeset, `#Dimension` = dim, `#Pop.`);
            end_if;
         else
            // ----------------------------------------------------------------
            // Einzelnes Objekt konvertieren
            // ----------------------------------------------------------------
            o:= doConvert(o);
         end_if;

         return(o);
      end_proc:

      // ----------------------------------------------------------------------
      // Szene "zerlegen" und auf den Attribut-Stack speichern
      // ----------------------------------------------------------------------
      attrStackPush([args()]);

      // ----------------------------------------------------------------------
      // Debug-Meldung: ggf. vor Beginn der Konvertierung
      // ----------------------------------------------------------------------
      if hasDebug or contains(attrStack[attrStackDepth][`#guessa`], `#Debug`) > 0  then
         print(Typeset, `#Dimension` = dim);
         print(Typeset, attrStack);
      end_if;

      // ----------------------------------------------------------------------
      // Colors=[...]: Auswerten und Farbgenerator initialisieren
      // ----------------------------------------------------------------------
      t:= select(attrStack[attrStackDepth][`#plota`], 
                 t->_lazy_and(type(t) = "_equal", lhs(t) = hold(Colors)));
      if nops(t) > 0 then
         dummy:= t[-1];
         t:= map(rhs(dummy), proc(t)
                local c;
                begin c:= plot::easy_helper_attrHashColor(t);
                      if c = FAIL then t else c end_if;
                end_proc);
         if testtype(t, Type::ListOf(Type::ListOf(DOM_FLOAT,3,4))) then
            %if 10*version()[1]+version()[2] < 44
               then nextColor2:= (__c:= -1; ()-> (__c:= __c +1; t[(__c mod nops(userColorList))+1]));
               else nextColor2:= RGB::generator(t);
            end_if;
            %if 10*version()[1]+version()[2] < 44
               then nextColor3:= (__c:= -1; ()-> (__c:= __c +1; t[(__c mod nops(userColorList))+1]));
               else nextColor3:= RGB::generator(t);
            end_if;
            currColor2:= nextColor2();
            currColor3:= nextColor3();
            currColorUsed:= FALSE;
            // Wenn wir auerhalb einer spezifischen Szene sind, dann
            // soll die Farbliste an alle UnterSzenen vererbt werden. 
            if not inScene then
               userColorList:= t;
            end_if;
            result := subs(result, dummy=null());
         end_if;
       end_if;
      
      // ----------------------------------------------------------------------
      // Mesh|Submesh=n|[n,m]: In die Default-Attribute fr Objekte aufnehmen.
      // ----------------------------------------------------------------------
      t:= select(attrStack[attrStackDepth][`#plota`], 
                 t->_lazy_and(type(t) = "_equal", (lhs(t) = hold(Mesh) or lhs(t) = hold(Submesh))));
      if nops(t) > 0 then
         userSceneAttributes:= userSceneAttributes, op(t);
         if not inScene then
            userDefaultAttributes:= userDefaultAttributes, op(t);
         end_if;
         map(t, dummy->(result:=subs(result, dummy=null())));
      end_if;
      
      // ----------------------------------------------------------------------
      // Argumente unter Beibehaltung der Reihenfolge konvertieren, sofern es
      // sich um guess-Objekte handelt. plot-Objekte werden "durchgeschleift",
      // guess-Attribute und guess-Bereiche ausgefiltert.
      // ----------------------------------------------------------------------
      result:= op(map(result, doGuess));

      // ----------------------------------------------------------------------
      // Falls die Dimension der Szene noch nicht ermittelt werden konnte, dann
      // erst schauen, ob has3D hint gesetzt wurde. Sonst die Dimension einfach
      // diktatorisch auf '2' setzen.
      // ----------------------------------------------------------------------
      if has3D and dim = hold(dim) then
         setSceneDimension(3);
      end_if;
      if dim = hold(dim) then
         setSceneDimension(2);
      end_if;

      // ----------------------------------------------------------------------
      // Den Koordinaten-Ursprung in die Szene mit einzeichnen
      // ----------------------------------------------------------------------
      if hasOrigin and not moreThanOneScene then
         result:= result,
             hold(piecewise)(
                [dim=2, hold(plot::Point2d)(0,0,   Visible=FALSE, Name="#Origin")],
                [dim=3, hold(plot::Point3d)(0,0,0, Visible=FALSE, Name="#Origin")]
             );
      end_if;

      // ----------------------------------------------------------------------
      // Die generierten plot-Objekte enthalten ggf. noch piecewise-Konstrukte,
      // die nun durch nochmalige Evaluierung aufgelst werden mssen.
      // ----------------------------------------------------------------------
      hasUndefined:= has([result], undefined); 
      result:= eval(result);

      // ----------------------------------------------------------------------
      // debug-Meldung
      // ----------------------------------------------------------------------
      if hasDebug or contains(attrStack[attrStackDepth][`#guessa`], `#Debug`) > 0 then
         print(Typeset, hold(``)._concat("-" $ 100));
         print(Typeset, `#Scene:`);
         map([result], (e) -> print(Typeset, hold(``) ==> hold(``).expr2text(e)));
         print(Typeset, hold(``)._concat("-" $ 100));
      end_if;

      // ----------------------------------------------------------------------
      // Ist ein Fehler bei der Evaluierung der piecewise-Objekte aufgetreten,
      // angezeigt durch 'undefined', so "werfen" wir den zugehrigen Fehler.
      // Sptere Fehlermeldungen von plot::new liessen kaum Rckschlsse auf
      // die Fehlerursache zu.
      // ----------------------------------------------------------------------
      if has([result], undefined) then
         if not hasUndefined and getlasterror()<>[0,""] then 
            lasterror(); // Exotischer Fall!?!
         end_if;
      end_if;

      return(result);
   end_proc;
  
   doScene(op(map([args()],
                  x -> if not contains({DOM_FUNC_ENV, DOM_PROC}, domtype(x)) then
                         subs(x, guessAttributes)
                       else
                         x
                       end_if)))
 end_proc:
