/***  plot::ExprAttribute -- domain for attributes containing expressions
 *
 */

alias(ExprAttribute = plot::ExprAttribute):

ExprAttribute := newDomain(hold(ExprAttribute)):
ExprAttribute::create_dom:=hold(ExprAttribute):
ExprAttribute::interface:={}:

/* plot::ExprAttribute is used to express the following types of
   attributes:

namedText:
   <Header>text</Header>

Expr:
   <Expr>expr</Expr>

optExpr:
   <Expr Opt="XMin">real</Expr>

namedExpr:
   <ParameterName>expr</ParameterName>

IExpr:
   <IExpr>expr</IExpr>

VExpr:
   <VExpr Val='real'>expr</VExpr>

namedVExpr:
   <XMin Val=real>expression</XMin>

namedCompound:
   <XAxisTextOrientation>
      other elements
   </XAxisTextOrientation>



namedCompound examples:
<TextOrientation>
   <IExpr>expr</IExpr>
...
</TextOrientation>

<XAxisTextOrientation>
   <VExpr Val='real'>expr</VExpr>
...
</XAxisTextOrientation>

<XTicksAt>
  <TickPos>
    <VExpr Val='real'>expr</VExpr>
    <Title>text</Title>
  </TickPos>
  ...
</XTicksAt>

<ColorFunc Opt="FillColorFunction">
   <IExpr>expr</IExpr>
   ...
</ColorFunc>

*/


/*
 *   Internal representation:
 *
 *   2 lists:
 *
 *   1)  the encoded MuPAD expression in a list
 *
 *   2)  [elementName, {attributes (as equations)},
 *        string or element of ExprAttribute]
 */

//-------------------------------------------------
// print: just print the expression
ExprAttribute::print := x -> op(extop(x, 1)):
//-------------------------------------------------

//-------------------------------------------------
// expr2text: just print the expression
ExprAttribute::expr2text := x -> expr2text(op(extop(x, 1))):
//-------------------------------------------------

//-------------------------------------------------
// getExpr:  just return the included expression
// used to be expr, but that maps through and make plain
// expressions where we don't want them, e.g. in piecewise
ExprAttribute::getExpr := e -> if testtype(e, dom) then
                                 op(extop(e, 1));
                               elif testtype(e, DOM_POLY) then
                                 expr(e)
                               else
                                 e
                               end_if:
// make it easier to find illegal calls
ExprAttribute::expr := e -> error("expr must not be called"):
//-------------------------------------------------

//-------------------------------------------------
// expr:  just return the included expression
ExprAttribute::indets := e -> indets(extop(e, 1)):
//-------------------------------------------------

ExprAttribute::freeIndets := e -> freeIndets(extop(e, 1), args(2..args(0))):

//-------------------------------------------------
// subs:  substitute values into attribute set
ExprAttribute::subs :=
   (e, eq) -> new(dom, subsop(extop(e), [2,2]=subs(op(extop(e, 2), 2), eq))):
//-------------------------------------------------


/*
 *   constructor methods:
 */

//-------------------------------------------------
ExprAttribute::generic :=
  proc(elemName : DOM_STRING, set : DOM_SET)
    local children;
  begin
    children := args(3..args(0));
    new(dom, [children], [elemName, set,
                          if children <> null() then
                            expr2text(children)
                          else
                            null()
                          end_if])
  end_proc:
//-------------------------------------------------


//-------------------------------------------------
//  expr returned an ident infinity; replace it by infinity
ExprAttribute::clean := () -> op(subs([args()], hold(infinity)=infinity)):
//-------------------------------------------------



//-------------------------------------------------
ExprAttribute::namedText :=
  proc(elemName : DOM_STRING, text : DOM_STRING)
  begin
    new(dom, [text], [elemName, {}, text])
  end_proc:
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::Expr := e -> new(dom, [ExprAttribute::clean(e)], ["Expr", {}, expr2text(e)]):
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::optExpr :=
  proc(s:DOM_STRING, e)
  begin
    new(dom, [ExprAttribute::clean(e)], ["Expr", {"Opt" = s}, expr2text(e)])
  end_proc:
//-------------------------------------------------

//-------------------------------------------------
ExprAttribute::optProp :=
  proc(s:DOM_STRING)
  begin
    new(dom, [ExprAttribute::clean(args(2..args(0)))], ["Prop", {"Opt" = s,
                                           "Val"=args(2..args(0))}])
  end_proc:
//-------------------------------------------------

//-------------------------------------------------
ExprAttribute::optColors :=
  proc()
  begin
    new(dom, [[ExprAttribute::clean(args())]], ["Prop", {"Opt" = "Colors", "Val"=args()}])
  end_proc:
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::namedExpr :=
  proc(s:DOM_STRING, e)
  begin
    new(dom, [ExprAttribute::clean(e)], [s, {}, expr2text(e)])
  end_proc:
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::IExpr := e -> new(dom, [ExprAttribute::clean(e)], ["IExpr", {}, expr2text(e)]):
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::VExpr :=
  e -> new(dom, [ExprAttribute::clean(e)], ["VExpr", {"Val" = e}, expr2text(e)]):
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::namedVExpr :=
  proc(s:DOM_STRING, e)
  begin
    new(dom, [ExprAttribute::clean(e)], [s, {"Val" = e}, expr2text(e)])
  end_proc:
//-------------------------------------------------


//-------------------------------------------------
ExprAttribute::namedCompound :=
  proc(s : DOM_STRING, attributes : DOM_SET, expression : DOM_LIST)
  begin
    new(dom, args(3), [s, attributes, [args(4..args(0))]])
  end_proc:
//-------------------------------------------------



//-------------------------------------------------
ExprAttribute::MuPlotML:=
proc(e)
  local rep;
begin
  rep := extop(e, 2);
  if nops(rep) = 2 then
    xmlprint::beginElement(plot::MuPlotML::OutputFile,
                           rep[1], map(op(rep[2]), float), Empty);
    xmlprint::raw(plot::MuPlotML::OutputFile, "\n");
  else
    xmlprint::beginElement(plot::MuPlotML::OutputFile,
                           rep[1], map(op(rep[2]), float));
    if domtype(rep[3]) = DOM_LIST then
      map(rep[3], ExprAttribute::MuPlotML);
    else
      if testtype(rep[3], DOM_STRING) then
        xmlprint::data(plot::MuPlotML::OutputFile, rep[3])
      else
        (rep[3])::dom::MuPlotML(rep[3])
      end_if;
    end_if;
    xmlprint::endElement(plot::MuPlotML::OutputFile, rep[1]);
    xmlprint::raw(plot::MuPlotML::OutputFile, "\n");
  end_if;
end_proc:
//-------------------------------------------------

unalias(ExprAttribute):