/** plot::inspector:
 *
 *  Domain containing some functions for the creaion of new
 *  inspector entries for non-primitive plot domains.
 */

// make sure attributes.mu is loaded, so that all type-checking procedures
// are defined
plot::attributes:

plot::inspector := newDomain(hold(plot::inspector)):

plot::inspector::create_dom := hold(plot::inspector):

plot::inspector::info := "plot::inspector -- creating new inspector entries":

plot::inspector::interface := {hold(PropCat), hold(PropInfo)}:

plot::inspector::BasePropertyType :=
  table("String"      = plot::checkText,
        "Color"       = plot::checkColor,
        "ColorList"   = plot::checkColorList,
        "Float"       = plot::checkReal,
        "PosFloat"    = plot::checkPosOutputSize,
        "NonNegFloat" = plot::checkNonNegOutputSize,
        "PosSize"     = plot::checkPosOutputSize,
        "NonNegSize"  = plot::checkNonNegOutputSize,
        "TipAngle"    = plot::checkPosReal,
        "ScenePos"    = plot::checkOutputPos,
        "Int"         = plot::checkInteger,
        "PosInt"      = plot::checkPosInteger,
        "NonNegInt"   = plot::checkNonNegInteger,
//        "Enum"        = FAIL,  set implicitly when using a set
        "Font"        = plot::checkFont
        ):
plot::inspector::BasePropertyMupType :=
  table("String"      = DOM_STRING,
        "Color"       = Type::ListOf(Type::Real, 3, 4),
        "Float"       = Type::Real,
        "PosFloat"    = Type::Positive,
        "NonNegFloat" = Type::NonNegative,
        "PosSize"     = Type::Positive,
        "NonNegSize"  = Type::NonNegative,
        "TipAngle"    = Type::Positive,
        "ScenePos"    = Type::NonNegative,
        "Int"         = Type::Integer,
        "PosInt"      = Type::PosInt,
        "NonNegInt"   = Type::NonNegInt
//        "Enum"        = FAIL,  set implicitly when using a set
//        "Font"        = FAIL has to be implemented
        ):

  
plot::inspector::EnumPropertyType :=
  table("ArrowLength"      = {Fixed, Proportional, Logarithmic},
        "AnimationStyle"   = {Loop, RunOnce, BackAndForth},
        "Axes"             = {Boxed, Frame, Origin, Automatic, None},
        "BackgroundStyle"  = {Flat, LeftRight, TopBottom, Pyramid},
        "Bool"             = {TRUE, FALSE},
        "ColorType"        = {Flat, Monochrome, Dichromatic, Rainbow,
                              Functional},
        "ColorAttributeName" = {LineColor, FillColor, PointColor, Colors},
        "CoordinateType"   = {LogLog, LinLog, LogLin, LinLin},
        "Scaling"          = {Constrained, Unconstrained, Automatic},
//      "CoordinateType2d" = FAIL,
//      "CoordinateType3d" = FAIL,
        "GridLineStyle"    = {Solid, Dashed, Dotted},
        "HorAlignment"     = {Left, Center, Right},
        "LegendPlacement"  = {Top, Bottom},
        "LineStyle"        = {Solid, Dashed, Dotted},
        "OutputUnits"      = {unit::cm, unit::dm, unit::km, unit::m, unit::mm, unit::pt, unit::inch},
        "PointStyle"       = {Squares, FilledSquares, Circles, FilledCircles,
                              Crosses, XCrosses, Diamonds, FilledDiamonds,
                              Stars},
        "PointStyle3d"     = {FilledSquares, FilledCircles},
        "Layout" = {Vertical, Horizontal, Tabular, Relative, Absolute},
        "Shading"          = {Flat, Smooth},
        "TicksLabelStyle"  = {Horizontal, Vertical, Diagonal, Shifted},
        "TicksNumber"      = {None, Low, Normal, High},
        "AxisTitleAlignment"   = {Begin, Center, End},
        "TitleOrientation" = {Horizontal, Vertical},
        "TipStyle"         = {Filled, Open, Closed},
        "Extension"        = {Finite, SemiInfinite, Infinite},
        "FillStyle"        = {Winding, EvenOdd},
        "FillPattern"      = {Solid, HorizontalLines, VerticalLines,
                              DiagonalLines, FDiagonalLines,
                              CrossedLines, XCrossedLines},
        "VertAlignment"    = {Top, BaseLine, Center, Bottom},
        "Lighting"         = {None, Automatic, Explicit}
        ):

       // ccr begin
plot::inspector::EnumPropertyMupType :=
 map(plot::inspector::EnumPropertyType,
     proc(set)
     begin
       Type::ElementOf(set);
     end_proc):
       // ccr end

plot::inspector::ElemPropertyType :=
  table("Title"               = plot::elementText,
        "Ticks"               = plot::checkTicksAt,
        "TextOrientation"     = plot::checkTextOrientation,
        "Expression"          = plot::elementExpr,
        "ValExpression"       = plot::elementVExpr,
        "Matrix2d"            = plot::checkMatrix2d,
        "Matrix3d"            = plot::checkMatrix3d,
        "Points2d"            = plot::checkPoints,
        "Points3d"            = plot::checkPoints,
        "Intensity"           = plot::elementOptExpr,
        "Angle"               = plot::elementOptExpr,
        "SpotAngle"           = plot::elementOptExpr
        ):

plot::inspector::ElemPropertyMupType :=
  table("Title"               = DOM_STRING,
//     "Ticks"               = plot::checkTicksAt,   
//      "TextOrientation"     = plot::checkTextOrientation,
        "Expression"          = Type::Arithmetical,
        "ValExpression"       = Type::Real,
//      "Matrix2d"            = plot::checkMatrix2d,
//      "Matrix3d"            = plot::checkMatrix3d,
        "Points2d"            = Type::ListOf(DOM_LIST),
        "Points3d"            = Type::ListOf(DOM_LIST)
//      "Intensity"           = plot::elementOptExpr,
//      "Angle"               = plot::elementOptExpr,
//      "SpotAngle"           = plot::elementOptExpr
        ):



plot::inspector::PropertyType :=
  table(plot::inspector::BasePropertyType,
        plot::inspector::EnumPropertyType,
        plot::inspector::ElemPropertyType):

plot::inspector::PropertyMupType :=
  table(plot::inspector::BasePropertyMupType,
        plot::inspector::EnumPropertyMupType,
        plot::inspector::ElemPropertyMupType):

alias(EA = plot::ExprAttribute):

plot::inspector::structTable := table():

plot::inspector::new :=
  proc(objType : DOM_STRING)
  begin
    sysassign(plot::inspector::structTable[objType],
              EA::namedCompound("ObjType", {"Type"=objType}, [hold(namedCompound)("Type"=objType, "...")],
                                args(2..args(0))))
  end_proc:


plot::inspector::PropCat :=
  proc(Name : DOM_STRING)
  begin
    EA::namedCompound("PropCat", {"Name"=args(1)}, [hold(PropCat)("Name"=args(1), "...")], args(2..args(0)))
  end_proc:

plot::inspector::PropInfo :=
  proc()
  begin
    EA::generic("PropInfo", {"Name"=args(1), args(2..args(0))})
  end_proc:

plot::inspector::Enum :=
  proc(s : DOM_SET)
    local tmp, i;
  begin
    tmp := DOM_SET::sort(s);
    _concat(tmp[1], (" ", i) $ i in [op(tmp, 2..nops(tmp))])
  end_proc:


// called by plot::createPlotDomain
plot::buildInspectorEntry :=
proc(objName : DOM_STRING, dim : DOM_INT, def : DOM_LIST,
     elemType : DOM_STRING)
  local idx, newDef, addInspectorEntry, list2PropCat, newAttr,
        chkFunc, defaultVal, oldAttr, readFunc, autoEntries;
begin
  addInspectorEntry :=
  proc(Name, entry)
    local i, j, mkPropInfo;
  begin
    mkPropInfo :=
    proc(Name, entry)
      local attr;
    begin
      if entry[2] <> "XMLAttribute" then
        attr := "Dom" = entry[2], "Opt"=Name;
        if entry[2] = "Prop" and domtype(entry[3]) = DOM_SET then
          attr := attr, "Type"="Enum"
        end_if;
      else
        attr := "Dom" = Name
      end_if;
      if domtype(entry[3]) = DOM_LIST then
        // different values for 2D and 3D by different property types
        if contains(plot::all2dObjects, objName) then
          attr := attr, "Type"=entry[3][1];
        else
          attr := attr, "Type"=entry[3][2];
        end_if;        
      elif contains(plot::inspector::PropertyType, entry[3]) then
        attr := attr, "Type"=entry[3];
      elif domtype(entry[3]) = DOM_SET then
        attr := attr, "Enum" = plot::inspector::Enum(entry[3]);
        // add set entries to stdlib::OPTIONS
//        sysassign(stdlib::OPTIONS, stdlib::OPTIONS union entry[3]);
        sysassign(plot::extraValues, plot::extraValues union entry[3]);
      elif domtype(entry[3]) = DOM_LIST then
        // different values for 2D and 3D by different explicit sets
        if contains(plot::all2dObjects, objName) then
          attr := attr, "Enum" = plot::inspector::Enum(entry[3][1]);
          // add set entries to stdlib::OPTIONS
//          sysassign(stdlib::OPTIONS, stdlib::OPTIONS union entry[3][1]);
          sysassign(plot::extraValues, plot::extraValues union entry[3][1]);
        else
          attr := attr, "Enum" = plot::inspector::Enum(entry[3][2]);
          // add set entries to stdlib::OPTIONS
//          sysassign(stdlib::OPTIONS, stdlib::OPTIONS union entry[3][2]);
          sysassign(plot::extraValues, plot::extraValues union entry[3][2]);
        end_if;
      else
        // should be FAIL, but we ignore everything else
      end_if;
      if entry[4] <> "" then
        attr := attr, "Info" = entry[4]
      end_if;
      if entry[5] = TRUE then
        attr := attr, "Recalc" = TRUE
      end_if;
      
      
      plot::inspector::PropInfo(Name, attr);
    end_proc:

    if domtype(entry) = DOM_SET then
      // library interface for the given set of attributes;
      // just allow it
    elif domtype(entry[1]) = DOM_STRING then
      for i from 1 to nops(newDef) do
        if newDef[i][1] = entry[1] then
          newDef[i] := newDef[i].[mkPropInfo(Name, entry)];
          return();
        end_if;
      end_for;
      // not found append to newDef
      newDef := newDef.[[entry[1], mkPropInfo(Name, entry)]];
    else
      // list with 2 string entries
      assert(testtype(entry[1], Type::ListOf(DOM_STRING, 2, 2)));
      for i from 1 to nops(newDef) do
        if newDef[i][1] = entry[1][1] then
          for j from 2 to nops(newDef[i]) do
            if domtype(newDef[i][j]) = DOM_LIST and
              newDef[i][j][1] = entry[1][2] then
              newDef[i][j] := newDef[i][j].[mkPropInfo(Name, entry)];
              return();
            end_if
          end_for;
          // not found append to newDef[i]
          newDef[i] := newDef[i]. [[entry[1][2], mkPropInfo(Name, entry)]];
          return();
        end_if;
      end_for;
      // not found append to newDef
      newDef := newDef.[[entry[1][1],
                        [entry[1][2], mkPropInfo(Name, entry)]]];
    end_if;
  end_proc:

  if elemType = "Obj" then
    newDef := [["Definition"], ["Animation"], ["Annotation"]];
  elif elemType = "Transform" then
    newDef := [["Definition"], ["Animation"]];
  else
    // not known
    error("unknown element type")
  end_if;
  
  for idx in def do
    readFunc := FAIL;
    if domtype(idx) = DOM_LIST then
      // definition of a new attribute
      if domtype(idx[2]) = DOM_LIST and idx[2][1] = "Library" then
        // definition of a library interface attribute
        if nops(idx[2]) = 4 and testtype(idx[2][3], Type::Function) and
          testtype(idx[2][4], Type::Function) and
          testtype(idx[3], DOM_SET) or testtype(idx[3], DOM_LIST) then
          chkFunc  := idx[2][3];
          readFunc := idx[2][4];
        else
          error("invalid library attribute definition for ".idx[1]);
        end_if;
      elif contains({"Expr", "ColorFunc", "ExprSeq"}, idx[3][2]) or
        idx[3][3] = "Expression" then
        if idx[3][2] = "Expr" and idx[3][3] <> FAIL then
          chkFunc := idx[3][3];
          if chkFunc = plot::elementOptFunctionExpr or
            chkFunc = plot::elementOptTextExpr then
            sysassign(plot::functions,
                      plot::functions union {idx[1]})
          end_if;
        elif idx[3][2] = "ExprSeq" then
          if idx[2][2] <> NIL then
            warning("Attributes of type ExprSeq must set their default values via styleSheet;\ndefault value '".expr2text(idx[2][2])."' for attribute '".idx[1]."' skipped")
          end_if;
          chkFunc := plot::checkList;
        elif idx[3][2] = "ColorFunc" then
          chkFunc := plot::checkColorFunction;
          sysassign(plot::colorFunctions,
                    plot::colorFunctions union {idx[1]})
        else
          chkFunc := plot::elementOptExpr;
        end_if;
      elif "Prop" = idx[3][2] then
        if domtype(idx[3][3]) <> DOM_SET then
          if contains(plot::inspector::PropertyMupType, idx[3][3]) then
            chkFunc := plot::elementOptProp(
                         plot::inspector::PropertyMupType[idx[3][3]]);
          else
            error("invalid inspector type definition for ".idx[1]);
          end_if;
        else
          chkFunc := plot::elementOptPropSet(idx[3][3])
        end_if;
      elif contains(plot::inspector::PropertyType, idx[3][3]) then
        chkFunc := plot::inspector::PropertyType[idx[3][3]];
      else
        error("invalid inspector type definition for ".idx[1]);
      end_if;

      if idx[2][2] <> NIL then
        defaultVal := chkFunc(objName, idx[1], idx[2][2]);
        if domtype(op(defaultVal)) = DOM_STRING then
          error("illegal default value '".expr2text(idx[2][2]).
                "' for attribute ".idx[1]." of object type plot::".objName);
        else
          defaultVal := [op(defaultVal, 2)];
        end_if;
      else
        defaultVal := [NIL]
      end_if:
      
      // build new entry for attributes table
      newAttr := [idx[2][1], op(defaultVal), chkFunc, {}, readFunc, idx[3]];

      if not contains(plot::attributes, idx[1]) then
        // new entry, insert entry into plot::attributes
        sysassign(plot::attributes[idx[1]], newAttr);
      else
        // make sure newAttr is consistent with existing entry
        // but make sure that escaped check-procedures have no
        // reference to their creators 
        oldAttr := plot::attributes[idx[1]];
        oldAttr := subsop(oldAttr, 4={}, [3, 12]=NIL);
        if oldAttr[5] <> FAIL then
          oldAttr := subsop(oldAttr, [5,12]=NIL);
        end_if;
        newAttr := subsop(newAttr, 4={}, [3, 12]=NIL);
        if newAttr[5] <> FAIL then
          newAttr := subsop(newAttr, [5,12]=NIL);
        end_if;
        
        if oldAttr = newAttr then
          // they are consistent, everything is fine
        else
          if subsop(newAttr, 2=NIL) = subsop(oldAttr, 2=NIL) then
            // specification only differs in default value; put it in the style sheet
            warning("attribute '".idx[1]."' already in use with different default value;\nPut default value into style sheet!")
          else
            error("attribute '".idx[1]."' already in use by ".expr2text(op(oldAttr[4]))." with different semantics")
          end_if
        end_if;
      end_if;
      
      // insert option name to global set stdlib::OPTIONS
//      sysassign(stdlib::OPTIONS, stdlib::OPTIONS union {idx[1]});

      if idx[2][1] <> "Library" then
        // insert entry into inspector
        addInspectorEntry(idx[1], plot::attributes[idx[1]][6]);
        if idx[2][1] = "Mandatory" then
          sysassign(plot::allMandatoryAttributes,
                    plot::allMandatoryAttributes union {idx[1]}):
        elif idx[2][1] = "Inherited" then
          sysassign(plot::allInheritedAttributes,
                    plot::allInheritedAttributes union {idx[1]}):
        else // Optional
          sysassign(plot::allOptionalAttributes,
                    plot::allOptionalAttributes union {idx[1]}):
       end_if
      else
        sysassign(plot::allLibraryAttributes,
                  plot::allLibraryAttributes union {idx[1]}):
      end_if;
    else
      if not contains(plot::attributes, idx) then
          error("invalid attribute name '".expr2text(idx)."'")
      elif not contains(plot::allLibraryAttributes, idx) then
        // insert entry into inspector
        addInspectorEntry(idx, plot::attributes[idx][6]);
      end_if;
    end_if;
  end_for;

  // add some entries automatically
  if elemType = "Obj" then
    autoEntries :=
    [ParameterName, ParameterBegin, ParameterEnd, Name, AffectViewingBox,
     Visible, TimeBegin, TimeEnd, VisibleBeforeBegin, VisibleAfterEnd,
     Frames, Title, TitlePositionX, TitlePositionY,
     if dim = 3 then TitlePositionZ else null() end,
     LegendEntry, LegendText, TitleFont, TitleAlignment]
  elif elemType = "Transform" then
    autoEntries :=
    [ParameterName, ParameterBegin, ParameterEnd, Name, AffectViewingBox,
     TimeBegin, TimeEnd, Frames]
  else
    // not known
    error("unknown element type")
  end_if;
  
  for idx in autoEntries do
    addInspectorEntry(idx, plot::attributes[idx][6]);
  end_for;

  // build inspector entry from entries in newDef
  list2PropCat := x -> if domtype(x) = DOM_LIST then
                         plot::inspector::PropCat(op(x))
                       else
                         x
                       end_if:

  newDef := map(newDef, map, list2PropCat);
  newDef := map(newDef, list2PropCat);
  plot::inspector::new(objName, op(newDef));
end_proc:
