//   

/*--
faclib::canzas -- split the product of all monic irreducible factors of degree 
                  i via Cantor-Zassenhaus method (see Keith O. Geddes, Stephen
                  R. Czapor and George Labahn, Algorithms for computer algebra,
                  page 373, Kluwer Academic Publishers, 1992), output is a set
                  of factors in polynomial form.

faclib::canzas(fpoly,i,matr,p,x) 
fpoly - a univatiate polynomial, it is the product of all monic irreducible
        factors of degree i
i - a positive integer number 
matr - a list of polynomials, matr[i]=divide(x^(i*p),fpoly0, Rem) and
         fpoly0=fpoly*(another polynomial)      
p - the prime number>2, all the related polynomials are in Zp
x - the indeterminate

--*/

faclib::canzas:=
proc(fpoly, i, matr, p, x)
  local d, dim, f, j, m, rp, pp, rnd: DOM_PROC;
  save SEED;
begin
  SEED:= 1;
  userinfo(2,"Equal-degree factorization for degree ".expr2text(i));
  userinfo(3,"Degree of input: ".expr2text(degree(fpoly)));
  if degree(fpoly)=i then
    return({fpoly})
  end_if;
  dim:= degree(fpoly)-1;
  delete m;
  if type(matr)=DOM_TABLE then // ddf_shoup was used 
    matr:=[divide(matr[m],fpoly,Rem)$m=1..nops(matr)-1];
    if nops(matr)<dim then // compute remaining powers 
      for j from nops(matr)+1 to dim do
        matr:=append(matr,divide(matr[j-1]*matr[1],fpoly,Rem))
      end_for
    end_if
  elif dim<nops(matr) then
    // compute matr[i]=divide(x^(i*p),fpoly,Rem) 
    matr:=[divide(matr[m],fpoly,Rem) $m=1..dim];
  end_if;

  rnd:= random(p);
  
  repeat
    d:=powermod(rnd(), dim, dim-1)+1;
    // generates a random polynomial over Zp 
    // of degree lower than that of fpoly    
    rp:=poly(rnd() +_plus(rnd() * _power(x,m)$m=1..d)\
             +_power(x,d+1),[x],IntMod(p));
    userinfo(10, "Using random polynomial ".expr2text(rp));
    pp:=rp;
    // use matr to compute rp^(p^(i-1)) mod fpoly 
    /* because of the characteristic,
       (a0 + ... + an*x^n)^p = a0 + a1*x^p + ... + an*x^(p*n) */
    /* we compute rp, rp^p, rp^(p^2) ... (after the i-th iteration,
       rp equals the original rp^(p*i) */
    /* multiplying pp all of these, we obtain
       pp = rp * rp^p * ... * rp^(p*(i-1))
          = rp^(1+p+p^2+...+p*(i-1)) */     
    for m from 1 to i-1 do
      rp:=poly(coeff(rp,0),[x],IntMod(p))+
           _plus(multcoeffs(matr[j],coeff(rp,j))$j=1..dim);
      pp:=divide(rp*pp, fpoly, Rem);
    end_for;

    /* since (p-1)/2 * (1+p+...+p^(i-1)) = p^i - 1, we
       now get the original rp^((p^i-1)/2) */

    pp:=faclib::powermod_poly(pp,(p-1)/2,fpoly)-poly(1,[x],IntMod(p));
    pp:=faclib::univ_mod_gcd(fpoly,pp);
    if degree(pp)>0 and pp<>fpoly then
      userinfo(4,"Dividing into factors of degree ".
               expr2text(degree(pp)).
               " and ".expr2text(degree(fpoly)-degree(pp)));
      f:=faclib::canzas(pp,i,matr,p,x) union
      faclib::canzas(divide(fpoly,pp,Quo),i,matr,p,x);
    else
      userinfo(4,"Split did not succeed");
      f:=FAIL
    end_if;
  until f<>FAIL /*nops(f)=degree(fpoly)/i*/ end_repeat;
  f;
end_proc:

faclib::canzas_lin:=
proc(fpoly, p, x) // split into linear factors (case i=1) 
  local pp, f, rnd;
  save SEED;
begin
  SEED:= 1;
// return list instead of sets because it is more efficient 
  if degree(fpoly)=1 then
    if lcoeff(fpoly) <> 1 then
      fpoly:= multcoeffs(fpoly, 1/lcoeff(fpoly))
    end_if;
    return([fpoly])
  end_if;
  rnd:= random(p);
  repeat
    pp:=poly(x + rnd(),[x],IntMod(p));
    pp:=faclib::powermod_poly(pp, (p-1)/2, fpoly)-poly(1,[x],IntMod(p));
    pp:=faclib::univ_mod_gcd(pp, fpoly);
    if degree(pp)>0 and degree(pp)<degree(fpoly) then
      f:=faclib::canzas_lin(pp,p,x) .
      faclib::canzas_lin(divide(fpoly,pp,Quo),p,x)
    else
      f:=FAIL
    end_if
  until /* nops(f)=degree(fpoly) */ f<>FAIL end_repeat;
  f
end_proc:

