/*--
faclib::ddf -- return the partial distinct degree factorization of q(x) in Zp,
               (see Alkiviadis G. Akritas, Elements of computer algebra with
               applications, page 299, John Wiley & Sons, Inc., 1989), output is
               given in form [[[f1,the distinct degree of f1],..,[fn, the
               distinct degree of fn]], Q-matrix(in a list of polynomial form,
               if it is calculated)].

faclib::ddf(a,p,x) 
a - a square free monic univariate polynomial in Zp
p - a prime number
x - the indeterminate

faclib::matrix_Q -- compute a list of polynomials, its i'th element equals 
                    x^(i*p) mod a(x) in Zp, their coefficients construct a
                    matrix (usually called Q-matrix)

faclib::matrix_Q(a,p,dim,x)
a - a square free univariate polynomial in Zp
p - a prime number
dim - a positive integer number, it is equal to degree(a)-1 
x - the indeterminate
--*/

faclib::ddf:=proc(a,p,x)
local d, dim, f, fs, j, m, r, dega, a0;
begin
    userinfo(2,"Distinct-degree factorization");
    /* first look for the degree 1 factors : they necessarily divide
      x^p-x, thus a1(x)=gcd(a(x),x^p-x) is the product of linear factors 
	(cf Geddes pages 368-369)
    */
    f:=poly(x,[x],IntMod(p));
    f:=faclib::powermod_poly(f,p,a)-f;
    if degree((f:=faclib::univ_mod_gcd(f,a)))>0 then // there are linear factors 
       if degree((a:=divide(a,f,Quo)))=0 then // all factors are linear 
          return([[[f,1]],[]]);
       elif degree(a)<4 then // there can be only one non-linear factor 
            return([[[f,1],[a,degree(a)]],[]]);
       else fs:=[[f,1]]; // linear factors 
       end_if;
    elif degree(a)=2 then return([[[a,2]],[]]) // it can have only one factor of deg 2 
    elif degree(a)=3 then return([[[a,3]],[]]) // it can have only one factor of deg 3 
    else fs:=[];
    end_if;                    
    d:=2; // degree of factors currently extracted 
    dim:=degree(a)-1;
    // Q-matrix method is used for nonlinear factors(see Alkiviadis G. Akritas, 
    // Elements of computer algebra with applications, page 300, John Wiley &   
    // Sons, Int., 1989) to compute r^p mod a                                   
    a0:=a; // a0 is the product of all non linear factors 
    m:=faclib::matrix_Q(a,p,dim,x);
    r:=m[1]; // m[i] = x^(i*p) mod a 
    dega := degree(a);
    while d <= dega/2 do 
          userinfo(4,"Extracting factors of degree ".expr2text(d));
          // r <- r(x <- x^p) mod a, thus r = x^(p^d) mod a 
          r:=poly(coeff(r,0),[x],IntMod(p))+\
                  _plus(multcoeffs(m[j],coeff(r,j))$ j=1..dim);
          if degree((f:=faclib::univ_mod_gcd(r-poly(x,[x],IntMod(p)),a)))>0 then
             fs:=append(fs,[f,d]);
             // no need to compute a/f if the factorization is finished 
             if degree(f)=degree(a) then dega:=0; break end_if;
             a:=divide(a,f,Quo); 
             dega := degree(a);
             r:=divide(r,a,Rem); 
          end_if; 
          d:=d+1;
    end_while;
    if dega=0 then 
       [fs,m]
    else [append(fs,[a,dega]),m]
    end_if;
end_proc:

faclib::matrix_Q:=proc(a,p,dim,x)
local i, m;
begin
    // there are several methods to compute this matrix (see page 422-428, The 
    // art of computer programming, Donald E. Knuth, Addison-Wesley Publishing 
    // Company, 1981), here use a relative direct and simple one               
    m:=[faclib::powermod_poly(poly(x,[x],IntMod(p)),p,a), (0 $ i=2..dim)];
    for i from 2 to dim do
        m[i] := divide(m[i-1]*m[1],a,Rem);
    end_for;
    return(m);
end_proc:



