/*--
faclib::mrestore -- restore the leading coefficients to factors of the 
                    univariate polynomial (see P. S. Wang, "An improved
                    multivariate polynomial factoring algorithm", Math. Comp.
                    32, page 1215-1231, (1978)).  According to the input, output
                    is given in form: appropriate readjustment of input p and
                    ufs, a list of correct multivariate leading coefficients(in
                    expression form) corresponding to the univariate factors.
 
faclib::mrestore(p,ufs,nu,lfs,lgs,ucont,bn,elfs)
p - a primitive square free multivariate polynomial in Z[x1..xn]
ufs - a list of nu univariate polynomials in Zpl[x1], which are pairwise
      relatively prime in Zpl[x1], such that u[1]*..*u[nu]=p mod <I, pl> 
nu - a positive integer number, it is the number of polynomials in ufs
lfs - a list of polynomials, they are the factors of lcoeff(p[x1]))
      factorization
lgs - a list of nonnegative integer numbers, they are the powers of
      lcoeff(p[x1])) factorization
ucont - an integer number, it is the icontent(p(x1,a2..an))
bn - a list of equations [x2=a2,..,xn=an], representing an evaluation
     homomorphism used, mathematically viewed as the idea I=<x2-a2,..,xn-an>
elfs - a list of integer numbers, they are the values of lfs at bn
--*/

faclib::mrestore:=proc(p,ufs,nu,lfs,lgs,ucont,bn,elfs)
local e, eh, h, g, i, j, k, lu, nl, s, t, tt; 
begin
    // if lcoeff=1 then it is straightforward 
    if lfs=[] then 
       if ucont=1 then 
          return(p,ufs,[]);
       else return(multcoeffs(p,_power(ucont,nu-1)),\
                   map(ufs,multcoeffs,ucont),[]);
       end_if;
    end_if;
    nl:=nops(lfs); 
    s:=array(1..nl,1..nu,[[0$nu]$nl]);
    t:=[lcoeff(ufs[i])*ucont $i=1..nu];
    // restore the factors 
    for j from nl downto 1 do 
        e:=lgs[j];
        for i from 1 to nu do 
            if e=0 then 
               break; 
            end_if;
            for k from 1 to lgs[j]+1 do 
                if type((tt:=t[i]/elfs[j]))=DOM_INT then 
                   t[i]:=tt; 
                else break; 
                end_if;
            end_for;
            s[j,i]:=k-1; 
            e:=e-k+1;
        end_for;
        if e<>0 then 
           return(FAIL);
        end_if;
    end_for;
    h:=[_mult(_power(lfs[j],s[j,i])$j=1..nl)$i=1..nu];
    // restore the icontent 
    if ucont=1 then 
       return(p,ufs,[_mult(h[i],lcoeff(ufs[i])/subs(h[i],bn))$i=1..nu]);
    else for i from 1 to nu do 
             lu:=lcoeff(ufs[i]); 
             eh:=subs(h[i],bn); 
             g:=igcd(lu,eh);
             h[i]:=_mult(h[i],lu/g);  
             ufs[i]:=multcoeffs(ufs[i],eh/g);
             ucont:=ucont*g/eh;
         end_for;
         if ucont=1 then 
            return(p,ufs,h);
         else multcoeffs(p,_power(ucont,nu-1)),\
              map(ufs,multcoeffs,ucont),map(h,_mult,ucont);
         end_if;
    end_if;
end_proc:
        
