/*--
faclib::plift -- lift the factors of the univariate polynomial a in Zp[x] to Z[x], the
                 basic idea of the algorithm can be found on page 100-101,
                 Computer algebra symbolic and algebraic computation, edited by
                 Buchberger, G. E. Collins and R. Loos, Springer-Verlag, Wien,
                 1982.  

                 During the lifting process 'smaller' combinations of the lifted
                 factors are checked whether they are already true factors.

                 If there are modular factors left after the lifting, either 
                 faclib::combine or faclib::knapsack are called to compute the
                 final factorcombination.
                 
                 Output is the factors in polynomial form.

faclib::plift(a,lc,f,nf,np,x,p,pos_g)
a - a primitive square free univariate polynomial in Z[x]
lc - lcoeff(a)
f - a list of polynomials in Zp[x], they are the factors of a in Zp[x]
nf - the number of polynomials in f
np - a nonnegative integer number, the first np polynomials of f may be the 
    true factors of a in Z[x]
x - the indeterminate
p - the prime number
pos_g - a set of positive integer numbers, they are the possible degrees that
        factors of a in Z[x] may have
--*/

faclib::plift:=
proc(a,lc,f,nf,np,x,p,pos_g,bf=FAIL)
  local d, d1, fc, ff, fl, fo, ft, i, j, la, qua, m2, pj, help_a, po,
      q, q1, s, t, v, vo, w, z, l, ptruefs, oldpj, oldf, remains, kmax,
      min_pos_g, nmax, nc, maxcombine, ll, USE_HOEIJ, HAVE_HOEIJBOUND, tmp;
begin
  USE_HOEIJ := bool(nops(f) > 12);
  HAVE_HOEIJBOUND := FALSE;
  if USE_HOEIJ then
      maxcombine := 3; // no more than 3 factors should be tested during the
  else
      maxcombine := nops(f)+1;
  end_if;
  // lifting process
  f[1]:=multcoeffs(f[1],lc mod p);
  help_a:=a;
  la:=lc; min_pos_g:=min(op(pos_g));
  if bf=FAIL then
    bf:=max(faclib::bound(a), p+1)
  end_if;
  userinfo(1,"bound on factor coefficients is",float(bf));
 
  d1:=degree(f[1]);
  w[nf]:=f[nf];
  for i from nf-1 downto 2 do
    w[i]:=f[i]*w[i+1];
  end_for;
  q:=poly(1,[x],IntMod(p));
  v:=[];
  // compute v, such that b[1]*v[1]+..+b[nf]*v[nf]=1 mod p, with 
  // degree(v[j])<degree(f[j]), where in terms of the given list 
  // of polynomials f[1],..,f[nf], the polynomials b[i](i=1..nf) 
  // are defined by b[i]=f[1]*..*f[i-1]*f[i+1]*..*f[nu]          
  for i from 1 to nf-1 do
    q1:=faclib::mod_gcd_ex(f[i],w[i+1]);
    v:=append(v,divide(q1[2]*q,f[i],Rem));
    q:=divide(q1[1]*q,w[i+1],Rem);
  end_for;
  v:=append(v,q);
  l:=faclib::bound2list(bf,p,1);
  delete l[1];
  
  help_a:=a;
  po:=p;
  fl:=[1$i=1..np];
  ptruefs:={$1..np}; // remaining polynomials that can be true factors 
  remains:={$1..nf}; // remaining polynomials 
  kmax:=1;
  nmax:=nf;
  nc:=0;
  ft:=[]; // true factors 
  while nops(l) > 0 do // po<=bf do
    oldf:=f;
    if (qua:=bool(l[1] mod 2 = 0)) then
      pj:=po*po
    else
      pj:=po*p
    end_if;
    oldpj:=pj;
    fo:=map(f,poly,IntMod(pj));
    if not iszero((d:=poly((poly
           ((lc-lcoeff(fo[1]))*_power(x,d1),[x],
             IntMod(pj))+fo[1])*_mult(fo[i]$i=2..nf)-poly(a,IntMod(pj)),
             Expr))) then
      d:=poly(multcoeffs(d,1/po),IntMod(po));
      for i from 1 to nf do
        f[i]:=fo[i]-poly(multcoeffs(poly(divide(v[i]*d,f[i],Rem),
                                         Expr),po),IntMod(pj));
      end_for;
    else
      f:=map(f,poly,IntMod(pj));
    end_if;
    f[1]:=poly((lc-lcoeff(f[1]))*_power(x,d1),[x],IntMod(pj))+f[1];
    userinfo(1,"lifted to ".p."^".l[1]." at time ".time());
    j:=0;
    m2:=sqrt(float(pj/2));
    // during lifting, factors are tested if they 
    // are already the true factors of a in Z     
    for i in ptruefs do
      userinfo(5, "Checking ".expr2text(f[i]).
               " whether it can be a true factor");
      if i=1 then
        ff:=poly(f[1],Expr)
      else
        ff:=poly(multcoeffs(f[i],la),Expr);
      end_if;
      /* check only if factor did not change during lifting,
                     except for the last time */
      if lc=1 and nops(l) >= 1 //pj<bf 
              and f[i]<>poly(oldf[i],IntMod(pj)) then
        t:=[]
      else
        t:=faclib::true_fs(help_a,la,multcoeffs
                           (ff,sign(lcoeff(ff))/icontent(ff)),pj,m2);
      end_if;
      if t<>[] then
        j:=j+1;
        fl[i]:=t[1];
        ft:=append(ft,fl[i]);
        ptruefs:=ptruefs minus {i};
        remains:=remains minus {i};
        help_a:=t[2];
        la:=lcoeff(help_a);
      end_if;
    end_for;
    // if there are true factors, then the bound can be reduced 
    if j>0 then
      userinfo(1,"*** found ".j." factor(s)");
      case nf-j
        of 0 do
          return(op(fl));
        of 1 do
          if nf=np then
            for i from 1 to nops(fl) do
              if fl[i]=1 then
                fl[i]:=help_a;
                return(op(fl));
              end_if;
            end_for;
          else
            return(op(fl),help_a);
          end_if;
      end_case;
      // deleted the following because it gave incomplete factorizations
      // bf:=faclib::bound(help_a);
      // userinfo(1,"new bound is",float(bf));
      l:=faclib::bound2list(bf,p,l[1]);
      pj:=p^l[1];
    end_if;
    nc+binomial(nops(remains),kmax+1);
    if %<=nmax then
      nc:=%;
      // limit the number of combinations to 3
      kmax:=min(maxcombine,kmax+1); // kmax+1
    end_if;
    nmax:=4*nmax; // maximum number of combination steps 
    if kmax>=2 and nops(l)>1 then
      if degree(help_a) = 0 then
        return(op(ft))
      end_if;
      s:=faclib::combine(help_a,la,[f[op(remains,i)]$i=1..nops(remains)],
           op(f[op(remains,1)],[3,1]),pos_g,kmax,p,l[1],FALSE);
      if nops(s)>=2 then
        return(op(ft),op(map(s,op@faclib::ufactor,x)))
      end_if;
    end_if;

    if pj<>oldpj then // lift down 
      w:=map(w,poly,IntMod(pj));
      z:=map(z,poly,IntMod(pj));
      f:=map(f,poly,IntMod(pj));
    end_if;
    w[nf]:=f[nf];
    for i from nf-1 downto 2 do
      w[i]:=f[i]*w[i+1]
    end_for;
    // for quadratic lifting, v need to be modified 
    z[1]:=f[1];
    for i from 2 to nf-1 do
      z[i]:=f[i]*z[i-1]
    end_for;
    vo:=map(v,poly,IntMod(pj));
    s:=poly(multcoeffs(poly(vo[1]*w[2]+_plus(z[i-1]*vo[i]*w[i+1]
    $i=2..nf-1)+z[nf-1]*vo[nf],Expr)
    -poly(1,[x]),1/po),IntMod(po));
    for i from 1 to nf do
      v[i]:=vo[i]-poly(multcoeffs(poly(divide(v[i]*s,
      poly(f[i],IntMod(po)),Rem),Expr),po),IntMod(pj))
    end_for;
    po:=pj;


    //------------------------------------------------------
    // decide whether to use van Hoeij's algorithm
    //------------------------------------------------------
    if nops(l) = 1 then // the lifting is done

        // first adjust the modular factors that still need treatment
        fc:=[_if(fl[i]=1, f[i], null()) $ i=1..np,
            f[i+np]$i=1..nf-np];
        if fc=[] then // we are done.
            return(op(ft))
        elif USE_HOEIJ and nops(fc) > 12 then 
            // branch into knapsack
            if not HAVE_HOEIJBOUND then
                // need to determine the lifting-bound.
                tmp := faclib::knapsack(help_a, fc, p, l[1], hold(GetBound));
                HAVE_HOEIJBOUND := TRUE;
                maxcombine := 0; //  no more combine-steps in
                // the lifting-process
                i := 1;
                while l[i] < tmp do
                    i := i+1;
                    l := l.[2*l[i-1]];
                end_while;
            end_if; // not HAVE_HOEIJBOUND
            // either no more lifting necessary or already done
            if nops(l)=1 then 
                // now try to combine the factors with Hoeij's algrithm
                tmp := faclib::knapsack(help_a, fc, p, l[1]);
                if tmp <> FAIL then
                    return (op(ft),op(tmp))    
                end_if; // fc <> FAIL
            end_if; // nops(l) = 1 
        end_if; // Hoeij-specific treatment       
    end_if; // nops(l) = 1 
    ll := l[1]; 
    delete l[1];
  end_while;

  // some of factors need to combine 
  fc:=subsop(fc,1=multcoeffs(fc[1],1/lcoeff(fc[1])));
  pj:=op(fc[1],[3,1]);
  fc:=faclib::combine(help_a,la,fc,pj,pos_g,nops(fc),p,ll,TRUE);
  return (op(ft),op(fc));
end_proc:

/* from a bound bf and a prime p, returns a list l such that
  l[1]<=l0, l[i+1]=l[i]+1 or 2*l[i] and p^l[nops(l)]>bf

  Example: bound2list(2e448,101,1) -> [1, 2, 3, 6, 7, 14, 28, 56, 112, 224]
          bound2list(2e377,101,6) -> [5, 10, 11, 22, 23, 46, 47, 94, 95, 190]
*/
faclib::bound2list :=
proc(bf,p,l0)
  local l,pj;
begin
  pj:=ceil(float(ln(bf)/ln(p)));
  if p>=7 and pj mod 2 = 1 then
    pj:=pj+1; // make pj even to end by a quadratic step 
    if pj>=14 and pj mod 4 = 2 then
      pj:=pj+2
    end_if // two quadratic steps 
  end_if;
  l:=null();
  while pj>l0 do
    l:=pj,l;
    if pj mod 2 = 1 then
      pj:=pj-1
    else
      pj:=pj/2
    end_if
  end_while;
  [pj,l];
end_proc:
