 
//    

// kg, 24/05/95 

/*++
faclib::sqrfree_ffield -- square-free factorization for finite fields

faclib::sqrfree_ffield(p, n, m)

p    - univariate monic polynomial over finite field GF(n^m)
n, m - n^m is the number of field elements, n is a prime

faclib::sqrfree_ffield(p,n,m) returns a list [f1,e1,...,fk,ek] where the fi's
are the square-free factors with order ei.

The algorithm used is described in K.O.Geddes et.al. "Algorithms for Computer
Algebra", Kluwer 1992, p345 (SquareFreeFF).

NOTE:
It is assumed that for each indeterminate x of p the following holds:
p is primitive when viewed as polynomial in x. This is true if p was
'pre-factorized' by faclib::pre_factor.
++*/

faclib::sqrfree_ffield:= proc(p, n, m)
    local X, T, deg, res, i, d, w, y, z;
begin
    userinfo(1,"Univariate squarefree factorization over finite field called");
    userinfo(2,"Degree of input: ".expr2text(degree(p)));
    X:= op(p,2);
    T:= op(p,3);
    
    deg:= 1;
    res:= [];
    
    d:= polylib::Dpoly(p);
    if iszero(d) then
        p:= faclib::sqrfree_ffield(faclib::recip_pow(p, n, m), n, m);
        res:= [ (p[2*i-1], p[2*i]*n) $ i=1..(nops(p) div 2) ]
    else
        d:= gcd(p, d);
        w:= divide(p, d, Exact);
        while degree(w) <> 0 do
            userinfo(3,"Looking for squarefree factors of multiplicity ".
		     expr2text(deg));
            y:= gcd(w, d);
            z:= divide(w, y, Exact);
            if degree(z) <> 0 then
		res:= append(res, z, deg)
            end_if;
            deg:= deg + 1;
            w:= y;
            d:= divide(d, y, Exact)
        end_while;
        if degree(d) <> 0 then
            d:= faclib::sqrfree_ffield(faclib::recip_pow(d, n, m), n, m);
            res:= append(res, (d[2*i-1], d[2*i]*n) $ i=1..(nops(d) div 2))
        end_if;
    end_if;
    res
end_proc:

/*--
faclib::recip_pow -- reciprocal power

faclib::recip_pow(p, n, m)

p    - univariate non-zero polynomial over GF(n^m)
n, m - n^m is the number of field elements, n is a prime

Returns p^(1/n).
--*/

faclib::recip_pow:= proc(p, n, m)
    local x, i;
begin
    if m <> 1 then
        p:= mapcoeffs(p, _power, n^(m-1))
    end_if;
    x:= poly(op(p,[2,1]), op(p, 2..3));
    
    _plus(multcoeffs(x^(degree(nthterm(p,i)) / n), nthcoeff(p,i))
          $ i=1..nterms(p))
end_proc:


/*-- 
faclib::sqrfrmv -- multivariate squarefree factorization over finite fields

faclib::sqrfrmv(p,n,m)

p    - multivariate non-zero primitive polynomial over GF(n^m)
n,m  - n^m is the number of field elements, n is a prime

Returns a list [f1,e1,..,fk,ek] where p=f1^e1* ... * fk^ek and each fi is
 squarefree, and the fi are relatively prime

 Source: Bernardin, On squarefree factorization
 of multivariate polynomials over a finite field 

--*/




faclib::sqrfrmv := proc(p,n,m)
local i,o,allzero, gk, result, spd, Pvec, j, k,C ;

begin
    userinfo(1,"Multivariate squarefree factorization over finite field called");
    userinfo(2,"Degree of input: ".expr2text(degree(p)));
    if nops(op(p,2))=1 then return(faclib::sqrfree_ffield(p,n,m)); end_if;   
    o:=nops(op(p,2));   

    // easy squarefreeness test 

    result:=gcd(D([1],p),p);
    if degree(result)=0 then  // easy and frequent case 
             userinfo(2,"Polynomial is squarefree");
             return([p,1])
    end_if;



    // By assumption, p is primitive, so steps 1 and 2 are not necessary 

    // Combine steps 3 and 4 

    gk:=[];
    for i from 1 to o do 
        userinfo(3,"Main variable: ".expr2text(op(p,[2,i])));
	if not iszero(D([i],p)) then
		spd:=faclib::SqfPosDer(p,i,n);
		gk:=zip(gk,op(spd,1),_mult,poly(1,op(p,2..3)));
		// gk is automatically extended to necessary length 
		p:=op(spd,2);	 
		if degree(p)=0 then
			break;
		end_if;
	end_if;
    end_for;

    if degree(p)=0 then 
		result:=[];
		for j from 1 to nops(gk) do
			if degree(gk[j])>0 then
				result:=result.[gk[j],j];
			end_if;
		end_for;
		return(result);
    end_if;

    // Step 5 

    // all partial derivatives vanish 
    // find the maximum k such that p is a polynomial in n^k 

    userinfo(2, "Handling factors whose multiplicity is a power ".
	        "of the characteristic");
    allzero:=TRUE;
    k:=0;
    repeat
	k:=k+1;
	p:=faclib::recursmap(p,n,m);
	for i from 1 to o do
		if not iszero(D([i],p)) then
			allzero:=FALSE;
			break;
		end_if;
	end_for;
    until allzero=FALSE end_repeat;

    // Step 6 

    Pvec:=faclib::sqrfrmv(p,n,m);

    // Step 7 

    result:=[];
  
    for j from 2 to nops(Pvec) step 2 do
	Pvec[j]:=Pvec[j]*n^k;
    end_for;

    for i from 1 to nops(gk) do
	for j from 1 to nops(Pvec) step 2 do
		C:=gcd(gk[i],Pvec[j]);
		if degree(C)>0 then
			result:=result.[C,Pvec[j+1]+i];
			gk[i]:=divide(gk[i],C,hold(Exact));
			Pvec[j]:=divide(Pvec[j],C,hold(Exact));
		end_if;
	end_for;
	if degree(gk[i])>0 then
        	result:=result.[gk[i],i];
	end_if;
    end_for;
    
    for j from 1 to nops(Pvec) step 2 do
	if degree(Pvec[j])>0 then		
   		result:= result.[Pvec[j],Pvec[j+1]];
	end_if;
    end_for;
    result
    end_proc:
    
    	

faclib::SqfPosDer:=proc(a,j,p)
local b,c,w,v,u,i,g,h;

// returns list [[g1..gk],c] 

begin
	b:=D([j],a);
	c:=gcd(a,b);
	w:=divide(a,c,hold(Exact));
	v:=divide(b,c,hold(Exact));
	u:=v-D([j],w);
	i:=1;
	g:=[];
	while i<p-1 and not iszero(u) do
                userinfo(4,"Looking for factors whose multiplicity is ".
			   "congruent to ".expr2text(i)." modulo ".
			   expr2text(p));
		h:=gcd(w,u);
		g:=append(g,h);
		w:=divide(w,h,hold(Exact));
		c:=divide(c,w,hold(Exact));
		v:=divide(u,h,hold(Exact));
		u:=v-D([j],w);
		i:=i+1;
	end_while;
	[append(g,w),c]
end_proc:





faclib::recursmap:=proc(p,n,m) 
// computes the n-th root of p 
local k;
begin 
	if nops(op(p,2))=1 then 
		return(faclib::recip_pow(p,n,m));
	end_if;
	_plus(poly(faclib::recursmap(coeff(p,k*n),n,m),op(p,2..3))*poly(op(p,[2,1])^k,op(p,2..3)) 
    		 $ k=0..degree(p,op(p,[2,1])) div n) 
end_proc: 
         
// end of file 
