/*
   gcdlib::gcdexModComposite(A, B, x)

   A, B: polynomials in x over IntMod(n), n not a prime 
        (the algorithm would also work for primes)
   x   : variable

   returns polynomials g, s, t such that s*A + t*B = g

   g is a "gcd" of A and B: it is of minimal degree among those 
   polynomials for which the equation is solver, and its leading coefficient 
   contains as few prime factors of n as possible

   Algorithm: 
   this is a self-invented algorithm. 

   We use the Euclidean algorithm; whenever we would have to divide a by b 
   where lcoeff(b) is a zero divisor, we multiply a by a factor such that lcoeff(a)
   is a unit times the same zero divisor. If this is the case, then 
   a=qb +r where q=U*x^k for some k and  some unit u
   r will not have smaller degree than b, but in any case when going from (a, b) to (b, r), 
   the maximum of degrees decreases, and so the algorithm terminates.

*/
gcdlib::gcdexModComposite:=
proc(A: DOM_POLY, B: DOM_POLY, x)
  local n,  a, b, s, t, u, v, q, tmp, gcb, m, lc;
  
begin
  assert(op(A, [3, 0]) = IntMod);
  assert(op(A, 2) = [x]);
  
  n:= op(A, [3, 1]);
  
  a:= A; b:= B;
  s:=a^0; t:=s-s; u:=t; v:=s;
   // loop invariant : a = s*A+t*B, b = u*A+v*B
  while not iszero(b) do
    assert(s*A+t*B = a);
    assert(u*A+v*B = b);
    if degree(a) < degree(b) or iszero(a) then
      q:= poly(0, op(A, 2..3)), a
    elif (gcb:= igcd(lcoeff(b), n)) = 1 then
      userinfo(10, "Applying Euclidean step");
      q:=divide(a,b)
    else
      userinfo(10, "Multiplying by a suitable zero-divisor");
      m:= ilcm(gcb, lcoeff(a))/lcoeff(a);
      // take a multiple of a such that the lcoeff(a) = 0 or = U*lcoeff(B), U unit
      lc:= m*lcoeff(a);
      a:= multcoeffs(a, m);
      s:= multcoeffs(s, m);
      t:= multcoeffs(t, m);
      if iszero(lc mod n) then
        // start over, loop invariant holds
        assert(s*A+t*B = a);
        assert(u*A+v*B = b);
        next
      end_if;
      assert(modp(lcoeff(a) - lc, n) = 0);
      // n, lcoeff(b) and lc are multiples of gcb, thus lc = lcoeff(b)*X is solvable mod n:
      // divide off gcb to obtain lc/gcb = lcoeff(b)/gcb * X (mod n/gcb)
      // since lcoeff(b)/gcb and n/gcb are relatively prime, we can use arithmetic mod n/gcb
      q:= poly(modp((lc/gcb) / (lcoeff(b)/gcb), (n/gcb)) * x^(degree(a) -degree(b)), [x], IntMod(n));
      q:= q, a-b*q
    end_if;  
      
   
    userinfo(10, expr2text(a)." = ".expr2text(b)."*".expr2text(q[1]).
             " + ".expr2text(q[2]));
    a:=b;
    b:= q[2];
    q:= q[1];
    tmp:=s; s:=u; u:=tmp-q*u;
    tmp:=t; t:=v; v:=tmp-q*v;
    // loop invariant holds again
    assert(s*A+t*B = a);
    assert(u*A+v*B = b);
    userinfo(10, expr2text(a)." = ".expr2text(s). "*". expr2text(A).
             " + ".expr2text(t)."*".expr2text(B))
  end_while;
  tmp:= 1/(lcoeff(a)/igcd(lcoeff(a), n)) mod n;
  multcoeffs(a,tmp), multcoeffs(s,tmp), multcoeffs(t,tmp);
  
end_proc:

// end of file