

/*
  gcdlib::gcdexModularRecursive(f, g, x, inds)

  f, g - polynomials in two variables x, y, with rational coefficients
     x - variable 
  inds - list of variables [x, y]

  returns h, s, t, d such that
  h = gcd(f, g) (not necessarily monic!)
  1/d * (s*f + t*g) = h

  the return values are polynomials in *one* variable x; 
  the variable y may appear in the denominator of the coefficients there

*/

gcdlib::gcdexModularRecursive:=
proc(f, g, x, inds)
  local h, i, y, ff, gg, ss, tt, s, t, d, DD, DDyi, resu, resuyi, nony;
begin
  // currently, this algorithm is only used in the case of two variables:
  assert(nops(inds) = 2);
  assert(op(f, 2) = inds);
  assert(op(g, 2) = inds);

  h:= gcd(f, g);
  f:= divide(f, h, Quo); // really an exact division
  g:= divide(g, h, Quo); // really an exact division
 
  y:= op(inds, 2);
  nony:= [op(inds, 1)];
 
  /* for more than two variables, this would become
    y:= op(inds, nops(inds));
    nony:= [op(inds, 1..nops(inds) - 1)];  
  */  
  
  // precompute the resultant to avoid Cauchy interpolation
  resu:= poly(polylib::resultant(f, g, x), inds);

  DD:= poly(1, inds); // product of all y-i so far
  i:= 0;
  s:= t:= poly(0, inds);
  repeat
    i:= i+1;
    // solve ss*f + tt*g = 1 modulo (y-i)
    ff:= evalp(f, y=i);
    gg:= evalp(g, y=i);
    [d, ss, tt]:= [gcdex(ff, gg, x)];
    // for three and more variables, this would be
    //  [d, ss, tt, denominator]:= [gcdlib::gcdexModularRecursive(ff, gg, x, nony)]
    if degree(d) > 0 then next; end_if;
    // given s*f + t*g = 1 (modulo yp)
    //      ss*f + tt*g = 1 (modulo y-i)
    // instead, solve ss*f + tt*g = resultant
    resuyi:= evalp(resu, y=i);
    ss:= poly(ss, nony) * resuyi; // TODO evalp only once
    tt:= poly(tt, nony) * resuyi;
    // if we ever use this for more than two variables, we should do
    /*    if nops(inds) > 2 then
            ss:= divide(ss, denominator, Exact);
            tt:= divide(tt, denominator, Exact);
          end_if;
     */
    assert(ss <> FAIL and tt <> FAIL);
    // find sn \equiv s (mod yp) and sn \equiv ss (mod y-i)
    DDyi:= op(evalp(DD, y = i), 1);
    s:= poly(ss - evalp(s, y = i), inds)/ DDyi * DD + s;
    t:= poly(tt - evalp(t, y = i), inds)/ DDyi * DD + t;
    // reconstruct s, t
    DD:= DD* poly(y-i, inds); 
  until degree((d:= mapcoeffs(s*f + t*g, expand, ArithmeticOnly)), x) = 0 and not iszero(d) end_repeat;
  // now s*f + t*g = d
  // for the original f and the original g (=f*h, g*h), this means 
  // s*f*h + t*g*h = d*h
  // => s/d * f*h + t/d * g*h  = h
  
  h:= poly(h, [x]);
  h,  poly(s, [x]), poly(t, [x]), d
end_proc:

/*
  gcdlib::gcdexModular(f, g, x, inds)

  f, g - polynomials in two variables x, y, with rational coefficients
     x - variable 
  inds - list of variables [x, y]

  returns h, s, t such that h=gcd(f, g) and s*f + t*g = h

  the return values are univariate polynomials in x

*/

gcdlib::gcdexModular:=
proc()
  local result, lc, d;
begin
  result:= gcdlib::gcdexModularRecursive(args());
  assert(nops(result) = 4);
  lc:= lcoeff(op(result, 1));
  d:= expr(op(result, 4));
  op(result, 1)/lc, normal(op(result, 2)/lc/d, Expand = FALSE), normal(op(result, 3)/lc/d, Expand = FALSE)
end_proc:  
  