

/*--
gcdlib::heu_gcd_QI -- compute heuristic gcd of 2 polynomials over the Gaussian integers

gcdlib::heu_gcd_QI(p1, p2)

p1,p2 - primitive non-zero polynomials over the Gaussian integers

gcdlib::heu_gcd computes the gcd of two non-zero primitive polynomials over Expr whose coefficients are complex numbers with integer real and imaginary part.
The procedure returns the gcd or FAIL resp. FALSE if the heuristic fails. The output must be made primitive!

See K.O.Geddes et al "Alg. for Computer Algebra", Kluwer 1992, pp320
This algorithm has also been analyzed  in Liao/Fateman, "Evaluation of the Heuristic Polynomial GCD"
See also Bernard Parisse, A correct proof of the heuristic gcd algorithm, Theorem 1

--*/

// magic numbers
// bound on the total degree of multivariate polynomials handled 
alias(BOUND = 20000):


/*
   experiments indicate that in our current test base, the first try succeeds in most cases; the second try succeeds in most of the remaining cases; and so on.
   Since mostly small univariate examples survive many tries and such input can be handled slightly more quickly by other algorithms, 
   NUMBEROFTRIES = 4 might mean a neglegible improvement or maybe not.
*/
alias(NUMBEROFTRIES = 6):


/*
   The following multiplier is suggested in most papers. It may be varied, but should be neither too small (too many tries needed) 
   nor too large (algorithm fails due to coefficient size soon). In addition, the continued fraction expansions of its positive integer powers should have small coefficients.

*/
alias(MULTIPLIER = 73794 / 27011):


gcdlib::heu_gcd_QI:=
proc(a, b)
  local x, xv, u, g, tries, ca, cb, aa, bb, bound, genpolyQI;
begin
  // local method genpolyQI
  // does the same as genpoly, but accepts Gaussian integer arguments
  genpolyQI:= 
  proc(z, xv, x)
  begin
    genpoly(Re(z), xv, x) + I*genpoly(Im(z), xv, x)
  end_proc;
  
  x:= op(a,[2,1]);
  
  // make polynomials primitive
  ca:= gcd(coeff(a));
  cb:= gcd(coeff(b));
  a:= multcoeffs(a, 1/ca);
  b:= multcoeffs(b, 1/cb);

  
// xv is large enough only if a and b are really primitive polynomials!
  xv:= ceil(2*min(norm(a), norm(b)) + 2);
  
  for tries from 1 to NUMBEROFTRIES do
    if nops(op(a,2)) = 1 then
      // gcd over the complex numbers
      g:= igcd(evalp(a,x=xv), evalp(b,x=xv))
    else
      // we try to estimate how large the Gaussian integers will be in the end. 
      // If a has degree N w.r.t. x, then plugging in x=xv will give a polynomial of norm up to xv*(xv^N) [= norm(a) * (2*norm(a))^N]
      // plugging two times this into a second variable - if M is the degree w.r.t. that variable -- will give norm xv*(2*xv^N)^M etc. 
      // in the end, we arrive at something like norm(a)^(2^k * \prod_i degree(a, x_i) ) where k is the number of variables and x_1, .., x_k are the variables
      // on the other hand, the running time of a modular algorithm would depend on norm(a), too. So we take logarithms and divide by norm(a), to get a crude
      // estimate of the additional running time factor involved
      bound:= max(_mult((2*degree(a, u)) $u in op(a, 2)), _mult((2*degree(b, u)) $u in op(b, 2)));
      if bound > BOUND then 
        return(FALSE)
      end_if;
      aa:= evalp(a, x=xv);
      if iszero(aa) then 
        if divide(a, b, Exact) <> FAIL then 
          return(b) 
        else
          // use another xv
          xv:= trunc(xv * MULTIPLIER);
          next
        end_if   
      end_if;   
      bb:= evalp(b, x=xv);
      if iszero(bb) then 
        if divide(b, a, Exact) <> FAIL then 
           return(a) 
         else 
           // use another xv
           xv:= trunc(xv * MULTIPLIER);
           next
         end_if;
      end_if;   
      // use heu_gcd_QI recursively
      g:= gcdlib::heu_gcd_QI(aa, bb);
      if g = FALSE then 
        return(FALSE) 
      end_if  
    end_if;
    
    if g <> FAIL then
      g:= genpolyQI(g, xv, x);
      g:= multcoeffs(g, 1/gcd(coeff(g)));
      if divide(a, g, hold(Exact)) <> FAIL then
        if divide(b, g, hold(Exact)) <> FAIL then
          return(g * gcd(ca, cb))
        end_if
      end_if
    end_if;
    /* some authors also suggest to do 
     xv:= trunc(xv^(5/4) * MULTIPLIER); 
     in this moment */
    xv:= trunc(xv * MULTIPLIER);
  end_for;
  FAIL
end_proc:

// end of file 
