//    

// kg, 09/12/93 

/*++
gcdlib::int_mod_gcd -- compute gcd of integer polynomials using modular
	algorithm

gcdlib::int_mod_gcd(a, b)

a,b - non-zero primitive polynomials of type 'Expr' over the integers

gcdlib::int_mod_gcd computes the gcd of 2 non-zero polynomials over the
integers via a modular gcd algorithm. The arguments are not checked.
The result is not made primitive.

See K.O.Geddes et al "Alg. for Computer Algebra", Kluwer 1992, pp307
++*/

gcdlib::int_mod_gcd:= proc(a, b)
    local x, g, n, m, lim, p, q, ap, bp, cp, ce, h, oldh;
begin
    // get last indet 
    x:= op(a,[2, nops(op(a,2))]);

    // get coefficient bound for gcd 
    g:= igcd(lcoeff(a), lcoeff(b));
    n:= min(degree(a,x), degree(b,x));
    lim:= (2^(n+1))* abs(g) * min(norm(a), norm(b));

    // CRA interpolation loop 
    // old values (1993):  1000003, 4294967029, 4611686018427380059, 21267647932558653966460912964485510037

    // the following code would be as good as the currently used one
    /*  
      p:= 1073741827; // nextprime(2^30)
       if p < lim then
          p:= 4611686018427388039; // nextprime(2^62)
       end_if;
    */
    
    // some experiments showed that nextprime(2^62) is a good starting point  
    p:= 4611686018427388039; 
    
    q:= 1;
    h:= poly(0, op(a,2..3));

    while TRUE do
	// get modular gcd 
	ap:= poly(a, IntMod(p));
	bp:= poly(b, IntMod(p));
	cp:= gcdlib::mod_gcd(ap, bp);
	m:= degree(cp,x);

	// normalize such that g mod p = lcoeff(cp) 
	cp:= multcoeffs(cp, (g mod p)/lcoeff(cp));

	// test for unlucky homomorphism 
	if m < n then
	    q:= p;
	    h:= poly(cp, Expr);
	    n:= m;
	    oldh:=FAIL
	elif m = n then

	    // update coeffs using integer chinese reminder 
	    if q = 1 then
		h:= poly(cp, Expr);
		oldh:=FAIL;
	    else
		ce:= poly(cp, Expr);
		/* combine h and cp:
		  compute integer chinese reminder of hi mod q and ci mod p
		  for the coeffs hi of h and ci of cp. The individual coeff
		  of the result is given by:
			p * mods((hi-ci)/p, q) + ci
		*/
		oldh:=h;
		h:= multcoeffs(mapcoeffs(multcoeffs(h - ce, 1/p), mods, q), p)
			+ ce;
	    end_if;
	    q:= p * q;
	end_if;

	// test for completion 
	/* if oldh and h are equal, oldh differed from the correct
	  result by a multiple of p (which is unlikely if it is not
	  the correct result itself) */

	if q > lim or oldh=h then

	    // test division 
	    if divide(a, h, Exact) <> FAIL then
	    	if divide(b, h, Exact) <> FAIL then
		    return(h)
		end_if
	    end_if

	elif m = 0 then
	    return(poly(1, op(a,[2..3])))
	end_if;

	// get next modulus 
	repeat
	    p:= nextprime(p+2);
	until domtype(g/p) <> DOM_INT end_repeat;
    end_while
end_proc:

// end of file 
