//    

// kg, 09/12/93 

/*--
gcdlib::mod_sparse_gcd -- compute the gcd of IntMod-polynomials using
	sparse Zippel interpolation

gcdlib::mod_sparse_gcd(p1, p2)

p1,p2 - non-zero polynomials over IntMod

gcdlib::mod_sparse_gcd computes the gcd of 2 non-zero polynomials
over IntMod(p). p must be prime. The arguments are not checked further.
Sparse Zippel interpolation is used to calculate the gcd.
gcdlib::mod_sparse_gcd returns FAIL if the starting point is bad.

See R.Zippel "Probabilistic Algorithms for Sparse Polynomials",
in: Symbolic & Algebraic Comp. (Ed E.W.Ng), Springer 1979, pp216

The main source of the algorithm is
R. Zippel, "Interpolating Polynomials from their Values",
J. Symbolic Computation (1990) 9, pp. 375-403

--*/



gcdlib::mod_sparse_gcd:= proc(aa, bb, seed)
  local a, b, p, T, Tg, Th, y, x, X, nv, av, bv, av0, bv0, cc, ch,
  gl, glv, glv0, d, gv, gc, gcn, h, ph, q, lim, v0, v, V,
  tn, t, sv, svn, i, j, k, S, Sn, rand, vanderm;
  save SEED;
begin

  ch := UNKNOWN;
  a:=aa;
  b:=bb;
  X:= op(a,2);
  nv:= nops(X)-1;
  T:= X, op(a,3);    // X = alle Variablen
  p:= op(T,[2,1]);
  y:= X[nv+1];       // y = Hauptvariable
  delete X[nv+1];
  rand:= random(p);
  SEED:= seed;
  
    /* compute contents of a and b, viewed as polynomials in y over
  Zp[x1,...,xn] */
  Th:= [y], polylib::Poly(X, IntMod(p)); // univ. poly ber Hauptvariable, 
                                         // mit Koeffs in den anderen Variablen
  av:= poly(a, Th);
  bv:= poly(b, Th);
  gl:= poly(1, X, IntMod(p));
  
  if lcoeff(av) = gl or lcoeff(bv) = gl then
    cc:= poly(1, T);
  else
    av0:= gcd(coeff(av));                // poly in den verbleibenden Vars
    bv0:= gcd(coeff(bv));                // poly in den verbleibenden Vars
    cc:= poly(gcd(av0,bv0), T);          // gcd of all coeffs = content bzgl. Hauptvariable y
    
    // remove contents from a and b 
    av:= mapcoeffs(av, divide, av0, Exact);   // av = prim. univ poly in y
    bv:= mapcoeffs(bv, divide, bv0, Exact);   // bv = prim. univ poly in y
    gl:= gcd(lcoeff(av), lcoeff(bv));
    
    // view a and b again as polynomials in X over Zp 
    a:= poly(av, T);
    b:= poly(bv, T);
  end_if;
  
  // pick random nv-Tupel v0 
  repeat
    v0:= [ (X[i]=rand()) $ i=1..nv ];      // random values for the 'other variables' (<> y)
    glv:= evalp(gl, op(v0));              
  until glv <> 0 end_repeat;
  
  // init dense interpolation for x1 
  x:= X[1];
  if nv = 1 then
    Th:= T;
    av0:= a;
    bv0:= b;
    glv0:= gl;
  else
    Th:= [x,y], IntMod(p);
    V:= op(v0,2..nv);
    av0:= evalp(a, V);    // poly in [x, y]
    bv0:= evalp(b, V);    // poly in [x, y]
    glv0:= evalp(gl, V);  // poly in [x]
  end_if;
  
  v:= op(v0[1],2);     // random values for x
  V:= { v };
  h:= gcdlib::univ_mod_gcd(evalp(av0, v0[1]), evalp(bv0, v0[1]));  // univ gcd of polys in [y]
  if not iszero(h) then
    h:= poly(multcoeffs(h, glv / lcoeff(h)), Th)    // ansatz for the wanted gcd = poly(..., [x, y])
  end_if;
  q:= poly(x - v, Th);    // poly(x - value, [x, y])
  
  lim:= min(degree(a,x), degree(b,x));    // max degree in [x] for the wanted gcd
  
  // dense interpolation loop for x1 
  while lim > 0 do
    
    // pick random value for x1 
    repeat
      v:= x=rand();
      glv:= evalp(glv0, v);
    until not contains(V, op(v,2)) and glv <> 0 end_repeat;
    
    gv:= gcdlib::univ_mod_gcd(evalp(av0, v), evalp(bv0, v));  // poly(..[y])
    if not iszero(gv) then
      gv:= multcoeffs(gv, glv / lcoeff(gv))
    end_if;
    
    // dense polynomial Newton interpolation 
    lim:= lim-1;
    
    // compute Newton coefficient 
    gv:= gv - evalp(h, v);
    v:= op(v,2);
    _mult(op(map(V, -_plus, -v)));
    gv:= multcoeffs(gv, 1/%);

    // update polynomial 
    if not iszero(gv) then
      h:= h + poly(gv, Th) * q;
    end_if;
    
    if lim = 0 then break end_if;
    
    if iszero(gv) then
      
      // get primitive part of h 
      ph:= poly(h, [y], polylib::Poly([x], IntMod(p)));
      ph:= mapcoeffs(ph, divide, gcd(coeff(ph)), Exact);
      ph:= poly(ph, Th);
      
      // test division 
      if divide(av0, ph, Exact) <> FAIL then
        if divide(bv0, ph, Exact) <> FAIL then
          if nv = 1 then return(cc * ph) end_if;
          break
        end_if
      end_if;
    end_if;
    
    q:= poly(x-v, Th) * q;
    V:= V union { v };
  end_while;
  
  // loop over x2,...,xn 
  for k from 1 to nv-1 do
    
    // built skeleton from h 
    d:= degree(h,y);
    S:= [ [] $ i=0..d ];
    for j from 1 to nterms(h) do
      t:= nthterm(h,j);
      d:= degree(t,y)+1;
      S[d]:= append(S[d], t)
    end_for;
    t:= max(nops(S[i]) $ i=1..nops(S));
    
    // init dense interpolation for x(k+1) 
    Tg:= Th;
    x:= X[k+1];
    if nv = k+1 then
      Th:= T;
      av0:= a;
      bv0:= b;
      glv0:= gl;
    else
      Th:= [op(X,1..k+1),y], IntMod(p);
      V:= op(v0,k+2..nv);
      av0:= evalp(a, V);
      bv0:= evalp(b, V);
      glv0:= evalp(gl, V);
    end_if;
    
    v:= op(v0[k+1],2);
    V:= { v };
    h:= poly(h, Th);
    q:= poly(x - v, Th);
    
    lim:= min(degree(a,x), degree(b,x));
    
    // dense interpolation loop for x(k+1) 
    while lim > 0 do
      // pick random evaluation point for dense interpolation 
      repeat
        v:= x=rand();
        glv:= evalp(glv0, v);
      until not contains(V, op(v,2)) and evalp(glv, op(v0,1..k)) <> 0
      end_repeat;
      
      av:= evalp(av0, v);
      bv:= evalp(bv0, v);

      /* repeat sparse interpolation step until sparse evaluation
      points allow a solution */
      repeat
        
        // pick random sparse evaluation point (k-tupel) 
        repeat
          sv:= [ rand() $ i=1..k ];
        until evalp(glv, (X[i]=sv[i]) $ i=1..k) <> 0 end_repeat;
        
        // get corresponding gcd's at sv^i for i=0..t-1 
        gc:= [];
     // ==============================================================
     // Walter 20.11.09: replaced the loop
     //     'for j from 0 to t-1 do'
     //  by 'for j from 1 to t do'
     // The original version implied (for j = 0) that X[i] = 1
     // for all i = 1..k. There are examples where, at the point
     // X[1] = X[2] = .. = 1, the univariate gcd of evalp(av, svn)
     // and evalp(bv, svn) is a bigger polynomial than at other
     // points. To cure this problem, we build a Vandermonde-type
     // system of equations (for Ch[1], Ch[2], ..) using 
     //  Ch[1]*sv[1]  + Ch[2]*sv[2]  +..+ Ch[k]*sv[k]   = gcn[1]
     //  Ch[1]*sv[1]^2+ Ch[2]*sv[2]^2+..+ Ch[k]*sv[k]^2 = gcn[2]
     //            ...
     //  Ch[1]*sv[1]^k+ Ch[2]*sv[2]^k+..+ Ch[k]*sv[k]^k = gcn[k]
     // instead of the proper (transposed) Vandermonde system
     //  ch[1]*sv[1]^0    + ch[2]*sv[2]^0    +..+ ch[k]*sv[k]^0     = gcn[1]
     //  ch[1]*sv[1]^1    + ch[2]*sv[2]^1    +..+ ch[k]*sv[k]^1     = gcn[2]
     //            ...
     //  ch[1]*sv[1]^(k-1)+ ch[2]*sv[2]^(k-1)+..+ ch[k]*sv[k]^(k-1) = gcn[k]
     // This ensures that only random points are inserted for the
     // variables X[i], thus reducing the risk of hitting an exceptional
     // point with a larger gcd.
     // Further down, we use gcdlib::vanderm_mod_solve (that solves for
     // Ch[i]) and obtain the wanted solution Ch[i] by Ch[i]=ch[i]/sv[i].
     // ==============================================================
       
        for j from 1 to t do
          svn:= (X[i]=stdlib::powermod(sv[i],j,p)) $ i=1..k;
          gcn:= gcdlib::univ_mod_gcd(evalp(av, svn), evalp(bv, svn));
          if not iszero(gcn) then
            gcn:= multcoeffs(gcn, evalp(glv, svn)/lcoeff(gcn))
          end_if;
          gc:= append(gc, gcn)
        end_for;
        
	/* solve lin. equ. for each coeff of h (viewed as polynomial in x1) */
        gv:= poly(0, Tg);
        svn:= ((X[i]=sv[i]) $ i=1..k), y=1;
        for j from 1 to nops(S) do
          
          // assemble and solve transpose Vandermonde system 
          Sn:= S[j];
          tn:= nops(Sn);
          vanderm:= [ evalp(Sn[i], svn) $ i=1..tn ];
       // Walter 20.11.09: added the following if select...
          if select(vanderm, iszero) <> [] then
             break;
          end_if;
          if tn = 1 then
            // Walter 20.11.09: replaced
            // gv:= gv + multcoeffs(Sn[1], coeff(gc[1], j-1));
            gv:= gv + multcoeffs(Sn[1], coeff(gc[1], j-1)/vanderm[1]);
          elif tn > 1 then
            ch:= gcdlib::vanderm_mod_solve(
                vanderm, // [ evalp(Sn[i], svn) $ i=1..tn ],
                [ coeff(gc[i], j-1) $ i=1..tn ], p);
            if ch = FAIL then break end_if;

            // the solution are the coeffs of the j-th coeff of gv 
         // Walter 20.11.09: replaced
         // gv:= gv + _plus(multcoeffs(Sn[i], ch[i]) $ i=1..tn);
            gv:= gv + _plus(multcoeffs(Sn[i], ch[i]/vanderm[i]) $ i=1..tn);
          end_if
        end_for;
      until ch <> FAIL end_repeat;
      
      // dense polynomial Newton interpolation 
      lim:= lim-1;
      
      // compute Newton coefficient 
      gv:= gv - evalp(h, v);
      v:= op(v,2);
      _mult(op(map(V, -_plus, -v)));
      gv:= multcoeffs(gv, 1/%);
      
      // update polynomial 
      if not iszero(gv) then
        h:= h + poly(gv, Th) * q;
      end_if;
      
      if lim = 0 then break end_if;
      
      if iszero(gv) then
        
        // get primitive part of h 
        ph:= poly(h, [y], polylib::Poly([op(X,1..k+1)], IntMod(p)));
        ph:= mapcoeffs(ph, divide, gcd(coeff(ph)), Exact);
        ph:= poly(ph, Th);
        
        // test division 
        if divide(av0, ph, Exact) <> FAIL then
          if divide(bv0, ph, Exact) <> FAIL then
            if nv = k+1 then return(cc * ph) end_if;
            break
          end_if
        end_if;
      end_if;
      
      q:= poly(x-v, Th) * q;
      V:= V union { v };
    end_while;
  end_for;
  
  // get primitive part of h 
  ph:= poly(h, [y], polylib::Poly([op(X,1..nv)], IntMod(p)));
  ph:= mapcoeffs(ph, divide, gcd(coeff(ph)), Exact);
  ph:= poly(ph, Th);
  
  // test division 
  if divide(a, ph, Exact) <> FAIL then
    if divide(b, ph, Exact) <> FAIL then
      return(cc * ph)
    end_if
  end_if;
  userinfo(1, "The sparse modular gcd fails");
  FAIL
end_proc:

// end of file 
