//    

// hornp, 11.07.2002
// kg, 15/03/95 

/*++
polylib::randpoly -- create random polynomial

randpoly([vars] [, dom] [, option=value...] [, Monic])

vars   - list of variables
dom    - coefficient domain (Expr, IntMod(n) or domain)
option - one of 'Coeffs', 'Terms', 'Degree', 'Monic'
value  - expression

randpoly creates a random polynomial with variables vars and coefficient
domain dom. The polynomial is created by adding a given number of terms,
where the degrees of the variables are in a given range.

- The option Terms=n with a positive integer n gives the number of terms
  (default is 6). The random generator may deliver a coefficient of 0, 
  so the number of termsof the output may be smaller than n. 
  If Terms=infinity, the output is a polynomial of the given or a smaller 
  degree, with the coefficients being chosen independently.
  If the number of terms is bigger than possible given by the degree and the
  number of indeterminates, a warning is raised and a dense polynomial is 
  returned; the warning is not raised if that number of terms was not explicitely given.

- The option Degree=n with a nonnegative integer n gives the maximal
  degree of each variable (default is 5).

- The option Coeffs=f with a function f returning a random coefficient is
  used to create the coefficients.
  
If the option Coeffs=f is missing and dom is Expr the coefficients of the
single terms will be integers in the range -999..999; if dom is a
user-defined domain it must have a method "random" to create the
coefficients.

If the option Monic is specified, the leading coefficient of the result is 1.

If the argument dom is missing Expr is used as default. If the argument
vars is missing [x] is used as default.
++*/

polylib::randpoly:= proc()
    local cgen, nt, dg, i, xx, opts, v, d,
        monic, l, t1, nv, DG, NT, WarnTerms;
begin
    userinfo(10, "polylib::randpoly called");
    opts:= [ args() ];
    
    // get vars and coeffs domain 
    v:= [hold(x)];
    d:= hold(Expr);
    if nops(opts) <> 0 then
	    if domtype(opts[1]) = DOM_LIST then
    	    v:= opts[1];
    	    delete opts[1];
    	end_if
    end_if;
    userinfo(20, "Variables are ".expr2text(v));
    if nops(opts) <> 0 then
    	if type(opts[1]) <> "_equal" and opts[1]<>Monic then
    	    d:= opts[1];
    	    delete opts[1];
    	end_if
    end_if;
    userinfo(20, "Coefficient ring is ".expr2text(d));

    // get options 
    WarnTerms:= FALSE;
    nt:= 6;
    dg:= 5;
    cgen:= FAIL;
    monic := FALSE;
    for i in opts do
        if (i) = hold(Monic) then
            monic := TRUE;
            next;
        end_if;
        if type(i) <> "_equal" then error("unknown option") end_if;
        case op(i,1)
        of hold(Terms) do
            nt:= op(i,2);
            WarnTerms:= TRUE;
            break;
        of hold(Degree) do
            dg:= op(i,2); break;
        of hold(Coeffs) do
            cgen:= op(i,2); break;
        otherwise error("unknown option");
        end_case;
    end_for;
    
    // test args 
    if testargs() then
        case domtype(d)
        of DOM_IDENT do
            if d <> hold(Expr) then
              error("illegal coefficient ring")
            end_if;
            break;
        of DOM_EXPR do
            if op(d,0) <> hold(IntMod) then
              error("illegal coefficient ring")
            end_if;
            if nops(d) <> 1 then
              error("illegal coefficient ring")
            end_if;
            if not testtype(op(d,1), Type::PosInt) then
                error("illegal modulus")
            end_if;
            break;
        of DOM_DOMAIN do
            break;
        otherwise
            error("illegal coefficient ring");
        end_case;
        if not testtype(nt, Type::PosInt) then
    	    if nt <> infinity then 
            error("illegal number of terms");
          end_if;
        end_if;
        if not testtype(dg, Type::NonNegInt) then 
          error("illegal degree")
        end_if;
    end_if;
    
    // random generator for coeffs 
    case domtype(d)
    of DOM_IDENT do
	    if cgen = FAIL then cgen:= random(-999..999) end_if;
	    break;
    of DOM_EXPR do
	    if cgen = FAIL then cgen:= random(op(d,1)) end_if;
	    break;
    otherwise
	    if cgen = FAIL then 
    	    cgen:= d::random;
    	    if cgen = FAIL then 
    		error("domain entry \"random\" missing") 
    	    end_if;
    	end_if;
    end_case;
   
    if nt <> infinity and nt > (dg+1)^nops(v) then
      if WarnTerms then
        warning("More terms requested than possible with requested degree,\n".
                "creating dense polynomial.")
      end_if;
      nt := infinity;
    end_if;

   
    if nt <> infinity then
        // random generator for exponents
   	    if monic then
            NT := nt-1;
            DG := (dg+1)^nops(v)-1;
   	    else
            NT := nt;
            DG := (dg+1)^nops(v);
        end_if;
   	        
   	    // create terms 
   	    // The idea of the algoritm is: 
   	    // 1. Start with the zero polynomial
   	    // 2. For each term requested, randomly 
   	    //    select a monomial not used yet
   	    // 3. Create a coefficient at random for this monomial
   	    //
   	    // - For multivariate polynomials, we use a (dg+1)-adic
   	    //   representation for the exponents.
   	    l := table();
        for i from 0 to NT-1 do
          // guess an exponent
          t1 := random(i-1..DG-1)();
          if t1 < NT or contains(l, t1) then
            t1 := i;
          end_if;
          l[t1] := cgen();
        end_for;
        // convert l to a list suitable for poly:
        nv := nops(v);
        if nv = 1 then
          l := map([op(l)], eq -> [op(eq, 2), [op(eq, 1)]]);
        else
          l := map([op(l)], eq -> [op(eq, 2),
            (numlib::g_adic(op(eq, 1), dg+1).[0 $ nv])[1..nv]]);
        end_if;
    else 
        // create dense polynomial 
        // We at first create a list of all possible exponents
        // and then add random coefficients.
        l := combinat::cartesianProduct([$0..dg] $ xx in v);
   	    l := map(l, o -> [cgen(), o]);
    end_if; 

    // set the lcoeff to 1
    if monic then
        l := select(l, o->bool(op(o,2) <> [dg $ xx in v]));
        l := l.[[1, [dg $ xx in v]]];
    end_if;
    return(poly(l, v, d));
end_proc:

// end of file 
