/*++
polylib::resultant -- compute the resultant of two polynomials

resultant(p, q [, X] [, x])

p, q - polynomials or expressions
X    - list of indeterminates if p,q are expressions
x    - indeterminate

The resultant is computed via the subresultant algorithm.
++*/

polylib::resultant:=
proc(p, q)
  local x, X, T, i, coeff_division, r;
begin
  if args(0) < 2 then
    error("wrong no of args")
  end_if;

  // allow overloading
  if p::dom::resultant <> FAIL then
    return( p::dom::resultant(args()) )
  end_if;
  if q::dom::resultant <> FAIL then
    return( q::dom::resultant(args()) )
  end_if;

/* SUBROUTINE */
/*--
coeff_division -- return coefficient division function

coeff_division(l)

l - non-empty list of polynomials of same type

Returns the 'best' coefficient division function for the polynomials
in l.
--*/

  coeff_division:=
  proc(x)
    local d;
  begin
    d:= op(x[1], 3);
    if domtype(d) = DOM_DOMAIN then
      d:= slot(d, "_divide");
      if d = FAIL then
        error("method '_divide' missing")
      else
        d
      end_if
    elif d = hold(Expr) then
      if gcdlib::coeff_type(x) = DOM_EXPR then      
        // test for polynomial coeffs
        d:= poly(_plus(op(map(x, coeff))));
        if d <> FAIL then
          subs(
            proc(a, b)
              name polylib::coeff_division;
            begin
              // the remainder is mathematically zero, but maybe not syntactically. 
              // Thus divide(..., Exact) might give FAIL
              divide(a, b, #d, Quo)
            end_proc,
            #d = op(d, 2))
        else
          proc(a, b)
            name polylib::coeff_division;
          begin
            normal(a / b)
          end_proc
        end_if   
      else
        // numerical coeffs only
        _divide
      end_if;
    else
      _divide
    end_if
  end_proc :

/* END OF SUBROUTINE */

  
  
  
// args = polynomials?
  if domtype(p) = DOM_POLY then
    T:= op(p,3); X:= op(p,2);
    if domtype(q) <> DOM_POLY then
      error("no polynomial")
    end_if;
    if T <> op(q,3) or X <> op(q,2) then
      error("polynomial types differ")
    end_if;
    if args(0) = 3 then
      x:= args(3);
      i:= contains(X, x);
      if i = 0 then
        error("unknown indet")
      end_if;
    elif args(0) > 3 then
      error("wrong no of args")
    else
      i:= 1;
      x:= X[1];
    end_if;
    if nops(X) = 1 then
      r:=polylib::presultant(p, q, x, coeff_division([p, q]));
      delete coeff_division;
      return(r)
    end_if;
    r:= polylib::presultant(p, q, x, coeff_division([p, q]));
    delete coeff_division;
    return(r);
  elif domtype(q) = DOM_POLY then
    error("Either both arguments must be polynomials or both expressions")
  end_if;

  if testargs() then
    if not testtype(p, Type::Arithmetical) or
      not testtype(q, Type::Arithmetical) then
      error("Arguments must be polynomials or arithmetical expressions")
    end_if
  end_if;
  
  // parse args for expressions
  case args(0)
    of 2 do
      X:= indets([p, q], RatExpr);
      if X = {} then X:= [#x] else X:= [op(X)] end_if;
      x:= X[1];
      break;
      
    of 3 do
      if domtype(args(3)) = DOM_LIST then
        X:= args(3);
        x:= X[1];
      else
        x:= args(3);
        X:= [op(indets([p, q], RatExpr) union {x})];
      end_if;
      break;

    of 4 do
      X:= args(3);
      x:= args(4);
      if domtype(X) <> DOM_LIST then 
        error("Third argument must be a list of variables") 
      end_if;
      if contains(X, x) = 0 then
        error("Indeterminate is not in the list of variables")
      end_if;  
      break;

    otherwise
      error("wrong no of args");
  end_case;

    // special cases 
  case type(p)
    of "_mult" do
      return(polylib::resultant(op(p,1), q, X, x) *
             polylib::resultant(subsop(p, 1=1), q, X, x))
    of "_power" do
      x:= polylib::resultant(op(p,1), q, X, x);
      return((if x = 0 then 0 else x^op(p,2) end_if))
  end_case;

  case type(q)
    of "_mult" do
      return(polylib::resultant(p, op(q,1), X, x) *
             polylib::resultant(p, subsop(q, 1=1), X, x))
    of "_power" do
      x:= polylib::resultant(p, op(q,1), X, x);
      return((if x = 0 then 0 else x^op(q,2) end_if))
  end_case;

  // convert to polynomials
  if nops(X) = 1 then
    p:= poly(p, [x]);
    q:= poly(q, [x]);
    if p = FAIL or q = FAIL then
      return(FAIL)
    end_if;
    r:=polylib::presultant(p, q, x, coeff_division([p, q]));
  else
    // we have extract the denominator here ?!
    p:= poly(p, X);
    q:= poly(q, X);
    if p = FAIL or q = FAIL then 
      return(FAIL) 
    end_if;
    r:= expr(polylib::presultant(p, q, x, coeff_division([p, q]) ))
  end_if;
  delete coeff_division;
  r;
end_proc:

// end of file 
