/* --------------------------------------------------------------- 
  polylib::shift -- computes a shifted polynomial
------------------------------------------------------------------ 
  
* Call: polylib::shift(f, n)

* Parameters: f -- a univariate polynomial of type DOM_POLY
              n -- an element of f's basefield
              
* Return value: f(x+n) as polynomial of the same kind as f.
  
* Example 1:
  >> f := poly( a0+a1*x+a2*x^2, [x])
  
                                  2
                         poly(a2 x  + a1 x + a0, [x])
  >> polylib::shift(f, 1)
  
                        2
               poly(a2 x  + (a1 + 2 a2) x + (a0 + a1 + a2), [x])
  >> polylib::shift(f, 0)
  
                                  2
                         poly(a2 x  + a1 x + a0, [x])                                                                                  

* Example 2:
  >> A := Dom::AlgebraicExtension(Dom::Rational, c^2+1, c)
  
                                                     2
             Dom::AlgebraicExtension(Dom::Rational, c  + 1 = 0, c)
  >> f := poly( 1+(c-1)*x+3/c*x^2, [x], A)
  
               2
  poly((-3 c) x  + (c - 1) x + 1, [x], Dom::AlgebraicExtension(
  
                     2
     Dom::Rational, c  + 1 = 0, c))
  >> polylib::shift(f, A(c))
  
               2
  poly((-3 c) x  + (c + 5) x + 2 c, [x],
  
                                             2
     Dom::AlgebraicExtension(Dom::Rational, c  + 1 = 0, c))
  >> polylib::shift(f, -3)  
  
               2
  poly((-3 c) x  + (19 c - 1) x - (30 c - 4), [x],
  
                                             2
     Dom::AlgebraicExtension(Dom::Rational, c  + 1 = 0, c))                                                                              

* Implementation Details:
  polylib::shift uses the formula
  f(x+n) = \sum_{j=0}^N x^j ( \sum_{i=j}^N a_i \binom ij n^{i-j}  )
  where f = \sum_{j=0}^N a_j\,x^j
  for efficiency-reasons a 'local', optimized function is used
  to compute the binomials.
--------------------------------------------------------------- */

polylib::shift := proc(f : DOM_POLY, n)
local l, X, F, mb, N, i, j;
begin
  userinfo(9,"polylib::shift called.");
  // check the parameters
  X := op(f,2);
  if nops(X) <> 1 then 
    return(FAIL);
  end_if;
  F := op(f,3);
  if F <> Expr and op(F,0) <> IntMod then
    if coerce(n, F) = FAIL or coerce(n+1, F)=FAIL then
      return(FAIL);
    end_if;
  end_if;
  // for speed, faster binomials
  mb := (n,k) -> _if(k>(n-k), 
    _mult(i $ i=k+1..n)/_mult(i$i=2..n-k), 
    _mult(i $ i=n-k+1..n)/_mult(i$i=2..k)):
  N := degree(f);
  l := [coeff(f, All)];
  poly([[_plus(l[i+1]*mb(i,j)*n^(i-j) $ i=j..N), j] $ j=0..N], X, F);
end_proc;
