/*++
polylib::sqrfree -- square-free factorization of polynomials

sqrfree(p)

p - polynomial or polynomial expression
Recollect - determines whether factors with the same power are collected

Returns the square-free factorization of a polynomial or expression
over the rationals.

If the polynomial is of type Expr ist coefficients must be rationals.
If the polynomial is of type IntMod(n) then n must be a prime number.
If the coefficients are from a domain that domain must be a UFD of 
characteristic 0 or a finite field.
++*/

polylib::sqrfree:= proc(p)
  local pp, pr, t, i, j, recollect, ret, options, unextend, ratsubsts, minpol, T, inds, Args;
begin

  pp := p; // to later read off its type

  // to support the old interface, where a Boolean parameter
  // was given as an optional second argument to set the "Recollect" parameter:
  Args := [args()];
  if nops(Args) = 2 and domtype(Args[2]) = DOM_BOOL then
    Args[2] := "Recollect" = Args[2];
  end_if;

  options := prog::getOptions(2, Args,
    table(
      "Recollect"             = TRUE,
      "UseAlgebraicExtension" = FALSE),
    TRUE,
    table(
      "Recollect"             = DOM_BOOL,
      "UseAlgebraicExtension" = DOM_BOOL))[1];

  // recollect - local method
  recollect:=
  if options["Recollect"] then
    proc(f)
      local t;
    begin
      // re-collect factors with equal powers
      t:= table();
      for i from 3 to nops(f) step 2 do
        if contains(t, f[i]) then
          t[f[i]]:= t[f[i]] * f[i-1]
        else
          t[f[i]]:= f[i-1]
        end_if
      end_for;
      [ f[1], (op(t, [j,2]), op(t, [j,1])) $ j=1..nops(t) ]
    end_proc;

  else
    // do not recollect
    id
  end_if;
  
  // if we should create algebraic extensions, we must go back at the end
  unextend := id;
  if options["UseAlgebraicExtension"] then
    p := poly(p);
    T := op(p, 3);
    if T = Expr then
      if map({coeff(p)}, domtype) minus {DOM_INT, DOM_RAT} = {} then
        T := Dom::Rational;
      else
        T := Dom::Fraction(Dom::Polynomial(Dom::Rational));
      end_if;
    end_if;
    inds := indets(poly2list(p));
    [pr, ratsubsts, minpol] := [rationalize(poly2list(p), FindRelations=["_power", "exp"], MinimalPolynomials)];
    if ratsubsts = {} or
        traperror((T := fp::fold((p, d) -> Dom::AlgebraicExtension(d, poly(p, [op(indets(p) minus inds)], d)),
          T)(op(minpol)))) <> 0 or
        traperror((p := poly(pr, op(p, 2), T))) <> 0 or p = FAIL then
      p := pp; // go back, don't try this
    else // conversion worked
      unextend :=
      if pp::dom = DOM_POLY then
        proc(f)
          local i;
        begin
          [if op(pp, 3) = Expr then
             expr(f[1])
           else
             coerce(f[1], op(pp, 3))
           end_if, (poly(f[i], op(pp, 3)), f[i+1]) $ i = 2..nops(f)-1 step 2] | ratsubsts
        end_proc;
      else
        proc(f)
          local i;
        begin
          [expr(f[1]), (expr(f[i]), f[i+1]) $ i = 2..nops(f)-1 step 2] | ratsubsts
        end_proc;
      end_if;
    end_if;
  end_if;

  // m a i n   p r o g r a m
  
  if p::dom::sqrfree <> FAIL then
    p:= p::dom::sqrfree(p);
    if domtype(p) = DOM_LIST then
      return(Factored::create( p,"squarefree" ))
    else
      return(p)
    end_if
  end_if;
  if domtype(p) = DOM_POLY then
    if testargs() then
      t:= op(p,3);
      if domtype(t) = DOM_DOMAIN then
        if t::hasProp <> FAIL then
          if t::hasProp(Cat::FactorialDomain) then
            if t::characteristic <> 0 then
              if t::hasProp(Cat::Field) then
                if not t::hasProp(Dom::IntegerMod) then
                  if not t::hasProp(Dom::GaloisField) then
                    if not t::hasProp(Dom::AlgebraicExtension) then
                      error("not a finite field")
                    elif t::degreeOverPrimeField=UNKNOWN then
                      error("not a finite field")
                    end_if
                  end_if
                end_if
              else
                error("not a field")
              end_if
            end_if
          else
            error("not a factorial domain")
          end_if
        end_if
      else
        if t<>hold(Expr) and not isprime(op(t,1)) then
          error("no prime modulus")
        end_if
      end_if
    end_if;
    ret := recollect(unextend(faclib::sqrfree_poly(p)));
    ret := Factored::create(ret, "squarefree");
  else
    ret := recollect(unextend(faclib::factor_expr(p, table(Adjoin = {}, MaxDegree = 1, Full = FALSE, Domain = Expr), TRUE)));
    ret := Factored::create(ret, "squarefree");
  end_if

end_proc:
