/*

    polylib::subresultant(f, g, x <, i>)
    polylib::subresultant(f, g <, i>)

    f, g - polynomials or polynomial expressions
       x - variable
       i - nonnegative integer

*/


polylib::subresultant:=
proc(f, g, x, i)
  local result, sgn, T, convertToExpressions: DOM_BOOL;
begin
  if args(0) < 2 then
    error("Wrong number of arguments")
  end_if;

  if type(f) <> DOM_POLY then
    if args(0) < 3 then 
      error("Variable missing")
    end_if;  
    
    f:= poly(f, [x]);
    g:= poly(g, [x]);
    if f = FAIL or g = FAIL then 
      error("Cannot convert the input to polynomials")
    end_if;
    convertToExpressions:= TRUE;
    
  elif type(g) <> DOM_POLY then
    g:= poly(g, op(f, 2..3));
    if g = FAIL then
      error("Cannot convert the input to polynomials")
    end_if;  
    if args(0) < 3 then
      x:= op(f, [2, 1])  
    end_if;
    convertToExpressions:= FALSE; // this is a matter of taste !?
    
  elif op(g, 2) <> op(f, 2) or op(g, 3) <> op(f, 3) then   
    error("Input must be polynomials over the same ring in the same indeterminates")
  else
    // both f, g are polynomials
    if args(0) < 3 then
      x:= op(f, [2, 1])  
    end_if;      
      
    convertToExpressions:= FALSE
  end_if;
  
  if nops(op(f, 2)) <> 1 then
    T:= select(op(f, 2), _unequal, x), op(f, 3);
    f:= poly(f, [x]);
    g:= poly(g, [x]);
  else
    T:= FAIL
  end_if;  
    
  if type(x) = DOM_INT then
    i:= x;
    x:= op(f, [2, 1]);
  elif args(0) < 4 then
    i:= FAIL 
  elif type(i) <> DOM_INT then   
    error("Fourth argument must be a non-negative integer")
  elif i < 0 or i > degree(f) then
    error("Fourth argument must be between zero and the maximum of degrees of the polynomials")  
  end_if;  
    
  if degree(g) > degree(f) then 
    // we adopt the convention that in this case, we just interchange f and g
    sgn:= (-1)^(degree(f)*degree(g)); 
    [f, g]:= [g, f];
  else   
    sgn:= 1
  end_if;  


  if type(i) = DOM_INT then
    // return *one* subresultant
    
    // special cases (see the spec)
    if i = degree(f) then
      result:= f
    elif i = degree(f) - 1 and degree(f) > degree(g) then
      result:= g
    elif i > degree(g) then
      // this is only possible if degree(f) - degree(g) >= 2
      result:= poly(0, op(g, 2..3))
    elif i = degree(g) then
      result:= multcoeffs(g, lcoeff(g)^(degree(f) - degree(g) - 1))
    else	
      // stop the Euclidean algorithm somewhere
      result:= polylib::subresultantDucos(f, g, x, i);
      result:= result[i];
    end_if;
    result:= multcoeffs(result, sgn);
    if convertToExpressions then
      result:= expr(result)
    elif T <> FAIL then
      result:= poly(result, T)
    end_if;
          
  else
    
    result:= polylib::subresultantDucos(f, g, x);
    result:= map(result, _mult, sgn);
    if convertToExpressions then
      result:= map(result, expr)
    elif T <> FAIL then
      result:= map(result, poly, T)
    end_if  
  end_if;
  
  result
end_proc:
  

/*
   polylib::subresultantDucos(P, Q, x)

   computes a table of all nonzero subresultants of P and Q

   See Lionel Ducos, Optimizations of the Subresultant Algorithm, 1998, page 8

*/
polylib::subresultantDucos:=
proc(PP: DOM_POLY, QQ: DOM_POLY, x, stopAtDegree = -1: DOM_INT)
  local S, s, d, e, delta, A, B, C, a, c, n, Dp, H, j, se, contP, contQ, P, Q, rat, substitutions;
begin
 
  rat:= rationalize(map([PP, QQ], poly2list), FindRelations = ["_power", "exp"]);
  [P, Q]:= map(op(rat, 1), poly, op(PP, 2..3)); 
  substitutions:= op(rat, 2);

  // Multiply by common denominator first. Note that we otherwise could not carry out exact divisions below, and would be forced to use normal
  contP:= lcm(op(map({coeff(P)}, denom)));
  contQ:= lcm(op(map({coeff(Q)}, denom)));
  Q:= normal(multcoeffs(Q, contQ));
  P:= normal(multcoeffs(P, contP));
  contP:= 1/contP;
  contQ:= 1/contQ;
  
  S:= table(poly(0, [x]));
  S[degree(P)]:= P;
  S[(e:= degree(Q))]:= multcoeffs(Q, lcoeff(Q)^(degree(P) - degree(Q) - 1));
  s:= lcoeff(Q)^(degree(P)-degree(Q));
  A:= Q;
  B:= pdivide(P, -Q, Rem);
  while not iszero(B) and e > stopAtDegree do
    d:= degree(A);
    e:= degree(B);
    assert(d>=e);
    if d = degree(Q) then
      assert(testeq(expr(divide(A, S[d], Rem)), 0, Steps = 0) <> FALSE)
    else
      assert(A = S[d])
    end_if;
    S[d-1]:= B;
    delta:= d-e;
    if delta > 1 then
      // C:= optimized calculation of S_e
      // a simple method would be 
      // C:= multcoeffs(B, (lcoeff(B)/s)^(delta-1));
      // optimized version, due to Lazard     
      n:= delta-1;
      assert(n<>0);
      a:= floor(log(2, n));
      a:= 2^a;
      c:= lcoeff(S[d-1]);
      n:= n-a;
      while a>1 do
        a:= a div 2;
        c:= normal(c^2/ lcoeff(S[d]));
        if n>=a then c:= normal(c*lcoeff(S[d-1])/lcoeff(S[d])); n:= n-a end_if;
      end_while;  
      C:= multcoeffs(S[d-1], c);
      C:= mapcoeffs(C, divide, lcoeff(S[d]), Quo); // really an exact division
      // end of Lazard's optimization
      S[e]:= C;
    else 
      C:= B;
    end_if;  
    if e=0 then 
      break 
    end_if;
    // B:= optimized calc. of S_{e-1}
    // simple version:
    // B:= multcoeffs(pdivide(A, -B, Rem), s^(-delta)/lcoeff(A));
    // optimized version:
    e:= degree(S[d-1]);
    c:= lcoeff(S[d-1]);
    se:= lcoeff(S[e]);
    H:= table();
    for j from 0 to e-1 do 
      H[j]:= poly(se*x^j, [x])
    end_for;  
    H[e]:= poly(se*x^e, [x]) - S[e];
    for j from e+1 to d-1 do           
      H[j]:= poly(x, [x])*H[j-1];
      // divide with remainder: however, divide(H[j], S[d-1], Rem) would not normalize
      H[j]:= H[j] - mapcoeffs(multcoeffs(S[d-1], coeff(H[j], e)), divide, c, Quo) // really an exact division
    end_for;
    Dp:= mapcoeffs(_plus(coeff(A, j)*H[j] $j=0..d-1), divide, lcoeff(A), Quo); // really an exact division
    assert(Dp <> FAIL);
    B:= (-1)^(d-e+1)*mapcoeffs(c*(x*H[d-1]+Dp) - coeff(H[d-1], e-1)*S[d-1], divide, lcoeff(S[d]), Quo); // really an exact division
    assert(B <> FAIL);
    // end of optimized version
    A:= C;
    s:= lcoeff(A)
  end_while;
  
  S[degree(P)]:= PP;
  if degree(P) > degree(Q) then
    S[degree(P)-1]:= QQ
  end_if;
  
  // multiply by content again
  for s in S do 
    d:= op(s, 1);
    if d < degree(Q) then
      S[d]:= normal(S[d] * contP^(degree(Q) - d) * contQ^(degree(P)-d))
    end_if  
  end_for;
  subs(S, substitutions, EvalChanges)
end_proc:


// end of file